function FF = statistics_of_features(fileName, win, step)

% This function computes the average and std values for the following audio
% features:
% - energy entropy
% - short time energy
% - spectral rolloff
% - spectral centroid
% - spectral flux
% 
% ARGUMENTS:
% fileName: the name of the .wav file in which the signal is stored
% win: the processing window (in seconds)
% step: the processing step (in seconds)
%
% RETURN VALUE:
% F: a 12x1 array containing the 12 feature statistics
%

[x, fs] = wavread([fileName '.wav']);



EE = Energy_Entropy_Block(x, win*fs, step*fs, 20);
E = ShortTimeEnergy(x, win*fs, step*fs);
Z = zcr(x, win*fs, step*fs, fs);
R = SpectralRollOff(x, win*fs, step*fs, 0.80, fs);
C = SpectralCentroid(x, win*fs, step*fs, fs);
F = SpectralFlux(x, win*fs, step*fs, fs);
PF = PeakFeature(x, win*fs, step*fs, 20, step, 0); 

% % Define variables
% Tw = win*fs;            % analysis frame duration (ms)
% Ts = step*fs;           % analysis frame shift (ms)
% alpha = 0.97;           % preemphasis coefficient
% M = 20;                 % number of filterbank channels 
% CC = 12;                 % number of cepstral coefficients
% L = 22;                 % cepstral sine lifter parameter
% LF = 40;               % lower frequency limit (Hz)
% HF = 500;              % upper frequency limit (Hz)
% 
% [ MFCCs, FBEs, frames ] = ...
%                 mfcc( x, fs, Tw, Ts, alpha, @hamming, [LF HF], M, CC+1, L );
            
            

FF = [];
FF = [FF statistic(EE, 1, length(EE), 'std')];
FF = [FF statistic(EE, 1, length(EE), 'stdbymean')];
FF = [FF statistic(Z, 1, length(Z), 'std')];
FF = [FF statistic(Z, 1, length(Z), 'stdbymean')];
FF = [FF statistic(R, 1, length(R), 'std')];
FF = [FF statistic(R, 1, length(R), 'stdbymean')];
FF = [FF statistic(C, 1, length(C), 'std')];
FF = [FF statistic(C, 1, length(C), 'stdbymean')];
FF = [FF statistic(F, 1, length(F), 'std')];
FF = [FF statistic(F, 1, length(F), 'stdbymean')];
FF = [FF statistic(E, 1, length(E), 'std')];
FF = [FF statistic(E, 1, length(E), 'stdbymean')];
FF = [FF statistic(PF, 1, length(PF), 'std')];
FF = [FF statistic(PF, 1, length(PF), 'stdbymean')];
FF = [FF statistic(PF, 1, length(PF), 'mean')];
FF = [FF statistic(PF, 1, length(PF), 'median')];

 FeaVec2 = PLP_FeatureExtraction(x, fs);
 FF = [FF FeaVec2'];




function [PF] = PeakFeature(f, winLength, winStep, numofShortBlocks, stStep, show_flag)

        
        Features = Energy_Entropy_Block(f, winLength, winStep, numofShortBlocks);
        
        FSize = length(Features);
        minH = mean(Features)*1.5;%max(Features(2,:))/3; 
        
        [pks,locs] = findpeaks(Features,'MINPEAKHEIGHT',minH); 
        LL = length(locs);
        if(show_flag)
           figure(1);
           plot((1:1:FSize)*stStep, Features,'r-');
           hold on           
           for ll = 1:LL
              plot(locs(ll)*stStep,pks(ll),'g.','MarkerSize',20);
           end
        end
        
        sumD = 0; 
        for ll = 1:LL-1 
            sumD = sumD + locs(ll+1)-locs(ll); 
        end
        minD = sumD / (LL-1); 
        [pks,locs] = findpeaks(Features,'MINPEAKHEIGHT',minH,'MINPEAKDISTANCE',round(minD)); 
        strN = 1:length(locs);   
        LL = length(locs);
        strS(1) = 0; 
        for ll = 2:LL 
            strS(ll) = (locs(ll)-locs(ll-1))*stStep;
        end
        strS(1) = mean(strS(2:LL)); 
        
        if(show_flag)
           figure(2);
           plot((1:1:FSize)*stStep, Features,'r-');
           hold on 
           for ll = 1:LL
              plot(locs(ll)*stStep,pks(ll),'g.','MarkerSize',20);
              text(locs(ll)*stStep,pks(ll),sprintf('#%d,%.1fs', strN(ll),strS(ll)));
           end
        end
        
        PF = strS(2:LL);
        



function [Entropy] = Energy_Entropy_Block(f,winLength,winStep,numOfShortBlocks)

f = f / max(abs(f));
Eol = sum(f.^2);
L = length(f);

if (winLength==0)
    winLength = floor(L);
    winStep = floor(L);
end


numOfBlocks = (L-winLength)/winStep + 1;
curPos = 1;
for (i=1:numOfBlocks)
    curBlock = f(curPos:curPos+winLength-1);
    curBlock = curBlock + 1e-9;
    for (j=1:numOfShortBlocks)        
        s(j) = sum(curBlock((j-1)*(winLength/numOfShortBlocks)+1:j*(winLength/numOfShortBlocks)).^2)/Eol;
    end
    Entropy(i) = -sum(s.*log2(s));
    curPos = curPos + winStep;
end

function E = ShortTimeEnergy(signal, windowLength,step);
signal = signal / max(max(signal));
curPos = 1;
L = length(signal);
numOfFrames = floor((L-windowLength)/step) + 1;
%H = hamming(windowLength);
E = zeros(numOfFrames,1);
for (i=1:numOfFrames)
    window = (signal(curPos:curPos+windowLength-1));
    window = window + 1e-9;
    E(i) = (1/(windowLength)) * sum(abs(window.^2));
    curPos = curPos + step;
end


function Z = zcr(signal,windowLength, step, fs);
signal = signal / max(abs(signal));
curPos = 1;
L = length(signal);
numOfFrames = floor((L-windowLength)/step) + 1;
%H = hamming(windowLength);
Z = zeros(numOfFrames,1);
for (i=1:numOfFrames)
    window = (signal(curPos:curPos+windowLength-1));  
    window = window + 1e-9;
    window2 = zeros(size(window));
    window2(2:end) = window(1:end-1);
    Z(i) = (1/(2*windowLength)) * sum(abs(sign(window)-sign(window2)));
    curPos = curPos + step;
end

function mC = SpectralRollOff(signal,windowLength, step, c, fs)
signal = signal / max(abs(signal));
curPos = 1;
L = length(signal);
numOfFrames = (L-windowLength)/step + 1;
H = hamming(windowLength);
m = [0:windowLength-1]';
for (i=1:numOfFrames)
    window = (signal(curPos:curPos+windowLength-1));  
    window = window + 1e-9;
    FFT = (abs(fft(window,512)));
    FFT = FFT(1:255);
    totalEnergy = sum(FFT);
    curEnergy = 0.0;
    countFFT = 1;
    while ((curEnergy<=c*totalEnergy) && (countFFT<=255))
        curEnergy = curEnergy + FFT(countFFT);
        countFFT = countFFT + 1;
    end
    mC(i) = ((countFFT-1))/(fs/2);
    curPos = curPos + step;
end

function C = SpectralCentroid(signal,windowLength, step, fs)
signal = signal / max(abs(signal));
curPos = 1;
L = length(signal);
numOfFrames = floor((L-windowLength)/step) + 1;
H = hamming(windowLength);
m = ((fs/(2*windowLength))*[1:windowLength])';
C = zeros(numOfFrames,1);
for (i=1:numOfFrames)
    window = (signal(curPos:curPos+windowLength-1));
    window = window + 1e-9;
    window = H.*window;    
    FFT = (abs(fft(window,2*windowLength)));
    FFT = FFT(1:windowLength);  
    FFT = FFT / max(FFT);
    C(i) = sum(m.*FFT)/sum(FFT);
    if (sum(window.^2)<0.010)
        C(i) = 0.0;
    end
    curPos = curPos + step;
end
C = C / (fs/2);

function F = SpectralFlux(signal,windowLength, step, fs)
signal = signal / max(abs(signal));
curPos = 1;
L = length(signal);
numOfFrames = floor((L-windowLength)/step) + 1;
H = hamming(windowLength);
m = [0:windowLength-1]';
F = zeros(numOfFrames,1);
for (i=1:numOfFrames)
    window = (signal(curPos:curPos+windowLength-1));   
    window = window + 1e-9;
    window = H.*window;  
    FFT = (abs(fft(window,2*windowLength)));
    FFT = FFT(1:windowLength);  
    FFT = FFT / max(FFT);
    if (i>1)
        F(i) = sum((FFT-FFTprev).^2);
    else
        F(i) = 0;
    end
    curPos = curPos + step;
    FFTprev = FFT;
end



