function [H,h,h_norm] = petropy(x,n,tau,method,accu)
% The petropy function calculates the permutation entropy of data series.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Permutation Entropy
%
% H = PETROPY(X,N,TAU,METHOD,ACCU) computes the permutation entropy H of 
% a scalar vector X, using permutation order N, time lags from TAU and 
% METHOD to treat equal values. The ACCU parameter describes the accuracy 
% of the values in X by the number of decimal places.
%
% x      - data vector (Mx1 or 1xM)
% n      - permutation order
% tau    - time lag scalar OR time lag vector (length = n-1)
% method - method how to treat equal values
%   'noise' - add small noise
%   'same'  - allow same rank for equal values
%   'order' - consider order of appearance (first occurence --> lower rank)
% accu   - maximum number of decimal places in x 
%         (only used for method 'noise')
%
% References:
%
% Bandt, C.; Pompe, B. Permutation Entropy: A Natural Complexity 
% Measure for  Time Series. Phys. Rev. Lett. 88 (2002) 17, 174102
%
% Riedl, M.; Mller, A.; Wessel, N.: Practical considerations of 
% permutation entropy. The European Physical Journal Special Topics 
% 222 (2013) 2, 249262

% Modified by Antonio Ravelo. Feb_2015 
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% example:

 %           [H_PE31,PE31,PE31_norm] =petropy(senal_rr,3,1);%1
 %           [H_PE41,PE41,PE41_norm]=petropy(senal_rr,4,1);%2
 %           [H_PE51,PE51,PE51_norm]=petropy(senal_rr,5,1);%3
 %           [H_PE61,PE61,PE61_norm]=petropy(senal_rr,6,1);%4
 %           [H_PE71,PE71,PE71_norm]=petropy(senal_rr,7,1);%5
 %           [H_PE32,PE32,PE32_norm]=petropy(senal_rr,3,2);%6
 %           [H_PE42,PE42,PE42_norm]=petropy(senal_rr,4,2);%7
 %           [H_PE52,PE52,PE52_norm]=petropy(senal_rr,5,2);%8
 %           [H_PE62,PE62,PE62_norm]=petropy(senal_rr,6,2);%9
 %           [H_PE72,PE72,PE72_norm]=petropy(senal_rr,7,2);%10
 %           [H_PE33,PE33,PE33_norm]=petropy(senal_rr,3,3);%11
 %           [H_PE43,PE43,PE43_norm]=petropy(senal_rr,4,3);%12
 %           [H_PE53,PE53,PE53_norm]=petropy(senal_rr,5,3);%13
 %           [H_PE63,PE63,PE63_norm]=petropy(senal_rr,6,3);%14
 %           [H_PE73,PE73,PE73_norm]=petropy(senal_rr,7,3);%15
 %           [H_PE34,PE34,PE34_norm]=petropy(senal_rr,3,4);%16
 %           [H_PE44,PE44,PE44_norm]=petropy(senal_rr,4,4);%17
 %           [H_PE54,PE54,PE54_norm]=petropy(senal_rr,5,4);%18
 %           [H_PE64,PE64,PE64_norm]=petropy(senal_rr,6,4);%19
 %           [H_PE74,PE74,PE74_norm]=petropy(senal_rr,7,4);%20


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

if nargin < 5, accu = 4; end
if nargin < 4, method = 'order'; end

x = x(:);
M = length(x);
equal = false;

if n*log10(n)>15, error('permutation dimension too high'); end
if (length(tau)) > 1 && (length(tau) ~= n-1), error('time lag vector has to have n-1 entries'); end
%if ((n-1)*min(tau) >=M) || max(tau) >= M, error('too few data points for desired dimension and lags'); end
if ((n-1)*min(tau) >=M) || max(tau) >= M, H=0; h=0; h_norm=0; disp('too few data points for desired dimension and lags'); return, end


switch lower(method)
    case {'noise'}
        %disp('Method: add small noise')
        x = x + rand(M,1)*10^(-accu-1);
    case 'equal'
        %disp('Method: allow equal ranks')
        equal = true;
    case 'order'
        %disp('Method: consider order of occurrence')
    otherwise
        error('unknown method')
end

if length(tau) > 1
    tau = reshape(tau,length(tau),1);
    tau = sort([0;tau]);
    % build n x (M-tau(n))-matrix from shifted values in x
    shift_mat = zeros(n,M-tau(n));
    for ii=1:n
        shift_mat(ii,:) = x(tau(ii)+1:M-tau(n)+tau(ii));
    end
else
    % vectorized
    shift_mat_ind = reshape(0:tau:(n-1)*tau,[],1) * ones(1,M-(n-1)*tau) +...
        ones(n, 1) * reshape(1:(M-(n-1)*tau),1,[]);
    shift_mat = x(shift_mat_ind);
end

if equal
    % allow equal values the same index
    ind_mat = zeros(size(shift_mat));
    for ii=1:size(ind_mat,2)
        [basura,basura2,ind_mat(:,ii)]=unique(shift_mat(:,ii),'first');
    end
else
    % sort matrix along rows to build rank orders, equal values retain
    % order of appearance
    [basura3, sort_ind_mat] = sort(shift_mat,1);
    ind_mat = zeros(size(sort_ind_mat));
    for ii=1:size(ind_mat,2)
        ind_mat(sort_ind_mat(:,ii),ii) = 1:n;
    end
end
% assign unique number to each pattern (base-n number system)
ind_vec = n.^(0:n-1) * (ind_mat-1);

% find first occurence of unique values in 'ind_vec' and use
% difference to determine length of sequence of the same numbers; e.g.
% sort_ind_vec = [21 21 11 19 11], unique_values = [11 19 21],
% ia = [1 3 4]: 11 occurs on places #1 and #2, 19 on #3 and 21 on #4 and #5
[basura5,ia,basura4] = unique(sort(ind_vec), 'first');

if size(ia,2)==1, ia=ia.'; end
permpat_num = diff([ia (length(ind_vec)+1)]);
permpat_num = permpat_num/sum(permpat_num);

% compute permutation entropy
H = -sum(permpat_num .* log2(permpat_num));
h=(1/(n-1))*H;
h_norm=H/(log2(factorial(n)));

