#!/usr/bin/python2

'''
   Copyright 2016 Christian Thomae

   Licensed under the Apache License, Version 2.0 (the "License");
   you may not use this file except in compliance with the License.
   You may obtain a copy of the License at

       http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.
'''

from __future__ import print_function

import os
import sys

# Keras <www.keras.io>
from keras.layers import Input, Convolution1D, MaxPooling1D, GRU, BatchNormalization, GaussianNoise, Dropout, Dense, merge, TimeDistributed, Lambda
from keras.models import Model
from keras.regularizers import l2
from keras import backend as K

# SciPy
from scipy.io import wavfile

# NumPy
import numpy as np

def create_model():
    batch_size = 1

    raw_input = Input(batch_shape=(batch_size, None, 1))

    rinput = GaussianNoise(1e-6)(raw_input) # denoise -> data augmentation
    rinput = BatchNormalization()(rinput)

    c4 = Convolution1D(16, 16, border_mode='same', W_regularizer=l2(0.01), activation='relu')(rinput)
    c8 = Convolution1D(16, 32, border_mode='same', W_regularizer=l2(0.01), activation='relu')(rinput)
    c16 = Convolution1D(16, 64, border_mode='same', W_regularizer=l2(0.01), activation='relu')(rinput)
    c32 = Convolution1D(16, 128, border_mode='same', W_regularizer=l2(0.01), activation='relu')(rinput)

    concat = merge([c4, c8, c16, c32], mode='concat')
    filtered = MaxPooling1D(pool_length=256, stride=128)(concat)
    filtered = BatchNormalization()(filtered)

    e1 = GRU(16, return_sequences=True, W_regularizer=l2(0.01), U_regularizer=l2(0.01))(filtered)
    e2 = GRU(16, return_sequences=True, go_backwards=True, W_regularizer=l2(0.01), U_regularizer=l2(0.01))(filtered)
    encoded = merge([e1, e2], mode='concat')
    encoded = BatchNormalization()(encoded)

    e1 = GRU(16, return_sequences=True, W_regularizer=l2(0.01), U_regularizer=l2(0.01))(encoded)
    e2 = GRU(16, return_sequences=True, go_backwards=True, W_regularizer=l2(0.01), U_regularizer=l2(0.01))(encoded)
    encoded = merge([e1, e2], mode='concat')
    encoded = BatchNormalization()(encoded)

    att_input = Input(batch_shape=(batch_size, 32))
    att = Dense(64, W_regularizer=l2(0.01))(att_input)
    att = Dense(8, W_regularizer=l2(0.01))(att)
    att = Dense(1, W_regularizer=l2(0.01))(att)
    attention = Model(input=att_input, output=att)

    attention = TimeDistributed(attention)(encoded)

    def TimeSoftMax(x):
        e = K.exp(x - K.max(x, axis=1, keepdims=True))
        s = K.sum(e, axis=1, keepdims=True)
        return e / s

    attention = Lambda(TimeSoftMax)(attention)

    def WeightedAverage(x):
        x = K.sum(x[0]*K.repeat_elements(x[1], 32, 2), axis=1)
        return x

    encoded = merge([encoded, attention], mode=WeightedAverage, output_shape=lambda x: (batch_size, 32))
    encoded = Dropout(0.5)(encoded)

    # decoder
    decoded = Dense(64, W_regularizer=l2(0.01))(encoded)
    decoded = Dropout(0.5)(decoded)
    decoded = Dense(8, W_regularizer=l2(0.01))(decoded)
    result = Dense(1, activation='tanh', W_regularizer=l2(0.01))(decoded)

    model = Model(input=[raw_input], output=result)
    model.compile('sgd', 'mse')
    model.load_weights('./weights_k1sclass.h5py')

    return model

def main():
    # create model
    model = create_model()

    # random
    data = np.random.randn(1, 10000, 1)

    # prediction
    c = 1 if np.mean(model.predict(data, batch_size=3, verbose=0)) > 0 else -1

    print(np.int8(c)) # correct textual representation
    # 0 == unk;  1 == abnorm; -1 == norm

if __name__ == '__main__':
    main()

