function y_hat = classifyProcessedAudio(FFTs)
%
% Classifies a pre-processed audio file and returns a label. The
% classification depends on pre-computed data. The data required are found
% in Sparse_Coding_Dictionaries.mat and SVM_Classifiers.mat.
%
% Sparse_Coding_Dictionaries.mat has dictionaries that were learned on
% audio files that were pre-processed by the function 'preprocessAudio.m'.
% There are 5 dictionaries: one for each sub-cycle of a heartbeat (S1,
% systole, S2, and diastole), and one for a complete cycle.
%
% SVM_Classifiers.mat contains support vector machines that were trained
% using the coefficients of the processed audio files in the sparse domain.
% A support vector machine was trained on each cycle and sub-cycle using
% MATLAB's statistical toolbox function 'fitcsvm'. In addition, there is a
% sixth SVM that combines the results from the cycle and sub-cycle SVMs to
% predict a single label.
%
% This function takes the pre-processed audio, computes sparse coefficient
% vectors using the dictionaries, and produces a label based on the outcome
% of the multiple SVMs.
%
% INPUTS:
% FFTs: A struct containing the processed information of an audio file. The
%       function 'preprocessAudio.m' will generate the appropriate struct
%       to be used by this function.
%
% OUTPUTS:
% y_hat: integer value where
%                     1 = abnormal recording
%                    -1 = normal recording
%                     0 = unsure (too noisy)
%
% Version 1.0
%
%
% Written by: Brad Whitaker, April 7 2016
%             b.whitaker@gatech.edu
%
%
% Last Modified: April 15, 2016

    %% Load Necessary Data
    load SparseCodingDictionaries;
    load SVMs;
    train_tol = 0.1;

    %% Calculate Sparse Coefficients of each segmented portion

    dict_opts_S1.in_iter = size(FFTs.S1,2);
    dict_opts_S1.tol = train_tol;
    coef_S1 = gen_multi_infer(DICT_S1, FFTs.S1, @l1ls_wrapper, dict_opts_S1);
    coef_S1 = mean(coef_S1,2);

    dict_opts_systole.in_iter = size(FFTs.systole,2);
    dict_opts_systole.tol = train_tol;
    coef_systole = gen_multi_infer(DICT_systole, FFTs.systole, @l1ls_wrapper, dict_opts_systole);
    coef_systole = mean(coef_systole,2);

    dict_opts_S2.in_iter = size(FFTs.S2,2);
    dict_opts_S2.tol = train_tol;
    coef_S2 = gen_multi_infer(DICT_S2, FFTs.S2, @l1ls_wrapper, dict_opts_S2);
    coef_S2 = mean(coef_S2,2);

    dict_opts_diastole.in_iter = size(FFTs.diastole,2);
    dict_opts_diastole.tol = train_tol;
    coef_diastole = gen_multi_infer(DICT_diastole, FFTs.diastole, @l1ls_wrapper, dict_opts_diastole);
    coef_diastole = mean(coef_diastole,2);

    dict_opts_cycle.in_iter = size(FFTs.cycle,2);
    dict_opts_cycle.tol = train_tol;
    coef_cycle = gen_multi_infer(DICT_cycle, FFTs.cycle, @l1ls_wrapper, dict_opts_cycle);
    coef_cycle = mean(coef_cycle,2);

    %% Classify Each Segment
    
    subcycle_scores = zeros(5,1);

    [~,~,subcycle_scores(1)] = svmpredict(ones(1,size(coef_S1,2))',coef_S1',SVM_S1,'-q');
    [~,~,subcycle_scores(2)] = svmpredict(ones(1,size(coef_systole,2))',coef_systole',SVM_systole,'-q');
    [~,~,subcycle_scores(3)] = svmpredict(ones(1,size(coef_S2,2))',coef_S2',SVM_S2,'-q');
    [~,~,subcycle_scores(4)] = svmpredict(ones(1,size(coef_diastole,2))',coef_diastole',SVM_diastole,'-q');
    [~,~,subcycle_scores(5)] = svmpredict(ones(1,size(coef_cycle,2))',coef_cycle',SVM_cycle,'-q');

    %% Label

    [y_hat,~,~] = svmpredict(1,subcycle_scores',SVM_combined,'-q');

end