 function [alarmResult]=challenge(recordName,alarm_type)
% Inputs:
%   recordName
%       String specifying the record name to process
%   alarmType
%       String specifying the alarm type. Alarm types are:
%             Asystole, Bradycardia, Tachycardia,
%             Ventricular_Tachycardia, Ventricular_Flutter_Fib
% Outputs:
%  alarmResult
%       Integer value where 0 = false alarm and 1 is a true alarm. 
%
% Version 0.51
% Written by Qiao Li, November 10, 2014.
% Modified: Ikaro Silva February 11, 2015
% Last Modified: Hai-Yan Yu, April 5, 2015
%
% %Example using training data- 
% alarmResult=challenge('./challenge/set-p/100','Asystole')
% alarmResult=challenge('../a796s','Asystole')

% Name of file containing answers
answers = 'answers.txt';

%Get all ECG, blood pressure and photoplethysmogram signals
[~,signal,Fs,siginfo]=rdmat(recordName);
alarmResult=1;
description=squeeze(struct2cell(siginfo));
description=description(4,:);

% Resample signal to 125Hz
Fs=Fs(1);
if Fs~=125
   signal = resample(signal,125,Fs);
   Fs=125;
end

%Run WABP on the record, which by default will analyze the first ABP, ART, or BP signal
N=[];
N0=[];
abp_ind=get_index(description,'ABP');          
%get_index() is defined at the end of this file
ann_abp=[];
features=[];
BEATQ=[];
R=[];
if(~isempty(abp_ind))
  ann_abp=wabp(signal(:,abp_ind),0,1);
   if length(ann_abp)>=3 
        [features] = abpfeature(signal(:,abp_ind),ann_abp);
        [BEATQ R] = jSQI(features, ann_abp, signal(:,abp_ind));
   end
end

%Run WABP on the record of 'PLETH' to analyze photoplethysmogram signal
ppg_ind=get_index(description,'PLETH');
ann_ppg=[];
if (~isempty(ppg_ind))
    y=quantile(signal(:,ppg_ind),[0.05,0.5,0.95]);
    ann_ppg=wabp(signal(:,ppg_ind),0,(y(3)-y(1))/120);
    % Analyze the signal quality index of PPG 
    if ~isempty(ann_ppg)
        [psqi]=ppgSQI(signal(:,ppg_ind),ann_ppg);
    end
end

%Run WABP on the record of RESP to analyze RESP signal
resp_ind=get_index(description,'RESP');
ann_resp=[];
respsqi=[];
if (~isempty(resp_ind))
    y=quantile(signal(:,resp_ind),[0.05,0.5,0.95]);
    ann_resp=wabp(signal(:,resp_ind),1600,(y(3)-y(1))*120);
    if ~isempty(ann_resp)
        [respsqi]=kSQI(signal(:,resp_ind),ann_resp);
    end
end

%Run WABP on the record of MCL to analyze photoplethysmogram signal
mcl_ind=get_index(description,'MCL');
ann_mcl=[];
if (~isempty(mcl_ind))
    y=quantile(signal(:,mcl_ind),[0.05,0.5,0.95]);
    ann_mcl=wabp(signal(:,mcl_ind),0,(y(3)-y(1))/120);
    if ~isempty(ann_mcl)
        [mclsqi]=kSQI(signal(:,mcl_ind),ann_mcl);
    end
end

%Run WABP on the record of ECG'II' to analyze photoplethysmogram signal
II_ind=get_index(description,'II'); % Not III
III_ind=get_index(description,'III');
II_ind=setdiff(II_ind,III_ind);
ann_II=[];
if (~isempty(II_ind))
    z=quantile(signal(:,II_ind),[0.05,0.5,0.95]);
   s=length(signal(:,II_ind));
   ann_II=wabp(signal(1:s,II_ind),1600,(z(3)-z(1))/120);
    if ~isempty(ann_II) && length(ann_II)<3
             IIsqi =0.1;       
    elseif ~isempty(ann_II) && length(ann_II)>=3
            [IIsqi]=ppgSQI(signal(:,II_ind),ann_II);
    end
end

%Run WABP on the record of ECG'V' to analyze photoplethysmogram signal
V_ind=get_index(description,'V');
aVF_ind=get_index(description,'aVF');
V_ind=setdiff(V_ind,aVF_ind);
ann_V=[];
if (~isempty(V_ind))
    y=quantile(signal(:,V_ind),[0.05,0.5,0.95]);
    ann_V=wabp(signal(:,V_ind),0,(y(3)-y(1))/120);
   if ~isempty(ann_V) && length(ann_V)<3
             Vsqi =0.1;      
    elseif ~isempty(ann_V) && length(ann_V)>=3
            [Vsqi]=ppgSQI(signal(:,V_ind),ann_V);
    end
end

%Run WABP on the record of ECG'III' to analyze photoplethysmogram signal
III_ind=get_index(description,'III');
ann_III=[];
if (~isempty(III_ind))
    y=quantile(signal(:,III_ind),[0.05,0.5,0.95]);
    ann_III=wabp(signal(:,III_ind),0,(y(3)-y(1))/120);
     if ~isempty(ann_III) && length(ann_III)<3
             IIIsqi =0.1;     
    elseif ~isempty(ann_III) && length(ann_III)>=3
            [IIIsqi]=ppgSQI(signal(:,III_ind),ann_III);
     end   
end

%Make decisions
N_d=Fs*5*60; % alarm position
N0_d=N_d-Fs*16+1; % 16s before the alarm

% select the beats in the segment
n_abp_beats=intersect(find(ann_abp>=N0_d),find(ann_abp<=N_d));
n_ppg_beats=intersect(find(ann_ppg>=N0_d),find(ann_ppg<=N_d));
%
n_resp_beats=intersect(find(ann_resp>=N0_d),find(ann_resp<=N_d));
n_V_beats=intersect(find(ann_V>=N0_d),find(ann_V<=N_d));
n_III_beats=intersect(find(ann_III>=N0_d),find(ann_III<=N_d));
n_II_beats=intersect(find(ann_II>=N0_d),find(ann_II<=N_d));
n_mcl_beats=intersect(find(ann_mcl>=N0_d),find(ann_mcl<=N_d));

hr_max_abp=NaN;
max_rr_abp=NaN;
hr_max_ppg=NaN;
max_rr_ppg=NaN;
%
 hr_max_resp=NaN;
 max_rr_resp=NaN;
hr_max_II=NaN;
max_rr_II=NaN;
hr_max_V=NaN;
max_rr_V=NaN;
hr_max_III=NaN;
max_rr_III=NaN;
hr_max_mcl=NaN;
max_rr_mcl=NaN;

% calculate the heart rate
if length(n_abp_beats)>=2
    hr_max_abp=60*Fs/min(diff(ann_abp(n_abp_beats)));
    max_rr_abp=max(diff(ann_abp(n_abp_beats)))/Fs;
end
if length(n_ppg_beats)>=2
    hr_max_ppg=60*Fs/min(diff(ann_ppg(n_ppg_beats)));
    max_rr_ppg=max(diff(ann_ppg(n_ppg_beats)))/Fs;
end

if length(n_resp_beats)>=2
    hr_max_resp=60*Fs/min(diff(ann_resp(n_resp_beats)));
    max_rr_resp=max(diff(ann_resp(n_resp_beats)))/Fs;
end


if length(n_II_beats)>=2
    hr_max_II=60*Fs/min(diff(ann_II(n_II_beats)));
    max_rr_II=max(diff(ann_II(n_II_beats)))/Fs;
end

if length(n_V_beats)>=2
    hr_max_V=60*Fs/min(diff(ann_V(n_V_beats)));
    max_rr_V=max(diff(ann_V(n_V_beats)))/Fs;
end

if length(n_III_beats)>=2
    hr_max_III=60*Fs/min(diff(ann_III(n_III_beats)));
    max_rr_III=max(diff(ann_III(n_III_beats)))/Fs;
end

if length(n_mcl_beats)>=2
    hr_max_mcl=60*Fs/min(diff(ann_mcl(n_mcl_beats)));
    max_rr_mcl=max(diff(ann_mcl(n_mcl_beats)))/Fs;
end

% calculate low heart rate of 5 consecutive beats for Bradycardia
low_hr_abp=NaN;
low_hr_ppg=NaN;
%
 low_hr_resp=NaN;
 low_hr_V=NaN;
 low_hr_III=NaN;
low_hr_II=NaN;
 low_hr_mcl=NaN;

if length(n_abp_beats>=5)
    for i=1:length(n_abp_beats)-4
        low_hr_abp(i)=60*Fs/((ann_abp(n_abp_beats(i+4))-ann_abp(n_abp_beats(i)))/4);
    end
end
low_hr_abp=min(low_hr_abp);

if length(n_ppg_beats>=5)
    for i=1:length(n_ppg_beats)-4
        low_hr_ppg(i)=60*Fs/((ann_ppg(n_ppg_beats(i+4))-ann_ppg(n_ppg_beats(i)))/4);
    end
end
low_hr_ppg=min(low_hr_ppg);

if length(n_resp_beats>=5)
    for i=1:length(n_resp_beats)-4
        low_hr_resp(i)=60*Fs/((ann_resp(n_resp_beats(i+4))-ann_resp(n_resp_beats(i)))/4);
    end
end
low_hr_resp=min(low_hr_resp);


if length(n_II_beats>=5)
    for i=1:length(n_II_beats)-4
        low_hr_II(i)=60*Fs/((ann_II(n_II_beats(i+4))-ann_II(n_II_beats(i)))/4);
    end
end
low_hr_II=min(low_hr_II);

if length(n_V_beats>=5)
    for i=1:length(n_V_beats)-4
        low_hr_V(i)=60*Fs/((ann_V(n_V_beats(i+4))-ann_V(n_V_beats(i)))/4);
    end
end
low_hr_V=min(low_hr_V);

if length(n_III_beats>=5)
    for i=1:length(n_III_beats)-4
        low_hr_III(i)=60*Fs/((ann_III(n_III_beats(i+4))-ann_III(n_III_beats(i)))/4);
    end
end
low_hr_III=min(low_hr_III);

if length(n_mcl_beats>=5)
    for i=1:length(n_mcl_beats)-4
        low_hr_mcl(i)=60*Fs/((ann_mcl(n_mcl_beats(i+4))-ann_mcl(n_mcl_beats(i)))/4);
    end
end
low_hr_mcl=min(low_hr_mcl);


% calculate high heart reate of 17 consecutive beats for Tachycardia
high_hr_abp=NaN;
high_hr_ppg=NaN;
%
high_hr_resp=NaN;
 high_hr_II=NaN;
 high_hr_V=NaN;
 high_hr_III=NaN;
 high_hr_mcl=NaN;


if length(n_abp_beats>=17)
    for i=1:length(n_abp_beats)-16
        high_hr_abp(i)=60*Fs/((ann_abp(n_abp_beats(i+16))-ann_abp(n_abp_beats(i)))/16);
    end
end
high_hr_abp=max(high_hr_abp);

if length(n_ppg_beats>=17)
    for i=1:length(n_ppg_beats)-16
        high_hr_ppg(i)=60*Fs/((ann_ppg(n_ppg_beats(i+16))-ann_ppg(n_ppg_beats(i)))/16);
    end
end
high_hr_ppg=max(high_hr_ppg);

if length(n_resp_beats>=17)
    for i=1:length(n_resp_beats)-16
        high_hr_resp(i)=60*Fs/((ann_resp(n_resp_beats(i+16))-ann_resp(n_resp_beats(i)))/16);
    end
end
high_hr_resp=max(high_hr_resp);

if length(n_II_beats>=17)
    for i=1:length(n_II_beats)-16
        high_hr_II(i)=60*Fs/((ann_II(n_II_beats(i+16))-ann_II(n_II_beats(i)))/16);
    end
end
high_hr_II=max(high_hr_II);

if length(n_V_beats>=17)
    for i=1:length(n_V_beats)-16
        high_hr_V(i)=60*Fs/((ann_V(n_V_beats(i+16))-ann_V(n_V_beats(i)))/16);
    end
end
high_hr_V=max(high_hr_V);

if length(n_III_beats>=17)
    for i=1:length(n_III_beats)-16
        high_hr_III(i)=60*Fs/((ann_III(n_III_beats(i+16))-ann_III(n_III_beats(i)))/16);
    end
end
high_hr_III=max(high_hr_III);

if length(n_mcl_beats>=17)
    for i=1:length(n_mcl_beats)-16
        high_hr_mcl(i)=60*Fs/((ann_mcl(n_mcl_beats(i+16))-ann_mcl(n_mcl_beats(i)))/16);
    end
end
high_hr_mcl=max(high_hr_mcl);


% calculate the signal quality index
if ~isempty(ann_abp)
    abpsqi=1-sum(sum(BEATQ(intersect(n_abp_beats,1:length(BEATQ)),:)))...
        /numel(BEATQ(intersect(n_abp_beats,1:length(BEATQ)),:));
else
    abpsqi=0;
end

if ~isempty(ann_ppg)
    ppgsqi=mean(psqi(intersect(n_ppg_beats,1:length(psqi))));
else
    ppgsqi=0;
end

if ~isempty(ann_resp)
    RESPsqi=mean(respsqi(intersect(n_resp_beats,1:length(respsqi))));
else
    RESPsqi=0;
end


if ~isempty(ann_II)
    ecgiisqi=mean(IIsqi(intersect(n_II_beats,1:length(IIsqi))));
else
    ecgiisqi=0;
end

if ~isempty(ann_V)
    ecg5sqi=mean(Vsqi(intersect(n_V_beats,1:length(Vsqi))));
else
    ecg5sqi=0;
end

if ~isempty(ann_III)
    ecg3sqi=mean(IIIsqi(intersect(n_III_beats,1:length(IIIsqi))));
else
    ecg3sqi=0;
end

if ~isempty(ann_mcl)
    MCLsqi=mean(mclsqi(intersect(n_mcl_beats,1:length(mclsqi))));
else
    MCLsqi=0;
end


% SQI threshold
sqi_th = 0.85;

% Alarm threshold (seconds)
ASY_th = 4;
BRA_th = 40;
TAC_th = 140;
VTA_th = 100;
VFB_th = 250;
%
tolerance = 5; % tolerance = 5 bmp
switch alarm_type
    case 'Asystole'
        % if the signal quality is good enough and the maximum RR interval
        % is less than the Asystole threshold, set the alarm as 'F'
        if ((abpsqi>=sqi_th && max_rr_abp<ASY_th) ...
           || (ppgsqi>=sqi_th && max_rr_ppg<ASY_th) ...  
           ||(RESPsqi>=sqi_th && max_rr_resp<ASY_th)...
            || (MCLsqi>=sqi_th && max_rr_mcl<ASY_th)...
           ||(ecgiisqi>=sqi_th && max_rr_II<ASY_th)...          
            || (ecg5sqi>=sqi_th && max_rr_V<ASY_th)... 
            || (ecg3sqi>=sqi_th && max_rr_III<ASY_th) )
            alarmResult=0;
        end
    case 'Bradycardia'
        if ((abpsqi>=sqi_th && low_hr_abp-tolerance>BRA_th) ...
                || (ppgsqi>=sqi_th && low_hr_ppg-tolerance>BRA_th) ...
                || (RESPsqi>=sqi_th && low_hr_resp-tolerance>BRA_th)...
                || (MCLsqi>=sqi_th && low_hr_mcl-tolerance>BRA_th) ...
                || (ecgiisqi>=sqi_th && low_hr_II-tolerance>BRA_th)...           
                || (ecg5sqi>=sqi_th && low_hr_V-tolerance>BRA_th)...
                || (ecg3sqi>=sqi_th && low_hr_III-tolerance>BRA_th))
            alarmResult=0;
        end
    case 'Tachycardia'
        if ((abpsqi>=sqi_th && high_hr_abp+tolerance<TAC_th)...
                || (ppgsqi>=sqi_th && high_hr_ppg+tolerance<TAC_th)...
                || (RESPsqi>=sqi_th && high_hr_resp+tolerance<TAC_th)...
                || (MCLsqi>=sqi_th && high_hr_mcl+tolerance<TAC_th)...
                || (ecgiisqi>=sqi_th && high_hr_II+tolerance<TAC_th)...
                || (ecg5sqi>=sqi_th && high_hr_V+tolerance<TAC_th)...
                || (ecg3sqi>=sqi_th && high_hr_III+tolerance<TAC_th))
            
            alarmResult=0;
        end
    case 'Ventricular_Tachycardia'
        % suppress false alarm using hr_max & sqi
        if ((abpsqi>=sqi_th && hr_max_abp+tolerance<VTA_th)...
                || (ppgsqi>=sqi_th && hr_max_ppg+tolerance<VTA_th)...
                || (RESPsqi>=sqi_th && hr_max_resp+tolerance<VTA_th)...
                || (MCLsqi>=sqi_th && hr_max_mcl+tolerance<VTA_th)...
                ||(ecgiisqi>=sqi_th && hr_max_II+tolerance<VTA_th)...
                || (ecg5sqi>=sqi_th && hr_max_V+tolerance<VTA_th)...
                || (ecg3sqi>=sqi_th && hr_max_III+tolerance<VTA_th))
             alarmResult=0;
        end
    case 'Ventricular_Flutter_Fib'
        % suppress false alarm using hr_max & sqi
        if (abpsqi>=sqi_th && hr_max_abp+tolerance<VFB_th) ...
                || (ppgsqi>=sqi_th && hr_max_ppg+tolerance<VFB_th)...
                || (RESPsqi>=sqi_th && hr_max_resp+tolerance<VFB_th)...
                || (MCLsqi>=sqi_th && hr_max_mcl+tolerance<VFB_th)...
                || (ecgiisqi>=sqi_th && hr_max_II+tolerance<VFB_th)...
                || (ecg5sqi>=sqi_th && hr_max_V+tolerance<VFB_th)...
                || (ecg3sqi>=sqi_th && hr_max_III+tolerance<VFB_th)            
            alarmResult=0;
        end
    otherwise
        error(['Unknown alarm type: ' alarm_type])
end

% Write result to answers.txt
fid = fopen(answers, 'a');
if (fid == -1)
    error('Could not open answer file');
end

% Get base name of record (without directories)
i = strfind(recordName, filesep);
if (~isempty(i))
    basename = recordName(i(end)+1 : end);
else
    basename = recordName;
end

fprintf(fid, '%s,%d\n', basename, alarmResult);
fclose(fid);

end

%%%%%%%%%%%% Helper Functions %%%%%%%%%%%%%%%%%%%%%
function ind=get_index(description,pattern)
M=length(description);
tmp_ind=strfind(description,pattern);
ind=[];
for m=1:M
    if(~isempty(tmp_ind{m}))
        ind(end+1)=m;
    end
end
end

