function alarmResult=challenge(recordName,alarm_type)
%
%  alarmResult=challenge(recordName,alarm_type)
%
% Sample entry for the 2015 PhysioNet/CinC Challenge.
%
% Inputs:
%   recordName
%       String specifying the record name to process
%   alarmType
%       String specifying the alarm type. Alarm types are:
%             Asystole, Bradycardia, Tachycardia,
%             Ventricular_Tachycardia, Ventricular_Flutter_Fib
%
%
% Outputs:
%   alarmResult
%       Integer value where 0 = false alarm and 1 is a true
%       alarm. 
%
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% Dependencies:
%
%       1) This function does not requires that you have the WFDB
%       App Toolbox installed. 
%       A matlab function 'rdmat' can read the data instead of using WFDB
%       Toolbox.
%
%       2) The CHALLENGE function requires that you have downloaded the challenge
%       data 'set-p' in a subdirectory of the current directory. The subdirectory
%       should be called '/challenge/2015/set-p/' . The 'set-p' dataset can
%       be downloaded from PhysioNet at:
%           http://physionet.org/physiobank/database/challenge/2015/
%          
%         This dataset is used by the generateValidationSet.m script to
%         create the annotations on your traing set that will be used to 
%         verify that your entry works properly in the PhysioNet testing 
%         environment. 
%
% Version 0.5
%
%
% Written by Qiao Li, November 10, 2014.
% Last Modified: Ikaro Silva February 11, 2015
%
%
%
% %Example using training data- 
% alarmResult=challenge('./challenge/set-p/100','Asystole')
%
% Default results:
% Asystole:                 TP: 0.148	FP: 0.402 	FN: 0.033	TN: 0.418
% Bradycardia:              TP: 0.506	FP: 0.202 	FN: 0.011	TN: 0.281
% Tachycardia:              TP: 0.814	FP: 0.029 	FN: 0.121	TN: 0.036
% Ventricular_Flutter_Fib:	TP: 0.086	FP: 0.466 	FN: 0.017	TN: 0.431
% Ventricular_Tachycardia:	TP: 0.235	FP: 0.545 	FN: 0.026	TN: 0.194
% Average:                  TP: 0.358	FP: 0.329 	FN: 0.042	TN: 0.272
% Gross:                    TP: 0.349	FP: 0.379 	FN: 0.043	TN: 0.229

% Name of file containing answers
answers = 'answers.txt';

%Get all ECG, blood pressure and photoplethysmogram signals
[~,signal,Fs,siginfo]=rdmat(recordName);
alarmResult=1;
description=squeeze(struct2cell(siginfo));
description=description(4,:);

% Resample signal to 125Hz
% Fs=Fs(1);
% if Fs~=125
%     signal=resample(signal,125,Fs);
%     Fs=125;
% end

%%Users can access the raw samples of the record by running the following
%command if WFDB Toolbox installed:
%[tm,signal]=rdsamp(recordName);
%
%%For more information please see the help in RDSAMP

%Run WABP on the record, which by default will analyze the first ABP, ART, or BP signal
N=[];
N0=[];
abp_ind=get_index(description,'ABP');
ann_abp=[];
features=[];
BEATQ=[];
R=[];
if(~isempty(abp_ind))
   ann_abp=wabp(signal(:,abp_ind),0,1);
   % Analyze the signal quality index of ABP using jSQI
   if length(ann_abp)>=3 % at least 3 abp beats detected
        [features] = abpfeature(signal(:,abp_ind),ann_abp);
        [BEATQ R] = jSQI(features, ann_abp, signal(:,abp_ind));
   end
end

%Run WABP on the record of 'PLETH' to analyze photoplethysmogram signal
ppg_ind=get_index(description,'PLETH');
ann_ppg=[];
if (~isempty(ppg_ind))
    y=quantile(signal(:,ppg_ind),[0.05,0.5,0.95]);
    ann_ppg=wabp(signal(:,ppg_ind),0,(y(3)-y(1))/120);
    % Analyze the signal quality index of PPG 
    if ~isempty(ann_ppg)
        [psqi]=ppgSQI(signal(:,ppg_ind),ann_ppg);
    end
end

%Make decisions

% set valid data segment for decision making, 16s before the alarm
N_d=Fs*5*60; % alarm position
N0_d=N_d-Fs*16+1; % 16s before the alarm

% select the beats in the segment
n_abp_beats=intersect(find(ann_abp>=N0_d),find(ann_abp<=N_d));
n_ppg_beats=intersect(find(ann_ppg>=N0_d),find(ann_ppg<=N_d));

hr_max_abp=NaN;
max_rr_abp=NaN;
hr_max_ppg=NaN;
max_rr_ppg=NaN;

% calculate the heart rate
if length(n_abp_beats)>=2
    hr_max_abp=60*Fs/min(diff(ann_abp(n_abp_beats)));
    abp_diff=ann_abp(n_abp_beats);
    abp_diff(end+1)=N_d;
    max_rr_abp=max(diff(abp_diff))/Fs;
    clear abp_diff
end
if length(n_ppg_beats)>=2
    hr_max_ppg=60*Fs/min(diff(ann_ppg(n_ppg_beats)));
    ppg_diff=ann_ppg(n_ppg_beats);
    ppg_diff(end+1)=N_d;
    max_rr_ppg=max(diff(ppg_diff))/Fs;
    clear ppg_diff
end
  
% calculate low heart rate of 5 consecutive beats for Bradycardia
low_hr_abp=NaN;
low_hr_ppg=NaN;
if length(n_abp_beats)>=5
    for i=1:length(n_abp_beats)-4
        low_hr_abp(i)=60*Fs/((ann_abp(n_abp_beats(i+4))-ann_abp(n_abp_beats(i)))/4);
    end
end
low_hr_abp=min(low_hr_abp);
if length(n_ppg_beats)>=5
    for i=1:length(n_ppg_beats)-4
        low_hr_ppg(i)=60*Fs/((ann_ppg(n_ppg_beats(i+4))-ann_ppg(n_ppg_beats(i)))/4);
    end
end
low_hr_ppg=min(low_hr_ppg);
        
% calculate high heart reate of 17 consecutive beats for Tachycardia
high_hr_abp=NaN;
high_hr_ppg=NaN;
if length(n_abp_beats)>=17
    for i=1:length(n_abp_beats)-16
        high_hr_abp(i)=60*Fs/((ann_abp(n_abp_beats(i+16))-ann_abp(n_abp_beats(i)))/16);
    end
end
high_hr_abp=max(high_hr_abp);
if length(n_ppg_beats)>=17
    for i=1:length(n_ppg_beats)-16
        high_hr_ppg(i)=60*Fs/((ann_ppg(n_ppg_beats(i+16))-ann_ppg(n_ppg_beats(i)))/16);
    end
end
high_hr_ppg=max(high_hr_ppg);

% calculate the signal quality index
if ~isempty(ann_abp)
    abpsqi=1-sum(sum(BEATQ(intersect(n_abp_beats,1:length(BEATQ)),:)))/numel(BEATQ(intersect(n_abp_beats,1:length(BEATQ)),:));
else
    abpsqi=0;
end
if ~isempty(ann_ppg)
    ppgsqi=mean(psqi(intersect(n_ppg_beats,1:length(psqi))));
else
    ppgsqi=0;
end

% SQI threshold
sqi_th = 0.9;
ASY_sqi_th = 0.7;
BRA_sqi_th = 0.92;
TAC_sqi_th = 0.94;
% Alarm threshold (seconds)
ASY_th = 4;
BRA_th = 40;
TAC_th = 140;
VTA_th = 100;
VFB_th = 250;
tolerance = 5; % tolerance = 5 bmp
ASY_tolerance = 0.5;
BRA_tolerance = 10;
TAC_tolerance = 7;
switch alarm_type
    case 'Asystole'
        % if the signal quality is good enough and the maximum RR interval
        % is less than the Asystole threshold, set the alarm as 'F'
        if (abpsqi>=ASY_sqi_th && max_rr_abp<(ASY_th-ASY_tolerance)) || (ppgsqi>=ASY_sqi_th && max_rr_ppg<(ASY_th-ASY_tolerance))...
                || (abpsqi>=ASY_sqi_th-0.1 && max_rr_abp<(ASY_th/2)) || (ppgsqi>=ASY_sqi_th-0.1 && max_rr_ppg<(ASY_th/2))
            alarmResult=0;
        elseif ~(abpsqi>=ASY_sqi_th+0.2) && ~(ppgsqi>=ASY_sqi_th+0.2)
            alarmResult=asystoles_problem_20150403(recordName, Fs, ASY_th);
        end
    case 'Bradycardia'
        % if the signal quality is good enough and the low heart rate
        % is greater than the Bradycardia threshold, set the alarm as 'F'
        if (abpsqi>=BRA_sqi_th && low_hr_abp-BRA_tolerance>BRA_th) || (ppgsqi>=BRA_sqi_th && low_hr_ppg-BRA_tolerance>BRA_th)
            alarmResult=0;
        else % check ECG 
            [low_hr_ecg_1,low_hr_ecg_2]=bradycardia_problem_20150404(recordName, Fs);
            if (abs(low_hr_ecg_1-low_hr_ecg_2) < (BRA_th*0.05)) && (min([low_hr_ecg_1 low_hr_ecg_2])-(BRA_tolerance*0.3)>BRA_th)
                alarmResult=0;
            elseif  (isnan(low_hr_ecg_1) || isnan(low_hr_ecg_2)) && ...
                    ((abpsqi>=BRA_sqi_th-0.2 && low_hr_abp-BRA_tolerance>BRA_th) || (ppgsqi>=BRA_sqi_th-0.2 && low_hr_ppg-BRA_tolerance>BRA_th))
                alarmResult=0;
            elseif (abs(low_hr_ecg_1-low_hr_ecg_2) < (BRA_th*0.4)) && ...
                    ((low_hr_ecg_1-(BRA_tolerance*0.5)>BRA_th) || (low_hr_ecg_2-(BRA_tolerance*0.5)>BRA_th))
                alarmResult=0;
            elseif (low_hr_ecg_1-BRA_tolerance>BRA_th) && (low_hr_ecg_2-BRA_tolerance>BRA_th)
                alarmResult=0;
            end            
        end
    case 'Tachycardia'
        % if the signal quality is good enough and the high heart rate
        % is less than the Tachycardia threshold, set the alarm as 'F'
        if (abpsqi>=TAC_sqi_th && high_hr_abp+TAC_tolerance<TAC_th) || (ppgsqi>=TAC_sqi_th && high_hr_ppg+TAC_tolerance<TAC_th)
            alarmResult=0;
        else 
            [high_hr_ecg_1,high_hr_ecg_2]=tachycardia_problem_20150405(recordName, Fs);
            if (abs(high_hr_ecg_1-high_hr_ecg_2) < (TAC_th*0.05)) && (max([high_hr_ecg_1 high_hr_ecg_2])+TAC_tolerance<TAC_th)
                alarmResult=0;
            else
                [low_hr_ecg_1,low_hr_ecg_2]=bradycardia_problem_20150404(recordName, Fs);
                if (abs(high_hr_ecg_1-low_hr_ecg_1) < (TAC_th*0.05)) && (high_hr_ecg_1+TAC_tolerance<TAC_th)
                    alarmResult=0;
                elseif (abs(high_hr_ecg_2-low_hr_ecg_2) < (TAC_th*0.05))&& (high_hr_ecg_2+TAC_tolerance<TAC_th)
                    alarmResult=0;                    
                end
            end
        end
    case 'Ventricular_Tachycardia'
        % suppress false alarm using hr_max & sqi
        if (abpsqi>=sqi_th && hr_max_abp+tolerance<VTA_th) || (ppgsqi>=sqi_th && hr_max_ppg+tolerance<VTA_th)
            alarmResult=0;
        else 
            alarmResult=ventricular_t_problem_20150407(signal, recordName, Fs);
        end
    case 'Ventricular_Flutter_Fib'
        % suppress false alarm using hr_max & sqi
        if (abpsqi>=0.8 && hr_max_abp+tolerance<VFB_th) || (ppgsqi>=0.8 && hr_max_ppg+tolerance<VFB_th)
            alarmResult=0;
        end
    otherwise
        error(['Unknown alarm type: ' alarm_type])
end

% Write result to answers.txt
fid = fopen(answers, 'a');
if (fid == -1)
    error('Could not open answer file');
end

% Get base name of record (without directories)
i = strfind(recordName, filesep);
if (~isempty(i))
    basename = recordName(i(end)+1 : end);
else
    basename = recordName;
end

fprintf(fid, '%s,%d\n', basename, alarmResult);
fclose(fid);

end

%%%%%%%%%%%% Helper Functions %%%%%%%%%%%%%%%%%%%%%
function ind=get_index(description,pattern)
M=length(description);
tmp_ind=strfind(description,pattern);
ind=[];
for m=1:M
    if(~isempty(tmp_ind{m}))
        ind(end+1)=m;
    end
end
end
