function alarmResult=challenge(recordName,alarm_type)

# alldata=challenge(recordName,alarm_type)
%
%  alarmResult=challenge(recordName,alarm_type)
%
% Sample entry for the 2015 PhysioNet/CinC Challenge.
%
% Inputs:
%   recordName
%       String specifying the record name to process
%   alarmType
%       String specifying the alarm type. Alarm types are:
%             Asystole, Bradycardia, Tachycardia,
%             Ventricular_Tachycardia, Ventricular_Flutter_Fib
%
%
% Outputs:
%   alarmResult
%       Integer value where 0 = false alarm and 1 is a true
%       alarm. 

% %Example using training data- 
% alarmResult=challenge('./challenge/set-p/100','Asystole')


  ## alldata contains signal, recordname. ....


  ## Name of file containing answers
  answers = 'answers.txt';

  ##Get all ECG, blood pressure and photoplethysmogram signals
  [tm,signal,Fs,siginfo]=rdmat(recordName);
  alarmResult=1;
  description=squeeze(struct2cell(siginfo));
  description=description(4,:);
  numchannels=size(signal,2);
  lastletter=recordName(end); ## 's' or 'l' serves to decide length of
  ## annotations used to compare gqrs and osea annotations
  

  newFs=125; ##Change Fs to 125
  signal=signal(1:2:end,:);

  ## eliminate pacemaker spikes at the end of the record (if they are
  ## present)
 
  ## changed rapr
  if(strcmp(alarm_type,"Ventricular_Tachycardia")==0) # no ventric_tachycardia
    init=newFs*4.5*60; # 4.5 minutes after start
  else
    init=newFs*3.5*60;
  endif

  ## replace Nans
  signal(isnan(signal))=-5;

  ## remove baseeline from ECG signals 
  ## extend signal before applying medfilt to avoid a bump at the end
  LL=length(signal);
  if(LL<5.1*60*newFs) ## its a short signal
    signal(5*60*newFs+1:5*60*newFs+newFs,:)=flipud(signal(5*60*newFs-newFs:5*60*newFs-1,:));
  endif
  baseline=medfilt1(signal(:,1),newFs);
  signal(:,1)=signal(:,1)-baseline;
  baseline=medfilt1(signal(:,2),newFs);
  signal(:,2)=signal(:,2)-baseline;

  ## remove pacemaker spikes from ECG 
  auxsignal=zeros(length(signal)-init+1,2);
  auxsignal(:,1)=detectpacespikes(signal(init:end,1),newFs,0.25);
  auxsignal(:,2)=detectpacespikes(signal(init:end,2),newFs,0.25);
  sig1=signal(:,1);
  sig1(init:end)=auxsignal(:,1);
  sig2=signal(:,2);
  sig2(init:end)=auxsignal(:,2);
  ## save both ECG channels in wfdb format 
  mat2wfdb( sig1, "ECG1", newFs);
  mat2wfdb( sig2, "ECG2", newFs);

  ## export alldata for debuggging
  alldata.signal=signal;
  alldata.signal=signal;
  alldata.recordName=recordName;
  alldata.alarmType=alarm_type;
  alldata.numchannels=numchannels;
  alldata.description=description;
  alldata.tm=tm;


  ## Assuming ECG are the first 2 channels (and only those)

  ## run qrs detectors on ecg channels, abp and pleth--------------------- 

  if(strcmp(alarm_type,"Ventricular_Flutter_Fib")==1)
    start=280*newFs; #20 sec before the alarm
    annFIB1=fibrillationpeaks(sig1(start+1:end),newFs);
    annFIB1=start+ annFIB1;
    annFIB2=fibrillationpeaks(sig2(start+1:end),newFs);
    annFIB2=start+ annFIB2;
    ## export for debugging
    alldata.annFIB1=annFIB1;
    alldata.annFIB2=annFIB2;
  endif

  ## run gqrs on each ECG lead
  gqrs("ECG1",5*60*newFs,[],1);
  annQRS1=[];
  try
    annQRS1=rdann("ECG1",'qrs');
  catch 
    printf("No gqrs detections on 1st channel\n")
  end_try_catch
  gqrs("ECG2",5*60*Fs,[],1);
  annQRS2=[];
  try
    annQRS2=rdann("ECG2",'qrs');
  catch 
    printf("No gqrs detections on 2nd channel\n")
  end_try_catch

  ## export for debugging
  alldata.annQRS1=annQRS1;
  alldata.annQRS2=annQRS2;
  
  ## run OSEA on each ECG lead
  [dir, name] = fileparts (recordName);
  if(isempty(dir))
    dir=pwd;
  endif
  dir=[dir,'/'];
  [annOSEA1 chan1type]=get_osea_annot("ECG1",0, dir);
  [annOSEA2 chan2type]=get_osea_annot("ECG2",0,dir);
  alldata.annOSEA1=annOSEA1;
  alldata.annType1=chan1type; # beat type 'N', 'V', ...
  alldata.annOSEA2=annOSEA2;
  alldata.annType2=chan2type; # beat type 'N', 'V', ...  
  ##

  
  ## run WABPOTHER on ABP and/or PLETH
    
  ## PLETH
  ppgInd=get_index(description,'PLETH');
  annPLETH=[];
  if(~isempty(ppgInd))
    ppgsignal=signal(:,ppgInd(1));
    ## low-pass filter ppgsignal (is 7 the right option???)
    ppgsignal=medfilt1(ppgsignal,7);
    ## remove baseline wander
    ppgsignal=median(ppgsignal)+ppgsignal-medfilt1(ppgsignal,newFs);
    y=quantile(ppgsignal,[0.05,0.5,0.95]);
    annPLETH=wabpother(ppgsignal,0,(y(3)-y(1))/120);
  endif
  alldata.annPLETH=annPLETH;
  
  ## ABP
  abpInd=get_index(description,'ABP');
  annABP=[];
  if(~isempty(abpInd))
    abpsignal=signal(:,abpInd(1));
    abpsignal=median(abpsignal)+abpsignal-medfilt1(abpsignal,newFs);
    y=quantile(abpsignal,[0.05,0.5,0.95]);
    annABP=wabpother(abpsignal,0,(y(3)-y(1))/120);
  endif
  alldata.annABP=annABP;

  ##------------------------------end running detectors------------------



  ## simple ('stupid') solution choose the result of the channel 
  ## with highest quality index
  Qindex=zeros(1,6);
  Results=zeros(1,6);

  ## number of Pulsatil channels available
  k=0;
  if(~isempty(ppgInd)) k+=1; endif
  if(~isempty(abpInd)) k+=1; endif
  alldata.numpulsatil=k;

  ## thresholds used for decisions -------------------
  N_d=newFs*5*60; ## alarm sample
  tolasys=0.2;
  tolrr=5;
  tolbradyECGrr=5;# we also tried 3 but some false negative
  tolbradyPulserr=5;
  toltachy=5;
  switch alarm_type
    case 'Asystole'
      alarm_thresholdECG=4-tolasys; # max rr interval
      alarm_thresholdPulse=4-1.3*tolasys;
      N0=floor(N_d-newFs*14+1);
      limitTrust=0.82;
    case 'Bradycardia'
      alarm_thresholdECG = 40+tolbradyECGrr; # (hr)
      alarm_thresholdPulse = 40+tolbradyPulserr; # (hr)
      N0=floor(N_d-newFs*10);
      limitTrust=0.9;
   case 'Tachycardia'
    alarm_thresholdECG = 140-toltachy; # (hr) 
    alarm_thresholdPulse = 140-toltachy; # (hr)
     N0=floor(N_d-newFs*17.3); #(60/140*17=7.2857
     limitTrust=0.9;
   case 'Ventricular_Tachycardia'
      alarm_thresholdECG = 100-tolrr; #(hr)
      alarm_thresholdPulse=100-tolrr; #(hr)
      N0=floor(N_d-newFs*10); # changed 17/8
      limitTrust=0.82;
    case 'Ventricular_Flutter_Fib'
    alarm_thresholdECG = 140-toltachy; # (hr)
      alarm_thresholdPulse = 200;
      N0=floor(N_d-newFs*14);
      limitTrust=0.9;
  endswitch


  ## evalute signal quality on the different channels------------------------

  ## ECG 1st channel ########################################
  ## gqrs
  
  if(lastletter=='s')
      LECG=N_d-N0-newFs;
    else
      LECG=N_d-N0;
    end

    ## changed rapr 2/8
  ##if(strcmp(alarm_type,'Ventricular_Tachycardia')==0)
    Results(1)=detect(alarm_type,annQRS1,N0,N_d,alarm_thresholdECG, newFs);
    printf("result of gqrs on channel 1 is %d\n",  Results(1));
  ##endif
  
  ## check ECG quality indices for this lead(compare gqrs and OSEA annotations)
  thres=0.15*newFs; #samples
  if((length(annQRS1(annQRS1>N0))>1)&&(length(annOSEA1(annOSEA1>N0))>1))
    Qindex(1)=ECGsqi(annQRS1(annQRS1>N0),annOSEA1(annOSEA1>N0),thres,N0,LECG);
  endif # else Qindex(1)=0;
  ecg1flat=flatLineECG(signal(N0:N_d,1),0.000001, 4);
  flatsececg1=detectHighFlatOnSeconds(ecg1flat,newFs, 0.8);
  ##fractionecg1flat=sum(ecg1flat)/(N_d-N0)
  if(flatsececg1>1)
    Qindex(1)=0;
    printf("ECG1 signal is flat\n")
  endif
  printf("ECG 1st channel quality index is %f  \n", Qindex(1));

  ## OSEA
  Results(2)=detect(alarm_type,annOSEA1,N0,N_d, alarm_thresholdECG, newFs, chan1type);
  printf("result of OSEA on channel 1 is %d\n",  Results(2));




  ## ECG 2nd channel ################ (same as 1st channel)
  ## gqrs
  #if(strcmp(alarm_type,'Ventricular_Tachycardia')==0)
    Results(3)=detect(alarm_type,annQRS2,N0,N_d, alarm_thresholdECG, newFs);
    printf("result of gqrs on channel 2 is %d\n", Results(3));
  #endif
  ## check ECG quality indices for this lead(compare gqrs and OSEA annotations)
  if((length(annQRS2(annQRS2>N0))>1)&&(length(annOSEA2(annOSEA2>N0))>1))
     Qindex(2)=ECGsqi(annQRS2(annQRS2>N0),annOSEA2(annOSEA2>N0),thres,N0,LECG);
  endif
  ecg2flat=flatLineECG(signal(N0:N_d,2),0.000001, 4);
  flatsececg2=detectHighFlatOnSeconds(ecg2flat,newFs, 0.8);
  ##fractionecg2flat=sum(ecg2flat)/(N_d-N0)
  if(flatsececg2>2)
    Qindex(2)=0;
    printf("ECG2 signal is flat\n")
  endif
  printf("ECG 2nd channel quality index is %f  \n", Qindex(2));

  
  ## OSEA
  Results(4)=detect(alarm_type,annOSEA2,N0,N_d, alarm_thresholdECG, newFs, chan2type);
  printf("result of OSEA on channel 2 is %d\n", Results(4));

  ## changed rapr 2/8
  if(strcmp(alarm_type,'Ventricular_Tachycardia')==1)
    Results(1,7)=detectvt(annOSEA1,N0,N_d, alarm_thresholdECG,newFs,chan1type);
    printf("result of vt detector on channel 1 is %d\n",  Results(7));
    Results(1,8)=detectvt(annOSEA2,N0,N_d, alarm_thresholdECG, newFs,chan2type);
    printf("result of vt detector on channel 2 is %d\n",  Results(8));
  ## for Ventricular_Flutter_Fib use (also) 'annFib1' and 'annFib2'
  elseif(strcmp(alarm_type, 'Ventricular_Flutter_Fib')==1)
    Results(1,7)=detect(alarm_type,annFIB1,N0,N_d, alarm_thresholdECG, newFs);
    printf("result of FIB detector on channel 1 is %d\n",  Results(7));
    Results(1,8)=detect(alarm_type,annFIB2,N0,N_d, alarm_thresholdECG, newFs);
    printf("result of FIB detector on channel 2 is %d\n",  Results(8));
  endif





  ## PLETH
  if((~isempty(ppgInd)))
    ## changed rapr 2/8
    ##if(strcmp(alarm_type,'Ventricular_Tachycardia')==0)
      [Results(5) n_ppg_beats]=detect(alarm_type,annPLETH,N0,N_d,
                                    alarm_thresholdPulse, newFs);
      printf("result of PLETH detections is %d\n", Results(5));
    ##else
    ##  n_ppg_beats=intersect(find(annPLETH>=N0),find(annPLETH<N_d));
    ##endif    
    if(~isempty(n_ppg_beats))
      
      #ppgsqi=mean(psqi(intersect(n_ppg_beats,1:length(psqi))));
      psqi=ourppgSQI(ppgsignal,annPLETH,125,n_ppg_beats(1),
                     n_ppg_beats(end));

      simpleppsqi=ppgSQI2(ppgsignal,annPLETH,125,n_ppg_beats(1),
			 n_ppg_beats(end));

      psqi=(psqi+simpleppsqi)/2;
      Qindex(3)=mean(psqi);
      ## if last detection is far from 5 min penalise it 
      if((abs(annPLETH(n_ppg_beats(end))-N_d)>1.5*newFs) &&
         (strcmp(alarm_type,'Asystole')==0)&&
        (strcmp(alarm_type,'Bradycardia')==0))
        Qindex(3)=0;
      endif
    endif
    ppgamplify=120;
    ppgflat=flatLinePulsatil(ppgamplify*signal(N0:N_d,ppgInd(1)),0.04, 10);
    flatsecppg=detectHighFlatOnSeconds(ppgflat,newFs,0.8);
    ##fractionppgflat=sum(ppgflat)/(N_d-N0)
    if(flatsecppg>2)
      Qindex(3)=0;
      printf("PLETH signal is flat\n")
    endif
    printf("PLETH signal quality index is %f\n",Qindex(3));
  endif

  ## ABP
  if((~isempty(abpInd)))
    ## changed rapr 2/8
    ##if(strcmp(alarm_type,'Ventricular_Tachycardia')==0)
      [Results(6) n_beats]=detect(alarm_type,annABP,N0,N_d, alarm_thresholdPulse,
				  newFs);
      printf("result of ABP detections is %d\n", Results(6));

    
    if(~isempty(n_beats))

      abpsqi=ourppgSQI(abpsignal,annABP,125,n_beats(1),
                     n_beats(end));
      simpleabpsqi=ppgSQI2(abpsignal,annABP,125,n_beats(1),
                     n_beats(end));
      abpsqi=(abpsqi+simpleabpsqi)/2;
      Qindex(4)=mean(abpsqi);
     ## if last detection is far from 5 min penalise it 
     if(abs(annABP(n_beats(end))-N_d)>1.5*newFs)
       Qindex(4)=0;
      endif
    endif
    abpflat=flatLinePulsatil(signal(N0:N_d,abpInd(1)),0.04,10);
    flatsecabp=detectHighFlatOnSeconds(abpflat,newFs,0.8);
    ##fractionabpflat=sum(abpflat)/(N_d-N0)
    if(flatsecabp>2)
      Qindex(4)=0;
      printf("ABP signal is flat\n")
    endif
    printf("ABP signal quality index is %f\n",Qindex(4));
  endif

  ## end evalute signal quality--------------------------------------------

  ann{1,1}=annQRS1;
  ann{1,2}=annOSEA1;
  ann{2,1}=annQRS2;
  ann{2,2}=annOSEA2;
  ann{3,1}=annPLETH;
  ann{3,2}=annABP;
  Q(1,1)=Qindex(1);
  Q(1,2)=Qindex(1);
  Q(2,1)=Qindex(2);
  Q(2,2)=Qindex(2);
  Q(3,1)=Qindex(3);
  Q(3,2)=Qindex(4);

  [Qannot]=Qualityannot(ann,Q,thres,N0,LECG, newFs);
  ##Qannot


  ## Decision


  ## default value for alarmResult
  alarmResult=1;

  [Q ind]=max(Qannot);

  if((strcmp(alarm_type,'Asystole')==1)||(strcmp(alarm_type,'Bradycardia')==1)
     ||(strcmp(alarm_type,'Tachycardia')==1) )
     
    ## take results of PLETH or ABP if they have the best quality index
    ## and it is >limitTrust
    if((ind>4) && Q>limitTrust) 
      alarmResult=Results(ind);
    elseif(ind==1 || ind==2) # 1st channel ECG
      if(Results(1)==Results(2))
        alarmResult=Results(2);
      else
        alarmResult=1;
      endif
    elseif(ind==3 || ind==4) # 2nd channel ECG
      if(Results(3)==Results(4))
        alarmResult=Results(4);
      else 
        alarmResult=1;
      endif
    else
      alarmResult=1;
    endif

  elseif(strcmp(alarm_type,'Ventricular_Tachycardia')==1)

    [Q ind]=max(Qannot);
    if(Q>limitTrust)
      if(Results(ind)==0) # use normal detections only to detect false positives
        alarmResult=0;
      else ## use OSEA annotations (only ventricular bets)
        alarmResult=max([Results(7),Results(8)]);
      endif
    else
      alarmResult=max([Results(7),Results(8)]);
    endif
  elseif(strcmp(alarm_type,'Ventricular_Flutter_Fib')==1)
     [Q ind]=max(Qannot);
     if(Q>limitTrust)
       if(ind>4) # PLETH or ABP
         alarmResult=Results(ind);
       elseif(ind==1 || ind==2) # 1st channel ECG
         if(Results(1)==Results(2))
           alarmResult=Results(2);
         else
           alarmResult=Results(7);
         endif
       elseif(ind==3 || ind==4) # 2nd channel ECG
         if(Results(3)==Results(4))
           alarmResult=Results(4);
         else
           alarmResult=Results(8);
         endif
       endif
     elseif((Results(7)==0)&&(Results(8)==0)) ## if signals quality is
       ## bad,  use FIB detector
       alarmResult=0
     else
       alarmResult=1;
     endif
  endif       
  
  printf("\n Alarm result is %d\n",alarmResult);

  ## Write result to answers.txt
  fid = fopen(answers, 'a');
  if (fid == -1)
    error('Could not open answer file');
  end

  ## Get base name of record (without directories)
  i = strfind(recordName, filesep);
  if (~isempty(i))
    basename = recordName(i(end)+1 : end);
  else
    basename = recordName;
  endif
     
  fprintf(fid, "%s,%d\n", basename, alarmResult);
  fclose(fid);


endfunction

 
##%%%%%%%%%% Helper Functions %%%%%%%%%%%%%%%%%%%%%

function ind=get_index(description,pattern)
  M=length(description);
  tmp_ind=strfind(description,pattern);
  ind=[];
  for m=1:M
    if(~isempty(tmp_ind{m}))
      ind(end+1)=m;
    end
  endfor
endfunction


function [res n_beats]=detect(alarm_type, ann, N0, N, threshold,newFs,chantype)
switch alarm_type
     case 'Asystole'
       [res n_beats]=detect_asystole(ann,N0,N,threshold,newFs);
     case 'Bradycardia'
       [res n_beats]=detect_bradycardia(ann,N0,N,threshold,newFs);
     case 'Tachycardia' 
       [res n_beats]=detect_tachycardia(ann,N0,N,threshold,newFs);
     case 'Ventricular_Tachycardia'
      [res n_beats]=detect_ventricular_tachycardia(ann,N0,N,threshold,newFs);
    case 'Ventricular_Flutter_Fib'
      [res n_beats]=detect_fibrillation(ann,N0,N,threshold,newFs);

  endswitch
endfunction

## detect Asystole
function [res n_beats]=detect_asystole(ann,N0,N,threshold,Fs)
  n_beats=intersect(find(ann>=N0),find(ann<=N));
  ## add 30*Fs to ann to check if annotations finnish too early
  ## add N0 to ann to check if annotations finnish too early
  max_rr=max(diff([N0;ann(n_beats)(:);5*60*Fs]))/Fs;
  ## debug
  #[~, loc]=max(diff([ann(n_beats)(:);30*Fs]))
  if(max_rr<threshold)
    res=0;
  else
    res=1;
  endif
endfunction

## detect Bradycardia
function [res n_beats]=detect_bradycardia(ann,N0,N,threshold, Fs)
  if(size(ann,1)>size(ann,2))
    ann=ann';
  endif
 
  if(length(find(ann>N0))>1)
    frontier=find(ann>=N0)(1); 
    n_beats=intersect(find(ann>=ann(frontier)),find(ann<=N));
    min_hr=NaN;
    min_beats=5;
    ## add N=5*60*Fs to ann to check if annotations finnish too early
    BEATS=[ann(n_beats),5*60*Fs];
    if(length(n_beats)>=min_beats)
      
      ## start checking HR 4 beats before N
      if(n_beats(1)>4)
        aux=[n_beats(1)-4:n_beats(1)-1];
        BEATS=[ann([n_beats(1)-4:n_beats(1)-1]),BEATS];
      endif

      for i=1:length(BEATS)-min_beats+1
	hr(i)=60*Fs/((BEATS(i+min_beats-1)-BEATS(i))/...
		     (min_beats-1));
      endfor
      min_hr=min(hr);

      if(min_hr<threshold)
	res=1;
      else
	res=0;
      endif
    else
      res=1;
    endif
  else
    res=1;
    n_beats=[];
  endif
endfunction

##detect tachycardia
function [res n_beats]=detect_tachycardia(ann,N0,N,threshold,Fs)
  min_beats=15; #17
  n_beats=intersect(find(ann>=N0),find(ann<=N));
  max_hr=NaN;
  if(length(n_beats)>=min_beats)
    for i=1:length(n_beats)-min_beats+1
      hr(i)=60*Fs/((ann(n_beats)(i+min_beats-1)-ann(n_beats)(i))/(min_beats-1));
    endfor
    [max_hr ord]=max(hr);
    max_hr;
    ann(n_beats)(ord);
  endif
  if(max_hr>threshold)
    res=1;
  else
    res=0;
  endif
endfunction


##detect ventricular tachycardia remove false positives using
## 'Normal annotations'
function [res n_beats]=detect_ventricular_tachycardia(ann,N0,N,threshold, Fs)

    min_beats=5;
    n_beats=intersect(find(ann>=N0),find(ann<=N));
    max_hr=NaN;
    if(length(n_beats)>=min_beats)

      ## start checking HR 4 beats before N
      if(n_beats(1)>4)
        n_beats=[[n_beats(1)-4:n_beats(1)-1](:);n_beats(:)];
      endif

      for i=1:length(n_beats)-min_beats+1
        hr(i)=...
        60*Fs/((ann(n_beats)(i+min_beats-1)-ann(n_beats)(i))/(min_beats-1));
      endfor
      max_hr=max(hr);
    endif
    if(max_hr>threshold)
      res=1;
    else
      res=0;
    endif

endfunction

## detect true positives using OSEA
function [res n_beats]=detectvt(ann,N0,N,threshold, Fs,annType)

    ## select ventricular beats
    aux=annType=='V';
    ann=ann(aux);

    min_beats=5;
    n_beats=[]; 
    n_beats=intersect(find(ann>=N0),find(ann<=N));
    max_hr=NaN;
    if(length(n_beats)>=min_beats)
      ## start checking HR 4 beats before N
      if(n_beats(1)>4)
        n_beats=[[n_beats(1)-4:n_beats(1)-1],n_beats];
      endif
      for i=1:length(n_beats)-min_beats+1
        hr(i)=...
        60*Fs/((ann(n_beats)(i+min_beats-1)-ann(n_beats)(i))/(min_beats-1));
      endfor
      [max_hr ord]=max(hr);
    endif
    if(max_hr>threshold)
      res=1;
    else
      res=0;
    endif
  
endfunction





function [res n_beats]=detect_fibrillation(ann,N0,N,threshold, Fs)

  if(length(ann)>2)
    n_beats=intersect(find(ann>=N0),find(ann<=N));
    max_hr=NaN;
    min_beats=10;
    if(length(n_beats)>=min_beats)
      for i=1:length(n_beats)-min_beats+1
        hr(i)=60*Fs/((ann(n_beats)(i+min_beats-1)-ann(n_beats)(i))/(min_beats-1));
      endfor
      [max_hr ord]=max(hr);
      #max_hr;
      # ann(n_beats)(ord);
    endif
    if(max_hr>threshold)
      res=1;
    else
      res=0;
    endif
  else
    res=0; # result is false alarm if there are few or no detections
    n_beats=[]; # its is just a result witout meaning (should be changed)  
endif

  
endfunction

