function alarmResult=challenge(recordName,alarm_type)
%
%  alarmResult=challenge(recordName,alarm_type)
%
% Entry for the 2015 PhysioNet/CinC Challenge.
%
% Inputs:
%   recordName
%       String specifying the record name to process
%   alarmType
%       String specifying the alarm type. Alarm types are:
%             Asystole, Bradycardia, Tachycardia,
%             Ventricular_Tachycardia, Ventricular_Flutter_Fib
%
% Outputs:
%   alarmResult
%       Integer value where 0 = false alarm and 1 is a true
%       alarm. 
%
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% Dependencies:
%
%       1) This function does not requires that you have the WFDB
%       App Toolbox installed. 
%       A matlab function 'rdmat' can read the data instead of using WFDB
%       Toolbox.
%
%       2) The CHALLENGE function requires that you have downloaded the challenge
%       data 'set-p' in a subdirectory of the current directory. The subdirectory
%       should be called '/challenge/2015/set-p/' . The 'set-p' dataset can
%       be downloaded from PhysioNet at:
%           http://physionet.org/physiobank/database/challenge/2015/
%          
%         This dataset is used by the generateValidationSet.m script to
%         create the annotations on your traing set that will be used to 
%         verify that your entry works properly in the PhysioNet testing 
%         environment. 
%
%
% Sample entry written by Qiao Li, November 10, 2014.
% Last Modified: Ikaro Silva February 11, 2015
%
% This entry written by Marcus Vollmer, March 06, 2015.
% Last Modified: Marcus Vollmer, March 06, 2015
%
% Version 0.1
%
% %Example using training data- 
% alarmResult=challenge('./challenge/set-p/100','Asystole')
%

% Name of file containing answers
answers = 'answers.txt';

% Set default values
alarmResult=1;

% Run mvqrs for multi-signal beat annotation
if ~(exist([recordName '_debug.mat'],'file')==2 && exist([recordName '.mvqrs'],'file')==2)
    mvqrs(recordName,'debugmode',1);
end
ann = rdann(recordName,'mvqrs');

% Get signal type
load([recordName '_debug.mat'],'ValidSignalsSetting')
ecg_idx = find(ValidSignalsSetting==0);
bp_idx  = find(ValidSignalsSetting==2);

load([recordName '_debug.mat'],'Fs')
load([recordName '_debug.mat'],'factor')


% Annotation for each signal
load([recordName '_debug.mat'],'Annotation')
load([recordName '_debug.mat'],'valid')



%Make decisions
% set valid data segment for decision making, 16s before the alarm
N_d=Fs*factor*5*60; % alarm position
N0_d=N_d-Fs*factor*16+1; % 16s before the alarm

% select the beats in the segment
n_beats=intersect(find(ann>=N0_d),find(ann<=N_d));
RR = diff(ann(n_beats))/(Fs*factor);

% calculate the heart rate
hr_max=NaN;
max_rr=NaN;
if length(n_beats)>=2
    hr_max=60/min(RR);
    max_rr=max(RR);
end

% calculate low heart rate of 5 consecutive beats for Bradycardia
low_hr=NaN;
if length(n_beats)>=5   
    low_hr = min(60./filter(ones(4,1),4,RR));
end
    
% calculate high heart reate of 17 consecutive beats for Tachycardia
high_hr=NaN;
if length(n_beats)>=17
    avg_16 = filter(ones(16,1),16,RR);
    high_hr = max(60./avg_16(16:end));
end


% Alarm threshold (seconds)
ASY_th = 4;
BRA_th = 40;
TAC_th = 140;
VTA_th = 100;
VFB_th = 250;
tolerance = 5; % tolerance = 5 bmp
switch alarm_type
    case 'Asystole'
        % if maximum RR interval is less than the Asystole threshold,
        % set the alarm as 'F' 
        if max_rr<=ASY_th
            alarmResult=0;
        end
    case 'Bradycardia'
        % if the low heart rate is greater than the Bradycardia threshold,
        % set the alarm as 'F'
        if low_hr-tolerance>BRA_th
            alarmResult=0;
        end
    case 'Tachycardia'
        % if the high heart rate is less than the Tachycardia threshold,
        % set the alarm as 'F'
        if high_hr+tolerance<TAC_th
            alarmResult=0;
        end
    case 'Ventricular_Tachycardia'
        % suppress false alarm using hr_max
        if hr_max+tolerance<VTA_th
            alarmResult=0;
        end
    case 'Ventricular_Flutter_Fib'
        % suppress false alarm using hr_max
        if hr_max+tolerance<VFB_th
            alarmResult=0;
        end
    otherwise
        error(['Unknown alarm type: ' alarm_type])
end

% Write result to answers.txt
fid = fopen(answers, 'a');
if (fid == -1)
    error('Could not open answer file');
end

% Get base name of record (without directories)
i = strfind(recordName, filesep);
if (~isempty(i))
    basename = recordName(i(end)+1 : end);
else
    basename = recordName;
end

fprintf(fid, '%s,%d\n', basename, alarmResult);
fclose(fid);

end