function alarmResult=challenge(recordName,alarm_type)
%
%  alarmResult=challenge(recordName,alarm_type)
%
% Sample entry for the 2015 PhysioNet/CinC Challenge. Alarms are classified
% to either true = 1 or false = 0 with Random Forest classifiers.
%
% Inputs:
%   recordName
%       String specifying the record name to process
%   alarmType
%       String specifying the alarm type. Alarm types are:
%             Asystole, Bradycardia, Tachycardia,
%             Ventricular_Tachycardia, Ventricular_Flutter_Fib
%
%
% Outputs:
%   alarmResult
%       Integer value where 0 = false alarm and 1 is a true
%       alarm. 
%
%
% Dependencies:
%
%       1) This function does not requires that you have the WFDB
%       App Toolbox installed. 
%       A matlab function 'rdmat' can read the data instead of using WFDB
%       Toolbox.
%
%       2) The CHALLENGE function requires that you have downloaded the challenge
%       data 'set-p' in a subdirectory of the current directory. The subdirectory
%       should be called '/challenge/2015/set-p/' . The 'set-p' dataset can
%       be downloaded from PhysioNet at:
%           http://physionet.org/physiobank/database/challenge/2015/
%          
%
% Written by Linda Eerikinen April 7, 2015
% Last modified: Linda Eerikinen August 20, 2015
%
%
%
% %Example using training data- 
% alarmResult=challenge('./challenge/set-p/100','Asystole')
%

% Name of file containing answers
answers = 'answers.txt';

% Select signals for feature computation
signalArray = signalSelection(recordName);

% Compute features for classification
physiological_features = computeFeatures(alarm_type, signalArray);

% A Random Forest classifier signs the alarm as 'true' or 'false' for the
% specified arrhythmia having a feature vector as an input
switch alarm_type
    case 'Asystole'
        load('asystoleRF.mat');
        features = [signalArray.sqi_features(1:3), physiological_features];
        class = predict(asystoleRF,features);
        alarmResult = str2double(class);
    case 'Bradycardia'
        load('bradycardiaRF.mat')
        features = [signalArray.sqi_features(1), signalArray.sqi_features(3), ...
            physiological_features];
        class = predict(bradycardiaRF,features);
        alarmResult = str2double(class);
    case 'Tachycardia'
        features = [signalArray.sqi_features(1), physiological_features];
        load('tachycardiaRF.mat')
        class = predict(tachycardiaRF,features);
        alarmResult = str2double(class);
    case 'Ventricular_Flutter_Fib'
        load('ventricularFlutterFibRF.mat')
        class = predict(ventricularFlutterFibRF,physiological_features);
        alarmResult = str2double(class);
    case 'Ventricular_Tachycardia'
        features = [signalArray.sqi_features(1:5), physiological_features];
        load('ventricularTachycardiaRF.mat')
        class = predict(ventricularTachycardiaRF, features);
        alarmResult = str2double(class);
    otherwise
        alarmResult = 1;
end

% Write result to answers.txt
fid = fopen(answers, 'a');
if (fid == -1)
    error('Could not open answer file');
end

% Get base name of record (without directories)
i = strfind(recordName, filesep);
if (~isempty(i))
    basename = recordName(i(end)+1 : end);
else
    basename = recordName;
end

fprintf(fid, '%s,%d\n', basename, alarmResult);
fclose(fid);


end


