function alarmResult=challenge_209(recordName,alarm_type)
%
%  alarmResult=challenge(recordName,alarm_type)
%
% Sample entry for the 2015 PhysioNet/CinC Challenge.
%
% Inputs:
%   recordName
%       String specifying the record name to process
%   alarmType
%       String specifying the alarm type. Alarm types are:
%             Asystole, Bradycardia, Tachycardia,
%             Ventricular_Tachycardia, Ventricular_Flutter_Fib
%
%
% Outputs:
%   alarmResult
%       Integer value where 0 = false alarm and 1 is a true
%       alarm. 
%
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% Dependencies:
%
%       1) This function does not requires that you have the WFDB
%       App Toolbox installed. 
%       A matlab function 'rdmat' can read the data instead of using WFDB
%       Toolbox.
%
%       2) The CHALLENGE function requires that you have downloaded the challenge
%       data 'set-p' in a subdirectory of the current directory. The subdirectory
%       should be called '/challenge/2015/set-p/' . The 'set-p' dataset can
%       be downloaded from PhysioNet at:
%           http://physionet.org/physiobank/database/challenge/2015/
%          
%         This dataset is used by the generateValidationSet.m script to
%         create the annotations on your traing set that will be used to 
%         verify that your entry works properly in the PhysioNet testing 
%         environment. 
%
% Version 0.5
%
%
% Written by Qiao Li, November 10, 2014.
% Last Modified: Ikaro Silva February 11, 2015
%
%
%
% %Example using training data- 
% alarmResult=challenge('./challenge/set-p/100','Asystole')
%

% Name of file containing answers
answers = 'answers.txt';

%Get all ECG, blood pressure and photoplethysmogram signals
[~,signal,Fs,siginfo]=rdmat(recordName);
alarmResult=1;
description=squeeze(struct2cell(siginfo));
description=description(4,:);

switch alarm_type
    case 'Asystole'
        load('OPTIMAL_DISCR_PCA_fft_c10_1');
        
    case 'Bradycardia'
        load('OPTIMAL_DISCR_PCA_fft_c10_2');
        
    case 'Tachycardia'
        load('OPTIMAL_DISCR_PCA_fft_c10_3');
        
    case 'Ventricular_Flutter_Fib'
        load('OPTIMAL_DISCR_PCA_fft_c10_4');
        
    case 'Ventricular_Tachycardia'
        load('OPTIMAL_DISCR_PCA_fft_c10_5');
        
    otherwise
        error(['Unknown alarm type: ' alarm_type])
end

resp_ind  = find(strcmp(description,'RESP'));
pleth_ind = find(strcmp(description,'PLETH'));
abp_ind   = find(strcmp(description,'ABP'));




if strcmp(DATA_OPT,'ecg')
    del_ind = [resp_ind pleth_ind abp_ind];
elseif strcmp(DATA_OPT,'bp')
    del_ind = [];
    signal  = signal(:,[pleth_ind abp_ind]);
elseif strcmp(DATA_OPT,'fused')
    del_ind = resp_ind;
end

if ~isempty(del_ind)
    signal(:,del_ind) = [];
end

signal(isnan(signal)) = 0;

% Resample signal to 100Hz
Fs=Fs(1);
if Fs~=100
    signal=resample(signal,100,Fs);
    Fs=100;
end

hopsize = 8;

% set valid data segment for decision making, 16s before the alarm
N_d=Fs*5*60;      % alarm position
N0_d=N_d-Fs*10+1; % 10s before the alarm

%Define interval search window
win = 0.3*Fs:2.0*Fs;

%Filter with [0.5 30] Hz band-pass
[b,a] = butter(2,[1 30]/(Fs/2));
signal = filtfilt(b, a, signal);

%Apply zscore to each channel
signal = zscore(signal);

indx = 1;
prob_arr = zeros(length(N0_d+1:hopsize:N_d-win(end)),size(win,2));
for l = N0_d+1:hopsize:N_d-win(end)
     win_sig = signal((l-win(end)):(l+win(end)),:);
     prob_arr(indx,:) = prod(prob_estimator(win_sig, win),1);
     indx = indx + 1;
end

if strcmp(METHOD_OPT,'no')
elseif strcmp(METHOD_OPT,'sum')
    prob_arr = prob_arr ./ repmat(sum(prob_arr,2),[1 size(prob_arr,2)]);
elseif strcmp(METHOD_OPT,'max')
    prob_arr = prob_arr ./ repmat(max(prob_arr,[],2),[1 size(prob_arr,2)]);
end

prob_arr = abs(fft2(log(prob_arr)));
prob_arr    = prob_arr(:) - DATA_MEAN_OPT(:);
X           = prob_arr' * U_OPT;
alarmResult = predict(DISCR_OPT,X);

% Write result to answers.txt
fid = fopen(answers, 'a');
if (fid == -1)
    error('Could not open answer file');
end

% Get base name of record (without directories)
i = strfind(recordName, filesep);
if (~isempty(i))
    basename = recordName(i(end)+1 : end);
else
    basename = recordName;
end

fprintf(fid, '%s,%d\n', basename, alarmResult);
fclose(fid);

end
