function alarmResult=challenge(recordName,alarm_type)
%
%  alarmResult=challenge(recordName,alarm_type)
%
% Sample entry for the 2015 PhysioNet/CinC Challenge.
%
% Inputs:
%   recordName
%       String specifying the record name to process
%   alarmType
%       String specifying the alarm type. Alarm types are:
%             Asystole, Bradycardia, Tachycardia,
%             Ventricular_Tachycardia, Ventricular_Flutter_Fib
%
%
% Outputs:
%   alarmResult
%       Integer value where 0 = false alarm and 1 is a true
%       alarm. 
%
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% Dependencies:
%
%       1) This function does not requires that you have the WFDB
%       App Toolbox installed. 
%       A matlab function 'rdmat' can read the data instead of using WFDB
%       Toolbox.
%
%       2) The CHALLENGE function requires that you have downloaded the challenge
%       data 'set-p' in a subdirectory of the current directory. The subdirectory
%       should be called '/challenge/2015/set-p/' . The 'set-p' dataset can
%       be downloaded from PhysioNet at:
%           http://physionet.org/physiobank/database/challenge/2015/
%          
%         This dataset is used by the generateValidationSet.m script to
%         create the annotations on your traing set that will be used to 
%         verify that your entry works properly in the PhysioNet testing 
%         environment. 
%
% Version 0.5
%
%
% Written by Qiao Li, November 10, 2014.
% Last Modified: Ikaro Silva February 11, 2015
%
%
%
% %Example using training data- 
% alarmResult=challenge('./challenge/set-p/100','Asystole')
%

% Name of file containing answers
answers = 'answers.txt';

%Get all ECG, blood pressure and photoplethysmogram signals
[~,signal,Fs,siginfo]=rdmat(recordName);
alarmResult=1;
description=squeeze(struct2cell(siginfo));
description=description(4,:);

resp_ind    = find(strcmp(description,'RESP'));
if ~isempty(resp_ind)
    signal(:,resp_ind) = [];
end

signal(isnan(signal)) = 0;

% Resample signal to 100Hz
Fs=Fs(1);
if Fs~=100
    signal=resample(signal,100,Fs);
    Fs=100;
end

hopsize = 8;

% set valid data segment for decision making, 16s before the alarm
N_d=Fs*5*60; % alarm position
N0_d=N_d-Fs*16+1; % 16s before the alarm

%Filter with [0.5 30] Hz band-pass
[b,a] = butter(2,[1 30]/(Fs/2));
signal = filtfilt(b, a, signal);

%Apply zscore to each channel
signal = zscore(signal);

switch alarm_type
    case 'Asystole'
        load('OPTIMAL_TREE_c5_1');
        
    case 'Bradycardia'
        load('OPTIMAL_TREE_c5_2');
        
    case 'Tachycardia'
        load('OPTIMAL_TREE_c5_3');
        
    case 'Ventricular_Flutter_Fib'
        load('OPTIMAL_TREE_c5_4');
        
    case 'Ventricular_Tachycardia'
        load('OPTIMAL_TREE_c5_5');
        
    otherwise
        error(['Unknown alarm type: ' alarm_type])
end

win = interv(1)*Fs:interv(2)*Fs;

indx = 1;
interv_arr  = zeros(length(N0_d+1:hopsize:N_d-win(end)),size(signal,2)+1);
quality_arr = interv_arr;
for l = N0_d+1:hopsize:N_d-win(end)
     win_sig = signal((l-win(end)):(l+win(end)),:);
     [interv_arr(indx,:), quality_arr(indx,:)] = single_interval_estimator(win_sig, win);
     indx = indx + 1;
end
interv_arr  = interv_arr(:,end);
quality_arr = quality_arr(:,end);

indx = 1;
interv_arr_ECG  = zeros(length(N0_d+1:hopsize:N_d-win(end)),3);
quality_arr_ECG = interv_arr_ECG;
for l = N0_d+1:hopsize:N_d-win(end)
     win_sig = signal((l-win(end)):(l+win(end)),1:2);
     [interv_arr_ECG(indx,:), quality_arr_ECG(indx,:)] = single_interval_estimator(win_sig, win);
     indx = indx + 1;
end
interv_arr_ECG  = interv_arr_ECG(:,end);
quality_arr_ECG = quality_arr_ECG(:,end);

xcorr_ecg_1 = xcorr(signal(N0_d:N_d,1),200);
[val, indx] = max(xcorr_ecg_1(1:180));
if indx == 180
    val = 0;
end
ratio_1_180 = val/max(xcorr_ecg_1);
[val, indx] = max(xcorr_ecg_1(1:110));
if indx == 110
    val = 0;
end
ratio_1_110 = val/max(xcorr_ecg_1);

xcorr_ecg_2 = xcorr(signal(N0_d:N_d,2),200);
[val, indx] = max(xcorr_ecg_2(1:180));
if indx == 180
    val = 0;
end
ratio_2_180 = val/max(xcorr_ecg_2);            
[val, indx] = max(xcorr_ecg_2(1:110));
if indx == 110
    val = 0;
end
ratio_2_110 = val/max(xcorr_ecg_2);            

ecg_ratio_180 = max([ratio_1_180 ratio_2_180]);
ecg_ratio_110 = max([ratio_1_110 ratio_2_110]);

pleth_ind = find(strcmp(description,'PLETH'));
abp_ind   = find(strcmp(description,'ABP'));
signal_tmp = signal(:,[pleth_ind abp_ind]);
indx = 1;
interv_arr_BP  = zeros(length(N0_d+1:hopsize:N_d-win(end)),size(signal_tmp,2)+1);
quality_arr_BP = interv_arr_BP;
for l = N0_d+1:hopsize:N_d-win(end)
     win_sig = signal_tmp((l-win(end)):(l+win(end)),:);
     [interv_arr_BP(indx,:), quality_arr_BP(indx,:)] = single_interval_estimator(win_sig, win);
     indx = indx + 1;
end
interv_arr_BP  = interv_arr_BP(:,end);
quality_arr_BP = quality_arr_BP(:,end);

pleth_ratio_180 = 0;
abp_ratio_180 = 0;
pleth_ratio_110 = 0;
abp_ratio_110 = 0;
        
if ~isempty(pleth_ind)
    xcorr_pleth = xcorr(signal(N0_d:N_d,pleth_ind),200);
    [val, indx] = max(xcorr_pleth(1:180));
    if indx == 180
        val = 0;
    end
    pleth_ratio_180 = val/max(xcorr_pleth);
    [val, indx] = max(xcorr_pleth(1:110));
    if indx == 110
        val = 0;
    end
    pleth_ratio_110 = val/max(xcorr_pleth);
end

if ~isempty(abp_ind)
    xcorr_abp = xcorr(signal(N0_d:N_d,abp_ind),200);
    [val, indx] = max(xcorr_abp(1:180));
    if indx == 180
        val = 0;
    end
    abp_ratio_180 = val/max(xcorr_abp);
    [val, indx] = max(xcorr_abp(1:110));
    if indx == 110
        val = 0;
    end
    abp_ratio_110 = val/max(xcorr_abp);
end
BP_ratio_180 = max([pleth_ratio_180  abp_ratio_180]);
BP_ratio_110 = max([pleth_ratio_110  abp_ratio_110]);

N0_d=N_d-Fs*5+1; % 5s before the alarm
corr_arr = xcorr(signal(N0_d:N_d,1),150);
for i=2:size(signal,2)
    corr_arr(:,i) = xcorr(signal(N0_d:N_d,i),150);
end

QUALITY_TH = (q-1)/8;

indx_0 = ceil(size(interv_arr,1)*QUALITY_TH+.001);

interv_fused    = medfilt1(interv_arr,median_val);
quality_fused   = quality_arr;
[~, indx]       = sort(quality_fused);
interv_fused_hq = interv_fused(indx(indx_0:end));

interv_ECG    = medfilt1(interv_arr_ECG,median_val);
quality_ECG   = quality_arr_ECG;
[~, indx]     = sort(quality_ECG);
interv_ECG_hq = interv_ECG(indx(indx_0:end));

interv_BP    = medfilt1(interv_arr_BP,median_val);
quality_BP   = quality_arr_BP;
[~, indx]    = sort(quality_BP);
interv_BP_hq = interv_BP(indx(indx_0:end));

X = zeros(1,32);

X(1)  = min(interv_fused_hq); %#ok<*AGROW>
X(2)  = max(interv_fused_hq);
X(3)  = std(interv_fused_hq);
X(4)  = std(interv_fused_hq)/mean(interv_fused_hq);
X(5)  = mad(interv_fused_hq,1);
X(6)  = sum(interv_fused_hq);            

X(7)  = min(interv_ECG_hq); %#ok<*AGROW>
X(8)  = max(interv_ECG_hq);
X(9)  = std(interv_ECG_hq);
X(10) = std(interv_ECG_hq)/mean(interv_ECG_hq);
X(11) = mad(interv_ECG_hq,1);
X(12) = sum(interv_ECG_hq);            

X(13) = min(interv_BP_hq); %#ok<*AGROW>
X(14) = max(interv_BP_hq);
X(15) = std(interv_BP_hq);
X(16) = std(interv_BP_hq)/mean(interv_BP_hq);
X(17) = mad(interv_BP_hq,1);
X(18) = sum(interv_BP_hq);

X(19) = mean(quality_fused);
X(20) = mean(quality_ECG);
X(21) = mean(quality_BP);

X(22) = median(quality_fused);
X(23) = median(quality_ECG);
X(24) = median(quality_BP);

X(25) = ecg_ratio_180;
X(26) = BP_ratio_180;

X(27) = mean(interv_fused_hq);
X(28) = mean(interv_ECG_hq);
X(29) = mean(interv_BP_hq);

X(30) = ecg_ratio_110;
X(31) = BP_ratio_110;

[max_corr, indx] = max(sum(corr_arr(1:120,:),2));
if indx==120
    max_corr = 0;
end
X(32) = max_corr;

[max_corr, indx] = max(max(corr_arr(1:120,:),[],2));
if indx==120
    max_corr = 0;
end
X(33) = max_corr;

alarmResult = do_tree(X,tree_struct);

% Write result to answers.txt
fid = fopen(answers, 'a');
if (fid == -1)
    error('Could not open answer file');
end

% Get base name of record (without directories)
i = strfind(recordName, filesep);
if (~isempty(i))
    basename = recordName(i(end)+1 : end);
else
    basename = recordName;
end

fprintf(fid, '%s,%d\n', basename, alarmResult);
fclose(fid);

end