function alarmResult=challenge(recordName,alarm_type)
%
%  alarmResult=challenge(recordName,alarm_type)
%
% Sample entry for the 2015 PhysioNet/CinC Challenge.
%
% Inputs:
%   recordName
%       String specifying the record name to process
%   alarmType
%       String specifying the alarm type. Alarm types are:
%             Asystole, Bradycardia, Tachycardia,
%             Ventricular_Tachycardia, Ventricular_Flutter_Fib
%
%
% Outputs:
%   alarmResult
%       Integer value where 0 = false alarm and 1 is a true
%       alarm.
%
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% Dependencies:
%
%       1) This function does not requires that you have the WFDB
%       App Toolbox installed.
%       A matlab function 'rdmat' can read the data instead of using WFDB
%       Toolbox.
%
%       2) The CHALLENGE function requires that you have downloaded the challenge
%       data 'set-p' in a subdirectory of the current directory. The subdirectory
%       should be called '/challenge/2015/set-p/' . The 'set-p' dataset can
%       be downloaded from PhysioNet at:
%           http://physionet.org/physiobank/database/challenge/2015/
%
%         This dataset is used by the generateValidationSet.m script to
%         create the annotations on your traing set that will be used to
%         verify that your entry works properly in the PhysioNet testing
%         environment.
%
% Version 0.5
%
%
% Written by Qiao Li, November 10, 2014.
% Last Modified: Ikaro Silva February 11, 2015
%
%
%
% %Example using training data-
% alarmResult=challenge('./challenge/set-p/100','Asystole')
%

% Name of file containing answers
answers = 'answers.txt';
debug=0;

%Get all ECG, blood pressure and photoplethysmogram signals
[aa,signal,Fs,siginfo]=rdmat(recordName);
description=squeeze(struct2cell(siginfo));
description=description(4,:);   % description = 'II'    'V'    'PLETH'    'RESP'

switch alarm_type
    case 'Asystole'
        Num_beat_min=25;
        alarmResult=1;
        Flag_A_True_ECG1=1;
        Flag_A_True_ECG2=1;
        Flag_A_True_ABP=1;
        Flag_A_True_PPG=1;
        Flag_A_True_ECG1_post=0;
        Flag_A_True_ECG2_post=0;
        Flag_A_True_ABP_post=0;
        Flag_A_True_PPG_post=0;
    case 'Bradycardia'
        Num_beat_min=19;
        alarmResult=1;
        Flag_B_True_ECG1=1;
        Flag_B_True_ECG2=1;
        Flag_B_True_ABP=1;
        Flag_B_True_PPG=1;
    case 'Tachycardia'
        Num_beat_min=30;
        alarmResult=1;
        Flag_T_True_ECG1=1;
        Flag_T_True_ECG2=1;
        Flag_T_True_ABP=1;
        Flag_T_True_PPG=1;
    case 'Ventricular_Tachycardia'
        Num_beat_min=30;
        alarmResult=0;
        Flag_VT_True_ECG1=0;
        Flag_VT_True_ECG2=0;
        Flag_VT_True_ABP=1;
        Flag_VT_True_PPG=1;
    case 'Ventricular_Flutter_Fib'
        Num_beat_min=15;
        alarmResult=1;
        Flag_VF_True_ECG1=1;
        Flag_VF_True_ECG2=1;
        Flag_VF_True_ABP=1;
        Flag_VF_True_PPG=1;
        Flag_VF_True_ECG1_post=0;
        Flag_VF_True_ECG2_post=0;
        Flag_VF_True_ABP_post=1;
        Flag_VF_True_PPG_post=1;
end

% Resample signal to 125Hz
Fs=Fs(1);
if Fs~=125
    signal=resample(signal,125,Fs);
    Fs=125;
end
Flag_post=0;
if size(signal,1)/Fs>300
    Flag_post=1;
end

T_A=8;
T_A_be=60;
T_A_post=30;
Num_beat_min_post=10;
Cons_tem_num=10;

start=(300-T_A_be)*Fs+1;  %
tail=300*Fs;
m=1;

%% read and process signals
% read and process ECG signal
ECG1=signal(:,1);
ECG2=signal(:,2);

ECG1_be=ECG1(start:tail);
ECG2_be=ECG2(start:tail);

[ECG1_be Flag_Exist_ECG1] = DetectNan(ECG1_be,T_A_be,Fs);
[ECG2_be Flag_Exist_ECG2] = DetectNan(ECG2_be,T_A_be,Fs);

if Flag_Exist_ECG1
    ECG1_be_fil=filtering(ECG1_be,5,45,Fs,1);
    ECG1_be_fil=polar2p(ECG1_be_fil,Fs,5);
    if strcmp(alarm_type,'Tachycardia')
        ECG1_R_peak_M1=FindPeak_T(ECG1_be_fil,Fs);
    elseif strcmp(alarm_type,'Ventricular_Tachycardia')
        ECG1_R_peak_M1=FindPeak_VT(ECG1_be_fil,Fs);
    elseif strcmp(alarm_type,'Ventricular_Flutter_Fib')
        ECG1_R_peak_M1=FindPeak_VF(ECG1_be_fil,Fs);
    else
        ECG1_R_peak_M1=FindPeak1(ECG1_be_fil,Fs);
    end
else
    ECG1_be_fil=ECG1_be;
    ECG1_R_peak_M1=[];
end

if Flag_Exist_ECG2
    ECG2_be_fil=filtering(ECG2_be,5,45,Fs,1);
    ECG2_be_fil=polar2p(ECG2_be_fil,Fs,5);
    if strcmp(alarm_type,'Tachycardia')
        ECG2_R_peak_M1=FindPeak_T(ECG2_be_fil,Fs);
    elseif strcmp(alarm_type,'Ventricular_Tachycardia')
        ECG2_R_peak_M1=FindPeak_VT(ECG2_be_fil,Fs);
    elseif strcmp(alarm_type,'Ventricular_Flutter_Fib')
        ECG2_R_peak_M1=FindPeak_VF(ECG2_be_fil,Fs);
    else
        ECG2_R_peak_M1=FindPeak1(ECG2_be_fil,Fs);
    end
else
    ECG2_be_fil=ECG2_be;
    ECG2_R_peak_M1=[];
end

Flag_DetAcc_ECG1=0;
Flag_DetAcc_ECG2=0;

if Flag_Exist_ECG1
    if length(ECG1_R_peak_M1)>Num_beat_min
        Flag_DetAcc_ECG1=1;
    end
end

if Flag_Exist_ECG2
    if length(ECG2_R_peak_M1)>Num_beat_min
        Flag_DetAcc_ECG2=1;
    end
end

if strcmp(alarm_type,'Ventricular_Tachycardia')
    if Flag_DetAcc_ECG1 && Flag_Exist_ECG1
        [Flag_DetAcc_ECG1,ECG1_template,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base]=DetAcc_VT(ECG1_be_fil,ECG1_R_peak_M1,Cons_tem_num,T_A_be,T_A,Fs);
    end
    if Flag_DetAcc_ECG2 && Flag_Exist_ECG2
        [Flag_DetAcc_ECG2,ECG2_template,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base]=DetAcc_VT(ECG2_be_fil,ECG2_R_peak_M1,Cons_tem_num,T_A_be,T_A,Fs);
    end
elseif strcmp(alarm_type,'Ventricular_Flutter_Fib')
    if Flag_DetAcc_ECG1 && Flag_Exist_ECG1
        [Flag_DetAcc_ECG1,ECG1_template,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base]=DetAcc_VF(ECG1_be_fil,ECG1_R_peak_M1,Cons_tem_num,T_A_be,T_A,Fs);
    end
    if Flag_DetAcc_ECG2 && Flag_Exist_ECG2
        [Flag_DetAcc_ECG2,ECG2_template,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base]=DetAcc_VF(ECG2_be_fil,ECG2_R_peak_M1,Cons_tem_num,T_A_be,T_A,Fs);
    end
elseif strcmp(alarm_type,'Tachycardia')
    if Flag_DetAcc_ECG1 && Flag_Exist_ECG1
        [Flag_DetAcc_ECG1,ECG1_template,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base]=DetAcc_T_ECG(ECG1_be_fil,ECG1_R_peak_M1,Cons_tem_num,Fs);
    end
    if Flag_DetAcc_ECG2 && Flag_Exist_ECG2
        [Flag_DetAcc_ECG2,ECG2_template,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base]=DetAcc_T_ECG(ECG2_be_fil,ECG2_R_peak_M1,Cons_tem_num,Fs);
    end
else
    if Flag_DetAcc_ECG1 && Flag_Exist_ECG1
        [Flag_DetAcc_ECG1,ECG1_template,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base]=DetAcc(ECG1_be_fil,ECG1_R_peak_M1,Cons_tem_num,Fs);
    end
    if Flag_DetAcc_ECG2 && Flag_Exist_ECG2
        [Flag_DetAcc_ECG2,ECG2_template,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base]=DetAcc(ECG2_be_fil,ECG2_R_peak_M1,Cons_tem_num,Fs);
    end
end

if strcmp(alarm_type,'Ventricular_Tachycardia')
    if Flag_DetAcc_ECG1 && Flag_DetAcc_ECG2
        if ECG1_HR_base/ECG2_HR_base > 3/2 || ECG1_HR_base/ECG2_HR_base < 2/3
            hr_diff=[abs(ECG1_HR_base-100) abs(ECG2_HR_base-100)];
            if hr_diff(1)<hr_diff(2)
                Flag_DetAcc_ECG2=0;
            else
                Flag_DetAcc_ECG1=0;
            end
        end
    end
else
    if Flag_DetAcc_ECG1 && Flag_DetAcc_ECG2
        if ECG1_HR_base/ECG2_HR_base > 3/2 || ECG1_HR_base/ECG2_HR_base < 2/3
            hr_diff=[abs(ECG1_HR_base-75) abs(ECG2_HR_base-75)];
            if hr_diff(1)<hr_diff(2)
                Flag_DetAcc_ECG2=0;
            else
                Flag_DetAcc_ECG1=0;
            end
        end
    end
end
    

switch alarm_type
    case 'Asystole'
        if Flag_DetAcc_ECG1
            Flag_A_True_ECG1(m)=Determine_A(ECG1_be_fil,ECG1_R_peak_M1,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base,T_A_be,T_A,Fs);
        end
        if Flag_DetAcc_ECG2
            Flag_A_True_ECG2(m)=Determine_A(ECG2_be_fil,ECG2_R_peak_M1,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base,T_A_be,T_A,Fs);
        end
    case 'Bradycardia'
        if Flag_DetAcc_ECG1
            [Flag_B_True_ECG1(m),ECG1_B_HR]=Determine_B(ECG1_be_fil,ECG1_R_peak_M1,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base,T_A_be,T_A,Fs);
        end
        if Flag_DetAcc_ECG2
            [Flag_B_True_ECG2(m),ECG2_B_HR]=Determine_B(ECG2_be_fil,ECG2_R_peak_M1,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base,T_A_be,T_A,Fs);
        end
    case 'Tachycardia'
        if Flag_DetAcc_ECG1
            Flag_T_True_ECG1(m)=Determine_T_ECG(ECG1_be_fil,ECG1_R_peak_M1,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base,T_A_be,T_A,Fs);
        end
        if Flag_DetAcc_ECG2
            Flag_T_True_ECG2(m)=Determine_T_ECG(ECG2_be_fil,ECG2_R_peak_M1,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base,T_A_be,T_A,Fs);
        end
    case 'Ventricular_Tachycardia'
        ECG1_Corr0=0;
        ECG1_Corr=[];
        ECG1_Shape_change=[];
        ECG1_VT_HR=[];
        if Flag_DetAcc_ECG1
            [Flag_VT_True_ECG1(m),ECG1_Corr0,ECG1_Corr,ECG1_Shape_change,ECG1_VT_HR]=Determine_VT_ECG(ECG1_be_fil,ECG1_R_peak_M1,ECG1_template,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base,T_A_be,T_A,Fs);
        end
        ECG2_Corr0=0;
        ECG2_Corr=[];
        ECG2_Shape_change=[];
        ECG2_VT_HR=[];
        if Flag_DetAcc_ECG2
            [Flag_VT_True_ECG2(m),ECG2_Corr0,ECG2_Corr,ECG2_Shape_change,ECG2_VT_HR]=Determine_VT_ECG(ECG2_be_fil,ECG2_R_peak_M1,ECG2_template,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base,T_A_be,T_A,Fs);
        end
    case 'Ventricular_Flutter_Fib'
        ECG1_Corr0=0;
        if Flag_DetAcc_ECG1
            [Flag_VF_True_ECG1(m),ECG1_Corr0,ECG1_Shape_change,ECG1_VF_HR]=Determine_VF_ECG(ECG1_be_fil,ECG1_R_peak_M1,ECG1_template,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base,T_A_be,T_A,Fs);
        end
        ECG2_Corr0=0;
        if Flag_DetAcc_ECG2
            [Flag_VF_True_ECG2(m),ECG2_Corr0,ECG2_Shape_change,ECG2_VF_HR]=Determine_VF_ECG(ECG2_be_fil,ECG2_R_peak_M1,ECG2_template,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base,T_A_be,T_A,Fs);
        end
end

data_be=[ECG1_be ECG2_be];
data_be_fil=[ECG1_be_fil ECG2_be_fil];
data_channel={'ECG1';'ECG2'};

 if Flag_post
        ECG1_post=ECG1(tail+1:end);
        ECG2_post=ECG2(tail+1:end);
        
        [ECG1_post Flag_Exist_ECG1_post] = DetectNan(ECG1_post,T_A_post,Fs);
        [ECG2_post Flag_Exist_ECG2_post] = DetectNan(ECG2_post,T_A_post,Fs);
   
         if Flag_Exist_ECG1_post
            ECG1_post_fil=filtering(ECG1_post,5,45,Fs,1);
            ECG1_post_fil=polar2p(ECG1_post_fil,Fs,5);
            if strcmp(alarm_type,'Tachycardia')
                ECG1_R_peak_post=FindPeak_T(ECG1_post_fil,Fs);
            elseif strcmp(alarm_type,'Ventricular_Tachycardia')
                ECG1_R_peak_post=FindPeak_VT(ECG1_post_fil,Fs);
            elseif strcmp(alarm_type,'Ventricular_Flutter_Fib')
                ECG1_R_peak_post=FindPeak_VF(ECG1_post_fil,Fs);
            else
                ECG1_R_peak_post=FindPeak1(ECG1_post_fil,Fs);
            end
        else
            ECG1_post_fil=ECG1_post;
            ECG1_R_peak_post=[];
        end
        
        if Flag_Exist_ECG2_post
            ECG2_post_fil=filtering(ECG2_post,5,45,Fs,1);
            ECG2_post_fil=polar2p(ECG2_post_fil,Fs,5);
            if strcmp(alarm_type,'Tachycardia')
                ECG2_R_peak_post=FindPeak_T(ECG2_post_fil,Fs);
            elseif strcmp(alarm_type,'Ventricular_Tachycardia')
                ECG2_R_peak_post=FindPeak_VT(ECG2_post_fil,Fs);
            elseif strcmp(alarm_type,'Ventricular_Flutter_Fib')
                ECG2_R_peak_post=FindPeak_VF(ECG2_post_fil,Fs);
            else
                ECG2_R_peak_post=FindPeak1(ECG2_post_fil,Fs);
            end
        else
            ECG2_post_fil=ECG2_post;
            ECG2_R_peak_post=[];
        end
        
        Flag_DetAcc_ECG1_post=0;
        Flag_DetAcc_ECG2_post=0;
        
        if Flag_Exist_ECG1_post
            if length(ECG1_R_peak_post)>Num_beat_min_post
                Flag_DetAcc_ECG1_post=1;
            end
        end
        
        if Flag_Exist_ECG2_post
            if length(ECG2_R_peak_post)>Num_beat_min_post
                Flag_DetAcc_ECG2_post=1;
            end
        end
        
        switch alarm_type
            case 'Asystole'
                if Flag_DetAcc_ECG1_post && Flag_Exist_ECG1_post
                    [Flag_DetAcc_ECG1_post,ECG1_template_post,ECG1_HR_base_post,ECG1_Range_base_post,ECG1_Amp_base_post]=DetAcc_post(ECG1_post_fil,ECG1_R_peak_post,Cons_tem_num,Fs);
                end
                if Flag_DetAcc_ECG2_post && Flag_Exist_ECG2_post
                    [Flag_DetAcc_ECG2_post,ECG2_template_post,ECG2_HR_base_post,ECG2_Range_base_post,ECG2_Amp_base_post]=DetAcc_post(ECG2_post_fil,ECG2_R_peak_post,Cons_tem_num,Fs);
                end
                
                if Flag_DetAcc_ECG1_post && Flag_DetAcc_ECG1                    
                    Flag_A_True_ECG1_post(m)=Determine_A_post(ECG1_HR_base,ECG1_Range_base,ECG1_HR_base_post,ECG1_Range_base_post);              
                end
                if Flag_DetAcc_ECG2_post && Flag_DetAcc_ECG2                    
                    Flag_A_True_ECG2_post(m)=Determine_A_post(ECG2_HR_base,ECG2_Range_base,ECG2_HR_base_post,ECG2_Range_base_post);
                end
                
            case 'Bradycardia'
%                 if Flag_DetAcc_ECG1_post && Flag_Exist_ECG1_post
%                     [Flag_DetAcc_ECG1_post,ECG1_template_post,ECG1_HR_base_post,ECG1_Range_base_post,ECG1_Amp_base_post]=DetAcc_post(ECG1_post_fil,ECG1_R_peak_post,Cons_tem_num,Fs);
%                 end
%                 if Flag_DetAcc_ECG2_post && Flag_Exist_ECG2_post
%                     [Flag_DetAcc_ECG2_post,ECG2_template_post,ECG2_HR_base_post,ECG2_Range_base_post,ECG2_Amp_base_post]=DetAcc_post(ECG2_post_fil,ECG2_R_peak_post,Cons_tem_num,Fs);
%                 end
%                 
%                 if Flag_DetAcc_ECG1_post && Flag_DetAcc_ECG1
%                     Flag_B_True_ECG1_post(m)=Determine_B_post(ECG1_post_fil,ECG1_R_peak_post,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base,Fs);
%                 end
%                 if Flag_DetAcc_ECG2_post && Flag_DetAcc_ECG2
%                     Flag_B_True_ECG2_post(m)=Determine_B_post(ECG2_post_fil,ECG2_R_peak_post,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base,Fs);
%                 end
                
            case 'Tachycardia'
%                 if Flag_DetAcc_ECG1
%                     Flag_T_True_ECG1(m)=Determine_T_ECG(ECG1_be_fil,ECG1_R_peak_M1,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base,T_A_be,T_A,Fs);
%                 end
%                 if Flag_DetAcc_ECG2
%                     Flag_T_True_ECG2(m)=Determine_T_ECG(ECG2_be_fil,ECG2_R_peak_M1,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base,T_A_be,T_A,Fs);
%                 end
            case 'Ventricular_Tachycardia'
%                 if Flag_DetAcc_ECG1_post && Flag_Exist_ECG1_post
%                     [Flag_DetAcc_ECG1_post,ECG1_template_post,ECG1_HR_base_post,ECG1_Range_base_post,ECG1_Amp_base_post]=DetAcc_VT_post(ECG1_post_fil,ECG1_R_peak_post,Cons_tem_num,Fs);
%                 end
%                 if Flag_DetAcc_ECG2_post && Flag_Exist_ECG2_post
%                     [Flag_DetAcc_ECG2_post,ECG2_template_post,ECG2_HR_base_post,ECG2_Range_base_post,ECG2_Amp_base_post]=DetAcc_VT_post(ECG2_post_fil,ECG2_R_peak_post,Cons_tem_num,Fs);
%                 end
%                 ECG1_Corr0_post=0;
%                 if Flag_DetAcc_ECG1_post && Flag_DetAcc_ECG1
%                     [Flag_VT_True_ECG1_post(m),ECG1_Corr0_post,ECG1_VT_HR_post]=Determine_VT_ECG_post(ECG1_post_fil,ECG1_R_peak_post,ECG1_template,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base,Fs);
%                 end
%                 ECG2_Corr0_post=0;
%                 if Flag_DetAcc_ECG2_post && Flag_DetAcc_ECG2
%                     [Flag_VT_True_ECG2_post(m),ECG2_Corr0_post,ECG2_VT_HR_post]=Determine_VT_ECG_post(ECG2_post_fil,ECG2_R_peak_post,ECG2_template,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base,Fs);
%                 end
            case 'Ventricular_Flutter_Fib'
                if Flag_DetAcc_ECG1_post && Flag_Exist_ECG1_post
                    [Flag_DetAcc_ECG1_post,ECG1_template_post,ECG1_HR_base_post,ECG1_Range_base_post,ECG1_Amp_base_post]=DetAcc_VF_post(ECG1_post_fil,ECG1_R_peak_post,Cons_tem_num,Fs);
                end
                if Flag_DetAcc_ECG2_post && Flag_Exist_ECG2_post
                    [Flag_DetAcc_ECG2_post,ECG2_template_post,ECG2_HR_base_post,ECG2_Range_base_post,ECG2_Amp_base_post]=DetAcc_VF_post(ECG2_post_fil,ECG2_R_peak_post,Cons_tem_num,Fs);
                end
                ECG1_Corr0_post=0;
                if Flag_DetAcc_ECG1_post && Flag_DetAcc_ECG1
                    [Flag_VF_True_ECG1_post(m),ECG1_Corr0_post,ECG1_Shape_change_post,ECG1_VF_HR_post]=Determine_VF_ECG_post(ECG1_post_fil,ECG1_R_peak_post,ECG1_template,ECG1_HR_base,ECG1_Range_base,ECG1_Amp_base,Fs);
                end
                ECG2_Corr0_post=0;
                if Flag_DetAcc_ECG2_post && Flag_DetAcc_ECG2
                    [Flag_VF_True_ECG2_post(m),ECG2_Corr0_post,ECG2_Shape_change_post,ECG2_VF_HR_post]=Determine_VF_ECG_post(ECG2_post_fil,ECG2_R_peak_post,ECG2_template,ECG2_HR_base,ECG2_Range_base,ECG2_Amp_base,Fs);
                end
        end          
        
        data_post=[ECG1_post ECG2_post];
        data_post_fil=[ECG1_post_fil ECG2_post_fil];
    end


% read and process ABP signal if exist
ABP_ind=get_index(description,'ABP');
if(~isempty(ABP_ind))
    ABP_be=signal(start:tail,ABP_ind);
    ABP_be_fil=filtering(ABP_be,5,35,Fs,1);
    ABP_foot=wabp(ABP_be_fil,0,1);
    
    if strcmp(alarm_type,'Bradycardia') || strcmp(alarm_type,'Tachycardia') || strcmp(alarm_type,'Ventricular_Flutter_Fib') || strcmp(alarm_type,'Ventricular_Tachycardia')
        abp=signal(:,ABP_ind);
        ann_abp=wabp(signal(:,ABP_ind),0,1);
        if length(ann_abp)>=3 % at least 3 abp beats detected
            [features] = abpfeature(signal(:,ABP_ind),ann_abp);
            [BEATQ R] = jSQI(features, ann_abp, signal(:,ABP_ind));
        end
        
        N_d=Fs*5*60; % alarm position
        N0_d=N_d-Fs*16+1; % 16s before the alarm
        % select the beats in the segment
        n_abp_beats=intersect(find(ann_abp>=N0_d),find(ann_abp<=N_d));
        
        if length(ann_abp)>=3 && ~isempty(BEATQ)
            abpsqi=1-sum(sum(BEATQ(intersect(n_abp_beats,1:length(BEATQ)),:)))/numel(BEATQ(intersect(n_abp_beats,1:length(BEATQ)),:));
        else
            abpsqi=0;
        end
        
        if length(n_abp_beats)>=2
            hr_max_abp=60*Fs/min(diff(ann_abp(n_abp_beats)));
        else
            hr_max_abp=0;
        end
        
        % calculate low heart rate of 5 consecutive beats for Bradycardia
        low_hr_abp=NaN;
        if length(n_abp_beats>=5)
            for i=1:length(n_abp_beats)-4
                low_hr_abp(i)=60*Fs/((ann_abp(n_abp_beats(i+4))-ann_abp(n_abp_beats(i)))/4);
            end
        end
        low_hr_abp=min(low_hr_abp);
        % calculate high heart reate of 17 consecutive beats for Tachycardia
        high_hr_abp=NaN;
        if length(n_abp_beats>=17)
            for i=1:length(n_abp_beats)-16
                high_hr_abp(i)=60*Fs/((ann_abp(n_abp_beats(i+16))-ann_abp(n_abp_beats(i)))/16);
            end
        end
        high_hr_abp=max(high_hr_abp);
    end
        
    Flag_DetAcc_ABP=0;
    if length(ABP_foot)>Num_beat_min
        Flag_DetAcc_ABP=1;
    end
    if Flag_DetAcc_ABP
        if strcmp(alarm_type,'Ventricular_Tachycardia')
            [Flag_DetAcc_ABP,ABP_template,ABP_HR_base,ABP_Range_base,ABP_Amp_base]=DetAcc_VT(ABP_be_fil,ABP_foot,Cons_tem_num,T_A_be,T_A,Fs);
        elseif strcmp(alarm_type,'Ventricular_Flutter_Fib')
            [Flag_DetAcc_ABP,ABP_template,ABP_HR_base,ABP_Range_base,ABP_Amp_base]=DetAcc_VF(ABP_be_fil,ABP_foot,Cons_tem_num,T_A_be,T_A,Fs);
        elseif strcmp(alarm_type,'Tachycardia')
            [Flag_DetAcc_ABP,ABP_template,ABP_HR_base,ABP_Range_base,ABP_Amp_base]=DetAcc_T_Pulse(ABP_be_fil,ABP_foot,Cons_tem_num,Fs);
        else
            [Flag_DetAcc_ABP,ABP_template,ABP_HR_base,ABP_Range_base,ABP_Amp_base]=DetAcc_pulse(ABP_be_fil,ABP_foot,Cons_tem_num,Fs);
        end
    end
    
    if Flag_DetAcc_ECG1 && Flag_DetAcc_ABP
        if ECG1_HR_base/ABP_HR_base > 3/2 || ECG1_HR_base/ABP_HR_base < 2/3
            hr_diff=[abs(ECG1_HR_base-75) abs(ABP_HR_base-75)];
            if hr_diff(1)<hr_diff(2)
                Flag_DetAcc_ABP=0;
            end
        end
    end
    
    if Flag_DetAcc_ECG2 && Flag_DetAcc_ABP
        if ECG2_HR_base/ABP_HR_base > 3/2 || ECG2_HR_base/ABP_HR_base < 2/3
            hr_diff=[abs(ECG2_HR_base-75) abs(ABP_HR_base-75)];
            if hr_diff(1)<hr_diff(2)
                Flag_DetAcc_ABP=0;
            end
        end
    end
    
    if Flag_DetAcc_ABP
        switch alarm_type
            case 'Asystole'
                Flag_A_True_ABP(m)=Determine_A(ABP_be_fil,ABP_foot,ABP_HR_base,ABP_Range_base,ABP_Amp_base,T_A_be,T_A,Fs);
            case 'Bradycardia'
                Flag_B_True_ABP(m)=Determine_B_Pulse(ABP_be_fil,ABP_foot,ABP_HR_base,ABP_Range_base,ABP_Amp_base,T_A_be,T_A,abpsqi,Fs);
            case 'Tachycardia'
                Flag_T_True_ABP(m)=Determine_T_Pulse(ABP_be_fil,ABP_foot,ABP_HR_base,ABP_Range_base,ABP_Amp_base,T_A_be,T_A,Fs);
            case 'Ventricular_Tachycardia'
                [Flag_VT_True_ABP(m),ABP_Corr0]=Determine_VT_Pulse(ABP_be_fil,ABP_foot,ABP_template,ABP_HR_base,ABP_Range_base,ABP_Amp_base,T_A_be,T_A,Fs);
            case 'Ventricular_Flutter_Fib'
                [Flag_VF_True_ABP(m),ABP_Corr0]=Determine_VF_Pulse(ABP_be_fil,ABP_foot,ABP_template,ABP_HR_base,ABP_Range_base,ABP_Amp_base,T_A_be,T_A,Fs);
        end
    end
    
    data_be=[data_be ABP_be];
    data_be_fil=[data_be_fil ABP_be_fil];
    data_channel=[data_channel;'ABP'];
    
    if Flag_post
            ABP_post=signal(tail+1:end,ABP_ind);
            ABP_post_fil=filtering(ABP_post,5,35,Fs,1);
            ABP_foot_post=wabp(ABP_post_fil,0,1);

            Flag_DetAcc_ABP_post=0;
            if length(ABP_foot_post)>Num_beat_min_post
                Flag_DetAcc_ABP_post=1;
            end
            if Flag_DetAcc_ABP_post
               switch alarm_type
                    case 'Asystole'
                        [Flag_DetAcc_ABP_post,ABP_template_post,ABP_HR_base_post,ABP_Range_base_post,ABP_Amp_base_post]=DetAcc_post(ABP_post_fil,ABP_foot_post,Cons_tem_num,Fs);
                        Flag_A_True_ABP_post(m)=Determine_A_post(ABP_HR_base,ABP_Range_base,ABP_HR_base_post,ABP_Range_base_post);
                        
                    case 'Bradycardia'
%                         [Flag_DetAcc_ABP_post,ABP_template_post,ABP_HR_base_post,ABP_Range_base_post,ABP_Amp_base_post]=DetAcc_post(ABP_post_fil,ABP_foot_post,Cons_tem_num,Fs);
%                         Flag_B_True_ABP_post(m)=Determine_B_post(ABP_post_fil,ABP_foot_post,ABP_HR_base,ABP_Range_base,ABP_Amp_base,Fs);
                        
                    case 'Tachycardia'
%                         Flag_T_True_ABP(m)=Determine_T_Pulse(ABP_be_fil,ABP_foot,ABP_HR_base,ABP_Range_base,ABP_Amp_base,T_A_be,T_A,Fs);
                    case 'Ventricular_Tachycardia'
%                         [Flag_DetAcc_ABP_post,ABP_template_post,ABP_HR_base_post,ABP_Range_base_post,ABP_Amp_base_post]=DetAcc_VT_post(ABP_post_fil,ABP_foot_post,Cons_tem_num,Fs);
%                         [Flag_VT_True_ABP_post(m),ABP_Corr0_post]=Determine_VT_Pulse_post(ABP_post_fil,ABP_foot_post,ABP_template,ABP_HR_base,ABP_Range_base,ABP_Amp_base,Fs);
                    case 'Ventricular_Flutter_Fib'
                        [Flag_DetAcc_ABP_post,ABP_template_post,ABP_HR_base_post,ABP_Range_base_post,ABP_Amp_base_post]=DetAcc_VF_post(ABP_post_fil,ABP_foot_post,Cons_tem_num,Fs);
                        [Flag_VF_True_ABP_post(m),ABP_Corr0_post]=Determine_VF_Pulse_post(ABP_post_fil,ABP_foot_post,ABP_template,ABP_HR_base,ABP_Range_base,ABP_Amp_base,Fs);
                end
            end
            
            data_post=[data_post ABP_post];
            data_post_fil=[data_post_fil ABP_post_fil];
        end
end

% read and process PPG signal if exist
PPG_ind=get_index(description,'PLETH');
if (~isempty(PPG_ind))
    PPG_be=signal(start:tail,PPG_ind);
    PPG_be_fil=filtering(PPG_be,5,35,Fs,1);
    y=quantile(PPG_be_fil,[0.05,0.5,0.95]);  % returns quantiles of the values in signal(:,PPG_ind)
    PPG_foot=wabp(PPG_be_fil,0,(y(3)-y(1))/120);
    
    if strcmp(alarm_type,'Bradycardia')   || strcmp(alarm_type,'Tachycardia')  || strcmp(alarm_type,'Ventricular_Flutter_Fib')  ||  strcmp(alarm_type,'Ventricular_Tachycardia')
        ppg=signal(:,PPG_ind);
        y=quantile(signal(:,PPG_ind),[0.05,0.5,0.95]);  % returns quantiles of the values in signal(:,ppg_ind)
        ann_ppg=wabp(signal(:,PPG_ind),0,(y(3)-y(1))/120);
        if ~isempty(ann_ppg)
            [psqi]=ppgSQI(signal(:,PPG_ind),ann_ppg);
        end
        N_d=Fs*5*60; % alarm position
        N0_d=N_d-Fs*16+1; % 16s before the alarm
        n_ppg_beats=intersect(find(ann_ppg>=N0_d),find(ann_ppg<=N_d));
        
        if ~isempty(ann_ppg)
            ppgsqi=mean(psqi(intersect(n_ppg_beats,1:length(psqi))));  % intersect: get the intersection of two sets
        else
            ppgsqi=0;
        end
        
        if length(n_ppg_beats)>=2
            hr_max_ppg=60*Fs/min(diff(ann_ppg(n_ppg_beats)));
        else
            hr_max_ppg=0;
        end
        
        % calculate low heart rate of 5 consecutive beats for Bradycardia
        low_hr_ppg=NaN;
        if length(n_ppg_beats>=5)
            for i=1:length(n_ppg_beats)-4
                low_hr_ppg(i)=60*Fs/((ann_ppg(n_ppg_beats(i+4))-ann_ppg(n_ppg_beats(i)))/4);
            end
        end
        low_hr_ppg=min(low_hr_ppg);
        % calculate high heart reate of 17 consecutive beats for Tachycardia
        high_hr_ppg=NaN;
        if length(n_ppg_beats>=17)
            for i=1:length(n_ppg_beats)-16
                high_hr_ppg(i)=60*Fs/((ann_ppg(n_ppg_beats(i+16))-ann_ppg(n_ppg_beats(i)))/16);
            end
        end
        high_hr_ppg=max(high_hr_ppg);
    end
    
    Flag_DetAcc_PPG=0;
    if length(PPG_foot)>Num_beat_min
        Flag_DetAcc_PPG=1;
    end
    if Flag_DetAcc_PPG
        if strcmp(alarm_type,'Ventricular_Tachycardia')
            [Flag_DetAcc_PPG,PPG_template,PPG_HR_base,PPG_Range_base,PPG_Amp_base]=DetAcc_VT(PPG_be_fil,PPG_foot,Cons_tem_num,T_A_be,T_A,Fs);
        elseif strcmp(alarm_type,'Ventricular_Flutter_Fib')
            [Flag_DetAcc_PPG,PPG_template,PPG_HR_base,PPG_Range_base,PPG_Amp_base]=DetAcc_VF(PPG_be_fil,PPG_foot,Cons_tem_num,T_A_be,T_A,Fs);
        elseif strcmp(alarm_type,'Tachycardia')
            [Flag_DetAcc_PPG,PPG_template,PPG_HR_base,PPG_Range_base,PPG_Amp_base]=DetAcc_T_Pulse(PPG_be_fil,PPG_foot,Cons_tem_num,Fs);
        else
            [Flag_DetAcc_PPG,PPG_template,PPG_HR_base,PPG_Range_base,PPG_Amp_base]=DetAcc_pulse(PPG_be_fil,PPG_foot,Cons_tem_num,Fs);
        end
    end
    
    if Flag_DetAcc_ECG1 && Flag_DetAcc_PPG
        if ECG1_HR_base/PPG_HR_base > 3/2 || ECG1_HR_base/PPG_HR_base < 2/3
            hr_diff=[abs(ECG1_HR_base-75) abs(PPG_HR_base-75)];
            if hr_diff(1)<hr_diff(2)
                Flag_DetAcc_PPG=0;
            end
        end
    end
    
    if Flag_DetAcc_ECG2 && Flag_DetAcc_PPG
        if ECG2_HR_base/PPG_HR_base > 3/2 || ECG2_HR_base/PPG_HR_base < 2/3
            hr_diff=[abs(ECG2_HR_base-75) abs(PPG_HR_base-75)];
            if hr_diff(1)<hr_diff(2)
                Flag_DetAcc_PPG=0;
            end
        end
    end
    
    if Flag_DetAcc_PPG
        switch alarm_type
            case 'Asystole'
                Flag_A_True_PPG(m)=Determine_A(PPG_be_fil,PPG_foot,PPG_HR_base,PPG_Range_base,PPG_Amp_base,T_A_be,T_A,Fs);
            case 'Bradycardia'
                Flag_B_True_PPG(m)=Determine_B_Pulse(PPG_be_fil,PPG_foot,PPG_HR_base,PPG_Range_base,PPG_Amp_base,T_A_be,T_A,ppgsqi,Fs);
            case 'Tachycardia'
                Flag_T_True_PPG(m)=Determine_T_Pulse(PPG_be_fil,PPG_foot,PPG_HR_base,PPG_Range_base,PPG_Amp_base,T_A_be,T_A,Fs);
            case 'Ventricular_Tachycardia'
                [Flag_VT_True_PPG(m),PPG_Corr0]=Determine_VT_Pulse(PPG_be_fil,PPG_foot,PPG_template,PPG_HR_base,PPG_Range_base,PPG_Amp_base,T_A_be,T_A,Fs);
            case 'Ventricular_Flutter_Fib'
                [Flag_VF_True_PPG(m),PPG_Corr0]=Determine_VF_Pulse(PPG_be_fil,PPG_foot,PPG_template,PPG_HR_base,PPG_Range_base,PPG_Amp_base,T_A_be,T_A,Fs);
        end
    end
    
    data_be=[data_be PPG_be];
    data_be_fil=[data_be_fil PPG_be_fil];
    data_channel=[data_channel;'PPG'];
    
    if Flag_post
        PPG_post=signal(tail+1:end,PPG_ind);
        PPG_post_fil=filtering(PPG_post,5,35,Fs,1);
        y=quantile(PPG_post_fil,[0.05,0.5,0.95]);  % returns quantiles of the values in signal(:,PPG_ind)
        PPG_foot_post=wabp(PPG_post_fil,0,(y(3)-y(1))/120);
        
        
        Flag_DetAcc_PPG_post=0;
        if length(PPG_foot_post)>Num_beat_min_post
            Flag_DetAcc_PPG_post=1;
        end
        if Flag_DetAcc_PPG_post
            switch alarm_type
                case 'Asystole'
                    [Flag_DetAcc_PPG_post,PPG_template_post,PPG_HR_base_post,PPG_Range_base_post,PPG_Amp_base_post]=DetAcc_post(PPG_post_fil,PPG_foot_post,Cons_tem_num,Fs);
                    Flag_A_True_PPG_post(m)=Determine_A_post(PPG_HR_base,PPG_Range_base,PPG_HR_base_post,PPG_Range_base_post);
                case 'Bradycardia'
%                     [Flag_DetAcc_PPG_post,PPG_template_post,PPG_HR_base_post,PPG_Range_base_post,PPG_Amp_base_post]=DetAcc_post(PPG_post_fil,PPG_foot_post,Cons_tem_num,Fs);
%                     Flag_B_True_PPG_post(m)=Determine_B_post(PPG_post_fil,PPG_foot_post,PPG_HR_base,PPG_Range_base,PPG_Amp_base,Fs);
                case 'Tachycardia'
                    %                         Flag_T_True_PPG(m)=Determine_T_Pulse(PPG_be_fil,PPG_foot,PPG_HR_base,PPG_Range_base,PPG_Amp_base,T_A_be,T_A,Fs);
                case 'Ventricular_Tachycardia'
%                     [Flag_DetAcc_PPG_post,PPG_template_post,PPG_HR_base_post,PPG_Range_base_post,PPG_Amp_base_post]=DetAcc_VT_post(PPG_post_fil,PPG_foot_post,Cons_tem_num,Fs);
%                     [Flag_VT_True_PPG_post(m),PPG_Corr0_post]=Determine_VT_Pulse_post(PPG_post_fil,PPG_foot_post,PPG_template,PPG_HR_base,PPG_Range_base,PPG_Amp_base,Fs);
                case 'Ventricular_Flutter_Fib'
                    [Flag_DetAcc_PPG_post,PPG_template_post,PPG_HR_base_post,PPG_Range_base_post,PPG_Amp_base_post]=DetAcc_VF_post(PPG_post_fil,PPG_foot_post,Cons_tem_num,Fs);
                    [Flag_VF_True_PPG_post(m),PPG_Corr0_post]=Determine_VF_Pulse_post(PPG_post_fil,PPG_foot_post,PPG_template,PPG_HR_base,PPG_Range_base,PPG_Amp_base,Fs);
            end
        end
        
        data_post=[data_post PPG_post];
        data_post_fil=[data_post_fil PPG_post_fil];
    end
end


% %% special processing for VF
% if strcmp(alarm_type,'Ventricular_Flutter_Fib')
%     Segment_t=10;
%     Segment_num=size(signal,1)/Fs/Segment_t;
%     SampEn1_0=zeros(Segment_num,1);
%     SampEn2_0=zeros(Segment_num,1);
%     Sd_ECG1_0=ones(Segment_num,1)*1000;
%     Sd_ECG2_0=ones(Segment_num,1)*1000;
%     Sd_abp_0=ones(Segment_num,1)*1000;
%     Sd_ppg_0=ones(Segment_num,1)*1000;
%     Mean_ECG1_0=zeros(Segment_num,1);
%     Mean_ECG2_0=zeros(Segment_num,1);
%     PeakLoca1=[];
%     PeakLoca2=[];
%     PeakLoca1_cur=[];
%     PeakLoca2_cur=[];
%     cur_PeakLoca1=[];
%     cur_PeakLoca2=[];
%     cur_PeakLoca11=[];
%     cur_PeakLoca22=[];
%     Indx1=[];
%     Indx2=[];
%     SampEn_thr1=0.05;
%     SampEn_thr2=1;
%     for i=1:Segment_num
%         SampEn1_0(i) = SampEn_fast(signal(Fs*Segment_t*(i-1)+1:Fs*Segment_t*i,1), 2, 0.2);
%         if SampEn1_0(i)>SampEn_thr1 & SampEn1_0(i)<SampEn_thr2
%             cur_ECG10=resample(signal(Fs*Segment_t*(i-1)+1:Fs*Segment_t*i,1),1000,Fs);
%             cur_ECG10 = wden(cur_ECG10, 'rigrsure', 's', 'mln', 3, 'coif5');
%             [C, L] = wavedec(cur_ECG10, 6, 'coif5'); % about 8 Hz
%             tretp  = upcoef('a', C(1:L(1)), 'coif5', 6);
%             indf   = 1 + floor((length(tretp) - length(cur_ECG10))/2) - floor(1*length(wfilters('coif5')));
%             cur_ECG10_tre = tretp(indf:indf + length(cur_ECG10) - 1);
%             cur_ECG1 = cur_ECG10 - cur_ECG10_tre;
%             cur_ECG1_sort=sort(cur_ECG1);
%             m_high=mean(cur_ECG1_sort(end-Fs+1:end))-mean(cur_ECG1);
%             m_low=mean(cur_ECG1)-mean(cur_ECG1_sort(1:Fs));
%             if 1.5*m_high<m_low
%                 cur_ECG1=-cur_ECG1;
%             end
%             PeakLoca1_cur = round(FindPeak0(cur_ECG1,1000)*Fs/1000);
%             Sd_ECG1_0(i)=std(diff(PeakLoca1_cur));
%             Mean_ECG1_0(i)=mean(diff(PeakLoca1_cur));
%             PeakLoca1=[PeakLoca1;PeakLoca1_cur+Fs*Segment_t*(i-1)+1];
%             Indx1=[Indx1;i];
%         end
%         
%         SampEn2_0(i) = SampEn_fast(signal(Fs*Segment_t*(i-1)+1:Fs*Segment_t*i,2), 2, 0.2);
%         if SampEn2_0(i)>SampEn_thr1 & SampEn2_0(i)<SampEn_thr2
%             cur_ECG20=resample(signal(Fs*Segment_t*(i-1)+1:Fs*Segment_t*i,2),1000,Fs);
%             cur_ECG20 = wden(cur_ECG20, 'rigrsure', 's', 'mln', 3, 'coif5');
%             [C, L] = wavedec(cur_ECG20, 6, 'coif5'); % about 8 Hz
%             tretp  = upcoef('a', C(1:L(1)), 'coif5', 6);
%             indf   = 1 + floor((length(tretp) - length(cur_ECG20))/2) - floor(1*length(wfilters('coif5')));
%             cur_ECG20_tre = tretp(indf:indf + length(cur_ECG20) - 1);
%             cur_ECG2 = cur_ECG20 - cur_ECG20_tre;
%             cur_ECG2_sort=sort(cur_ECG2);
%             m_high=mean(cur_ECG2_sort(end-Fs+1:end))-mean(cur_ECG2);
%             m_low=mean(cur_ECG2)-mean(cur_ECG2_sort(1:Fs));
%             if 1.5*m_high<m_low
%                 cur_ECG2=-cur_ECG2;
%             end
%             PeakLoca2_cur = round(FindPeak0(cur_ECG2,1000)*Fs/1000);
%             Sd_ECG2_0(i)=std(diff(PeakLoca2_cur));
%             Mean_ECG2_0(i)=mean(diff(PeakLoca2_cur));
%             PeakLoca2=[PeakLoca2;PeakLoca2_cur+Fs*Segment_t*(i-1)+1];
%             Indx2=[Indx2;i];
%         end
%         SampEn1=round(mean(SampEn1_0)*100)/100;
%         SampEn2=round(mean(SampEn2_0)*100)/100;
%     end
%     
%     Flag_falsedet_thr=0.1;
%     Amp_thr=2;
%     HR_ECG1=NaN;
%     Amp_ECG1=NaN;
%     Flag_falsedet1=0;
%     if (~isempty(PeakLoca1))
%         if  PeakLoca1(end)>length(ECG1)
%             PeakLoca1(end)=[];
%         end
%         [aa cur_indx1]=min(Sd_ECG1_0(1:300/Segment_t-1));
%         if aa<1000
%             cur_PeakLoca1=PeakLoca1(find(PeakLoca1>Fs*Segment_t*(cur_indx1-1) & PeakLoca1<Fs*Segment_t*cur_indx1));
%             ECG_template10=zeros(length(cur_PeakLoca1)-1,Fs);
%             Distance1=zeros(length(cur_PeakLoca1)-1,length(cur_PeakLoca1)-1);
%             for k=1:length(cur_PeakLoca1)-1
%                 ECG_template10(k,:)=resample(ECG1(cur_PeakLoca1(k):cur_PeakLoca1(k+1)),Fs,cur_PeakLoca1(k+1)-cur_PeakLoca1(k)+1);
%                 Range(k)=max(ECG1(cur_PeakLoca1(k):cur_PeakLoca1(k+1)))-min(ECG1(cur_PeakLoca1(k):cur_PeakLoca1(k+1)));
%                 for k2=1:length(cur_PeakLoca1)-1
%                     Range(k2)=max(ECG1(cur_PeakLoca1(k2):cur_PeakLoca1(k2+1)))-min(ECG1(cur_PeakLoca1(k2):cur_PeakLoca1(k2+1)));
%                     if Range(k)/Range(k2)>Amp_thr || Range(k)/Range(k2)>Amp_thr
%                         Distance1(k,k2)=1;
%                     end
%                 end
%             end
%             if sum(sum(Distance1))/(length(cur_PeakLoca1)-1).^2 <Flag_falsedet_thr
%                 Flag_falsedet1=1;
%             end
%             ECG_template1=mean(ECG_template10);
%             HR_ECG1=60*Fs/mean(diff(cur_PeakLoca1));
%             Amp_ECG1=max(ECG_template1)-min(ECG_template1);
%             
%             cur_PeakLoca11=PeakLoca1(find(PeakLoca1>Fs*Segment_t*(30-1) & PeakLoca1<Fs*Segment_t*30));
%             ECG_Seg30_corr1=zeros(length(cur_PeakLoca11)-1,1);
%             for k=1:length(cur_PeakLoca11)-1
%                 cursig = resample(ECG1(cur_PeakLoca11(k):cur_PeakLoca11(k+1)),Fs,cur_PeakLoca11(k+1)-cur_PeakLoca11(k)+1);
%                 xy = corrcoef(ECG_template1,cursig);
%                 ECG_Seg30_corr1(k) = xy(2,1);
%             end
%             ECG_A_corr1=mean(abs(ECG_Seg30_corr1));
%         end
%     else ECG_A_corr1=0;
%     end
%     
%     HR_ECG2=NaN;
%     Amp_ECG2=NaN;
%     Flag_falsedet2=0;
%     if (~isempty(PeakLoca2))
%         if  PeakLoca2(end)>length(ECG2)
%             PeakLoca2(end)=[];
%         end
%         [aa cur_indx2]=min(Sd_ECG2_0(1:300/Segment_t-1));
%         if aa<1000
%             cur_PeakLoca2=PeakLoca2(find(PeakLoca2>Fs*Segment_t*(cur_indx2-1) & PeakLoca2<Fs*Segment_t*cur_indx2));
%             ECG_template20=zeros(length(cur_PeakLoca2)-1,Fs);
%             Distance2=zeros(length(cur_PeakLoca2)-1,length(cur_PeakLoca2)-1);
%             for k=1:length(cur_PeakLoca2)-1
%                 ECG_template20(k,:)=resample(ECG2(cur_PeakLoca2(k):cur_PeakLoca2(k+1)),Fs,cur_PeakLoca2(k+1)-cur_PeakLoca2(k)+1);
%                 Range(k)=max(ECG2(cur_PeakLoca2(k):cur_PeakLoca2(k+1)))-min(ECG2(cur_PeakLoca2(k):cur_PeakLoca2(k+1)));
%                 for k2=1:length(cur_PeakLoca2)-1
%                     Range(k2)=max(ECG2(cur_PeakLoca2(k2):cur_PeakLoca2(k2+1)))-min(ECG2(cur_PeakLoca2(k2):cur_PeakLoca2(k2+1)));
%                     if Range(k)/Range(k2)>Amp_thr || Range(k)/Range(k2)>Amp_thr
%                         Distance2(k,k2)=1;
%                     end
%                 end
%             end
%             if sum(sum(Distance2))/(length(cur_PeakLoca2)-1).^2 <Flag_falsedet_thr
%                 Flag_falsedet2=1;
%             end
%             ECG_template2=mean(ECG_template20);
%             HR_ECG2=60*Fs/mean(diff(cur_PeakLoca2));
%             Amp_ECG2=max(ECG_template2)-min(ECG_template2);
%             
%             cur_PeakLoca22=PeakLoca2(find(PeakLoca2>Fs*Segment_t*(30-1) & PeakLoca2<Fs*Segment_t*30));
%             ECG_Seg30_corr2=zeros(length(cur_PeakLoca22)-1,1);
%             for k=1:length(cur_PeakLoca22)-1
%                 cursig = resample(ECG2(cur_PeakLoca22(k):cur_PeakLoca22(k+1)),Fs,cur_PeakLoca22(k+1)-cur_PeakLoca22(k)+1);
%                 xy = corrcoef(ECG_template2,cursig);
%                 ECG_Seg30_corr2(k) = xy(2,1);
%             end
%             ECG_A_corr2=mean(abs(ECG_Seg30_corr2));
%         end
%     else ECG_A_corr2=0;
%     end
%     
%     Falg_ECG1_VF=0;
%     Falg_ECG2_VF=0;
%     
%     N0_d_VF=Fs*(5*60-5);
%     
%     if Flag_falsedet1
%         cur_data=ECG1(N0_d_VF+1:N_d);
%         cur_data=cur_data-min(cur_data);
%         cur_index=find(cur_data>Amp_ECG1*0.6);
%         if length(find(diff(cur_index)>10))+1>12  && HR_ECG1>=VFB_th-20
%             Falg_ECG1_VF=1;
%         end
%         
%     end
%     
%     if Flag_falsedet2
%         cur_data=ECG2(N0_d_VF+1:N_d);
%         cur_data=cur_data-min(cur_data);
%         cur_index=find(cur_data>Amp_ECG2*0.8);
%         if length(find(diff(cur_index)>10))+1>12  && HR_ECG2>=VFB_th
%             Falg_ECG2_VF=1;
%         end
%     end
% end

  %% Determination 
    sqi_th = 0.6;
    sqi_th_T = 0.85;
    sqi_th_VF = 0.6;
    BRA_th = 40;
    TAC_th = 80;
    VFB_th = 170;
    tolerance = 5;

    
switch alarm_type
    case 'Asystole'
        alarmResult=~((~Flag_A_True_ECG1(m) && Flag_DetAcc_ECG1) || (~Flag_A_True_ECG2(m) &&  Flag_DetAcc_ECG2) || (~Flag_A_True_ABP(m) &&  Flag_DetAcc_ABP) || (~Flag_A_True_PPG(m) &&  Flag_DetAcc_PPG));
%         if Flag_post
%             if (Flag_A_True_ECG1_post(m) && Flag_DetAcc_ECG1) || (Flag_A_True_ECG2_post(m) && Flag_DetAcc_ECG2) || (Flag_A_True_ABP_post(m) && Flag_DetAcc_ABP_post && Flag_DetAcc_ABP && ~Flag_A_True_ABP(m)) || (Flag_A_True_PPG_post(m) && Flag_DetAcc_PPG_post && Flag_DetAcc_PPG  && ~Flag_A_True_PPG(m))
%                 alarmResult=0;
%             end
%         end
        
    case 'Bradycardia'
        if(~isempty(ABP_ind))
            if abpsqi>=sqi_th && low_hr_abp-tolerance>BRA_th
                alarmResult=0;
            end
        end
        if(~isempty(PPG_ind))
            if ppgsqi>=sqi_th && low_hr_ppg-tolerance>BRA_th
                alarmResult=0;
            end
        end
        
    case 'Tachycardia'
        alarmResult=~((~Flag_T_True_ECG1(m) && Flag_DetAcc_ECG1) || (~Flag_T_True_ECG2(m) &&  Flag_DetAcc_ECG2) || (~Flag_T_True_ABP(m) &&  Flag_DetAcc_ABP) || (~Flag_T_True_PPG(m) &&  Flag_DetAcc_PPG));
        if (Flag_T_True_ECG1(m) && Flag_DetAcc_ECG1 && ECG1_HR_base>110) || (Flag_T_True_ECG2(m) &&  Flag_DetAcc_ECG2 && ECG2_HR_base>110)
            alarmResult=1;
        end
        if(~isempty(ABP_ind))
            if abpsqi>=sqi_th_T && high_hr_abp+tolerance<TAC_th && Flag_DetAcc_ABP
                alarmResult=0;
            end
        end
        if(~isempty(PPG_ind))
            if ppgsqi>=sqi_th_T && high_hr_ppg+tolerance<TAC_th  && Flag_DetAcc_PPG
                alarmResult=0;
            end
        end
            
    case 'Ventricular_Tachycardia'
        alarmResult=(Flag_VT_True_ECG1(m) && Flag_DetAcc_ECG1 && min(ECG1_Corr)<0.82 && max(ECG1_Corr)>0.9 && max(ECG1_Shape_change)>0.65) || (Flag_VT_True_ECG2(m) &&  Flag_DetAcc_ECG2 && min(ECG2_Corr)<0.82 && max(ECG2_Corr)>0.9 && max(ECG2_Shape_change)>0.65);
        if (~isempty(ABP_ind))
            if Flag_VT_True_ECG1(m) && Flag_DetAcc_ECG1 && Flag_VT_True_ECG2(m) && Flag_DetAcc_ECG2 && Flag_VT_True_ABP(m) && Flag_DetAcc_ABP
                alarmResult=1;
            end
        end      
        if (~isempty(PPG_ind))
            if Flag_VT_True_ECG1(m) && Flag_DetAcc_ECG1 && Flag_VT_True_ECG2(m) && Flag_DetAcc_ECG2 && Flag_VT_True_PPG(m) && Flag_DetAcc_PPG
                alarmResult=1;
            end
        end
        if Flag_VT_True_ECG1(m) && Flag_DetAcc_ECG1 && max(ECG1_VT_HR)-ECG1_HR_base>15  && max(ECG1_Corr)-min(ECG1_Corr)>0.15
            alarmResult=1;
        end
        
        if Flag_VT_True_ECG2(m) && Flag_DetAcc_ECG2 && max(ECG2_VT_HR)-ECG2_HR_base>15  && max(ECG2_Corr)-min(ECG2_Corr)>0.15
            alarmResult=1;
        end
         
    case 'Ventricular_Flutter_Fib'
        if (~isempty(ABP_ind))
            if (abpsqi>=sqi_th_VF && hr_max_abp+tolerance<VFB_th)
                alarmResult=0;
            end
        end
        
        if (~isempty(PPG_ind))
            if (ppgsqi>=sqi_th_VF && hr_max_ppg+tolerance<VFB_th)
                alarmResult=0;
            end
        end
        
    otherwise
        error(['Unknown alarm type: ' alarm_type])
end

%% plot signal and thier features
if debug
    figure('Color', 'w', 'Units', 'Inch', 'Position', [1, 1, 12, 5.5]);
    switch alarm_type
        case 'Asystole'
            for j=1:size(data_be,2)
                subplot(size(data_be,2),1,j)
                plot((1:length(ECG1_be))/Fs+300-T_A_be,data_be_fil(:,j),'b-')
                switch data_channel{j,:}
                    case 'ECG1'
                        hold on
                        plot((ECG1_R_peak_M1)/Fs+300-T_A_be,ECG1_be_fil(ECG1_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG1',strcat('DecAcc=',num2str(Flag_DetAcc_ECG1)),strcat('True=',num2str(Flag_A_True_ECG1(m)))})
                        
                    case 'ECG2'
                        hold on
                        plot((ECG2_R_peak_M1)/Fs+300-T_A_be,ECG2_be_fil(ECG2_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG2',strcat('DecAcc=',num2str(Flag_DetAcc_ECG2)),strcat('True=',num2str(Flag_A_True_ECG2(m)))})
                        
                    case 'ABP'
                        hold on
                        plot((ABP_foot)/Fs+300-T_A_be,ABP_be_fil(ABP_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ABP',strcat('DecAcc=',num2str(Flag_DetAcc_ABP)),strcat('True=',num2str(Flag_A_True_ABP(m)))})
                        
                    case 'PPG'
                        hold on
                        plot((PPG_foot)/Fs+300-T_A_be,PPG_be_fil(PPG_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'PPG',strcat('DecAcc=',num2str(Flag_DetAcc_PPG)),strcat('True=',num2str(Flag_A_True_PPG(m)))})
                end
            end
        case 'Bradycardia'
            for j=1:size(data_be,2)
                subplot(size(data_be,2),1,j)
                plot((1:length(ECG1_be))/Fs+300-T_A_be,data_be_fil(:,j),'b-')
                switch data_channel{j,:}
                    case 'ECG1'
                        hold on
                        plot((ECG1_R_peak_M1)/Fs+300-T_A_be,ECG1_be_fil(ECG1_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG1',strcat('DecAcc=',num2str(Flag_DetAcc_ECG1)),strcat('True=',num2str(Flag_B_True_ECG1(m)))})
                        
                    case 'ECG2'
                        hold on
                        plot((ECG2_R_peak_M1)/Fs+300-T_A_be,ECG2_be_fil(ECG2_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG2',strcat('DecAcc=',num2str(Flag_DetAcc_ECG2)),strcat('True=',num2str(Flag_B_True_ECG2(m)))})
                        
                    case 'ABP'
                        hold on
                        plot((ABP_foot)/Fs+300-T_A_be,ABP_be_fil(ABP_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ABP',strcat('DecAcc=',num2str(Flag_DetAcc_ABP)),strcat('True=',num2str(Flag_B_True_ABP(m)))})
                        
                    case 'PPG'
                        hold on
                        plot((PPG_foot)/Fs+300-T_A_be,PPG_be_fil(PPG_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'PPG',strcat('DecAcc=',num2str(Flag_DetAcc_PPG)),strcat('True=',num2str(Flag_B_True_PPG(m)))})
                end
            end
            
        case 'Tachycardia'
            for j=1:size(data_be,2)
                subplot(size(data_be,2),1,j)
                plot((1:length(ECG1_be))/Fs+300-T_A_be,data_be_fil(:,j),'b-')
                switch data_channel{j,:}
                    case 'ECG1'
                        hold on
                        plot((ECG1_R_peak_M1)/Fs+300-T_A_be,ECG1_be_fil(ECG1_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG1',strcat('DecAcc=',num2str(Flag_DetAcc_ECG1)),strcat('True=',num2str(Flag_T_True_ECG1(m)))})
                        
                    case 'ECG2'
                        hold on
                        plot((ECG2_R_peak_M1)/Fs+300-T_A_be,ECG2_be_fil(ECG2_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG2',strcat('DecAcc=',num2str(Flag_DetAcc_ECG2)),strcat('True=',num2str(Flag_T_True_ECG2(m)))})
                        
                    case 'ABP'
                        hold on
                        plot((ABP_foot)/Fs+300-T_A_be,ABP_be_fil(ABP_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ABP',strcat('DecAcc=',num2str(Flag_DetAcc_ABP)),strcat('True=',num2str(Flag_T_True_ABP(m)))})
                        
                    case 'PPG'
                        hold on
                        plot((PPG_foot)/Fs+300-T_A_be,PPG_be_fil(PPG_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'PPG',strcat('DecAcc=',num2str(Flag_DetAcc_PPG)),strcat('True=',num2str(Flag_T_True_PPG(m)))})
                end
            end
            
        case 'Ventricular_Tachycardia'
            for j=1:size(data_be,2)
                subplot(size(data_be,2),1,j)
                plot((1:length(ECG1_be))/Fs+300-T_A_be,data_be_fil(:,j),'b-')
                switch data_channel{j,:}
                    case 'ECG1'
                        hold on
                        plot((ECG1_R_peak_M1)/Fs+300-T_A_be,ECG1_be_fil(ECG1_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG1',strcat('DecAcc=',num2str(Flag_DetAcc_ECG1)),strcat('True=',num2str(Flag_VT_True_ECG1(m)))})
                        
                    case 'ECG2'
                        hold on
                        plot((ECG2_R_peak_M1)/Fs+300-T_A_be,ECG2_be_fil(ECG2_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG2',strcat('DecAcc=',num2str(Flag_DetAcc_ECG2)),strcat('True=',num2str(Flag_VT_True_ECG2(m)))})
                        
                    case 'ABP'
                        hold on
                        plot((ABP_foot)/Fs+300-T_A_be,ABP_be_fil(ABP_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ABP',strcat('DecAcc=',num2str(Flag_DetAcc_ABP)),strcat('True=',num2str(Flag_VT_True_ABP(m)))})
                        
                    case 'PPG'
                        hold on
                        plot((PPG_foot)/Fs+300-T_A_be,PPG_be_fil(PPG_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'PPG',strcat('DecAcc=',num2str(Flag_DetAcc_PPG)),strcat('True=',num2str(Flag_VT_True_PPG(m)))})
                end
            end
            
        case 'Ventricular_Flutter_Fib'
            for j=1:size(data_be,2)
                subplot(size(data_be,2),1,j)
                plot((1:length(ECG1_be))/Fs+300-T_A_be,data_be_fil(:,j),'b-')
                switch data_channel{j,:}
                    case 'ECG1'
                        hold on
                        plot((ECG1_R_peak_M1)/Fs+300-T_A_be,ECG1_be_fil(ECG1_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG1',strcat('DecAcc=',num2str(Flag_DetAcc_ECG1)),strcat('True=',num2str(Flag_VF_True_ECG1(m)))})
                        
                    case 'ECG2'
                        hold on
                        plot((ECG2_R_peak_M1)/Fs+300-T_A_be,ECG2_be_fil(ECG2_R_peak_M1),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ECG2',strcat('DecAcc=',num2str(Flag_DetAcc_ECG2)),strcat('True=',num2str(Flag_VF_True_ECG2(m)))})
                        
                    case 'ABP'
                        hold on
                        plot((ABP_foot)/Fs+300-T_A_be,ABP_be_fil(ABP_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'ABP',strcat('DecAcc=',num2str(Flag_DetAcc_ABP)),strcat('True=',num2str(Flag_VF_True_ABP(m)))})
                        
                    case 'PPG'
                        hold on
                        plot((PPG_foot)/Fs+300-T_A_be,PPG_be_fil(PPG_foot),'ro','MarkerSize',4,'MarkerEdgeColor','r','MarkerFaceColor','r')
                        ylabel({'PPG',strcat('DecAcc=',num2str(Flag_DetAcc_PPG)),strcat('True=',num2str(Flag_VF_True_PPG(m)))})
                end
            end
    end
end

% Write result to answers.txt
fid = fopen(answers, 'a');
if (fid == -1)
    error('Could not open answer file');
end

% Get base name of record (without directories)
i = strfind(recordName, filesep);
if (~isempty(i))
    basename = recordName(i(end)+1 : end);
else
    basename = recordName;
end

fprintf(fid, '%s,%d\n', basename, alarmResult);
fclose(fid);

end

%%%%%%%%%%%% Helper Functions %%%%%%%%%%%%%%%%%%%%%
function ind=get_index(description,pattern)
M=length(description);
tmp_ind=strfind(description,pattern);
ind=[];
for m=1:M
    if(~isempty(tmp_ind{m}))
        ind(end+1)=m;
    end
end
end
