function [BeatQ, r] = wSQI(features, onset)
% WSQI  ABP waveform signal quality index.
%   [BEATQ, R] = wSQI(FEATURES, ONSET, ABP) returns a binary signal quality
%   assessment of each beat in ABP.  This algorithm relies on detecting
%   abnormalities of numeric values in FEATURES and ONSET.
%
%   In:   FEATURES <mx16> --- features extracted from ABP using abpfeature.m
%         ONSET    <nx1>  --- onset times of ABP using wabp.m
%         ABP      <px1>  --- arterial blood pressure waveform (125Hz sampled)
%
%   Out:  BEATQ    <nx6> --- SQI of each beat: 1=good, 0=bad
%           Col 1:  minus of cols 2 thru 6
%               2:  uAN     ABP amplitude is normal
%               3:  uSN     ABP slope is normal
%               4:  uKRTL   ABP the biggest rise time last 
%               5:  uSHTL   ABP high level time last  
%               6:  uWBT    ABP PPD and(DBPI)and(not PrP)  
%      
%         R        <1x1>  fraction of good beats in ABP 
%                   goodbeats is  wSQI>0.5
% 
%   Usage:
%   - FEATURES must be obtained using abpfeature.m
%   - ONSET    must be obtained using wabp.m
%
%   Written by Zhiqiang Sheng  on March 15, 2015.


%% old is 30
if length(onset)<40
    BeatQ = [];
    r = [];
    return
end

%%  get ABP features

        SBP = features(:,2);    SBPs = 0;
        DBP = features(:,4);    DBPs = 0;
        PBP = features(:,5);    PBPs = 0;
        PP  = features(:,7);    PPs = 0;
        MPPS = features(:,13);  MPPSs = 0;
        MNPS = features(:,14);  MNPSs = 0;
        MUSD = features(:,15);  
        MDAT = features(:,16);  

%% learning period 20s change the onsetset is 40 40*125ms*2*2 ~= 20s
    for i = 1 : 40
        SBPs = SBPs + SBP(i);
        DBPs = DBPs + DBP(i);
        PBPs = PBPs + PBP(i);
        PPs =  PPs + PP(i);
        MPPSs = MPPSs + MPPS(i);
        MNPSs = MNPSs + MNPS(i);
    end
    
    SBPa = mean(SBPs);
    DBPa = mean(DBPs);
    PBPa = mean(PBPs);
    PPa =  mean(PPs);
    MPPSa = mean(MPPSs);
    MNPSa = mean(MNPSs);

%% get the wSQI value

%% Initialize the variable to allocate memory

  OT = onset(1 :end-1);
  qt = length(OT);
  wSQI = ones(1,qt + 1);
  
for i = 1 : qt
%%  wSQI abpfeature
    
    SBPia = SBP(i) - SBPa;
    DBPi = DBP(i);
    MPPSia = MPPS(i) / MPPSa;
    MNPSia = MNPS(i) - MNPSa;
    MUSDi = MUSD(i);
    MDATi = MDAT(i);
    PBPia = PBP(i) / PBPa;
    DBPia = DBP(i) / DBPa;
    PPia = PP(i) / PPa;
    
    uATL = S(SBPia,20,60);
    uATS = Z(DBPi,0,20);
    uSTL = S(MPPSia,1,3);
    uSTS = S(MNPSia,1,3);
    uKRTL = S(MUSDi,200,500);
    uSHTL = S(MDATi,400,800);
    uPPD = Z(PBPia,0.5,0.9);
    uDBPI = S(DBPia,0.8,1.1);
    uPrP = Z(PPia,0.75,0.95);

%% SQI final
% & is minus of numbers
   
    an = [ (1 - uATL),(1 - uATS) ];
    sn = [ (1 - uSTL),(1 - uSTS) ];
    wbt = [ uPPD, uDBPI,(1 - uPrP)];
    uAN = min( an );
    uSN = min( sn );
    uWBT = min( wbt );
    sqg = [ uAN, uSN,(1 - uKRTL) ,(1 - uSHTL),(1 - uWBT)];
    % the min of lines or uSQG = min ( sqg' )';
    % uSQG = min ( sqg ,[] ,2 );
    uSQG = min (sqg);
    wSQI(i) = uSQG;

%% update the feature value real-time

        if (uSQG > 0.5)
        SBPa = uSQG * SBPa + (1 - uSQG) *SBP(i);
        DBPa = uSQG * DBPa + (1 - uSQG) *DBP(i);
        MPPSa = uSQG * MPPSa + (1 - uSQG) *MPPS(i);
        MNPSa = uSQG * MNPSa + (1 - uSQG) *MNPS(i);
        PPa = uSQG * PPa + (1 - uSQG) *PP(i);
        end
end       

%% Output
%BeatQ is the onset's wSQI
BeatQ   = wSQI';
% fraction of good beats overall  wSQI>0.5 is good
r = length(find(BeatQ(:,1) > 0.5))/qt;
end

%% Write by Zhiqiang Sheng March 14 2015
%%  Help Function
%% Function S
function s = S(x,a,b)
     c = (a + b)/2;
     d =  b - a;
     s = 0;
     if( a >= x )
         s = 0;
     end
     
     if( ( a < x ) && ( x <c ) )
         s = 2 * ((x - a) / d)^2;
     end
     
     if( ( c < x ) && ( x <b ))
         s = 1 - 2 * ((x - b) / d )^2;
     end
     
     if( b < x )
         s = 1;
     end

end
%% Fuction Z
function z = Z(x,a,b)
    z = 1 - S(x,a,b);
end
