function [S] = rdsamp(record, varargin)
% function [S] = rdsamp(record, ...)
% Read WFDB signal files for the specified record and return the samples
% as column vectors.
%
% Column 1 of the output is always a vector of the sample numbers;
% the remaining columns are vectors of the record's signals.
%
% One required argument:
% record: the name of the WFDB record
%
% Six optional arguments (default values in []s):
% 'begin' (time-format string): the time in the record from which
%    to begin reading ['00:00:00'] 
% 'maxt' (time-format string): upper limit, in time, on the amount
%    of output produced [no limit]
% 'stop' (time-format string): the time in the record at which to
%    stop reading [end of record]
% 'hires' (boolean): whether or not to read in high-resolution
%    mode; identical for ordinary records, but for
%    multifrequency records, the standard
%    decimation of oversampled signals to the frame
%    rate is suppressed (all other signals are
%    resampled at the highest sampling frequency) [false]
% 'phys' (boolean): whether or not to return signal values as those
%    of their respective physical units, instead of
%    as raw A/D units [false]
% 'sigs' (1 or more uint32's): the signal numbers of the signals to
%    be returned (0-indexed)
%    [all signal numbers, in order]
%
% Note that the values of 'begin', 'maxt', and 'stop' are expected
% to be "time-format strings" of the form 'AB:CD:EF' (where A
% through F represent single digits); e.g., '00:00:00' represents
% time zero, '00:00:05' five seconds, '00:05:00' five minutes, and
% so on.
%
% The value of 'maxt' is strictly an upper limit: if the time
% length between 'begin' and 'stop' is greater than 'maxt', then
% the actual stopping time is 'begin' + 'maxt'.
%
% The optional arguments are specified as parameter/value pairs,
% where the parameter is a string.
%
% Examples
%
%    % return all signals of the record 'mitdb/100'
%    rdsamp('mitdb/100')
%
%    % return all signals of 'mitdb/100', in physical units
%    rdsamp('mitdb/100','phys',true)
%
%    % return all signals of 'mitdb/101', in high-res. mode, beginning
%    % at 10 seconds from the start, and ending at 2 min. 30
%    % secs. from the start
%    rdsamp('mitdb/101','begin','00:00:10','stop','00:02:30','hires',true)
%
%    % return only the first (i.e. "number zero") signal of 'mitdb/101'
%    rdsamp('mitdb/101','sigs',1)
%
%    % return only the second, third, and fifth signals of 'mghdb/mgh001',
%    % as the second, third, and fourth columns respectively
%    rdsamp('mghdb/mgh001','sigs',1,2,4)
%
% See also WRSAMP, RDANN, WRANN, WFDBDESC, rdsamp(1) man page
%
% Copyright (c) 2009 by Michael Craig, All Rights Reserved
% Contact M. Craig (mic@mit.edu)
%
%    This program is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation; either version 2 of the License, or
%    (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
%    02111-1307  USA
%
% rdsamp.m and its dependencies are freely available from Physionet -
% http://www.physionet.org/ - please report any bugs to the authors above.


%
%LoadWFDBJava.load;


%
argsInfo = struct('begin', { {[]},    '1', 'string'  }, ...
                  'maxt',  { {[]},    '1', 'string'  }, ...
                  'stop',  { {[]},    '1', 'string'  }, ...
                  'hires', { {false}, '1', 'boolean' }, ...
                  'phys',  { {false}, '1', 'boolean' }, ...
                  'sigs',  { {},      '*', 'uint32'  });

%
p = varargin2struct(varargin, argsInfo(1), argsInfo(2));
verify_params(p, argsInfo(2:3));

%
S = rdsampMatlab.rdsamp(record, ...
                        p.begin{1}, ...
                        p.maxt{1}, ...
                        p.stop{1}, ...
                        p.hires{1}, ...
                        p.phys{1}, ...
                        cell2mat(p.sigs));
