function [] = wrsamp(cols, record, varargin)
%WRSAMP write signal data into WFDB-compatible records
% Create a record with the given data and name.
%
% Creation of the new record implies creation of the file named
% [record,'.dat'], and the file named [record,'.hea'].
%
% WRSAMP() can be used nearly as an inverse to RDSAMP(), except
% that whereas the first column returned by RDSAMP() is a vector of
% sample numbers, the columns passed to WRSAMP() are only signal
% data.
%
% Nothing is returned.
%
% Two required arguments:
% cols: the signals to write out, as column vectors
% record: the name of the record to create
%
% Seven optional arguments (default values in []s):
% 'inputname' (string): if non-null, this is used in the
%    description (the 'desc' field) of each signal [null]
% 'dither' (boolean): whether or not to dither the data in cols
%    before converting it to integer output [false]
% 'start' (uint): the number of the sample from which to begin
%    writing, with 0-based index; all data before it are ignored [0]
% 'stop' (uint): the number of the sample at which to stop writing;
%    only data before it are used [end of data, i.e. size(cols,1)+1]
% 'freq' (number): the sampling frequency of each of the signals, in
%    samples per second; only affects the header file written [250]
% 'gain' (0 or more numbers): the gain (in A/D units per millivolt) of the
%    signals; multiple values may be given, up to the number of
%    signals, and if N values are given for M signals and N < M,
%    all signals after the Nth will have the same gain as the Nth;
%    only affects the header file written [200]
% 'scale' (0 or more numbers): a factor by which to scale all
%    samples in the signals; multiple values may be given, behaving
%    exactly as 'gain' does above for multiple signals; affects the
%    data file by changing the signal data [1]
%
% The argument 'inputname' roughly corresponds to the "-i" option
% in the original wrsamp program, except that it is only used to
% change the descriptions stored in the header for the signals --
% it is NOT used to specify a file from which to read input (all
% signal data are passed in through cols.) If 'inputname' is not
% used, the description for column N is simply ['column ',N]; if it is
% used and has value V, the description is [V,', column ',N].
%
% Dithering, if performed, is done by adding, to each sample, a
% random value, selected from a triangular probability density
% function between -1 and +1.
%
% Examples
%
%    % just get some existing data
%    d = rdsamp('mitdb/100');
%
%    % write out a copy of record 'mitdb/100', named '100copy',
%    % consisting of local files '100copy.dat' and '100copy.hea';
%    % note that we skip the first column returned by rdsamp()
%    % (the sample numbers)
%    wrsamp(d(:,2:3), '100copy');
%
%    % triplicate the first signal of record 'mitdb/100', and write
%    % these to record '100dup'; skip the first ten samples; and
%    % give the first signal a gain of 1.0, and the latter two a
%    % gain of 2.0
%    wrsamp([d(:,2), d(:,2), d(:,2)], '100dup', 'start', 10, ...
%           'gain', 1.0, 2.0);
%
% See also RDSAMP, WRANN, WFDB_LICENSE wrsamp(1) man page
%
% Copyright (c) 2009 by Michael Craig, All Rights Reserved
% Contact M. Craig (mic@mit.edu)
%


% Modified by Ikaro Silva, 6/2012 

% FIXME: change 'dither' to 'S' (switch)
%

%
% call System.loadLibrary("wfdbjava") so that the WFDB constants
% below will be accessible
%
LoadWFDBJava.load;
wfdb.wfdb.wfdbquit;


%
%
argsInfo = struct('inputname', { {[]},                     '1', 'string'  }, ...
                  'dither',    { {false},                  '1', 'boolean' }, ...
                  'start',     { {0},                      '1', 'uint32'  }, ...
                  'stop',      { {0},                      '1', 'uint32'  }, ...
                  'freq',      { {wfdb.wfdb.WFDB_DEFFREQ}, '1', 'number'  }, ...
                  'gain',      { {wfdb.wfdb.WFDB_DEFGAIN}, '+', 'number'  }, ...
                  'scale',     { {1.0},                    '+', 'number'  });

%
%
p = varargin2struct(varargin, argsInfo(1), argsInfo(2));
verify_params(p, argsInfo(2:3));

%
%
wrsampMatlab.wrsamp(cols,              ...
                    record,            ...
                    p.inputname{1},    ...
                    p.dither{1},       ...
                    p.start{1},        ...
                    p.stop{1},         ...
                    p.freq{1},         ...
                    cell2mat(p.gain),  ...
                    cell2mat(p.scale));

                
