function varargout=wfdb_query(varargin)
%
% info=WFDB_QUERY(mode)
%
% Displays information about PhysioNet's Public Databases. Parameters are:
%
% mode  - (String) input, options are:
%       - 'all'  outputs a list of all databases in PhysioNet
%       - database_name Outputs list of records specific to a database
%         given from the ouput of using WFDB_QUERY('all'). The records in this list
%         can be imported directly into MATLAB by using RDSAMP.
%
%
% Outputs:
%
% If mode is equal to 'all' the output is a cell list containing the name
% of all public databases in PhysioNet based on cached data from the Toolbox.
%
% If mode is equal to one the database names form the out of 'all', then
% the output is a  cell list containing information about each signal in
% that particular database. Each row in the list consists of tab-separated columns
% with the following information:
%
% record name
% signal type, with an appended digit [1]
% signal name
% sampling frequency
% gain [2]
% length in samples [3]
% interval(s) within the record when the signal is present [4]
%
% [1] For example, ECG1, ECG2, etc.  The digit is missing if there are
%     never multiple signals of the given type in any record (e.g., PLETH).
% [2] '?' means signal is uncalibrated
% [3] total length of all segments containing the signal in the record
% [4] absent except for variable-layout records and 'Annot'.
% If the signal type is 'Annot', the next column is the annotator name,
% the number of annotations appears in lieu of the gain, and the times
% (in sample intervals) of the first and last annotations are shown in
% the last column.
%
%
% NOTE: Database information is cached (stored locally) in the WFDB Toolbox
% directory. This function connects with PhysioNet servers at:
%
%  http://physionet.org/physiobank/database/pbi/
%
% In order to check if these cached files are up-to-date. If they are not,
% up-to-date (or if they do not exist locally) the function automatically downloads
% these files.
%
%
% %Example 1- Display a list of all databases:
%
% db_list=wfdb_query;
%
% %Example 2- Display a list of all signals in the database 'fantasia':
% rec_list=wfdb_query('fantasia')
%
%
% %Exmample 3- Extract data from the first record from Example 2
%  rec_list=wfdb_query('fantasia')
%  rec1_info=textscan(rec_list{1},'%s');
%  rdsamp(rec{1}{1},'begin','00:00:10','stop','00:02:30')
% See also RDSAMP, WFDB_LICENSE, WFDBDESC,
%
% Copyright (c) 2012 by Ikaro Silva, All Rights Reserved
%

%Cached Database Info is expected to result in subdirectory of toolbox
%Each database file has the format {DatabaseAcronym}db.txt
%To get list of all the databases, parse directory information using this
%pattern

%NOTE: This function uses files derived from an index file that at
%     http://physionet.org/physiobank/database/physiobank-index
%
% %This information is based on the current format of physio-bank index, so
% %change to that index file might require to change this information as well


mode='all'; %Default option for input mode
inputs={'mode'};
for n=1:nargin
    if(~isempty(varargin{n}))
        eval([inputs{n} '=varargin{n};'])
    end
end

%Cache Toolbox configuration parameters to speed
config=wfdb_install('setup');

switch mode
    
    case 'all'
        %Generate list of databases
        varargout(1)=gen_db_list(config);
    otherwise
        %Query a specific database for it's info
        varargout(1)={gen_db_query(config,mode)};
        
end



function db_list=gen_db_list(config)
exception=[];
msgid='WFDBToolbox:wfdb_query:gen_db_list';
fname=[config.DB_CACHED_DIR config.DB_LIST];

persistent update_check
%Check if cached version is up to date
if(isempty(update_check))
    warning(msgid,'Checking if cache is up-to-date with PhysioNet')
    update_url([config.DB_LIST_URL config.DB_LIST],fname)    
end
%Set persistent variable to 1 to avoid re-checking if file is called often
%Check occurs only when MATLAB is restarted (or a specific clear command)
update_check=1; 
fid=fopen(fname,'r');
if(fid ~= -1)
    try
        %db_list=textscan(fid,'%s');
        db_list=textscan(fid,'%q','delimiter','\n','HeaderLines',0);
    catch exception
        % Close fid before throwing error or warning
        db_list=[];
    end
    fclose(fid);
    if(~isempty(exception))
        warnign(msgid,['Could not open database list: \n' ...
            fname '\n Make sure database name is presente in the first column of the output of wfdb_query(''all'')'])
        
    end
end



function db_info=gen_db_query(config,mode)

exception=[];
db_info=[];
msgid='WFDBToolbox:wfdb_query:gen_db_query';
%ASSUMES FILES END WITH TXT EXTENSION!!
fname=[config.DB_CACHED_DIR mode '.txt'];
persistent update_check

%Check if cached version is up to date
if(~strcmp(mode,update_check))
    update_url([config.DB_URL mode ],fname)    
    update_check{end+1}=mode;
end
%Set persistent variable to db_name to avoid re-checking file if called
%consecutively




fid=fopen(fname,'r');
if(fid ~= -1)
    try
        %Using default white space delimiter!
        db_info=textscan(fid,'%q','delimiter','\n','HeaderLines',0);
    catch exception
        % Close fid before throwing error or warning
    end
    fclose(fid);
    if(~isempty(exception))
        warning(msgid,['Could not open database info file: \n' fname])
        warning(msgid,['Error:' exception.message])
    end
    db_info=db_info{:};
end


function update_url(server_url,local_url)
msgid='WFDBToolbox:wfdb_query:update_url';
try
    url=java.net.URL(server_url);
    server_tmstamp=url.openConnection.getLastModified();
    if(exist(local_url,'file'))
        url2=java.io.File(local_url);
        local_tmstamp=url2.lastModified;
    else
        local_tmstamp=-inf;
    end
    if(local_tmstamp < server_tmstamp)
        warning(msgid,['Cached database info is out-of-date, downloading new cached file from '...
            ' PhysioNet''s server: \n ' server_url])
        [filestr,status]=urlwrite(server_url,local_url);
        if(~status)
            warning(msgid,['Could not update cached database information.'...
                '\nMake sure database name is presente in the first column of the output of wfdb_query(''all'')'])
        else
            warning(msgid,'Cached database information sucessfully updated!!')
        end
    end
catch exception
    warning(msgid,['Could not check if cached file is up-to-date. Error:\n'...
        exception.message])
end

