function [] = verify_params(paramStruct, verifInstructions)
% function [] = verify_params(paramStruct, verifInstructions)
% Verify that the "parameters" struct, typically created by
% VARARGIN2STRUCT(), matches the set of constraints imposed by
% verifInstructions.
%
% paramStruct is expected to be a 1x1 struct array, each of whose
% values is expected to be a (possibly empty) cell
% array. verifInstructions is to provide instructions for verifying
% the lengths of those cell arrays, and the types of their contents.
%
% verifInstructions should be a 1x2 struct array whose fields have
% names corresponding exactly to those in paramStruct.
%
% The values in verifInstructions(1) are strings indicating the
% lengths of the corresponding values in paramStruct: '1' indicates
% it should contain one member; '+', at least one; '*', any number.
% 'S' indicates that the parameter is a "switch," meaning that if
% its value type must be boolean, and if it occurs in the original
% varargin list, it need not have its value specified -- it will
% just be assumed to be true (but it may be specified explicitly
% as well.) E.g. if a parameter name is 'verbose' for a function f,
% a call such as f('verbose', ...) is equivalent to
% f('verbose', true, ...), and both are valid.
%
% The values in verifInstructions(2) are strings indicating the
% types of the contents of the corresponding values in paramStruct:
% 'string' indicates they should be strings; 'number' indicates
% they should be numbers; 'uint32' indicates they should be
% unsigned 32-bit integers; 'int8' indicates they should be signed
% 8-bit integers; 'boolean' indicates they should be booleans
% (simply 0 or 1.)
%
% Nothing is returned, but if an error is discovered, it is
% reported with the error function (causing the caller to
% terminate).
%
% Examples
%
%    % no error is produced
%    verify_params(struct('a', {{1}}, 'b', {{'hello', 'world'}}, ...
%                         'c', {{100, 105}}), ...
%                  struct('a', {'1', 'boolean'}, ...
%                         'b', {'*', 'string'}, ...
%                         'c', {'+', 'int8'}));
%
%    % an error is produced, because the second value for 'c' does
%    % not fit in an int8
%    verify_params(struct('a', {{1}}, 'b', {{'hello', 'world'}}, ...
%                         'c', {{100, 128}}), ...
%                  struct('a', {'1', 'boolean'}, ...
%                         'b', {'*', 'string'}, ...
%                         'c', {'+', 'int8'}));
%
%    % an error is produced, because 'a' has three values, but it
%    % should have exactly one.
%    verify_params(struct('a', {{1,2,3}}), ...
%                  struct('a', {'1', 'number'}));
%
% See also VARARGIN2STRUCT, RDSAMP, RDANN
%
% Copyright (c) 2009 by Michael Craig, All Rights Reserved
% Contact M. Craig (mic@mit.edu)
%
%    This program is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation; either version 2 of the License, or
%    (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
%    02111-1307  USA
%
% verify_params.m and its dependencies are freely available from Physionet -
% http://www.physionet.org/ - please report any bugs to the authors above.

%
%
%
err = struct('message', {}, 'identifier', {});
fnames = fieldnames(verifInstructions);


%
% verify that the fields of paramStruct and verifInstructions match
%
if length(fieldnames(paramStruct)) ~= length(fieldnames(verifInstructions))
  err(end+1) = struct('message', ['lengths of paramStruct and ' ...
                                  'verifInstructions do not match'], ...
                      'identifier', ['wfdbSwigMatlab:verify_params:' ...
                      'unmatched_fields']);
  return
end

errOccurred = false;
for i = 1 : length(fnames)
  if ~isfield(paramStruct, fnames{i})
    errOccurred = true;
    err(end+1) = struct('message', ...
                        sprintf('field "%s" is not in paramStruct', ...
                                fnames{i}), ...
                        'identifier', ...
                        ['wfdbSwigMatlab:verify_params:' ...
                        'unmatched_fields']);
  end
end
if errOccurred
  return
end


%
% verify that paramStruct consists of only 1xN cell arrays
%
errOccurred = false;
for i = 1 : length(fnames)
  if ~iscell(getfield(paramStruct, fnames{i}))
    errOccurred = true;
    err(end+1) = struct('message', ...
                        sprintf(['field "%s" in paramStruct is not ' ...
                        'a cell array'], fnames{i}), ...
                        'identifier', ...
                        ['wfdbSwigMatlab:verify_params:' ...
                        'param_is_not_cell_array']);
  elseif size(getfield(paramStruct, fnames{i}), 1) > 1
    errOccurred = true;
    err(end+1) = struct('message', ...
                        sprintf(['field "%s" in paramStruct has ' ...
                        'height > 1'], fnames{i}), ...
                        'identifier', ...
                        ['wfdbSwigMatlab:verify_params:' ...
                        'param_is_not_row_vector']);
  end
end
if errOccurred
  return
end


%
% verify that the sizes of paramStruct's fields are correct
% FIXME: implement (and document) '?'
%
for i = 1 : length(fnames)
  param = getfield(paramStruct, fnames{i});
  switch getfield(verifInstructions(1), fnames{i})
   case '1'
    if ~isscalar(param)
      err(end+1) = struct('message', ...
                          sprintf(['field "%s" in paramStruct ' ...
                                   'does not have length 1'], ...
                                  fnames{i}), ...
                          'identifier', ...
                          'wfdbSwigMatlab:verify_params:param_size_invalid');
    end
   case '?'
    if ~isscalar(param) && ~isempty(param)
      err(end+1) = struct('message', ...
                          sprintf(['field "%s" in paramStruct ' ...
                                   'does not have length <= 1'], ...
                                  fnames{i}), ...
                          'identifier', ...
                          ['wfdbSwigMatlab:verify_params:' ...
                          'param_size_invalid']);
    end
   case 'S'
    if ~isscalar(param)
      err(end+1) = struct('message', ...
                          sprintf(['field "%s" in paramStruct ' ...
                                   'does not have length 1'], ...
                                  fnames{i}), ...
                          'identifier', ...
                          'wfdbSwigMatlab:verify_params:param_size_invalid');
    end
    if ~strcmp(getfield(verifInstructions(2), fnames{i}), ...
               'boolean')
      err(end+1) = struct('message', ...
                          sprintf(['field "%s" in paramStruct ' ...
                                   'is a switch (''S'') but does ' ...
                                   'not have type boolean'], ...
                                  fnames{i}), ...
                          'identifier', ...
                          'wfdbSwigMatlab:verify_params:switch_param_not_boolean');
    end
   case '+'
    if length(param) < 1
      err(end+1) = struct('message', ...
                          sprintf(['field "%s" in paramStruct does ' ...
                          'not have length >= 1'], fnames{i}), ...
                          'identifier', ...
                          ['wfdbSwigMatlab:verify_params:' ...
                          'param_size_invalid']);
    end
   case '*'
    ;
   otherwise
    err(end+1) = struct('message', ...
                        sprintf(['invalid size specifier "%s" given ' ...
                                 'for field "%s" in verifInstructions'], ...
                                getfield(verifInstructions(1), ...
                                         fnames{i}), ...
                                fnames{i}), ...
                        'identifier', ...
                        'wfdbSwigMatlab:verify_params:invalid_size_spec');
  end
end


%
% verify that the types of paramStruct's fields are correct
% FIXME: implement (and document) 'boolean' and 'uint32'
%
for i = 1 : length(fnames)
  param = getfield(paramStruct, fnames{i});
  for j = 1 : length(param)
    switch getfield(verifInstructions(2), fnames{i})
     case 'string'
      if (~ischar(param{j}) || size(param{j}, 1) > 1) && ~isempty(param{j})
        err(end+1) = struct('message', ...
                            sprintf(['field "%s" in paramStruct has ' ...
                                     'non-string member'], fnames{i}), ...
                            'identifier', ...
                            ['wfdbSwigMatlab:verify_params:' ...
                            'param_type_invalid']);
        break
      end
     case 'number'
      if ~isnumeric(param{j}) || ~isscalar(param{j})
        err(end+1) = struct('message', ...
                            sprintf(['field "%s" in paramStruct has ' ...
                                     'a member that is not a scalar number'], ...
                                    fnames{i}), ...
                            'identifier', ...
                            ['wfdbSwigMatlab:verify_params:' ...
                            'param_type_invalid']);
        break
      end
     case 'uint32'
      if ~isnumeric(param{j}) || ~isscalar(param{j}) ...
            || (uint32(param{j}) ~= param{j})
        err(end+1) = struct('message', ...
                            sprintf(['field "%s" in paramStruct has ' ...
                                     'a member that is not a scalar, or ' ...
                                     'does not fit appropriately ' ...
                                     'in a uint32'], ...
                                    fnames{i}), ...
                            'identifier', ...
                            ['wfdbSwigMatlab:verify_params:' ...
                            'param_type_invalid']);
        break
      end
    case 'int8'
      if ~isnumeric(param{j}) || ~isscalar(param{j}) ...
            || (int8(param{j}) ~= param{j})
        err(end+1) = struct('message', ...
                            sprintf(['field "%s" in paramStruct has ', ...
                                     'a member that is not a scalar, or ' ...
                                     'does not fit appropriately ' ...
                                     'in an int8'], ...
                                    fnames{i}), ...
                            'identifier', ...
                            ['wfdbSwigMatlab:verify_params:' ...
                             'param_type_invalid']);
        break
      end
    case 'boolean'
      if ~islogical(param{j}) || ~isscalar(param{j})
        err(end+1) = struct('message', ...
                            sprintf(['field "%s" in paramStruct has ' ...
                            'non-boolean member'], fnames{i}), ...
                            'identifier', ...
                            ['wfdbSwigMatlab:verify_params:' ...
                            'param_type_invalid']);
        break
      end
     otherwise
      err(end+1) = struct('message', ...
                          sprintf(['Invalid type specifier "%s" given ' ...
                                   'for field "%s" in verifInstructions'], ...
                                  getfield(verifInstructions(2), ...
                                           fnames{i}), ...
                                  fnames{i}), ...
                          'identifier', ...
                          ['wfdbSwigMatlab:verify_params:' ...
                          'invalid_type_spec']);
      break
    end
  end
end
        
                                    
%
% report any error(s) caught.
%
% FIXME: even though we may have collected more than one struct in err,
%        the error function will only let us report one. this is
%        unfortunate, but i don't see any way around it yet.
%
if ~isempty(err)
  error(err(1));
end

