function [S] = varargin2struct(namedArgs, defaultStruct, sizeStruct)
% function [S] = varargin2struct(namedArgs, defaultStruct, sizeStruct)
% Convert a row vector of name/value pairs into a struct, with field
% names given by the names of the pairs, and field values given by
% the values of those pairs; fill in given defaults, for fields whose
% names did not appear in the row vector.
%
% The intended usage of this function is as a helper for functions
% that want to accept named, optional parameters. For example,
% RDSAMP() takes one required parameter, but has several optional
% named parameters:
%
%    rdsamp('mitdb/100', 'begin', '00:00:30', 'hires', true);
%
% In this call, the last four parameters form the row vector that
% would be passed to VARARGIN2STRUCT() (as namedArgs). 'begin' and
% 'hires' are names, and '00:00:30' and true are their values,
% respectively.
%
% Default values, for fields whose names do not appear in the row
% vector namedArgs, are given in the struct defaultStruct. namedArgs
% must not contain names without corresponding fields in
% defaultStruct.
%
% sizeStruct -- whose fields are expected to match those of
% defaultStruct -- indicates, with a simple string-based code, how
% many values each name is allowed to have. sizeStruct is only used
% here to determine whether a name is a "switch," (indicated by
% value 'S') meaning its value should be a single boolean and, if
% it appears in namedArgs, its value does not need to follow it, in
% which case it is given the value true. For example, because
% 'concise' is a "switch" in RDANN(), these two calls are equivalent:
%
%    rdann('mitdb/100', 'atr', 'concise', 'begin', '00:00:30');
%    rdann('mitdb/100', 'atr', 'concise', true, 'begin', '00:00:30');
%
% (Note that RDANN() takes two required parameters, here
% 'mitdb/100' and 'atr'.) When 'concise' appears, and the argument
% following it is the name of a field in defaultStruct, the value
% for 'concise' is simply assumed to be true. Note that the
% following call is valid, too, with different behavior:
%
%    rdann('mitdb/100', 'atr', 'concise', false, 'begin', '00:00:30');
%
% Precise behavior of VARARGIN2STRUCT() is somewhat difficult to
% specify; the idea is that it should "do the right thing." A name
% may have multiple values, which may be specified right after the
% name, or separately, as in these two equivalent RDSAMP() calls:
%
%    rdsamp('mitdb/100', 'sigs', 0, 1, 'hires', true);
%    rdsamp('mitdb/100', 'sigs', 0, 'hires', true, 'sigs', 1);
%
% As namedArgs is scanned from left to right: when a name appears,
% it is assumed that the parameter to its right is a value (unless
% it is a "switch"), and the parameters to the right of that are
% values until the next name of a field in defaultStruct is
% encountered. If the user desires to specify multiple values to a
% name, and those values are in turn names of other fields in
% defaultStruct, then separate occurrences of that name must occur;
% consider this call to RDANN():
%
%    rdann('mitdb/100', 'atr', 'type', 'chan', 'type', 'num', 'num', 0);
%
% This indicates that 'chan' and 'num' should both be values for the
% name 'type' (the name 'num' itself having the value 0.) Though
% 'chan' and 'num' normally would never appear as values for
% 'type', we make it so, here, only to illustrate functionality.
%
% Because the calls to VARARGIN2STRUCT() are rather complex, it
% makes sense to look at how they are done in functions such as
% RDANN() and RDSAMP().
%
% Examples
%
%    % the call used by RDANN
%    varargin2struct(varargin,
%                    struct('start',{{[]}}, 'stop',{{[]}}, ...
%                           'chan',{{}}, 'num',{{}}, 'subtype',{{}}, ...
%                           'type',{{}}, 'concise',{{false}}),
%                    struct('start',{'1'}, 'stop',{'1'}, ...
%                           'chan',{'?'}, 'num',{'?'}, 'subtype',{'?'}, ...
%                           'type',{'*'}, 'concise',{'S'}));
%
% See also VERIFY_PARAMS, RDSAMP, RDANN
%
% Copyright (c) 2009 by Michael Craig, All Rights Reserved
% Contact M. Craig (mic@mit.edu)
%
%    This program is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation; either version 2 of the License, or
%    (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
%    02111-1307  USA
%
% varargin2struct.m and its dependencies are freely available from Physionet -
% http://www.physionet.org/ - please report any bugs to the authors above.

%
% FIXME: check that namedArgs is a cell array, and nameMapping a
%        struct of strings
%
% FIXME: it may make sense to collect the arguments into matrices
%        rather than into cell arrays
%   
% FIXME: 'S' expects a default, but it should not!
%
%

%
%
%
S = struct();
name = [];
prevName = [];
values = {};
for i = 1 : length(namedArgs)
  
  % is this the first argument?
  if isempty(name)
    if isfield(defaultStruct, namedArgs{i})
      prevName = name; % this should be always redundant
      name = namedArgs{i};
      values = {}; % this should be always redundant
    else
      error('wfdbSwigMatlab:varargin2struct:invalidArgName', ...
            'Invalid name used for "named argument," "%s" (not in defaultStruct)', ...
            namedArgs{i});
    end
    continue
  end
  
  if isempty(values)
    if strcmp(getfield(sizeStruct, name), 'S') && isfield(defaultStruct, namedArgs{i})
      S = catfield(S, 2, name, { true });
      prevName = name;
      name = namedArgs{i};
      values = {};
    else
      values = { namedArgs{i} };
    end
  elseif isfield(defaultStruct, namedArgs{i})
    S = catfield(S, 2, name, values);
    prevName = name;
    name = namedArgs{i};
    values = {};
  else
    values = cat(2, values, { namedArgs{i} });
  end
  
end


%
% commit the final name->value pair
%
if isempty(values)

  if isempty(name)
    % zero args. passed in total
    ;
  elseif isempty(prevName)
    % one arg. passed in total
    if strcmp(getfield(sizeStruct, name), 'S')
      % the arg. is a name for a "switch" ('S') parameter
      S = catfield(S, 2, name, { true });
    else
      error('wfdbSwigMatlab:varargin2struct:onlyOneArg', ...
            'Only one argument was passed, a name without a value');
    end
  elseif strcmp(getfield(sizeStruct, name), 'S')
    S = catfield(S, 2, name, { true });
  else
    % here, name is actually a value for prevName, because it's the
    % last arg. in namedArgs
    S = catfield(S, 2, prevName, { name });
  end

else
  S = catfield(S, 2, name, values);
end


%
% fill in defaults as necessary
%
fields = fieldnames(defaultStruct);
for i = 1 : length(fields)
  if ~isfield(S, fields{i})
    S = setfield(S, fields{i}, ...
                    getfield(defaultStruct, fields{i}));
  end
end


  