function [S] = catfield(S, dim, field, values)
% function [S] = catfield(S, dim, field, values)
% Concatenate new data onto a specific field of the given struct.
%
% This is essentially a composition of the functions SETFIELD(),
% CAT(), and GETFIELD(). A call CATFIELD(S, D, F, V) assumes
% that the data stored in field F of struct S are a matrix or
% a cell array, and uses CAT() to concatenate V onto that data,
% along dimension D.
%
% Common usage is to append a single element onto a single-row
% field of a struct, so that dim is 2 and values is a scalar.
%
% If the field specified doesn't previously exist in S, it is
% newly created to hold the given values.
%
% Examples
%
%    % assuming s.a is empty or a row vector, append the value 15
%    % onto it, updating struct s
%    s = catfield(s, 2, 'a', 15)
%
%    % assuming s.a is empty or a row vector, append the values 15,
%    % 16, and 17 onto it, updating struct s
%    s = catfield(s, 2, 'a', [15 16 17])
%
%    % assuming s.b is empty or is a cell array of length 3, append
%    % the two rows {18 19 20} and {21 22 23} onto it, updating
%    % struct s
%    s = catfield(s, 1, 'b', {18 19 20; 21 22 23})
%
% See also CAT
%
% Copyright (c) 2009 by Michael Craig, All Rights Reserved
% Contact M. Craig (mic@mit.edu)
%
%    This program is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation; either version 2 of the License, or
%    (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
%    02111-1307  USA
%
% catfield.m and its dependencies are freely available from Physionet -
% http://www.physionet.org/ - please report any bugs to the authors above.



if isfield(S, field)
  S = setfield(S, field, cat(dim, getfield(S, field), values));
else
  S = setfield(S, field, values);
end


