function [str] = deepconvert(obj)
% function [str] = deepconvert(obj)
% Perform "deep" conversion of a Java object into a MATLAB struct:
% convert simple-typed fields directly, convert String objects to
% Matlab strings, and recurse on fields that contain other Java
% objects.
%
% NOTE: this does not work on objects that contain cyclic
% references, so care must be used not to avoid applying this
% function to such objects.
%
% Bugs
%
%    If an array is to be converted (at any depth during the
%    conversion) that contains some null values, an error will
%    occur.
%
%    If a 0-element array is to be converted, a 0x0 struct
%    array with no fields will be created; it would be more
%    proper for the returned struct array to have the fields
%    of the given array's element type.
%
% Examples
%
%    % convert a Java object of type Dimension into a struct
%    % with two fields, 'width' and 'height'
%    ds = deepconvert(java.awt.Dimension(5, 10))
%
%    ds = 
%
%        width: 5
%       height: 10
%
%
%
%    % convert a Java array of element-type Dimension into a
%    % struct array with two fields, 'width' and 'height'
%    da(1) = java.awt.Dimension(5, 10);
%    da(2) = java.awt.Dimension(15, 20);
%    das = deepconvert(da)
%
%    das =
%
%    1x2 struct array with field:
%        width
%        height
%
%
%
%    % convert a Java string into a single-row char array
%    cs = deepconvert(java.lang.String('hello'))
%
%    cs =
%
%    hello
%
%
%
% See also STRUCT2JAVAARRAY
%
% Copyright (c) 2009 by Michael Craig, All Rights Reserved
% Contact M. Craig (mic@mit.edu)
%
%    This program is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation; either version 2 of the License, or
%    (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
%    02111-1307  USA
%
% deepconvert.m and its dependencies are freely available from Physionet -
% http://www.physionet.org/ - please report any bugs to the authors above.


% FIXME: some cases should really be optimized, such as converting
%        an array of non-array objects (we only need to call
%        fieldnames() once, not N times!)

% FIXME: null values are not always handled correctly, e.g. when
%        an array is converted and some of the elements are null
%


if isjava(obj)
  
  if obj.getClass().isArray()

    % special behavior if obj is an array of String's: make it a
    % cell array of char vectors -- we can't make it a normal
    % "vector of vectors," because that would just be a matrix, so
    % all of the char vectors would need the same length.
    % FIXME: document this behavior in the help text above.
    % FIXME: comparing against the type signature -- though it's
    %        well documented in the javadocs -- feels unsafe
    if strcmp(obj.getClass().getName(), '[Ljava.lang.String;')
      str = {};
      for i = 1:obj.length()
        str{i} = deepconvert(obj(i));
      end
    else
      if obj.length() > 0
        for i = 1:obj.length()
          str(i) = deepconvert(obj(i));
        end
      else
        % FIXME: this creates a fieldless 0x0 struct array, which is
        %        not quite what we want. instead we'd like a 0x0
        %        struct array with the proper fields for the type (of
        %        which obj is an array). as far as i can tell, this
        %        is only possible with a matlab call like
        %        struct('field1', {}, 'field2', {}, ...), which does
        %        not give us the dynamism we require here. the only
        %        way i see how, currently, is using eval, which is
        %        tedious enough that i am punting on it for now.
        str = struct([]);
      end
    end
  
    % FIXME: there's probably a better way to do this...
  elseif obj.getClass().getName().equals('java.lang.String')
    str = obj.toCharArray()';
  else
    str = struct(obj);
    fnames = fieldnames(str);
    for i = 1:length(fnames)
      str = setfield(str, fnames{i}, deepconvert(getfield(str, ...
                                                        fnames{i})));
    end
  end
  
else
  
  str = obj;
  
end