function [anns] = rdann(record, annotator, varargin)
%RDANN read annotation files for WFDB records
% Read the annotation file for the specified record and annotator,
% and return it as a human-readable struct array, with one entry
% per annotation. A 'concise' option can be used to indicate that,
% instead, only the times of annotations should be returned; in
% this case, the return value is not a struct array, but a
% two-column matrix.
%
% The returned non-concise struct array has the following fields:
% timeInSeconds (number): the time of the annotation in seconds
% sampleNumber (uint): the time of the annotation in samples
% typeMnemonic (string): the mnemonic for the type of the annotation
% subtype (uint8): the annotation's subtyp field
% chan (uint8): the annotation's chan field
% num (uint8): the annotation's num field
% auxInfo (string): a string of auxiliary information for the annotation
%
% Two required arguments:
% record: the name of the WFDB record
% annotator: the name of the annotator for the given record
%
% Six optional arguments (default values in []s):
% 'start' (time-format string): the time in the record from which
%    to begin reading ['00:00:00']
% 'stop' (time-format string): the time in the record at which to
%    stop reading [end of record]
% 'chan' (int8): the "chan" number for annotations to be returned (all
%    other annotations are not returned) []
% 'num' (int8): the "num" number for annotations to be returned (all
%    other annotations are not returned) []
% 'subtype' (int8): the "subtyp" number for annotations to be returned (all
%    other annotations are not returned) []
% 'type' (1 or more strings): mnemonics of the types for annotations to be
%    returned (all other annotations are not returned) []
% 'concise' (0 or 1 boolean): return only two columns: the first of
%    annotation times, and the second of annotation sample numbers
%
% All six arguments are essentially simple filters on the data to
% be returned.
%
% Examples
%
%    % return all annotations of the record 'mitdb/100' with
%    % the annotator 'atr'
%    rdann('mitdb/100', 'atr')
%
%    % return all annotations of the record 'mitdb/100' with
%    % the annotator 'atr', and with the type 'A' or the type '+'
%    rdann('mitdb/100', 'atr', 'type', 'A', '+')
%
%    % return all annotations of 'mitdb/100' between 1 min. and
%    % 1 min. 30 secs., with chan 0 and subtype 0
%    rdann('mitdb/100', 'atr', 'start', '00:01:00', 'stop', ...
%          '00:01:30', 'chan', 0, 'subtype', 0)
%
% See also RDSAMP, WRSAMP, WRANN, WFDBDESC, rdann(1) man page,
%    http://physionet.org/physiobank/annotations.shtml
%
% Copyright (c) 2009 by Michael Craig, All Rights Reserved
% Contact M. Craig (mic@mit.edu)
%
%    This program is free software; you can redistribute it and/or modify
%    it under the terms of the GNU General Public License as published by
%    the Free Software Foundation; either version 2 of the License, or
%    (at your option) any later version.
%
%    This program is distributed in the hope that it will be useful,
%    but WITHOUT ANY WARRANTY; without even the implied warranty of
%    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
%    GNU General Public License for more details.
%
%    You should have received a copy of the GNU General Public License
%    along with this program; if not, write to the Free Software
%    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
%    02111-1307  USA
%
% rdann.m and its dependencies are freely available from Physionet -
% http://www.physionet.org/ - please report any bugs to the authors above.


%
argsInfo = struct('start',   { {[]},    '1', 'string'  }, ...
                  'stop',    { {[]},    '1', 'string'  }, ...
                  'chan',    { {},      '?', 'int8'    }, ...
                  'num',     { {},      '?', 'int8'    }, ...
                  'subtype', { {},      '?', 'int8'    }, ...
                  'type',    { {},      '*', 'string'  }, ...
                  'concise', { {false}, 'S', 'boolean' });

%
p = varargin2struct(varargin, argsInfo(1), argsInfo(2));
verify_params(p, argsInfo(2:3));

%
% FIXME: these three statements seem rather inelegant
%
if ~isempty(p.chan)
  p.chan{1} = java.lang.Byte(p.chan{1});
end

if ~isempty(p.num)
  p.num{1} = java.lang.Byte(p.num{1});
end

if ~isempty(p.subtype)
  p.subtype{1} = java.lang.Byte(p.subtype{1});
end


%
% Note that for any 1-element cell array A, the expressions
% cell2mat(A) and A{1} are equivalent. The benefit of using
% cell2mat is that 0-element cell arrays are converted to [],
% or null in Java.
%

% anns = deepconvert(rdannMatlab.rdann(record, ...
%                                     annotator, ...
%                                     p.start{1}, ...
%                                     p.stop{1}, ...
%                                     cell2mat(p.chan), ...
%                                     cell2mat(p.num), ...
%                                     cell2mat(p.subtype), ...
%                                     p.type));

%
LoadWFDBJava.load;
wfdb.wfdb.wfdbquit;

%
javaAnns = rdannMatlab.rdann(record, ...
                             annotator, ...
                             p.start{1}, ...
                             p.stop{1}, ...
                             cell2mat(p.chan), ...
                             cell2mat(p.num), ...
                             cell2mat(p.subtype), ...
                             p.type);
if p.concise{1}
  % FIXMEFIXME: this is inefficient enough that it should be
  %             done in java!
  anns = [];
  for i = 1:length(javaAnns)
    anns(i,1) = javaAnns(i).timeInSeconds;
    anns(i,2) = javaAnns(i).sampleNumber;
  end
else
  anns = deepconvert(javaAnns);
end
