function BatchTWA(varargin)
% BatchTWA.m
% Author: Alexander Khaustov; alexander dot khaustov at gmail dot com 
% Copyright (C) 2008 St.-Petersburg Institute of Cardiological Technics (Incart), www.incart.ru
% This software is released under the terms of the GNU General
% Public License (http://www.gnu.org/copyleft/gpl.html).
% 
% This is the routine to run if you want TWA analysis for several records
% at once 
%
% accepts a file with a list of ECG records in MIT WFDB format (no
% extensions in the list) and outputs alternans values into a single file
% (see filenames for fres in the code below)
%
% Expects annotations with extension .vf (verified) that contain q, s and t end labels
% for each beat (the format should be the same as produced by ecgpuwave); 
% if there's no such annotation file - ecgpuwave is used to generate 
% annotations
%
% Alter 'Metric' and 'Method' lists (see the code below) to choose between availiable
% methods of analysis: MMA (modified moving average), SM (spectral method) with invalid (i.e. noisy or ectopic) beat
% replacement ('standard'), SM with invalid beat replacement base on first difference alternans seris ('differences'), SM using
% lomb periodogram ('lomb')

    if ~nargin
        [fname, pathname] = uigetfile('*.*');
        if ~fname
            return;
        end;
        BatchTWA(strcat(pathname, fname));
        return;
    end;    

    global Param
    Param.Alignment = 'st';    

%     remove strings if you don't need a method
%     e.g.
%     Metric = cellstr({'SM', 'MMA'});
%     Method = cellstr({'standard', ''});
    Metric = cellstr({'SM', 'SM', 'SM', 'MMA'});
    Method = cellstr({'standard', 'differences', 'lomb', ''});
    File  = cellstr({'twa_SM_standard.txt', 'twa_SM_differences.txt', 'twa_SM_lomb.txt', 'twa_MMA.txt', });

    for i = 1:length(Metric)
        f = fopen(varargin{1});
        
        Param.Metric = Metric{i};
        Param.MethodForEctopy = Method{i};
        fres = fopen(File{i}, 'w');
    
        nextf = fgetl(f)
        while (nextf ~= -1)
            clear global TWARes ecg
        
            TWA(nextf);
        
            fprintf(fres, '%s ', nextf);
        
            if (strcmp(Param.Metric, 'MMA'))
                fprintf(fres, MMARes);
            else
                fprintf(fres, SMRes);
            
% Write the number of replaced/ignored beats to the results if you like  
%             global Align        
%             if (strcmp(Param.MethodForEctopy, 'standard') && ~isempty(Align.fid))
%                 NInvalid = length(Align.valid) - length(find(Align.valid));
%                 if NInvalid ~= 0
%                     fprintf(fres, '%d beats replaced', NInvalid);
%                 end;
%             end;
            end;        
            
            fprintf(fres, '\n');        
            nextf = fgetl(f)
        end;
        fclose all;
    end;
 return;
 
function s = MMARes

global TWARes           

if (~isempty(TWARes) && TWARes.successfull)
    
    % MMA metric is unstable, use some sort of
    % averaging to get the overall file result
    res = zeros(1, size(TWARes.VAltTrend, 2));
    IntvInARow = 4;
    for lead = 1:size(TWARes.VAltTrend, 2)
        for j = 1:size(TWARes.VAltTrend, 1) - IntvInARow + 1
            curr = min(TWARes.VAltTrend(j:j + IntvInARow - 1, lead));
            res(lead) = max(res(lead), curr);
        end;
    end;
    s = sprintf('%.2f ', max(res));
    %                 fprintf(fres, '%.2f ', max(max(TWARes.VAltTrend)));
else
    s = '0.00 ';
end;

return;

function s = SMRes

global TWARes Param

signifVAlt = [];
if (~isempty(TWARes))
    signifs = find(TWARes.(Param.MethodForEctopy).significant);
    signifVAlt = TWARes.(Param.MethodForEctopy).VAlt(signifs);
end;
if (length(signifVAlt))
    s = sprintf('%.2f ', max(signifVAlt));
else
    s = '0.00 ';   
end;

return;