%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%%% GetChallengeFeatures_CinC2021_v01
%%% Extract ECG Features from the specified input signal
%%%
%%% Inputs:
%%% Y  - The input signal
%%% Fs - Frequency sampling
%%%
%%% Outputs:
%%% features - Array of signal features
%%%
%%% Author:  Santiago Jiménez-Serrano [sanjiser@upv.es]
%%% Version: 1.0
%%% Date:    2020-03-26
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [features] = GetChallengeFeatures_CinC2021_v01(Y, Fs)
%GETCHALLENGEFEATURES_CINC2021_V01 Extract ECG Features from the specified input signal

    % 1st stage - Signal Filtering
    [Y, Fs] = SignalFiltering_CinC2021_v01(Y', Fs);

    % Get qrs index 
    [ ~, qrs_indices, qrs_values ] = getQRS( Y, Fs );
    QRSms = (qrs_indices / Fs)*1000; % QRS waves occurrences in ms
    
    %%% T    
    msROffset = 100;
    msWindow  = 300;
    [ t_indices, t_values, diff_qt_ms ] = getTWave_mini( Y, Fs, qrs_indices, msROffset, msWindow );
    
    %%% Patrón QRS
    msOffset  = 100;    
    [ patron_r, per_descartes_r, rmse_r, amp_r ] = getQRS_Pattern_mini( Y, Fs, qrs_indices, msOffset );
    
    % Patrón T
    msOffset  = 100;    
    [ patron_t, per_descartes_t, rmse_t, amp_t ] = getQRS_Pattern_mini( Y, Fs, t_indices, msOffset);
    
 
    
    % 
    fvolt_r = getMuSg(qrs_values);   % r voltages       stats
    fvolt_t = getMuSg(t_values);     % t voltages       stats
    fqt     = getMuSg(diff_qt_ms);   % qt interval (ms) stats
    
    fqt_volt    = [0, 0, 0, 0, 0, 0, 0];
    fqt_volt(1) = amp_t ./ amp_r;
    fqt_volt(2) = sign(fvolt_t(1))+1; %[0, 2]
    fqt_volt(3) = sign(fvolt_r(1))+1; %[0, 2]
    fqt_volt(4) = max(abs(diff(patron_r)));
    fqt_volt(5) = max(abs(diff(patron_t)));
    fqt_volt(6) = max(abs(diff(patron_r, 2)));
    fqt_volt(7) = max(abs(diff(patron_t, 2)));

    pattern_features = [per_descartes_r, rmse_r, per_descartes_t, rmse_t ];
    
    %img_pr = getImagePatron(patron_r);
    %img_pt = getImagePatron(patron_t);
    
    [fspect] = getSpectralF(Y, Fs);
    
    
    rr   = diff(QRSms);        % RR in ms
    rr   = Filter_Outliers(rr);  % Mejores resultados filtrando      
    rrd1 = diff(rr);
    rrd2 = diff(rrd1);
    
    
    
    


     % Todas estas son las óptimas
     f0    = getMuSgKuSk(rr);             % Todas estas las medidas aportan información
     f1    = getMuSgKu(rrd1);             % Filtramos media y proporcion, las demás son las que mejor resultado dan en conjunto
     f2    = getSgKu(rrd2);               % Filtramos media y proporción, las demás son las que mejor resultado dan en conjunto
     
     %%%%% fdt   = getDistanceToPnnCurve(rrd1);                     %  4 Features [01-04] ***** 08
     ff    = getChallengeFeaturesAux(qrs_indices, length(Y)); % 44 Features [05-48]
     fcx   = getRRComplexFeatures(rr);                        %  2 features [49-50]
     frr   = [f0, f1, f2];                                    %  9 features [51-59]
     pf    = getPoincareFeatures(rr);                         %  4 features [60-63]
     lf    = getLorenzFeatures(rr);                           %  8 features [64-71]
     ppc   = getPPC(Y, Fs, qrs_indices);                      %  1 features [  -72]
     
     [pcaf] = getFeaturesPCA(rrd1);
 
     
     % Final features vector
     features = [fspect, fvolt_r, fvolt_t, fqt, fqt_volt, pattern_features, ff, fcx, frr, pf, lf, ppc, pcaf]; % sin pnn-curve
     
     

end



%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Basic Stats Functions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [s] = getMu(v)
    if isempty(v), s = 0;
    else,          s = nanmean(v);
    end
end

function [s] = getMuSg(v)
    if isempty(v), s = [0, 0];
    else,          s = [nanmean(v), nanstd(v)];
    end
end

function [s] = getSgKu(v)
    if isempty(v), s = [0, 0];
    else,          s = [nanstd(v), kurtosis(v)];
    end
end

function [s] = getMuSgKu(v)
    if isempty(v), s = [0, 0, 0];
    else,          s = [nanmean(v), nanstd(v), kurtosis(v)];
    end
end

function [s] = getMuSgKuSk(v)
   
    if isempty(v)
        s = [NaN, NaN, NaN, NaN];
    else        
        s = [nanmean(v), nanstd(v), kurtosis(v), skewness(v)];
    end
end

function [ x, filter, nout ] = Filter_Outliers( x )

    % Umbrales para outliers
    m = median(x);
    s = std(x);
    
    th_up = m + (3.0 * s);
    th_dn = m - (3.0 * s);
    
    filter    = x<th_dn | x>th_up;
    nout      = sum(filter);
    x(filter) = [];

end


%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Dynamic Time Warping - Auxiliar Function
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function d=mydtw(s,t,w)
% s: signal 1, size is ns*k, row for time, colume for channel 
% t: signal 2, size is nt*k, row for time, colume for channel 
% w: window parameter
%      if s(i) is matched with t(j) then |i-j|<=w
% d: resulting distance

    if nargin<3
        w=Inf;
    end

    ns=size(s,1);
    nt=size(t,1);
    if size(s,2)~=size(t,2)
        warning('Error in dtw(): the dimensions of the two input signals do not match.');
        d = 100000;
        return;
    end
    w=max(w, abs(ns-nt)); % adapt window size

    %% initialization
    D=zeros(ns+1,nt+1)+Inf; % cache matrix
    D(1,1)=0;

    %% begin dynamic programming
    for i=1:ns
        for j=max(i-w,1):min(i+w,nt)
            oost=norm(s(i,:)-t(j,:));
            D(i+1,j+1)=oost+min( [D(i,j+1), D(i+1,j), D(i,j)] );

        end
    end
    d=D(ns+1,nt+1);

end



%%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% RR Features functions
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
function [f] = getChallengeFeaturesAux(rindx, n) % 44 Features


    Fs    = 300;
    QRSms = (rindx / Fs)*1000; % Posición de los QRS en mili-segundos
    
    %rindx
    %n = length(y);
    win = 2400;
    ste = 1200;
    
    M = [];
  
    for w = 1 : ste: n-win
        start = w;
        stop  = start+win;
        
        idx = (rindx >= start & rindx <= stop);
        qqq = QRSms(idx);
        rr  = diff(qqq);
        
        if length(rr) > 2
            M = [M; getChallengeFeaturesI(rr)];
        end

    end    
    
    [n, ~] = size(M);
    
    if n == 0
        f = zeros(1, 11*4); % 11 features each
    elseif n == 1
        f = [M, M, M, zeros(1, 11)];
    else
        f = [nanmin(M), nanmax(M), nanmean(M), nanstd(M)];
    end   

end
    

function [ f ] = getChallengeFeaturesI( rr )
    
    rrd1 = diff(rr, 1); % Poincaré
    rrd2 = diff(rr, 2); % Lorenz

    % Todas estas son las óptimas
    f0  = getMuSgKu(rr);           % Todas estas las medidas aportan información
    f1  = getSgKu(rrd1);           % Filtramos media y proporcion, las demás son las que mejor resultado dan en conjunto
    f2  = getSgKu(rrd2);           % Filtramos media y proporción, las demás son las que mejor resultado dan en conjunto
    fpf = getPoincareFeatures(rr); % Poincaré Features for this sequence
    
    f  = [f0, f1, f2, fpf];
end





function [results] = getRRComplexFeatures(rr)
    
    % Antes Features 1
    entRR=shannonEntropy(rr);
    
    % Antes Feature 2 (lzcrr)
    aux = rr>median(rr);
    
    if length(aux) < 3
        lzcrr = entRR;
    else
        lzcrr=calc_lz_complexity(aux, 'exhaustive', 1);    
    end 

    results = [entRR, lzcrr];    

end

function [ppc] = getPPC(y, Fs, qrs_idx)

    % Test for errors
    if nargin < 3
        qrs_idx = [];
    elseif length(qrs_idx) < 3
        ppc = 1;
        return;        
    end
    
    % Antes Feature 5
    %plot(y)
    
    % Test if it is a Noisy Record
    [~,nQRS]=getQRSpattern(y', qrs_idx, 1200, Fs);
    ppc=length(nQRS)/sum(nQRS);
end

function [ values ] = getPoincareFeatures( rr ) %% rr ha de estar en ms

    if isempty(rr)
        values = ones(1, 4);
        return;
    end

    rrd1 = diff(rr);
    
    % RMSSD (Antes Feature 6)
    RMSSD=sqrt(mean(rrd1.^2));
    
    % PNN50 (Antes Features 4)
    n    = length(rrd1);
    NN25 = sum(rrd1>25);
    NN50 = sum(rrd1>50); % > 50 ms... las medidas están en segundos
    NN75 = sum(rrd1>75);
    pNN25=NN25/n;
    pNN50=NN50/n;
    pNN75=NN75/n;    

    values = [RMSSD, pNN25, pNN50, pNN75];
    
end

function [results] = getLorenzFeatures(rr)
    
    if isempty(rr) || length(rr) < 3
        results = zeros(1, 8);
        return;
    end

    n     = length(rr);
    x     = rr(1:end-1);
    y     = rr(2:end);
    theta = atand(x ./ y);
    li    = sqrt(x.^2 + y.^2); % Suma de distancias al origen (sdo)
    L     = mean(li);
    VAI   = sum(abs(theta-45))./(n-1);
    VLI   = sqrt(sum((li-L).^2))./(n-1);
    
    rrd1 = diff(rr);
    x    = rrd1(1:end-1);
    y    = rrd1(2:end);
    
    % Suma de distancias al origen
    sdo = sqrt(x.^2 + y.^2); 
   
    % Suma de distancias entre puntos consecutivos
    sdp = sqrt((x(1:end-1)-x(2:end)).^2 + (y(1:end-1)-y(2:end)).^2);
    
    % Diferencias entre distancias de 3 en 3 puntos, con ventana de 1
    dife = sqrt((sdp(1:end-1) - sdp(2:end)).^2);
    
    
    results = [VAI, VLI, getMuSg(sdo), getMuSg(sdp), getMuSg(dife)];
end




function [f] = getDistanceToPnnCurve(rrd1ms)
    
	SS=load('mediaPNN.mat');
    mediaPNN=SS.mediaPNN;

    rrd1n  = length(rrd1ms);
    
    pnn   = 10:1:150;
    pnnMi = zeros(1, length(pnn));
    for pnni = 1:length(pnn)
        NN50 =sum(rrd1ms>pnn(pnni)); % > 50 ms... las medidas están en segundos
        pNN50=NN50/rrd1n;
        pnnMi(pnni) = (pNN50);
    end  

    
    da = mydtw(pnnMi, mediaPNN(1, :)); % Distance to the mean Atrial Curve
    dn = mydtw(pnnMi, mediaPNN(2, :)); % Distance to the mean Normal Curve
    do = mydtw(pnnMi, mediaPNN(3, :)); % Distance to the mean Others Curve
    dp = mydtw(pnnMi, mediaPNN(4, :)); % Distance to the mean Noise  Curve
    
    d = pnnMi-mediaPNN(1, :); d1 = sum(d.*d); % sqrt(diff(pnnMi, mean Atrial)^2)
    d = pnnMi-mediaPNN(2, :); d2 = sum(d.*d); % sqrt(diff(pnnMi, mean Normal)^2)
    d = pnnMi-mediaPNN(3, :); d3 = sum(d.*d); % sqrt(diff(pnnMi, mean Others)^2)
    d = pnnMi-mediaPNN(4, :); d4 = sum(d.*d); % sqrt(diff(pnnMi, mean Noise)^2)
    
    f = [da, dn, do, dp, d1, d2, d3, d4];
end






function [f] = getFeaturesPCA(rrd1)

   
    dmax = -Inf;
    dmin = +Inf;
    dsum = 0;
    n    = length(rrd1)-1;
    dd = [];
    
    for i = 1: n
        for j = i+1 : n
             if i ~= j
                p1 =[rrd1(i), rrd1(i+1)];
                p2 =[rrd1(j), rrd1(j+1)];

                dx = p1-p2;
                d = sqrt(sum((dx.*dx)));

                if(d > dmax), dmax = d; end
                if(d < dmin), dmin = d; end
                dsum = dsum + d;
                dd = [dd, d];
             end
        end
    end
    
    f = NaN(1, 7);
    
    if n > 1
    
        f(1) = dmax;
        f(2) = dmin;
        f(3) = abs(dmin-dmax);
        f(4) = mean(dd);
        f(5) = std(dd);
        f(6) = kurtosis(dd);
        f(7) = skewness(dd);    
    end
    
end



function [img] = getImagePatron(patron)

    
    patron = patron - min(patron);
    patron = patron ./ max(patron);
    
    
    
    img = zeros(10, 10);    
    sig = zeros(10, 1);
    
    
    sig( 1) = int32(mean(patron( 1:10))*10)+1;
    sig( 2) = int32(mean(patron(11:20))*10)+1;
    sig( 3) = int32(mean(patron(21:30))*10)+1;
    sig( 4) = int32(mean(patron(31:40))*10)+1;
    sig( 5) = int32(mean(patron(41:50))*10)+1;
    sig( 6) = int32(mean(patron(51:60))*10)+1;
    sig( 7) = int32(mean(patron(61:70))*10)+1;
    sig( 8) = int32(mean(patron(71:80))*10)+1;
    sig( 9) = int32(mean(patron(81:90))*10)+1;
    sig(10) = int32(mean(patron(91:end))*10)+1;
    
    %sig
    
    for i = 1 : 10
        if sig(i) < 1
            img( 1, i) = 1;
        elseif sig(i) > 10
            img(10, i) = 1;
        else
            img(sig(i), i) = 1;
        end
    end
    
    %plot(patron);
    %imtool(img);    
    %pause
    img = reshape(img, 1, 100);

end

function [r] = getSpectralF(Y, Fs)

    %[Pxx, F, T, FDom, PercentAreaFDom] = AnalisisEspectralPWelch(Y, Fs, true);
    
    [~, ~, FDom, PercentAreaFDom, Accum] = AnalisisEspectralPWelch_mini(Y, Fs);
%     FDom
%     PercentAreaFDom
%     Accum
%     size(Accum)
%     plot(Accum);
%     
%     pause;

    r = [FDom, PercentAreaFDom, Accum];

end









