%% This function implements a timeseries-filter
% The basic implementation is from Wichterle2004 (also used by Karlsson2012
% and Wiklund2008). The filter interpolates the value between the two
% adjacent values; if the difference between the value under consideration
% and the interpolated value is too big, it's sorted out
% NOTE: a linear interpolation can be done where the adjacent values contribute 
% equally to the interpolation (i.e. interpolValue=(value_before+value_after)/2); 
% BUT the DEFAULT is: interpolation incorporates the distance of the value which is
% interpolated from it's adjacent neighbors (i.e.
% interpolValue=(value_after-value_before)/(time_after-time_before)*(time_t
% o_be_interpolated-time_before)+value_before)
%
%
% input:
%   timeseries:             Timeseries to be filterd; an
%                           array of size 1xN is expected
%   positions:              Time instants which belong to the timeseries; an
%                           array of size 1xN is expected (equal size than
%                           timeseries)
%
%
% output:
%   usedIndices:            Vector to indicate which elements of the
%                           input series was used
%   filteredPositions:       Occurrence times which belong to the filtered
%                           time series
%   filteredTimeseries:     Filtered times series
% Options:
%   'Deviation':            Allowed deviation from interpolated value in
%                           percent;numerical input is expected (DEFAULT is 15%)


function [usedIndices,filteredPositions,filteredTimeseries]=...
    timeseriesFilterPercentage(timeseries,positions, varargin)

%% set parameters
maximumDeviation=15; %maximum deviation in time series in percent
okargs =   {'Deviation'};


%% evaluate input arguments (future versions)

if(nargin<2)%leave while loop if there are no more new variable arguments
    errordlg(['Too few arguments']);
    return;
end
    
i=1;
while(1)
    
    if(mod(size(varargin,2),2)~=0) %check if number of variable arguments is even
            errordlg(['Uneven number of arguments not supported']);
            return;
    end
    
    if(i>size(varargin,1))%leave while loop if there are no more new variable arguments
        break;
    end
    
    if(ischar(varargin{i}))%evaluate key word of next argument
        if(ismember(varargin{i},okargs))
            actualArgument=varargin{i};
        else
            errordlg(['Specified Option ''' varargin{i} ''' is not a valid option']);
            return;
        end
        
    else
        errordlg(['Specified Option ''' varargin{i} ''' is not a valid option']);
        return;
    end
    i=i+1;
    
    switch actualArgument %set values for the actual arguments
        case 'Deviation'%sampling frequency
            if(~isnumeric(varargin{i}))
                errordlg(['Deviation needs to be numeric (in percent)']);
                return;
            end
            maximumDeviation=varargin{i};
    end
end
    
actualTimeSeries=timeseries;%store input time series
actualPositions=positions;%store timestamps which belong to the time series
actualOriginalIndices=1:length(actualTimeSeries);%store orignal indices of each time series element




%% do calculation
while(1)%loop until no outlier is existent anymore
    
    extendedTimeSeries=[2*actualTimeSeries(1)-actualTimeSeries(2) actualTimeSeries 2*actualTimeSeries(end)-actualTimeSeries(end-1)];%add values to calculate meaningful convolutions even at the end and at the beginning of time series
    if(0)
        interpolatedTimeSeries = conv(extendedTimeSeries,[0.5 0 0.5],'valid');%calculate interpolated time series -> due to extension and the 'valid' option values should belong to same indices in vector actualTimeSeries
    else
        extendedPositions=[2*actualPositions(1)-actualPositions(2) actualPositions 2*actualPositions(end)-actualPositions(end-1)];%build vector of positions
        diff3Positions=conv(extendedPositions,[1 0 -1],'valid');%calculate difference between positions around value to be interpolated
        diff2Positions=conv(extendedPositions,[1 -1],'valid');%calculate distance from interpolated value to value before 
        diffValues=conv(extendedTimeSeries,[1 0 -1],'valid');%calculate difference between values before and after value to be interpolated
        interpolatedTimeSeries=extendedTimeSeries(1:end-2)+diffValues./diff3Positions.*(diff2Positions(2:end));%do interpolation
    end
    acceptanceRange=interpolatedTimeSeries*(maximumDeviation/100);%calculate the percentage value which is an allowed deviation
    outliers=abs(actualTimeSeries-interpolatedTimeSeries) > acceptanceRange;%find outliers
    if(sum(outliers)==0)%no more outliers --> break loop
        break;
    end
    actualTimeSeries=actualTimeSeries(~outliers);%remove outliers from timeseries
    actualPositions=actualPositions(~outliers);%remove outliers from actual positions
    actualOriginalIndices=actualOriginalIndices(~outliers);%remove outliers from original indices
end


%% write output
filteredTimeseries=actualTimeSeries;
filteredPositions=actualPositions;
usedIndices=actualOriginalIndices;
end