function [Signal,Info] = readPoly5(FileName)

% Former FUNCTION [Signal,Info] = ReadPoly5(FileName)
%
% FILENAME
% Name of the file to import into Matlab 5.2 & 5.3
% If no filename is supplied a selection box will be prompted
%
% SIGNAL
% Signal is a cell array containing the signals. In each cell a signal is stored
% Each cell contains fields referring to the signal. e.g. Signal{1}.Name contains
% the name of the stored data and Signal{1}.Data contains the sampled data.
%
% INFO
% Info is a structure containing the data from the master header of the file
% All fields in the Info structure correspond to a field in the header of the data
%
% ir. JG Stinstra 
% email j.g.stinstra@tn.utwente.nl
% September 1999, University of Twente
% ---
% file_vers=0 - txt or ascii file format readable (jh, October 2001)
% file_vers=1 or 2 - Poly5 and TMS32 file format readable (jh, April 2001)
% file_vers=3 - Beatscope out file format readable (jh, October 2001)
%   out-files must have 3 channels
% file_vers=4 - MDC et1_ and et2_ file format readable (jh, October 2001)
%   et1_ file must have 4 channels, et2_ file must have 6 channels

if nargin < 1,
   fltspc{1}='*.S0*;*.out;*.et1_*;*.et2_*;*.txt';
   % Promp a selection box
   [FName,FilePath] = uigetfile(fltspc,'Load data...');
   if FilePath==0 return; end;
   FileName = fullfile(FilePath,FName);
end;
clear fltspc

% Determine the file format
Version203 = ['POLY SAMPLE FILEversion 2.03' char([13 10 26])];
Version204 = ['POLY SAMPLE FILE version 2.04' char([13 10])];
et_Vers   = ['MDC V2.0'];% first 8 chars of header of et1_ & et2_ file
file_vers  = 0; Signal=0; Info=0;

FID = fopen(FileName,'r');
if FID < 0,
   error('Could not open file');
end

Version = fscanf(FID,'%31c',1);
if strcmp(Version,Version203)==1,
   StartHeader = 31; file_vers=1;
elseif strcmp(Version,Version204)==1,
   StartHeader = 32; file_vers=2;
elseif strcmp('.out',lower(FileName(find(FileName=='.'):find(FileName=='.')+3))) ==1
    StartHeader = 0; file_vers=3;
elseif strcmp(Version(1:8),et_Vers)==1,
    StartHeader = 0; file_vers=4;
end

if file_vers == 0
    %%% Read txt or ascii files
    fclose(FID);
    Signal=load(FileName);
elseif (file_vers == 1 | file_vers == 2),
    %%% Read Poly_5 or TMS_32 files
    clear Signal Info
    % Read header
    fseek(FID,StartHeader,-1);
    % Info=structurearray for HEADER!; mit jedem Info. wird neue Zeile an Info angehaengt
    Info.VersionNumber = fread(FID,1,'uint16');% Poly5: 204; PortiLab: 203 :-)
    N = fread(FID,1,'uchar'); Str = char(fread(FID,80,'uchar'))';
    Info.MeasurementName = Str(1:N);
    Info.SampleFrequency = fread(FID,1,'int16');
    Info.StorageRate = fread(FID,1,'int16');
    Info.StorageType = fread(FID,1,'schar');
    Info.NumberChannels = fread(FID,1,'int16');
    Info.SampleRecordCountLo = fread(FID,1,'int32');
    Info.SampleRecordCountHi = fread(FID,1,'int32');
    Info.CreationDateAndTime = fread(FID,7,'uint16');
    Info.NumberOfDataBlocks = fread(FID,1,'uint32');
    Info.NumberOfSampleRecords = fread(FID,1,'uint16');
    Info.SizeOfDataBlock = fread(FID,1,'uint16');
    Info.DeltaCompression = fread(FID,1,'uint16');

    if Info.DeltaCompression > 0,
       error('No compression supported in Poly_5/TMS_32');
    end

    % Read Signal Headers
    fseek(FID,64,0);
    Samples = Info.SampleRecordCountLo;
%     h=waitbar(0,'Read File Properties...');
    for p = 1:Info.NumberChannels
       N = fread(FID,1,'uchar'); Str = char(fread(FID,40,'uchar'))';
       Signal{p}.Name = Str(1:N);
       fseek(FID,4,0);
       N = fread(FID,1,'uchar'); Str = char(fread(FID,10,'uchar'))';
       Signal{p}.UnitName = Str(1:N);
       Signal{p}.UnitLow = fread(FID,1,'single');
       Signal{p}.UnitHigh = fread(FID,1,'single');
       Signal{p}.AdcLow = fread(FID,1,'single');
       Signal{p}.AdcHigh = fread(FID,1,'single');
       Signal{p}.Resolution = (Signal{p}.UnitHigh-Signal{p}.UnitLow)/(Signal{p}.AdcHigh-Signal{p}.AdcLow);
       fseek(FID,64,0);
       if  strcmp(Version,Version203) == 1
           if mod(p,2) == 0
               Signal{p/2}.Data = zeros(1,Samples);
           else
               Signal{p}.Data=zeros(1,1);
           end
       else
           Signal{p}.Data=zeros(1,Samples);
       end
       Signal{p}.SampleFrequency =Info.SampleFrequency;
%        waitbar(p/Info.NumberChannels,h)
    end
%     close(h)

    % Read the Signals
    K = 0;
    SDB = Info.SizeOfDataBlock/(2*Info.NumberChannels);
    %h=waitbar(0,'Read the Signals...');
    for b = 1:Info.NumberOfDataBlocks,
       L = fread(FID,1,'uint32');
       fseek(FID,82,0);
       if ~(K == L)
          error('Error in Poly File');
       end
       L = Samples-K;
       if (L>SDB)
          L = SDB;
       end
       %%%%%%%%%%%%%%%%%%%%%%
       %%%%%%%%%%%%%%%%%%%%%%
       if strcmp(Version,Version203) == 1,
           Buffer = fread(FID,L*Info.NumberChannels/2,'float');% 32 bit floating point TMS32
           Buffer = reshape(Buffer,Info.NumberChannels/2,L);
           for pp=1:Info.NumberChannels/2
               Signal{pp}.Data(K+1:K+L) = Signal{2*pp}.Resolution*Buffer(pp,:);
           end
       end
       if strcmp(Version,Version204) == 1,
           Buffer = fread(FID,L*Info.NumberChannels,'int16');% 16 bit integer Poly5
           Buffer = reshape(Buffer,Info.NumberChannels,L);
           for p = 1:Info.NumberChannels,
               Signal{p}.Data(K+1:K+L) = Signal{p}.Resolution*Buffer(p,:);
           end
       end
       K = K + Info.NumberOfSampleRecords;
       %waitbar(b/Info.NumberOfDataBlocks,h)
    end
    %close(h)
elseif file_vers == 3
    %%% Read Beatscope out-Files
    fseek(FID,0,-1);
    [out_file,cnt]=fscanf(FID,'%c');
    fclose(FID);
    t_file=strread(out_file,'%f','delimiter',' ','headerlines',2); %remove 2 lines (header)
    if mod(length(t_file),3)~=0,disp('file truncated...'),end
    Signal=zeros((length(t_file)-mod(length(t_file),3))/3,3);
    k=1;
    for i=0:length(t_file)-mod(length(t_file),3)-1
        Signal(k,mod(i,3)+1)=t_file(i+1);
        if i > 0 & mod(i+1,3) == 0, k=k+1;end
    end
elseif (file_vers == 4 & strcmp('.et1_',FName(find(FName=='.'):find(FName=='.')+4)) ==1)
    %%% Read MDC et1_ files
    fseek(FID,0,-1);
    [t_file,cnt]=fread(FID,'char');
    % 320+4*s+9+4*s+9+4*1+9+4*s
    sam_block=(cnt-351)/12;
    fseek(FID,320,-1);
    c1=fread(FID,sam_block,'float32');% R_Amp
    fseek(FID,9,0);
    c2=fread(FID,sam_block,'float32');% timepoints R's
    fseek(FID,9,0);
    c3=fread(FID,1,'float32');% unknown
    fseek(FID,9,0);
    c4=fread(FID,sam_block,'float32');% eval
    fclose(FID);
    Signal=zeros(sam_block,4);
    Signal(:,1)=c1; Signal(:,2)=c2;
    Signal(1,3)=c3; Signal(:,4)=c4;
    clear c1 c2 c3 c4 sam_block t_file
elseif (file_vers == 4 & strcmp('.et2_',FName(find(FName=='.'):find(FName=='.')+4)) ==1)
    %%% Read MDC et2_ files
    fseek(FID,0,-1);
    [t_file,cnt]=fread(FID,'char');
    % 330+4*s+9+4*s+9+4*s+9+4*s+9+4*1+9+4*s
    sam_block=(cnt-379)/20;
    fseek(FID,330,-1);
    c1=fread(FID,sam_block,'float32');% Sys
    fseek(FID,9,0);
    c2=fread(FID,sam_block,'float32');% timepoints Sys
    fseek(FID,9,0);
    c3=fread(FID,sam_block,'float32');% Dst
    fseek(FID,9,0);
    c4=fread(FID,sam_block,'float32');% timepoints Dst
    fseek(FID,9,0);
    c5=fread(FID,1,'float32');% unknown
    fseek(FID,9,0);
    c6=fread(FID,sam_block,'float32');% eval
    fclose(FID);
    Signal=zeros(sam_block,6);
    Signal(:,1)=c1; Signal(:,2)=c2; Signal(:,3)=c3;
    Signal(:,4)=c4; Signal(1,5)=c5; Signal(:,6)=c6;
    clear c1 c2 c3 c4 c5 c6 sam_block t_file
end% if file_vers

return% [signal,info]
