##########################
##   Import Libraries   ##
##########################


import torch
import torch.nn as nn
from torch import Tensor
from typing import Type, Any, Callable, Union, List, Optional

##########################
#####   Def layers   #####
##########################

def linear(dim_in, dim_out):
    return [nn.Linear(dim_in, dim_out)]

def linear_relu(dim_in, dim_out):
    return [nn.Linear(dim_in, dim_out),
            nn.ReLU(inplace=True)]

def dropout_linear(dim_in, dim_out, p_drop):
    return [nn.Dropout(p_drop),
            nn.Linear(dim_in, dim_out)]

def dropout_linear_relu(dim_in, dim_out, p_drop):
    return [nn.Dropout(p_drop),
            nn.Linear(dim_in, dim_out),
            nn.ReLU(inplace=True)]

def bn_dropout_linear(dim_in, dim_out, p_drop):    
    return [nn.BatchNorm1d(dim_in,momentum=0.2),
            nn.Dropout(p_drop),
            nn.Linear(dim_in, dim_out)]

def bn_linear_relu(dim_in, dim_out):
    return [nn.BatchNorm1d(dim_in),
            nn.Linear(dim_in, dim_out),
            nn.ReLU(inplace=True)]

def bn_dropout_linear_relu(dim_in, dim_out, p_drop):
    return bn_dropout_linear(dim_in, dim_out, p_drop) + [nn.ReLU(inplace=True)]
            
def conv_relu_maxpool(cin, cout, csize, cstride, cpad, msize, mstride, mpad):
    return [nn.Conv1d(cin, cout, csize, cstride, cpad),
            nn.ReLU(inplace=True),
            nn.MaxPool1d(msize, mstride, mpad)]

def conv_bn_relu_maxpool(cin, cout, csize, cstride, cpad, msize, mstride, mpad): 
    return [nn.Conv1d(cin, cout, csize, cstride, cpad),
            nn.ReLU(inplace=True),
            nn.BatchNorm1d(cout,momentum=0.2),
            nn.MaxPool1d(msize, mstride, mpad)]    

def conv_relu_avgpool(cin, cout, csize, cstride, cpad, msize, mstride, mpad):
    return [nn.Conv1d(cin, cout, csize, cstride, cpad),
            nn.ReLU(inplace=True),
            nn.AvgPool1d(msize, mstride, mpad)]

def conv_bn_relu_avgpool(cin, cout, csize, cstride, cpad, msize, mstride, mpad):
    return [nn.Conv1d(cin, cout, csize, cstride, cpad),
            nn.ReLU(inplace=True),
            nn.BatchNorm1d(cout),
            nn.AvgPool1d(msize, mstride, mpad)]

def conv2d_bn_relu_maxpool(cin, cout, csize, cstride, cpad, msize, mstride, mpad): 
    return [nn.Conv2d(cin, cout, csize, cstride, cpad),
            nn.ReLU(inplace=True),
            nn.BatchNorm2d(cout,momentum=0.2),
            nn.MaxPool2d(msize, mstride, mpad)]

def conv3x3(in_planes: int, out_planes: int, stride: int = 1, groups: int = 1, dilation: int = 1) -> nn.Conv1d:
    """3x3 convolution with padding"""
    return nn.Conv1d(in_planes, out_planes, kernel_size=3, stride=stride,
                     padding=dilation, groups=groups, bias=False, dilation=dilation)


def conv1x1(in_planes: int, out_planes: int, stride: int = 1) -> nn.Conv1d:
    """1x1 convolution"""
    return nn.Conv1d(in_planes, out_planes, kernel_size=1, stride=stride, bias=False)



##########################
####   Def Networks   ####
##########################

class Conv1DNet(nn.Module):
    def __init__(self, num_classes = 111,num_channels = 12):
        super(Conv1DNet, self).__init__()
        self.empty_tensor = torch.randn(1, num_channels, 2570)
        self.num_classes = num_classes
        self.num_channels = num_channels
        #MaxPooling Version
        
        self.conv_model_max = nn.Sequential(
            *conv_bn_relu_maxpool(cin=num_channels, cout=128,
                                csize=11, cstride=3, cpad=3,
                                msize=2, mstride=1, mpad=0),
            *conv_bn_relu_maxpool(cin=128, cout=256,
                                csize=11, cstride=3, cpad=3,
                                msize=2, mstride=1, mpad=0),
            *conv_bn_relu_maxpool(cin=256, cout=256,
                                csize=11, cstride=3, cpad=3,
                                msize=2, mstride=1, mpad=0),
            *conv_bn_relu_maxpool(cin=256, cout=256,
                                csize=11, cstride=3, cpad=3,
                                msize=2, mstride=1, mpad=0),
            *conv_bn_relu_maxpool(cin=256, cout=256,
                                csize=11, cstride=3, cpad=3,
                                msize=2, mstride=1, mpad=0),
            *conv_bn_relu_maxpool(cin=256, cout=256,       
                                csize=5, cstride=2, cpad=0,
                                msize=2, mstride=1, mpad=0))

        self.output_size =self.conv_model_max(self.empty_tensor).shape[2]*self.conv_model_max(self.empty_tensor).shape[1]
        self.fc_model = nn.Sequential(
            *bn_dropout_linear_relu(self.output_size, 1024, 0.8),
            *bn_dropout_linear_relu(1024, 512, 0.8),
            *bn_dropout_linear_relu(512, 256, 0.8),
            nn.Linear(256, self.num_classes))
        self.init_weights()

    def forward(self, inputs):
        conv_features = self.conv_model_max(inputs)
        conv_features = conv_features.reshape(conv_features.size(0), -1)
        return self.fc_model(conv_features)

    #bn[0]_dropout[1]_linear[2]_relu[3]_bn[4]_dropout[5]_linear[6]_relu[7]... Explanation
    def init_weights(self):
        initrange = 0.1
        self.fc_model[2].bias.data.fill_(0)
        self.fc_model[2].weight.data.uniform_(-initrange, initrange)
        self.fc_model[6].bias.data.fill_(0)
        self.fc_model[6].weight.data.uniform_(-initrange, initrange)
        self.fc_model[10].bias.data.fill_(0)
        self.fc_model[10].weight.data.uniform_(-initrange, initrange)
        self.fc_model[12].bias.data.fill_(0)
        self.fc_model[12].weight.data.uniform_(-initrange, initrange)



# 2D pour spectro
class Conv2DNet(nn.Module):
    def __init__(self, num_classes = 111):
        super(Conv2DNet, self).__init__()
        self.empty_tensor = torch.zeros(1, 12, 257,1286)
        self.num_classes = num_classes

        #MaxPooling Version
        
        self.conv_model_max_2d = nn.Sequential(
            *conv2d_bn_relu_maxpool(cin=12, cout=128,
                                csize=11, cstride=3, cpad=3,
                                msize=2, mstride=1, mpad=1),
            *conv2d_bn_relu_maxpool(cin=128, cout=256,
                               csize=11, cstride=3, cpad=3,
                                msize=2, mstride=1, mpad=1),
            *conv2d_bn_relu_maxpool(cin=256, cout=256,
                                csize=11, cstride=3, cpad=3,
                                msize=2, mstride=1, mpad=1),
            *conv2d_bn_relu_maxpool(cin=256, cout=256,
                                csize=11, cstride=3, cpad=5,
                                msize=2, mstride=1, mpad=1),
            *conv2d_bn_relu_maxpool(cin=256, cout=256,
                                csize=11, cstride=3, cpad=5,
                                msize=2, mstride=1, mpad=1),
            *conv2d_bn_relu_maxpool(cin=256, cout=256,       
                                csize=11, cstride=3, cpad=5,
                                msize=2, mstride=1, mpad=1))
        
        cv = self.conv_model_max_2d(self.empty_tensor)
        # cv is (B, C, H, W)
        print("cv",cv.shape)
        self.output_size =cv.shape[1]*cv.shape[2]*cv.shape[3]
        self.fc_model = nn.Sequential(
            *bn_dropout_linear_relu(self.output_size, 512, 0.5),
            *bn_dropout_linear_relu(512, 256, 0.5),
            *bn_dropout_linear_relu(256, 128, 0.5),
            nn.Linear(128, self.num_classes))
        self.init_weights()

    def forward(self, inputs):
        conv_features = self.conv_model_max_2d(inputs)
        conv_features = conv_features.reshape(conv_features.size(0), -1)
        return self.fc_model(conv_features)

    #bn[0]_dropout[1]_linear[2]_relu[3]_bn[4]_dropout[5]_linear[6]_relu[7]... Explanation
    def init_weights(self):
        initrange = 0.1
        self.fc_model[2].bias.data.fill_(0)
        self.fc_model[2].weight.data.uniform_(-initrange, initrange)
        self.fc_model[6].bias.data.fill_(0)
        self.fc_model[6].weight.data.uniform_(-initrange, initrange)
        self.fc_model[10].bias.data.fill_(0)
        self.fc_model[10].weight.data.uniform_(-initrange, initrange)
        self.fc_model[12].bias.data.fill_(0)
        self.fc_model[12].weight.data.uniform_(-initrange, initrange)



class BasicBlock(nn.Module):
    expansion: int = 1

    def __init__(
        self,
        inplanes: int,
        planes: int,
        stride: int = 1,
        downsample: Optional[nn.Module] = None,
        groups: int = 1,
        base_width: int = 64,
        dilation: int = 1,
        norm_layer: Optional[Callable[..., nn.Module]] = None
    ) -> None:
        super(BasicBlock, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm1d
        if groups != 1 or base_width != 64:
            raise ValueError('BasicBlock only supports groups=1 and base_width=64')
        if dilation > 1:
            raise NotImplementedError("Dilation > 1 not supported in BasicBlock")
        # Both self.conv1 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv3x3(inplanes, planes, stride)
        self.bn1 = norm_layer(planes)
        self.relu = nn.ReLU(inplace=True)
        self.conv2 = conv3x3(planes, planes)
        self.bn2 = norm_layer(planes)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x: Tensor) -> Tensor:
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class Bottleneck(nn.Module):
    # Bottleneck in torchvision places the stride for downsampling at 3x3 convolution(self.conv2)
    # while original implementation places the stride at the first 1x1 convolution(self.conv1)
    # according to "Deep residual learning for image recognition"https://arxiv.org/abs/1512.03385.
    # This variant is also known as ResNet V1.5 and improves accuracy according to
    # https://ngc.nvidia.com/catalog/model-scripts/nvidia:resnet_50_v1_5_for_pytorch.

    expansion: int = 4

    def __init__(
        self,
        inplanes: int,
        planes: int,
        stride: int = 1,
        downsample: Optional[nn.Module] = None,
        groups: int = 1,
        base_width: int = 64,
        dilation: int = 1,
        norm_layer: Optional[Callable[..., nn.Module]] = None
    ) -> None:
        super(Bottleneck, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm1d
        width = int(planes * (base_width / 64.)) * groups
        # Both self.conv2 and self.downsample layers downsample the input when stride != 1
        self.conv1 = conv1x1(inplanes, width)
        self.bn1 = norm_layer(width)
        self.conv2 = conv3x3(width, width, stride, groups, dilation)
        self.bn2 = norm_layer(width)
        self.conv3 = conv1x1(width, planes * self.expansion)
        self.bn3 = norm_layer(planes * self.expansion)
        self.relu = nn.ReLU(inplace=True)
        self.downsample = downsample
        self.stride = stride

    def forward(self, x: Tensor) -> Tensor:
        identity = x

        out = self.conv1(x)
        out = self.bn1(out)
        out = self.relu(out)

        out = self.conv2(out)
        out = self.bn2(out)
        out = self.relu(out)

        out = self.conv3(out)
        out = self.bn3(out)

        if self.downsample is not None:
            identity = self.downsample(x)

        out += identity
        out = self.relu(out)

        return out


class ResNet(nn.Module):

    def __init__(
        self,
        block: Type[Union[BasicBlock, Bottleneck]],
        layers: List[int],
        num_classes: int = 26,
        zero_init_residual: bool = False,
        groups: int = 1,
        width_per_group: int = 64,
        input_channels: int = 12,
        replace_stride_with_dilation: Optional[List[bool]] = None,
        norm_layer: Optional[Callable[..., nn.Module]] = None
    ) -> None:
        super(ResNet, self).__init__()
        if norm_layer is None:
            norm_layer = nn.BatchNorm1d
        self._norm_layer = norm_layer

        self.inplanes = 64
        self.dilation = 1
        if replace_stride_with_dilation is None:
            # each element in the tuple indicates if we should replace
            # the 2x2 stride with a dilated convolution instead
            replace_stride_with_dilation = [False, False, False]
        if len(replace_stride_with_dilation) != 3:
            raise ValueError("replace_stride_with_dilation should be None "
                             "or a 3-element tuple, got {}".format(replace_stride_with_dilation))
        self.groups = groups
        self.base_width = width_per_group
        self.conv1 = nn.Conv1d(input_channels, self.inplanes, kernel_size=7, stride=2, padding=3,
                               bias=False)
        self.bn1 = norm_layer(self.inplanes)
        self.relu = nn.ReLU(inplace=True)
        self.maxpool = nn.MaxPool1d(kernel_size=3, stride=2, padding=1)
        self.layer1 = self._make_layer(block, 64, layers[0])
        self.layer2 = self._make_layer(block, 128, layers[1], stride=2,
                                       dilate=replace_stride_with_dilation[0])
        self.layer3 = self._make_layer(block, 256, layers[2], stride=2,
                                       dilate=replace_stride_with_dilation[1])
        self.layer4 = self._make_layer(block, 512, layers[3], stride=2,
                                       dilate=replace_stride_with_dilation[2])
        self.avgpool = nn.AdaptiveAvgPool1d(1)
       # self.fc = nn.Linear(512 * block.expansion, num_classes)

        for m in self.modules():
            if isinstance(m, nn.Conv1d):
                nn.init.kaiming_normal_(m.weight, mode='fan_out', nonlinearity='relu')
            elif isinstance(m, (nn.BatchNorm1d, nn.GroupNorm)):
                nn.init.constant_(m.weight, 1)
                nn.init.constant_(m.bias, 0)

        # Zero-initialize the last BN in each residual branch,
        # so that the residual branch starts with zeros, and each residual block behaves like an identity.
        # This improves the model by 0.2~0.3% according to https://arxiv.org/abs/1706.02677
        if zero_init_residual:
            for m in self.modules():
                if isinstance(m, Bottleneck):
                    nn.init.constant_(m.bn3.weight, 0)  # type: ignore[arg-type]
                elif isinstance(m, BasicBlock):
                    nn.init.constant_(m.bn2.weight, 0)  # type: ignore[arg-type]

    def _make_layer(self, block: Type[Union[BasicBlock, Bottleneck]], planes: int, blocks: int,
                    stride: int = 1, dilate: bool = False) -> nn.Sequential:
        norm_layer = self._norm_layer
        downsample = None
        previous_dilation = self.dilation
        if dilate:
            self.dilation *= stride
            stride = 1
        if stride != 1 or self.inplanes != planes * block.expansion:
            downsample = nn.Sequential(
                conv1x1(self.inplanes, planes * block.expansion, stride),
                norm_layer(planes * block.expansion),
            )

        layers = []
        layers.append(block(self.inplanes, planes, stride, downsample, self.groups,
                            self.base_width, previous_dilation, norm_layer))
        self.inplanes = planes * block.expansion
        for _ in range(1, blocks):
            layers.append(block(self.inplanes, planes, groups=self.groups,
                                base_width=self.base_width, dilation=self.dilation,
                                norm_layer=norm_layer))

        return nn.Sequential(*layers)

    def _forward_impl(self, x: Tensor) -> Tensor:
        # See note [TorchScript super()]
        x = self.conv1(x)
        x = self.bn1(x)
        x = self.relu(x)
        x = self.maxpool(x)

        x = self.layer1(x)
        x = self.layer2(x)
        x = self.layer3(x)
        x = self.layer4(x)
        #x = self.avgpool(x)
       # x = torch.flatten(x, 1)
       # x = self.fc(x)

        return x

    def forward(self, x: Tensor) -> Tensor:
        return self._forward_impl(x)


def _resnet(
    arch: str,
    block: Type[Union[BasicBlock, Bottleneck]],
    layers: List[int],
    **kwargs: Any
) -> ResNet:
    model = ResNet(block, layers, **kwargs)
    return model


def resnet18(**kwargs: Any) -> ResNet:
    return _resnet('resnet18', BasicBlock, [2, 2, 2, 2],**kwargs)


def resnet34(**kwargs: Any) -> ResNet:
    return _resnet('resnet34', BasicBlock, [3, 4, 6, 3], **kwargs)


def resnet50(**kwargs: Any) -> ResNet:
    return _resnet('resnet50', Bottleneck, [3, 4, 6, 3], **kwargs)


def resnet101(**kwargs: Any) -> ResNet:
    return _resnet('resnet101', Bottleneck, [3, 4, 23, 3], **kwargs)


def resnet152(**kwargs: Any) -> ResNet:
    return _resnet('resnet152', Bottleneck, [3, 8, 36, 3],**kwargs)