# this file contains utility functions (like simple debug calculations etc.) that are not exactly necessary for the
# training of the model
import math
import numpy as np
import os


def convert_size(size_bytes):
    """
    This function takes a size in bytes and calculates readable file sizes.

    :param size_bytes: The size of file in byte
    :type size_bytes: int
    """

    if size_bytes == 0:
        return "0B"

    size_name = ("B", "KB", "MB", "GB", "TB", "PB", "EB", "ZB", "YB")
    i = int(math.floor(math.log(size_bytes, 1024)))
    p = math.pow(1024, i)
    s = round(size_bytes / p, 2)
    return "%s %s" % (s, size_name[i])


class ValueSaver:

    def __init__(self):

        # introduce the dictionary
        self.value_dict = {}

    def __len__(self):
        return len(self.value_dict)

    def __str__(self):
        return str(self.value_dict)

    def __repr__(self):
        return str(self.value_dict)

    def __call__(self, value, key=None):

        # set the key to the counter if the key is none
        if key is None:
            key = str(len(self.value_dict))

        # test the value if it is the right type
        if not isinstance(value, np.float) and not isinstance(value, int) and not isinstance(value, float):
            raise ValueError('The value to be saved is not a float.')

        # test if the key is a string
        if not isinstance(key, str):
            raise ValueError('The key needs to be a string.')

        # check if the key is already there
        if key in self.value_dict.keys():
            raise ValueError('The key is already taken.')

        # add the key-value pair to the dictionary
        self.value_dict[key] = value

    def __getitem__(self, key):
        if key in self.value_dict.keys():
            value = self.value_dict[key]
        else:
            raise ValueError('The key is not part of the saved values.')
        return value

    def to_array(self):

        data = np.array([ele[1] for ele in self.value_dict.items()])
        return data

    def mean(self):

        # check if the list has values
        if self.value_dict == {}:
            return None

        # get the data
        data = self.to_array()

        # calculate the mean
        mean = np.mean(data)
        return mean

    def max(self):

        # check if the list has values
        if self.value_dict == {}:
            return [None], float('-inf')

        # get the data
        data = self.to_array()

        # get the maximum
        max_ele = np.max(data)

        # find the key
        key_list = []
        for key, value in self.value_dict.items():
            if value == max_ele:
                key_list.append(key)

        return key_list, max_ele

    def min(self):
        # check if the list has values
        if self.value_dict == {}:
            return [None], float('+inf')

        # get the data
        data = self.to_array()

        # get the maximum
        min_ele = np.min(data)

        # find the key
        key_list = []
        for key, value in self.value_dict.items():
            if value == min_ele:
                key_list.append(key)

        return key_list, min_ele

    def reset(self):

        # reset the dictionary
        self.value_dict = {}


def convert_seconds(seconds):
    minutes, seconds = divmod(seconds, 60)
    hours, minutes = divmod(minutes, 60)
    return f'{hours:02.0f}h:{minutes:02.0f}m:{seconds:02.0f}s'


def list_subdir(path):
    if os.path.isdir(path):
        path_list = [os.path.join(path, d) for d in os.listdir(path) if os.path.isdir(os.path.join(path, d))]
    else:
        raise ValueError('[path] is no directory.')

    return path_list


def list_subfiles(path, file_type='*'):
    if os.path.isdir(path):
        path_list = []
        for file in os.listdir(path):
            if os.path.isfile(os.path.join(path, file)):
                if os.path.splitext(file)[-1] == file_type or file_type == '*':
                    path_list.append(os.path.join(path, file))
    else:
        raise ValueError('[path] is no directory.')

    return path_list
