#!/usr/bin/env python

import numpy as np
import wfdb
import math
import persim
from persim import persistent_entropy
from scipy.signal import butter, lfilter
from scipy import stats
from scipy.stats import skew, kurtosis
from wfdb import processing
from ripser import ripser
from scipy.io import loadmat 


def detect_peaks(ecg_measurements,signal_frequency,gain):

        """
        Method responsible for extracting peaks from loaded ECG measurements data through measurements processing.

        This implementation of a QRS Complex Detector is by no means a certified medical tool and should not be used in health monitoring. 
        It was created and used for experimental purposes in psychophysiology and psychology.
        You can find more information in module documentation:
        https://github.com/c-labpl/qrs_detector
        If you use these modules in a research project, please consider citing it:
        https://zenodo.org/record/583770
        If you use these modules in any other project, please refer to MIT open-source license.

        If you have any question on the implementation, please refer to:

        Michal Sznajder (Jagiellonian University) - technical contact (msznajder@gmail.com)
        Marta lukowska (Jagiellonian University)
        Janko Slavic peak detection algorithm and implementation.
        https://github.com/c-labpl/qrs_detector
        https://github.com/jankoslavic/py-tools/tree/master/findpeaks
        
        MIT License
        Copyright (c) 2017 Michal Sznajder, Marta Lukowska
    
        Permission is hereby granted, free of charge, to any person obtaining a copy
        of this software and associated documentation files (the "Software"), to deal
        in the Software without restriction, including without limitation the rights
        to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
        copies of the Software, and to permit persons to whom the Software is
        furnished to do so, subject to the following conditions:
        The above copyright notice and this permission notice shall be included in all
        copies or substantial portions of the Software.
        THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
        IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
        FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
        AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
        LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
        OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
        SOFTWARE.

        """


        filter_lowcut = 0.001
        filter_highcut = 15.0
        filter_order = 1
        integration_window = 30  # Change proportionally when adjusting frequency (in samples).
        findpeaks_limit = 0.35
        findpeaks_spacing = 100  # Change proportionally when adjusting frequency (in samples).
        refractory_period = 240  # Change proportionally when adjusting frequency (in samples).
        qrs_peak_filtering_factor = 0.125
        noise_peak_filtering_factor = 0.125
        qrs_noise_diff_weight = 0.25


        # Detection results.
        qrs_peaks_indices = np.array([], dtype=int)
        noise_peaks_indices = np.array([], dtype=int)


        # Measurements filtering - 0-15 Hz band pass filter.
        filtered_ecg_measurements = bandpass_filter(ecg_measurements, lowcut=filter_lowcut, highcut=filter_highcut, signal_freq=signal_frequency, filter_order=filter_order)

        filtered_ecg_measurements[:5] = filtered_ecg_measurements[5]

        # Derivative - provides QRS slope information.
        differentiated_ecg_measurements = np.ediff1d(filtered_ecg_measurements)

        # Squaring - intensifies values received in derivative.
        squared_ecg_measurements = differentiated_ecg_measurements ** 2

        # Moving-window integration.
        integrated_ecg_measurements = np.convolve(squared_ecg_measurements, np.ones(integration_window)/integration_window)

        # Fiducial mark - peak detection on integrated measurements.
        detected_peaks_indices = findpeaks(data=integrated_ecg_measurements,
                                                     limit=findpeaks_limit,
                                                     spacing=findpeaks_spacing)

        detected_peaks_values = integrated_ecg_measurements[detected_peaks_indices]

        return detected_peaks_values,detected_peaks_indices

 

def bandpass_filter(data, lowcut, highcut, signal_freq, filter_order):
        """
        Method responsible for creating and applying Butterworth filter.
        :param deque data: raw data
        :param float lowcut: filter lowcut frequency value
        :param float highcut: filter highcut frequency value
        :param int signal_freq: signal frequency in samples per second (Hz)
        :param int filter_order: filter order
        :return array: filtered data
        """
        nyquist_freq = 0.5 * signal_freq
        low = lowcut / nyquist_freq
        high = highcut / nyquist_freq
        b, a = butter(filter_order, [low, high], btype="band")
        y = lfilter(b, a, data)
        return y

def findpeaks(data, spacing=1, limit=None):
        """
        Janko Slavic peak detection algorithm and implementation.
        https://github.com/jankoslavic/py-tools/tree/master/findpeaks
        Finds peaks in `data` which are of `spacing` width and >=`limit`.
        :param ndarray data: data
        :param float spacing: minimum spacing to the next peak (should be 1 or more)
        :param float limit: peaks should have value greater or equal
        :return array: detected peaks indexes array
        """
        len = data.size
        x = np.zeros(len + 2 * spacing)
        x[:spacing] = data[0] - 1.e-6
        x[-spacing:] = data[-1] - 1.e-6
        x[spacing:spacing + len] = data
        peak_candidate = np.zeros(len)
        peak_candidate[:] = True
        for s in range(spacing):
            start = spacing - s - 1
            h_b = x[start: start + len]  # before
            start = spacing
            h_c = x[start: start + len]  # central
            start = spacing + s + 1
            h_a = x[start: start + len]  # after
            peak_candidate = np.logical_and(peak_candidate, np.logical_and(h_c > h_b, h_c > h_a))

        ind = np.argwhere(peak_candidate)
        ind = ind.reshape(ind.size)
        if limit is not None:
            ind = ind[data[ind] > limit]
        return ind

#This is the barcode-based TDA feature extractor. The inputs are a point-cloud and the maximum dimension of features to be computed 
# and the output is a numpy array of stats-based features extracted from the barcodes.

def extract(diagrams,maxdimension):
#def extract(PTCLOUD,maxdimension):
    
    #initialize some local variables
    N1 = N2 = MeanBC1 = MeanDC1 = SUMpersC1 = SDBC1 = SDDC1 = SDPersC1 = SKPersC1 = SUMpersC1 = SKBC1 = KurtBC1 = 0
    SKDC1 = KurtDC1 = MeanpersC1 = KurtpersC1 = MeanBC2 = MeanDC2 = SUMpersC2 = SDBC2 = SDDC2 = SDPersC2 = SKPersC2 = 0
    SUMpersC2 = SKBC2 = KurtBC2 = SKDC2 = KurtDC2 = MeanpersC2 = KurtpersC2 = 0
    
    #Start filling in variables
    N0 = len(diagrams[0])  
    C0d = np.array(diagrams[0][:len(diagrams[0])-1,1])
    if len(C0d)>1:        
        MeanDC0 = np.mean(C0d)
        SUMpersC0 = sum(C0d)
        SDDC0 = np.std(C0d)
        SKDC0 = skew(C0d)
        KurtDC0 = kurtosis(C0d)
    elif len(C0d)==1:
        MeanDC0 = np.mean(C0d)
        SUMpersC0 = sum(C0d)
        SDDC0 = 0
        SKDC0 = 0
        KurtDC0 = 0
    else:
        MeanDC0 = 0
        SUMpersC0 = 0
        SDDC0 = 0
        SKDC0 = 0
        KurtDC0 = 0
    d0 = diagrams[0][:len(diagrams[0])-1,1]
    hd0 = np.histogram(d0,bins = 'auto')
    d0 = d0[:len(d0)-1-len(np.where(d0>=hd0[1][len(hd0[1])-2])[0])]
    if len(d0)>1:
        MeanDC0noPers = np.mean(d0)
        SDDC0noPers = np.std(d0)
        SKDC0noPers = skew(d0)
        KurtDC0noPers = kurtosis(d0)
    elif len(d0)==1:
        MeanDC0noPers = d0
        SDDC0noPers = 0
        SKDC0noPers = 0
        KurtDC0noPers = 0
    else:
        MeanDC0noPers = 0
        SDDC0noPers = 0
        SKDC0noPers = 0
        KurtDC0noPers = 0
    MeanBC1noPers = SDBC1noPers = SKBC1noPers = KurtBC1noPers = MeanDC1noPers = SDDC1noPers = SKDC1noPers = KurtDC1noPers = 0
    SUMPersC1noPers = MeanPersC1noPers = SDPersC1noPers = SKPersC1noPers = KurtPersC1noPers = 0
    MeanBC2noPers = SDBC2noPers = SKBC2noPers = KurtBC2noPers = MeanDC2noPers = SDDC2noPers = SKDC2noPers = KurtDC2noPers = 0
    SUMPersC2noPers = MeanPersC2noPers = SDPersC2noPers = SKPersC2noPers = KurtPersC2noPers = 0
    #new_d = 0
    #new_d2 = 0
    pers_entropy = [0,0]
 
    if(len(diagrams[1])>0):
        pers_entropy = []
        for i in range(maxdimension+1):
            pers_entropy = np.append(pers_entropy,persim.persistent_entropy.persistent_entropy(diagrams[i]))

    if(len(diagrams[1])>0):
        N1 = len(diagrams[1])
   
        #print(N1)
        if(N1>1):
            d0=diagrams[1][:len(diagrams[1])-1,0]
            d1=diagrams[1][:len(diagrams[1])-1,1]
            d = np.array(d1-d0)
            #cons = np.array(d[1:])-np.array(d[:len(d)-1])
            #mark = np.argmax(cons)+1
            #diffs = d[mark] - d[:mark]
            #diffs_ave = np.average(diffs)
            C1b = np.array(diagrams[1][:,0])
            C1d = np.array(diagrams[1][:,1])
            MeanBC1 = np.mean(C1b)
            MeanDC1 = np.mean(C1d)
            SUMpersC1 = sum(C1d-C1b)
            SDBC1 = np.std(C1b)
            SDDC1 = np.std(C1d)
            SDPersC1 = np.std(C1d-C1b)
            SKPersC1 = skew(C1d-C1b)
            SKBC1 = skew(C1b)
            KurtBC1 = kurtosis(C1b)
            SKDC1 = skew(C1d)
            KurtDC1 = kurtosis(C1d)
            MeanpersC1 = np.mean(C1d-C1b)
            KurtpersC1 = kurtosis(C1d-C1b)
            hd = np.histogram(d,bins = 'auto')
            d = d[:len(d)-1-len(np.where(d>=hd[1][len(hd[1])-2])[0])]
            MeanBC1noPers = np.mean(d0)
            SDBC1noPers = np.std(d0)
            SKBC1noPers = skew(d0)
            KurtBC1noPers = kurtosis(d0)
            MeanDC1noPers = np.mean(d1)
            SDDC1noPers = np.std(d1)
            SKDC1noPers = skew(d1)
            KurtDC1noPers = kurtosis(d1)
            SUMPersC1noPers = sum(d)
            MeanPersC1noPers = np.mean(d)
            SDPersC1noPers = np.std(d)
            SKPersC1noPers = skew(d)
            KurtPersC1noPers = kurtosis(d)
            #if mark < len(d):
            #    new_d = (np.sum(d[d>=d[mark]])+np.sum(d[:mark][diffs<=diffs_ave*0.4]))/d[mark-1]
            #    new_d2 = len(d[d>=d[mark]])+len(d[:mark][diffs<=diffs_ave*0.4])
            #else:
            #    new_d = max(cons)/d[mark-2]
            #    new_d2 = 1
        else:
            MeanBC1 = diagrams[1][:,0]
            MeanDC1 = diagrams[1][:,1]
            SUMpersC1 = diagrams[1][:,1]-diagrams[1][:,0]
            SDBC1 = SDDC1 = SDPersC1 = SKPersC1 = SKBC1 = KurtBC1 = SKDC1 = KurtDC1 = KurtpersC1 = 100
            MeanpersC1 = diagrams[1][:,1]-diagrams[1][:,0]

            MeanBC1noPers = SDBC1noPers = SKBC1noPers = KurtBC1noPers = MeanDC1noPers = SDDC1noPers = SKDC1noPers = 100
            KurtDC1noPers = SUMPersC1noPers = MeanPersC1noPers = SDPersC1noPers = SKPersC1noPers = KurtPersC1noPers = 100
            
    if(len(diagrams)==3):
        N2 = len(diagrams[2])
        d0=diagrams[2][:len(diagrams[2])-1,0]
        d1=diagrams[2][:len(diagrams[2])-1,1]
        d = d1-d0
        if N2>1:

            C2b = np.array(diagrams[2][:,0])
            C2d = np.array(diagrams[2][:,1])
            MeanBC2 = np.mean(C2b)
            MeanDC2 = np.mean(C2d)
            SUMpersC2 = sum(C2d-C2b)
            SDBC2 = np.std(C2b)
            SDDC2 = np.std(C2d)
            SDPersC2 = np.std(C2d-C2b)
            SKPersC2 = skew(C2d-C2b)
            SKBC2 = skew(C2b)
            KurtBC2 = kurtosis(C2b)
            SKDC2 = skew(C2d)
            KurtDC2 = kurtosis(C2d)
            MeanpersC2 = np.mean(C2d-C2b)
            KurtpersC2 = kurtosis(C2d-C2b)

            hd = np.histogram(d,bins = 'auto')
            d = d[:len(d)-1-len(np.where(d>=hd[1][len(hd[1])-2])[0])]
            MeanBC2noPers = np.mean(d0)
            SDBC2noPers = np.std(d0)
            SKBC2noPers = skew(d0)
            KurtBC2noPers = kurtosis(d0)
            MeanDC2noPers = np.mean(d1)
            SDDC2noPers = np.std(d1)
            SKDC2noPers = skew(d1)
            KurtDC2noPers = kurtosis(d1)
            SUMPersC2noPers = sum(d)
            MeanPersC2noPers = np.mean(d)
            SDPersC2noPers = np.std(d)
            SKPersC2noPers = skew(d)
            KurtPersC2noPers = kurtosis(d)
        elif (N2==1):
            MeanBC2 = diagrams[2][:,0]
            MeanDC2 = diagrams[2][:,1]
            SUMpersC2 = diagrams[2][:,1]-diagrams[2][:,0]
            SDBC2 = SDDC2 = SDPersC2 = SKPersC2 = SKBC2 = KurtBC2 = SKDC2 = KurtDC2 = KurtpersC2 = 1000
            MeanpersC2 = diagrams[2][:,1]-diagrams[2][:,0]
            MeanBC2noPers = SDBC2noPers = SKBC2noPers = KurtBC2noPers = MeanDC2noPers = SDDC2noPers = SKDC2noPers = 1000
            KurtDC2noPers = SUMPersC2noPers = MeanPersC2noPers = SDPersC2noPers = SKPersC2noPers = KurtPersC2noPers = 1000
    
    
    
    
    #save the features as an array
    #this first one is with dimension 2 features
    #ECGFeat = np.array([N0, MeanDC0, SDDC0, SUMpersC0, SKDC0, KurtDC0, MeanBC1, MeanDC1, SDBC1, SDDC1, SUMpersC1, SKPersC1,N1, SDPersC1, SKBC1, KurtBC1,SKDC1,KurtDC1,MeanpersC1,KurtpersC1,  N2, SDPersC2,MeanBC2, MeanDC2, SDBC2, SDDC2,SKPersC2, SUMpersC2,SKBC2, KurtBC2,SKDC2,KurtDC2,MeanpersC2,KurtpersC2,MeanDC0noPers, SDDC0noPers,SKDC0noPers,KurtDC0noPers,MeanBC1noPers,SDBC1noPers,SKBC1noPers,KurtBC1noPers,MeanDC1noPers,SDDC1noPers,SKDC1noPers,KurtDC1noPers,SUMPersC1noPers,MeanPersC1noPers,SDPersC1noPers,SKPersC1noPers, KurtPersC1noPers,MeanBC2noPers,SDBC2noPers,SKBC2noPers,KurtBC2noPers,MeanDC2noPers,SDDC2noPers,SKDC2noPers,KurtDC2noPers,SUMPersC2noPers,MeanPersC2noPers,SDPersC2noPers,SKPersC2noPers, KurtPersC2noPers])
    
    #This one does not have dimension 2 features
    ECGFeat = np.array([N0, MeanDC0, SDDC0, SUMpersC0, SKDC0, KurtDC0, MeanBC1, MeanDC1, SDBC1, SDDC1, SUMpersC1, SKPersC1,N1, SDPersC1, SKBC1, KurtBC1,SKDC1,KurtDC1,MeanpersC1,KurtpersC1,MeanDC0noPers, SDDC0noPers,SKDC0noPers,KurtDC0noPers,MeanBC1noPers,SDBC1noPers,SKBC1noPers,KurtBC1noPers,MeanDC1noPers,SDDC1noPers,SKDC1noPers,KurtDC1noPers,SUMPersC1noPers,MeanPersC1noPers,SDPersC1noPers,SKPersC1noPers, KurtPersC1noPers,
                       pers_entropy[0],pers_entropy[1]])
    return ECGFeat 



def persland(diagram):
    diagram=sorted(diagram, key=lambda t: (t[0], -t[1]))
    diagram=np.array(diagram)
    k = 0
    L = []
    Peaks = []
    Valleys = []
    while len(diagram) != 0:
        i = 0
        p = 0
        pair = diagram[i]
        diagram = np.delete(diagram, i, axis = 0)
        L.append(np.array([[-math.inf,0], [pair[0],0], [0.5*(pair[0] + pair[1]), 0.5*(pair[1] - pair[0])]]))
        Peaks.append(np.array([[0.5*(pair[0] + pair[1]), 0.5*(pair[1] - pair[0])]]))
        Valleys.append(np.array([[pair[0],0]]))
        while all(L[k][-1] != np.array([math.inf, 0])):
            if len(diagram) == 0 or p==len(diagram):
                L[k] = np.append(L[k], np.array([[pair[1], 0]]), axis=0)
                #L[k] = np.append(L[k], np.array([[math.inf, 0]]), axis=0)
                break
            ran = range(p, len(diagram))
            if pair[1] >= max(diagram[:,1][ran]):
                L[k] = np.append(L[k], np.array([[pair[1], 0]]), axis=0)
                #L[k] = np.append(L[k], np.array([[math.inf, 0]]), axis=0)
            else:
                u = np.where(diagram[:,1] > pair[1])[0]
                j = min(u[np.where(u >= p)[0]])
                pair1 = diagram[j]
                diagram = np.delete(diagram, j, axis = 0)
                p = j
                if pair1[0] > pair[1]:
                    L[k] = np.append(L[k], np.array([[pair[1], 0]]), axis = 0)
                    Valleys[k] = np.append(Valleys[k], np.array([[pair[1],0], [pair1[0],0]]), axis = 0)
                if pair1[0] >= pair[1]:
                    L[k] = np.append(L[k], np.array([[pair1[0], 0]]), axis = 0)
                else:
                    b_1 = 0.5*(pair1[0] + pair[1])
                    d_1 = 0.5*(pair[1] - pair1[0])
                    L[k] = np.append(L[k], np.array([[b_1, d_1]]), axis=0)
                    diagram = np.append(diagram, np.array([[pair1[0],pair[1]]]), axis = 0)
                    ran = range(p, len(diagram))
                    diagram[ran] = sorted(diagram[ran], key=lambda t: (t[0], -t[1]))
                    Valleys[k] = np.append(Valleys[k], np.array([[b_1,d_1]]), axis = 0)
                b_2 = 0.5*(pair1[0] + pair1[1])
                d_2 = 0.5*(pair1[1] - pair1[0])
                L[k] = np.append(L[k], np.array([[b_2,d_2]]), axis = 0)
                Peaks[k] = np.append(Peaks[k], np.array([[b_2,d_2]]), axis = 0)
                pair = pair1
        Valleys[k] = np.append(Valleys[k], np.array([[pair[1], 0]]), axis = 0)
        k+=1
    
    #Features from the Persistence Landscapes
    
    Order = k #Maximum Number of Layers
    Num_Peaks = np.zeros(k)
    Num_Valls = np.zeros(k)

    for i in range(k):
        Num_Peaks[i] = len(Peaks[i]) #Number of Peaks Per Layer
        Num_Valls[i] = len(Valleys[i]) #Number of Valleys Per Layer
    if len(Num_Peaks)>1:
        Mean_Peaks = np.mean(Num_Peaks) #Mean of Peaks
        STD_Peaks = np.std(Num_Peaks) #Standard Deviation of Peaks
        Skew_Peaks = skew(np.asarray(Num_Peaks)) #Skewness of Peaks
        Kurt_Peaks = kurtosis(np.asarray(Num_Peaks)) #Kurtosis of Peaks
    else:
        Mean_Peaks = 0
        STD_Peaks = 0
        Skew_Peaks = 0
        Kurt_Peaks = 0
    if len(Num_Valls)>1:
        Mean_Valls = np.mean(Num_Valls) #Mean of Valleys     
        STD_Valls = np.std(Num_Valls) #Standard Deviation of Valleys
        Skew_Valls = skew(np.asarray(Num_Valls)) #Skewness Valleys
        Kurt_Valls = kurtosis(np.asarray(Num_Valls)) #Kurtosis of Peaks
    else:
        Mean_Valls = 0
        STD_Valls = 0
        Skew_Valls = 0
        Kurt_Valls = 0
    
    Areas = np.zeros(k)
    if k>1:
        Rel_Areas = np.zeros(k-1)
        for i in range(k):
            q = L[i]
            for j in range(1,len(q) - 1):
                b = 0.5*(q[j][1] + q[j+1][1])*(q[j+1][0] - q[j][0])
                Areas[i] = Areas[i] + b
        for i in range(0,k-2):
            Rel_Areas[i] = (Areas[i+1]-Areas[i+2])/Areas[1]
        Rel_Areas[k-2]=Areas[k-1]/Areas[1]
    else:
        Rel_Areas = np.array([1])
    
    Mean_Areas = np.mean(Rel_Areas)    
    STD_Areas = np.std(Rel_Areas)
    Skew_Areas = skew(np.asarray(Rel_Areas))
    Kurt_Areas = kurtosis(np.asarray(Rel_Areas)) 
    
    ECGFeat_Land = np.array([Order, Mean_Peaks, STD_Peaks, Skew_Peaks, Kurt_Peaks, Mean_Valls, STD_Valls, Skew_Valls, Kurt_Valls, Mean_Areas, STD_Areas, Skew_Areas, Kurt_Areas])
    
    return ECGFeat_Land




def get_12ECG_features_original(data, header_data,lead):

    tmp_hea = header_data[0].split(' ')
    ptID = tmp_hea[0]
    num_leads = int(tmp_hea[1])
    sample_Fs= int(tmp_hea[2])
    gain_lead = np.zeros(num_leads)
    
    
    for ii in range(num_leads):
        tmp_hea = header_data[ii+1].split(' ')
        gain_lead[ii] = int(tmp_hea[2].split('/')[0])
 
    # for testing, we included the mean age of 57 if the age is a NaN
    # This value will change as more data is being released
    for iline in header_data:
        if iline.startswith('#Age'):
            tmp_age = iline.split(': ')[1].strip()
            age = int(tmp_age if tmp_age != 'NaN' else 57)
        elif iline.startswith('#Sex'):
            tmp_sex = iline.split(': ')[1]
            if tmp_sex.strip()=='Female':
                sex =1
            else:
                sex=0
#        elif iline.startswith('#Dx'):
#            label = iline.split(': ')[1].split(',')[0]


    
#   We are only using data from lead1
    peaks,idx = detect_peaks(data[lead],sample_Fs,gain_lead[lead])
    if len(peaks)==0 or len(idx)==0:
        peaks,idx = detect_peaks(data[0],sample_Fs,gain_lead[0])
        mean_RR = np.mean(idx/sample_Fs*1000)
        mean_Peaks = np.mean(peaks*gain_lead[0])

    #   median
        median_RR = np.median(idx/sample_Fs*1000)
        median_Peaks = np.median(peaks*gain_lead[0])

    #   standard deviation
        std_RR = np.std(idx/sample_Fs*1000)
        std_Peaks = np.std(peaks*gain_lead[0])

    #   variance
        var_RR = stats.tvar(idx/sample_Fs*1000)
        var_Peaks = stats.tvar(peaks*gain_lead[0])

    #   Skewness
        skew_RR = stats.skew(idx/sample_Fs*1000)
        skew_Peaks = stats.skew(peaks*gain_lead[0])

    #   Kurtosis
        kurt_RR = stats.kurtosis(idx/sample_Fs*1000)
        kurt_Peaks = stats.kurtosis(peaks*gain_lead[0])
    else:
        #   mean
        mean_RR = np.mean(idx/sample_Fs*1000)
        mean_Peaks = np.mean(peaks*gain_lead[lead])

    #   median
        median_RR = np.median(idx/sample_Fs*1000)
        median_Peaks = np.median(peaks*gain_lead[lead])

    #   standard deviation
        std_RR = np.std(idx/sample_Fs*1000)
        std_Peaks = np.std(peaks*gain_lead[lead])

    #   variance
        var_RR = stats.tvar(idx/sample_Fs*1000)
        var_Peaks = stats.tvar(peaks*gain_lead[lead])

    #   Skewness
        skew_RR = stats.skew(idx/sample_Fs*1000)
        skew_Peaks = stats.skew(peaks*gain_lead[lead])

    #   Kurtosis
        kurt_RR = stats.kurtosis(idx/sample_Fs*1000)
        kurt_Peaks = stats.kurtosis(peaks*gain_lead[lead])

    features = np.hstack([mean_RR,mean_Peaks,median_RR,median_Peaks,std_RR,std_Peaks,var_RR,var_Peaks,skew_RR,skew_Peaks,kurt_RR,kurt_Peaks])

  
    return features





def get_12ECG_features(data, header_data):

    tmp_hea = header_data[0].split(' ')
    ptID = tmp_hea[0]
    num_leads = int(tmp_hea[1])
    sample_Fs= int(tmp_hea[2])
    gain_lead = np.zeros(num_leads)
    numpoints=1800
    skip=int(6)
    for ii in range(num_leads):
        tmp_hea = header_data[ii+1].split(' ')

    # for testing, we included the mean age of 57 if the age is a NaN
    # This value will change as more data is being released
    for iline in header_data:
        if iline.startswith('#Age'):
            tmp_age = iline.split(': ')[1].strip()
            age = int(tmp_age if tmp_age != 'NaN' else 57)
        elif iline.startswith('#Sex'):
            tmp_sex = iline.split(': ')[1]
            if tmp_sex.strip()=='Female':
                sex =1
            else:
                sex=0
    RRtemp = np.array([]) 
    for lead in np.arange(data.shape[0]):
        RRtemp = np.append(RRtemp,get_12ECG_features_original(data, header_data,lead))
    RRtemp = np.append([age,sex],RRtemp)
    
    PTCLOUD = np.array([np.ones(len(data[:,0]))])
    PTCLOUD1 = np.array([np.ones(3)])
    PTCLOUD2 = np.array([np.ones(4)])
    PTCLOUD3 = np.array([np.ones(5)])
    PTCLOUD4 = np.array([np.ones(4)])
    PTCLOUD5 = np.array([np.ones(4)])
    PTCLOUD6 = np.array([np.ones(4)])
    
    
    a = np.array(np.arange(numpoints))

    a = a[a%skip==1]
    for j in a:
        PTCLOUD = np.concatenate((PTCLOUD, [data[:,j]]))
        PTCLOUD1 = np.concatenate((PTCLOUD1, [data[:3,j]]))
        app = np.append(np.append(data[0:1,j],data[6:7,j]),np.append(data[10:11,j],data[11:12,j]))
        PTCLOUD2 = np.concatenate((PTCLOUD2, [app]))
        PTCLOUD3 = np.concatenate((PTCLOUD3, [data[7:,j]]))
        PTCLOUD4 = np.concatenate((PTCLOUD4, [data[3:7,j]]))
        PTCLOUD5 = np.concatenate((PTCLOUD5, [np.append(np.append(data[2:3,j],data[3:4,j]),np.append(data[5:6,j],data[8:9,j]))]))
        PTCLOUD6 = np.concatenate((PTCLOUD6, [np.append(np.append(data[1:2,j],data[4:5,j]),np.append(data[9:10,j],data[10:11,j]))]))

        
    PTCLOUD = np.delete(PTCLOUD,0,0)    
    PTCLOUD1 = np.delete(PTCLOUD1,0,0)
    PTCLOUD2 = np.delete(PTCLOUD2,0,0)
    PTCLOUD3 = np.delete(PTCLOUD3,0,0)
    PTCLOUD4 = np.delete(PTCLOUD4,0,0)
    PTCLOUD5 = np.delete(PTCLOUD5,0,0)
    PTCLOUD6 = np.delete(PTCLOUD6,0,0)
    
    
    
    maxdimension=1
    diagram = ripser(PTCLOUD, maxdim=maxdimension)['dgms']
    diagram1 = ripser(PTCLOUD1, maxdim=maxdimension)['dgms']
    diagram2 = ripser(PTCLOUD2, maxdim=maxdimension)['dgms']
    diagram3 = ripser(PTCLOUD3, maxdim=maxdimension)['dgms']
    diagram4 = ripser(PTCLOUD4, maxdim=maxdimension)['dgms']
    diagram5 = ripser(PTCLOUD5, maxdim=maxdimension)['dgms']
    diagram6 = ripser(PTCLOUD6, maxdim=maxdimension)['dgms']
    
    
    
    RRtemp = np.append(RRtemp, extract(diagram,maxdimension)[1:])
    
    TDAfeat1 = extract(diagram1,maxdimension)
    TDAfeat2 = extract(diagram2,maxdimension)
    TDAfeat3 = extract(diagram3,maxdimension)
    TDAfeat4 = extract(diagram4,maxdimension)
    TDAfeat5 = extract(diagram5,maxdimension)
    TDAfeat6 = extract(diagram6,maxdimension)
    
    
    
    TDAfeat = np.append(np.append(np.append(np.append(TDAfeat1,TDAfeat2),TDAfeat3),np.append(np.append(TDAfeat4,TDAfeat5),TDAfeat6)),[TDAfeat1[-4]+TDAfeat2[-4]+TDAfeat3[-4]+TDAfeat4[-4]+TDAfeat5[-4]+TDAfeat6[-4],TDAfeat1[-3]+TDAfeat2[-3]+TDAfeat3[-3]+TDAfeat4[-3]+TDAfeat5[-3]+TDAfeat6[-3],np.mean([TDAfeat1[-2],TDAfeat2[-2],TDAfeat3[-2],TDAfeat4[-2],TDAfeat5[-2],TDAfeat6[-2]]),np.mean([TDAfeat1[-1],TDAfeat2[-1],TDAfeat3[-1],TDAfeat4[-1],TDAfeat5[-1],TDAfeat6[-1]])])
    
    PERSLAND = persland(diagram[1])
    PERSLAND1 = persland(diagram1[1])
    PERSLAND2 = persland(diagram2[1])
    PERSLAND3 = persland(diagram3[1])
    PERSLAND4 = persland(diagram4[1])
    PERSLAND5 = persland(diagram5[1])
    PERSLAND6 = persland(diagram6[1])
    
    
    
    TDAfeat = np.append(TDAfeat,[PERSLAND[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND1[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND2[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND3[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND4[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND5[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND6[:9]])
    
    TDAfeat = np.append(TDAfeat,[PERSLAND[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND1[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND2[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND3[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND4[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND5[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND6[9:]])
    
    ####Ends features
    
    PTCLOUD = np.array([np.ones(len(data[:,0]))])
    PTCLOUD1 = np.array([np.ones(3)])
    PTCLOUD2 = np.array([np.ones(4)])
    PTCLOUD3 = np.array([np.ones(5)])
    PTCLOUD4 = np.array([np.ones(4)])
    PTCLOUD5 = np.array([np.ones(4)])
    PTCLOUD6 = np.array([np.ones(4)])
    
    a = np.array(np.arange((len(data[1,:])-numpoints),len(data[1,:])))

    a = a[a%skip==1]
    for j in a:
        PTCLOUD = np.concatenate((PTCLOUD, [data[:,j]]))
        PTCLOUD1 = np.concatenate((PTCLOUD1, [data[:3,j]]))
        app = np.append(np.append(data[0:1,j],data[6:7,j]),np.append(data[10:11,j],data[11:12,j]))
        PTCLOUD2 = np.concatenate((PTCLOUD2, [app]))
        PTCLOUD3 = np.concatenate((PTCLOUD3, [data[7:,j]]))
        PTCLOUD4 = np.concatenate((PTCLOUD4, [data[3:7,j]]))
        PTCLOUD5 = np.concatenate((PTCLOUD5, [np.append(np.append(data[2:3,j],data[3:4,j]),np.append(data[5:6,j],data[8:9,j]))]))
        PTCLOUD6 = np.concatenate((PTCLOUD6, [np.append(np.append(data[1:2,j],data[4:5,j]),np.append(data[9:10,j],data[10:11,j]))]))

    PTCLOUD = np.delete(PTCLOUD,0,0)    
    PTCLOUD1 = np.delete(PTCLOUD1,0,0)
    PTCLOUD2 = np.delete(PTCLOUD2,0,0)
    PTCLOUD3 = np.delete(PTCLOUD3,0,0)
    PTCLOUD4 = np.delete(PTCLOUD4,0,0)
    PTCLOUD5 = np.delete(PTCLOUD5,0,0)
    PTCLOUD6 = np.delete(PTCLOUD6,0,0)
    
    
    
    maxdimension=1
    diagram = ripser(PTCLOUD, maxdim=maxdimension)['dgms']
    diagram1 = ripser(PTCLOUD1, maxdim=maxdimension)['dgms']
    diagram2 = ripser(PTCLOUD2, maxdim=maxdimension)['dgms']
    diagram3 = ripser(PTCLOUD3, maxdim=maxdimension)['dgms']
    diagram4 = ripser(PTCLOUD4, maxdim=maxdimension)['dgms']
    diagram5 = ripser(PTCLOUD5, maxdim=maxdimension)['dgms']
    diagram6 = ripser(PTCLOUD6, maxdim=maxdimension)['dgms']
    
    
    
    RRtemp = np.append(RRtemp, extract(diagram,maxdimension)[1:])
    
    TDAfeat1 = extract(diagram1,maxdimension)
    TDAfeat2 = extract(diagram2,maxdimension)
    TDAfeat3 = extract(diagram3,maxdimension)
    TDAfeat4 = extract(diagram4,maxdimension)
    TDAfeat5 = extract(diagram5,maxdimension)
    TDAfeat6 = extract(diagram6,maxdimension)
    
    
    TDAfeat = np.append(TDAfeat,np.append(np.append(np.append(np.append(TDAfeat1,TDAfeat2),TDAfeat3),np.append(np.append(TDAfeat4,TDAfeat5),TDAfeat6)),[TDAfeat1[-4]+TDAfeat2[-4]+TDAfeat3[-4]+TDAfeat4[-4]+TDAfeat5[-4]+TDAfeat6[-4],TDAfeat1[-3]+TDAfeat2[-3]+TDAfeat3[-3]+TDAfeat4[-3]+TDAfeat5[-3]+TDAfeat6[-3],np.mean([TDAfeat1[-2],TDAfeat2[-2],TDAfeat3[-2],TDAfeat4[-2],TDAfeat5[-2],TDAfeat6[-2]]),np.mean([TDAfeat1[-1],TDAfeat2[-1],TDAfeat3[-1],TDAfeat4[-1],TDAfeat5[-1],TDAfeat6[-1]])]))
    
    PERSLAND = persland(diagram[1])
    PERSLAND1 = persland(diagram1[1])
    PERSLAND2 = persland(diagram2[1])
    PERSLAND3 = persland(diagram3[1])
    PERSLAND4 = persland(diagram4[1])
    PERSLAND5 = persland(diagram5[1])
    PERSLAND6 = persland(diagram6[1])
    
    
    
    TDAfeat = np.append(TDAfeat,[PERSLAND[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND1[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND2[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND3[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND4[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND5[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND6[:9]])
    
    TDAfeat = np.append(TDAfeat,[PERSLAND[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND1[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND2[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND3[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND4[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND5[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND6[9:]])
    
    ###Middle Features
    
    
    PTCLOUD = np.array([np.ones(len(data[:,0]))])
    PTCLOUD1 = np.array([np.ones(3)])
    PTCLOUD2 = np.array([np.ones(4)])
    PTCLOUD3 = np.array([np.ones(5)])
    PTCLOUD4 = np.array([np.ones(4)])
    PTCLOUD5 = np.array([np.ones(4)])
    PTCLOUD6 = np.array([np.ones(4)])
    
    a = np.array(np.arange((int(len(data[1,:])/2)-int(numpoints/2)),(int(len(data[1,:])/2)+int(numpoints/2))))
    
    a = a[a%skip==1]
    for j in a:
        PTCLOUD = np.concatenate((PTCLOUD, [data[:,j]]))
        PTCLOUD1 = np.concatenate((PTCLOUD1, [data[:3,j]]))
        app = np.append(np.append(data[0:1,j],data[6:7,j]),np.append(data[10:11,j],data[11:12,j]))
        PTCLOUD2 = np.concatenate((PTCLOUD2, [app]))
        PTCLOUD3 = np.concatenate((PTCLOUD3, [data[7:,j]]))
        PTCLOUD4 = np.concatenate((PTCLOUD4, [data[3:7,j]]))
        PTCLOUD5 = np.concatenate((PTCLOUD5, [np.append(np.append(data[2:3,j],data[3:4,j]),np.append(data[5:6,j],data[8:9,j]))]))
        PTCLOUD6 = np.concatenate((PTCLOUD6, [np.append(np.append(data[1:2,j],data[4:5,j]),np.append(data[9:10,j],data[10:11,j]))]))


    PTCLOUD = np.delete(PTCLOUD,0,0)    
    PTCLOUD1 = np.delete(PTCLOUD1,0,0)
    PTCLOUD2 = np.delete(PTCLOUD2,0,0)
    PTCLOUD3 = np.delete(PTCLOUD3,0,0)
    PTCLOUD4 = np.delete(PTCLOUD4,0,0)
    PTCLOUD5 = np.delete(PTCLOUD5,0,0)
    PTCLOUD6 = np.delete(PTCLOUD6,0,0)
    
    
    
    maxdimension=1
    diagram = ripser(PTCLOUD, maxdim=maxdimension)['dgms']
    diagram1 = ripser(PTCLOUD1, maxdim=maxdimension)['dgms']
    diagram2 = ripser(PTCLOUD2, maxdim=maxdimension)['dgms']
    diagram3 = ripser(PTCLOUD3, maxdim=maxdimension)['dgms']
    diagram4 = ripser(PTCLOUD4, maxdim=maxdimension)['dgms']
    diagram5 = ripser(PTCLOUD5, maxdim=maxdimension)['dgms']
    diagram6 = ripser(PTCLOUD6, maxdim=maxdimension)['dgms']
    
    
    
    RRtemp = np.append(RRtemp, extract(diagram,maxdimension)[1:])
    
    TDAfeat1 = extract(diagram1,maxdimension)
    TDAfeat2 = extract(diagram2,maxdimension)
    TDAfeat3 = extract(diagram3,maxdimension)
    TDAfeat4 = extract(diagram4,maxdimension)
    TDAfeat5 = extract(diagram5,maxdimension)
    TDAfeat6 = extract(diagram6,maxdimension)    
    
    TDAfeat = np.append(TDAfeat,np.append(np.append(np.append(np.append(TDAfeat1,TDAfeat2),TDAfeat3),np.append(np.append(TDAfeat4,TDAfeat5),TDAfeat6)),[TDAfeat1[-4]+TDAfeat2[-4]+TDAfeat3[-4]+TDAfeat4[-4]+TDAfeat5[-4]+TDAfeat6[-4],TDAfeat1[-3]+TDAfeat2[-3]+TDAfeat3[-3]+TDAfeat4[-3]+TDAfeat5[-3]+TDAfeat6[-3],np.mean([TDAfeat1[-2],TDAfeat2[-2],TDAfeat3[-2],TDAfeat4[-2],TDAfeat5[-2],TDAfeat6[-2]]),np.mean([TDAfeat1[-1],TDAfeat2[-1],TDAfeat3[-1],TDAfeat4[-1],TDAfeat5[-1],TDAfeat6[-1]])]))
    
    PERSLAND = persland(diagram[1])
    PERSLAND1 = persland(diagram1[1])
    PERSLAND2 = persland(diagram2[1])
    PERSLAND3 = persland(diagram3[1])
    PERSLAND4 = persland(diagram4[1])
    PERSLAND5 = persland(diagram5[1])
    PERSLAND6 = persland(diagram6[1])
    
    
    
    TDAfeat = np.append(TDAfeat,[PERSLAND[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND1[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND2[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND3[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND4[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND5[:9]])
    TDAfeat = np.append(TDAfeat,[PERSLAND6[:9]])
    
    TDAfeat = np.append(TDAfeat,[PERSLAND[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND1[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND2[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND3[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND4[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND5[9:]])
    TDAfeat = np.append(TDAfeat,[PERSLAND6[9:]])
    
    
    Allfeat = np.append(RRtemp,TDAfeat)
    """
    Allfeat = [Allfeat[k] for k in [43, 45, 7, 710, 73, 39, 9, 109, 3, 37, 260, 35, 71, 21, 884, 565, 11, 23, 135, 121, 880, 659, 133, 105, 131, 141, 320, 485, 95, 301, 145, 129, 402, 440, 385, 139, 41, 829, 97, 185, 38, 340, 220, 5, 790, 119, 103, 55, 19, 114, 13, 17, 873, 107, 630, 445, 336, 655, 99, 437, 653, 52, 91, 751, 545, 432, 123, 814, 643, 50, 639, 848, 174, 398, 600, 46, 116, 120, 802, 561, 72, 144, 127, 882, 433, 430, 482, 267, 51, 439, 74, 426, 886, 415, 85, 132, 511, 64, 79, 656, 57, 96, 487, 819, 10, 877, 608, 835, 81, 449, 75, 685, 53, 82, 83, 90, 828, 918, 521, 434, 12, 912, 311, 500, 441, 815, 874, 137, 652, 526, 61, 111, 93, 15, 499, 258, 631, 386, 572, 18, 830, 2, 304, 147, 143, 697, 660, 474, 536, 687, 136, 266, 40, 182, 269, 359, 77, 205, 20, 684, 360, 98, 522, 284, 65, 770, 86, 712, 421, 399, 670, 27, 827, 292, 809, 289, 609, 483, 813, 170, 649, 673, 236, 110, 223, 58, 76, 422, 920, 908, 669, 604, 914, 642, 199, 676, 892, 610, 562, 622, 100, 346, 444, 662, 25, 831, 206, 270, 787, 87, 933, 242, 921, 188, 452, 856, 281, 911, 825, 31, 250, 574, 757, 92, 678, 778, 633, 425, 278, 125, 502, 607, 726, 448, 740, 321, 431, 60, 805, 693, 70, 241, 22, 34, 274, 134, 627, 618, 0, 49, 47, 917, 708, 288, 818, 657, 635, 26, 748, 709, 512, 337, 878, 509, 361, 620, 495, 872, 501, 928, 101, 692, 221, 728, 24, 703, 423, 901, 922, 112, 293, 479, 852, 383, 680, 153, 520, 727, 484, 616, 286, 689, 115, 619, 753, 33, 436, 519, 585, 894, 166, 510, 885, 471, 59, 401, 480, 527, 67, 16, 883, 117, 427, 347, 737, 375, 900, 566, 443, 32, 28, 277, 582, 54, 248, 154, 543, 523, 294, 636, 291, 315, 741, 453, 623, 677, 246, 617, 273, 230, 29, 30, 463, 94, 887, 731, 312, 517, 380, 303, 313, 641, 797, 864, 363, 516, 424, 104, 69, 181, 535, 84, 552, 601, 895, 913, 547, 6, 414, 353, 626, 455, 505, 881, 664, 506, 666, 934, 559, 876, 263, 275, 314, 640, 796, 42, 584, 696, 342, 833, 118, 786, 799, 102, 62, 724, 803, 695, 671, 518, 817, 846, 490, 496, 550, 14, 367, 365, 586, 418, 124, 156, 599, 349, 865, 89, 774, 613, 48, 909, 931, 875, 891, 379, 926, 255, 171, 707, 457, 658, 167, 647, 907, 558, 113, 810, 534, 142, 648, 252, 454, 644, 683, 826, 155, 736, 503, 374, 450, 419, 217, 298, 551, 531, 164, 491, 804, 691, 36, 932, 910, 588, 629, 508, 783, 324, 606, 732, 262, 435, 290, 847, 489, 899, 688, 598, 832, 916, 150, 638, 504, 705, 576, 322, 310, 233, 238, 478, 492, 592, 364, 823, 355, 187, 905, 472, 464, 840, 612, 403]]
     
               
               [43, 45, 7, 710, 39, 73, 9, 109, 3, 37, 35, 260, 71, 21, 23, 11, 884, 565, 135, 121, 659, 133, 880, 141, 131, 105, 320, 485, 95, 139, 145, 301, 440, 829, 129, 41, 185, 385, 119, 13, 402, 340, 220, 38, 5, 790, 17, 19, 873, 97, 114, 107, 55, 336, 630, 445, 103, 174, 116, 52, 99, 751, 437, 600, 91, 432, 643, 545, 433, 123, 848, 72, 639, 882, 802, 814, 50, 653, 561, 46, 430, 57, 127, 655, 120, 144, 426, 74, 398, 877, 819, 51, 90, 500, 886, 267, 85, 81, 53, 656, 79, 439, 64, 415, 828, 82, 835, 132, 487, 10, 434, 449, 482, 685, 96, 815, 98, 75, 83, 2, 61, 311, 687, 441, 631, 912, 12, 526, 93, 499, 652, 18, 258, 92, 136, 521, 15, 608, 918, 143, 111, 830, 20, 874, 572, 170, 697, 511, 660, 137, 223, 182, 77, 684, 474, 289, 669, 670, 359, 892, 346, 86, 421, 609, 40, 386, 642, 444, 100, 284, 483, 536, 32, 360, 914, 422, 269, 266, 813, 110, 76, 856, 827, 622, 399, 770, 787, 522, 676, 147, 125, 278, 47, 712, 60, 304, 604, 292, 24, 0, 809, 509, 610, 274, 920, 321, 673, 34, 831, 933, 27, 65, 778, 633, 134, 423, 757, 495, 562, 921, 607, 281, 878, 649, 241, 69, 58, 709, 448, 928, 452, 206, 431, 25, 657, 205, 917, 805, 436, 708, 908, 484, 104, 693, 87, 620, 29, 872, 662, 414, 49, 26, 31, 188, 199, 383, 913, 585, 425, 689, 627, 230, 748, 70, 22, 337, 626, 911, 59, 250, 566, 726, 242, 703, 277, 616, 480, 293, 692, 236, 512, 502, 678, 922, 270, 33, 166, 574, 774, 852, 471, 479, 825, 894, 443, 680, 635, 618, 501, 361, 101, 28, 221, 347, 901, 728, 797, 286, 117, 424, 818, 900, 288, 535, 731, 741, 582, 887, 885, 619, 727, 677, 753, 905, 313, 641, 112, 506, 463, 666, 552, 294, 315, 740, 171, 584, 375, 520, 380, 510, 455, 342, 559, 864, 115, 16, 519, 647, 153, 303, 94, 648, 84, 275, 516, 724, 30, 273, 896, 636, 883, 453, 67, 363, 934, 490, 623, 617, 737, 427, 543, 547, 601, 246, 832, 140, 909, 54, 36, 505, 263, 517, 466, 312, 696, 664, 42, 527, 786, 644, 324, 62, 804, 518, 478, 508, 419, 298, 613, 881, 113, 695, 102, 314, 450, 118, 803, 550, 688, 683, 895, 586, 181, 454, 167, 325, 379, 799, 732, 187, 796, 916, 489, 629, 142, 367, 472, 353, 840, 523, 833, 907, 248, 876, 931, 891, 403, 910, 262, 268, 349, 14, 576, 810, 926, 6, 364, 291, 8, 734, 154, 204, 557, 124, 871, 89, 658, 534, 48, 592, 707, 767, 722, 826, 491, 691, 598, 435, 816, 328, 599, 504, 464, 932, 865, 496, 588, 736, 515, 679, 130, 252, 155, 401, 531, 287, 88, 492, 558, 310, 503, 765, 68, 365, 551, 261, 612, 624, 723, 675, 783, 66, 640, 758, 533, 704, 782, 126, 238, 713, 290, 256, 418, 282, 546, 322, 847, 390, 890, 164, 625, 754, 451, 280, 300, 817, 156, 395, 537, 234, 867, 217, 621, 893, 541, 899, 150, 668, 63, 462, 374, 255, 823, 705, 671, 410, 428, 213, 283, 800, 930, 606, 661, 556, 858, 904, 208, 189, 859, 851, 875, 56, 229, 470, 667, 846, 296, 638, 706, 214, 457, 700, 888, 694, 276, 456, 532, 192, 493, 245, 925, 352, 406, 721, 839, 200, 628, 138, 106, 834, 397, 411, 861, 855, 333, 544, 376, 4, 513, 563, 682, 297, 701, 78, 271, 404, 148, 413, 863, 80, 285, 235, 382, 211, 750, 869, 307, 44, 201, 344, 233, 762, 194, 822, 326, 494, 866, 488, 807, 686, 355, 172, 843, 702, 108, 929, 381, 824, 538, 412, 498, 225, 769, 468, 168, 446, 400, 497, 605, 632, 597, 868, 165, 730, 377, 779, 768, 316, 459, 794, 583, 898, 528, 177, 615, 384, 196, 735, 763, 259, 573, 711, 903, 335, 850, 733, 665, 579, 372, 923, 264, 460, 209, 178, 149, 720, 746, 838, 388, 540, 257, 801, 715, 853, 836, 226, 718, 760, 243, 776, 481, 239, 784, 594, 811, 232, 595, 348, 567, 237, 122, 577, 358, 637, 476, 530, 792, 571, 393, 373, 317, 614, 747, 716, 849, 265, 356, 771, 564, 159, 773, 529, 699, 249, 845, 244, 405, 857, 228, 396, 210, 370, 475, 725, 568, 408, 469, 467, 387, 795, 542, 420, 744, 368, 596, 514, 738, 309, 906, 247, 560, 180, 870, 924, 752, 719, 525, 681, 207, 743, 590, 461, 591, 844, 781, 458, 343, 160, 362, 739, 780, 645, 808, 295, 329, 391, 184, 330, 224, 553, 447, 742, 389, 862, 305, 251, 674, 179, 651, 175]]
    
    
               [511, 262, 435,  39, 608,  15,  45,  75,  43, 472,   0, 123, 609,
       135,  41,  29,  19, 127, 129,  77, 125,  87,  53, 607,  63, 111,
        27,  81,  57, 434,   3,  69,  67, 139,  51,  21,  49,  55,  99,
       436, 133, 137, 263,  31, 606, 141,   9, 260, 433,   7,  47, 626,
        33,   5,  17, 261,  79,  30, 223, 185, 437,  11, 453, 126, 454,
       438,  25, 182, 121,  23, 456, 610, 628,  59, 321, 113,  89, 548,
        56, 455, 304, 109, 145,  13,  65, 667,  35, 389, 258,  90, 336,
        91, 322, 264, 627, 131, 147, 280,  42, 303, 281, 668,  93, 629,
       220, 117, 494, 115,  54,  66, 265, 138,  71, 611, 375,  80, 721,
       143, 477, 552,  32, 114, 509, 682, 495, 379, 725, 684,   6, 283,
       705, 513, 119, 301, 103, 476, 128,  18, 104,  73, 766, 385, 765,
       562, 144,  44,   8, 388, 140, 762, 243, 107, 361, 102,  38, 342,
       649, 101, 282, 561,  74,  20, 533, 134, 395, 360, 105, 686,  40,
        97,  92, 647, 116, 650, 755,  85, 168, 759, 474,  96, 150, 187,
       298, 470,  68, 532, 761, 586, 205, 340, 558,  83,  14, 193, 244,
       149, 167, 242, 343, 706, 582, 188, 391,  98,  82, 320, 204, 471,
       110, 431, 603,  50, 599,  37, 452,  84,  78, 206, 384, 557, 203,
       698, 768, 756, 297, 430, 707,  36, 569, 106, 643, 124,  26, 770,
       432,  62,  95,  61, 184, 531, 122, 392, 359,  10, 737, 224, 741,
       112, 742, 469,  60, 225, 491, 483,  94, 130, 396,  86, 534, 583,
        88, 591, 515, 640, 412,  28, 416,   4, 136, 689, 514, 587, 619,
       688, 760, 241, 716,   2, 710, 769,  34, 422, 508, 272,  12, 772,
       427, 118, 571, 507, 338, 186, 551, 496, 699, 269, 380, 764, 778,
       601, 516, 226,  64, 142, 108, 468, 567, 410, 398, 565,  16, 100,
       369, 758, 566, 146, 666, 166, 685, 512, 222, 757, 473,  58, 596,
       475, 540, 711,  76, 290, 467, 268, 687, 605, 743, 493, 644, 465,
       693, 597, 444, 289, 570, 696, 329, 285, 177, 506, 189,  48, 394,
       604, 713, 479, 550, 616, 313, 480, 358, 446, 215, 641, 704, 738,
       132, 726, 773, 525, 621, 406, 426]]
    """
   
                
      #         [ 186, 188,  39, 189, 123,   0, 185,  75,  43,  15,  45, 135,  29,
      #  187,  53,  87, 206, 207, 129, 127, 204, 184,  27, 125,90,133,225,147,284,245,7,227,167]]
    #Allfeat = [Allfeat[k] for k in [0,1,8,15,25,31,35,43,44,49,50,51,56,69,71,75,76,83,85,88,89,94,97,101,111,112,120,121,123,125,133,138,140,145]]
    #Allfeat = [Allfeat[k] for k in [220,0,1,8,15,25,31,35,43,44,49,50,51,56,69,71,75,76,83,85,88,89,94,97,101,111,112,120,121,123,125,133,138,140,145,157,167,175,178,180,195,199,209,210,224,226,235,244,249,252,257,261,277,289,311,350,359]]
    
    return Allfeat
    
