# Copied from https://stanford.edu/~shervine/blog/keras-how-to-generate-data-on-the-fly

import os, glob
import numpy as np
from tensorflow.keras.utils import Sequence
from get_12ECG_features import get_12ECG_features, get_random_forest_features
#import matplotlib.pyplot as plt

class DataGenerator(Sequence):
    'Generates data for Keras'
    def __init__(self, patientIDs, input_directory, scaler, loadFunc, mapFunc, classes, batch_size=32, n_channels=12,
                 dim=4096, nRF_features=14, rfScaler=None, imputer=None):
        'Initialization'
        self.idList = patientIDs
        self.input_directory = input_directory
        self.scaler = scaler
        self.rfScaler = rfScaler
        self.imputer = imputer
        self.loadFunc = loadFunc
        self.mapFunc = mapFunc
        self.classes = classes
        self.batch_size = batch_size
        self.n_channels = n_channels
        self.n_classes = len(classes)
        self.dim = dim
        self.nRF_features = nRF_features
        self.shuffle = True
        #self.idList = [os.path.basename(x)[:-4] for x in glob.glob(input_directory + '\\*.mat')]
        self.nFiles = len(self.idList)
        self.on_epoch_end()

    def __len__(self):
        'Denotes the number of batches per epoch'
        return int(np.floor(self.nFiles / self.batch_size))

    def __getitem__(self, index):
        'Generate one batch of data'
        # Generate indexes of the batch
        indexes = self.indexes[index*self.batch_size:(index+1)*self.batch_size]

        # Find list of IDs
        list_IDs_temp = [self.idList[k] for k in indexes]

        # Generate data
        X, y = self.__data_generation(list_IDs_temp)

        return X, y

    def getLabel(self, header):
        for l in header:
            if l.startswith('#Dx:'):
                labels_act = np.zeros(self.n_classes)
                arrs = l.strip().split(' ')
                for arr in arrs[1].split(','):
                    cls = self.mapFunc(arr.rstrip(), self.classes)  # JCC
                    if cls:
                        class_index = self.classes.index(cls)  # Only use first positive index
                        labels_act[class_index] = 1
        return labels_act

    def on_epoch_end(self):
        'Updates indexes after each epoch'
        self.indexes = np.arange(self.nFiles)
        if self.shuffle == True:
            np.random.shuffle(self.indexes)

    # def plotData(self, data):
    #     plt.figure()
    #     plt.plot(data[:,1])
    #     plt.show()
    #     return

    def __data_generation(self, list_IDs_temp):
        'Generates data containing batch_size samples' # X : (n_samples, *dim, n_channels)
        # Initialization
        X = np.empty((self.batch_size, self.dim, self.n_channels))
        Z = np.empty((self.batch_size, self.nRF_features))
        y = np.empty((self.batch_size, self.n_classes))

        # Generate data
        for i, ID in enumerate(list_IDs_temp):
            # Store sample
            header_file = os.path.join(self.input_directory, ID + '.hea')
            recording, header = self.loadFunc(header_file)
            X[i,] = get_12ECG_features(recording, header, self.scaler)

            # self.plotData(X[i,])

            y[i,] = self.getLabel(header)

            if self.rfScaler and self.imputer:
                Z[i,] = get_random_forest_features(recording, header, self.rfScaler, self.imputer)

        if self.rfScaler and self.imputer:
            return [X, Z], y
        else:
            return X, y