function classifyResult = challenge(recordName)
%
% Sample entry for the 2017 PhysioNet/CinC Challenge.
%
% INPUTS:
% recordName: string specifying the record name to process
%
% OUTPUTS:
% classifyResult: integer value where
%                     N = normal rhythm
%                     A = AF
%                     O = other rhythm
%                     ~ = noisy recording (poor signal quality)
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% The challenge function requires that you have downloaded the challenge
% data 'training_set' in a subdirectory of the current directory.
%    http://physionet.org/physiobank/database/challenge/2017/
%
% This dataset is used by the generateValidationSet.m script to create
% the annotations on your training set that will be used to verify that
% your entry works properly in the PhysioNet testing environment.
%
%
% Version 1.0
%
%
% Written by: Chengyu Liu and Qiao Li January 20 2017
%             chengyu.liu@emory.edu  qiao.li@emory.edu
%
% Last modified by:
%
%

classifyResult = '~'; % default output noise

%% AF determination

%recordName = 'validation/A00001';
[tm,ecg,fs,siginfo] = rdmat(recordName);
[QRS,sign,en_thres] = qrs_detect2(ecg',0.55,0.65,fs);

if sign<0
    ecg=-ecg;
end

QRS_radius = fs/20;
QRS_height_threshold = 0.2;
RR_new_length = 20;
SQ_amplifying = 30;
SQ_smoothing = 3;
TP_distance_threshold = 0.3;

QRS = maximize_qrs(ecg, QRS, QRS_radius);	
[~, RR_lengths, ST_lengths, PQ_lengths, R_prevs, T_vals, P_vals, R_currs] = ...
    extract_all_info(ecg, QRS, QRS_height_threshold, RR_new_length, SQ_amplifying, SQ_smoothing, TP_distance_threshold);

n = size(RR_lengths,2);
periods = 6;

if periods > n
    return
end
    
row = 1;
X = [];
for i=periods:n
    column = 0;
    for j=(i-periods+1):i
        column = column+1;
        X(row, column) = RR_lengths(j);
        column = column+1;
        X(row,column) = ST_lengths(j);
        column = column+1;
        X(row,column) = ST_lengths(j)/RR_lengths(j);
        column = column+1;
        X(row,column) = PQ_lengths(j);
        column = column+1;
        X(row,column) = PQ_lengths(j)/RR_lengths(j);            

        column = column+1;
        X(row,column) = R_prevs(j);            
        column = column+1;
        X(row,column) = R_currs(j);

        column = column+1;            
        X(row,column) = T_vals(j);
        column = column+1;
        X(row,column) = T_vals(j)/R_prevs(j);
        column = column+1;
        X(row,column) = P_vals(j);
        column = column+1;
        X(row,column) = P_vals(j)/R_currs(j);        
    end

    for tmp = {RR_lengths ST_lengths PQ_lengths R_prevs R_currs T_vals P_vals}
        v = double(tmp{1}((i-periods+1):i));
        
        column = column+1;
        X(row,column) = std(v);
        column = column+1;
        X(row,column) = min(v);
        column = column+1;
        X(row,column) = mean(v);
        column = column+1;
        X(row,column) = max(v);
    end
    
    row = row + 1;
end
    
tr = load('model_full_tree_100');
ytest = predict(tr.model, X);
[~,pmax] = max(histc(ytest,[0,1,2,3]));

switch pmax
    case 1
       classifyResult = 'N';
    case 2
       classifyResult = 'A';
    case 3
       classifyResult = 'O';
    otherwise
       classifyResult = '~';
end
