function PredictedAnnotation = challenge(recordName)
%
% Sample entry for the 2017 PhysioNet/CinC Challenge.
%
% INPUTS:
% recordName: string specifying the record name to process
%
% OUTPUTS:
% classifyResult: integer value where
%                     N = normal rhythm
%                     A = AF
%                     O = other rhythm
%                     ~ = noisy recording (poor signal quality)
%
% Last modified by: Philip de Chazal 28/08/2017
%

load ClassifyAF_PdeC_Parameters
PredictedAnnotation = 'N'; % default output normal rhythm

[tm,ecg,fs,siginfo]=rdmat(recordName);

%%%%%%%%%%%%%%%%%
%1.  QRS detection
%%%%%%%%%%%%%%%%%

%First, median filter ECG to correct baseline
ecgbl1=medfilt1(ecg,round(0.2*fs));
ecgbl2=medfilt1(ecgbl1,round(0.6*fs));
ecg=ecg-ecgbl2;
[QRS,sign,en_thres] = qrs_detect2(ecg',0.25,0.6,fs);
if sign<0 
   [QRS,sign,en_thres] = qrs_detect2(-ecg',0.25,0.6,fs);  
end


%%%%%%%%%%%%%%%%%
%2. Feature extraction
%%%%%%%%%%%%%%%%%%%
if length(QRS)<6
    PredictedAnnotation='N';
    return
else
    RR=diff(QRS')/fs;
    
%Features
%2.1 Heart rate >100bpm and <60bpm
if median(RR)< 60/100
    HRflag=[1 0];
elseif median(RR)> 60/60
    HRflag=[0 1];
else
    HRflag=[0 0];
end

%2.2 Mean plus std HR
medianRR=median(RR);

RRs=sort(RR);
l=1+round(length(RR)*0.05);
u=round(length(RR)*0.95);
mintrimmedRR=RRs(l);
maxtrimmedRR=RRs(u);

stdRR=log(std(RR));

PSD_RR = PSD_features(RR);

normRR=(RR-mean(RR));

histRR=hist(normRR,[-0.5:0.05:0.5]);
histRR=histRR/length(RR);

diffRR=diff(RR);
PSD_diffRR = PSD_features(diffRR);
histdiffRR=hist(diffRR,[-0.5:0.05:0.5]);
histdiffRR=histdiffRR/length(diffRR);

%2.4 Other features
AFEv = comput_AFEv(RR); 
SNR =snr(ecg,fs);

%2.5 Assemble Feature vector
X=[HRflag medianRR stdRR mintrimmedRR maxtrimmedRR AFEv SNR PSD_RR PSD_diffRR histRR histdiffRR];end

M=length(X);

%%%%%%%%%%%%%%%%%%%%%
%3. Classification
%%%%%%%%%%%%%%%%%%%%%

%Find all the nans and infs in the data and if record has greater than 5%
%features missing then return the default classification
MVidx=find(~isfinite(X));
if length(MVidx)>=0.05*M
    return
end

%Replace missing values with mean feature values determined form the 8528
%training examples
X(MVidx)=mX(MVidx);

%Select feature used for processing (determined from training data)
X=X(Features);

%Append a bias to end of the data and process with a quadratic classifier
%(two layer classifier, first layer square non-linearity, 2nd layer linear
%detemined using linear regression.
Y=(([X,1]*W{2}).^2)*W{3};

%Find argmax of Y and return the corresponding class

[~,idx]=max(Y);
if idx==2
    PredictedAnnotation='N';
elseif idx==3
    PredictedAnnotation='A';
elseif idx==4
    PredictedAnnotation='O';
elseif idx==1
    PredictedAnnotation='~';
else
    error('Unknown class')
end

end

function PSD_intervals = PSD_features(Intervals)
        
        Data=zeros(1,256);                   % Zero-padded to length 256
        Data(1:length(Intervals))=Intervals-mean(Intervals);   % remove mean of segment to have zero-mean sequence
        PSD=abs(fft(Data)).^2; %Spectrum estimate
        PSD=reshape(PSD(1:128),4,128/4); %Remove upper half of specturm and reshape ready for averaging
        PSD_intervals=log(sum(PSD,1)/4);    %Features = log of PSD bin sums
        
end