function classifyResult = challenge(recordName)
%
% Sample entry for the 2017 PhysioNet/CinC Challenge.
%
% INPUTS:
% recordName: string specifying the record name to process
%
% OUTPUTS:
% classifyResult: integer value where
%                     N = normal rhythm
%                     A = AF
%                     O = other rhythm
%                     ~ = noisy recording (poor signal quality)
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% The challenge function requires that you have downloaded the challenge
% data 'training_set' in a subdirectory of the current directory.
%    http://physionet.org/physiobank/database/challenge/2017/
%
% This dataset is used by the generateValidationSet.m script to create
% the annotations on your training set that will be used to verify that
% your entry works properly in the PhysioNet testing environment.
%
%
% Version 1.0
%
%
% Written by: Chengyu Liu and Qiao Li January 20 2017
%             chengyu.liu@emory.edu  qiao.li@emory.edu
%
% Last modified by:
%
%%
debugFlag=0; 
graphS=0; saveGraphS=0;
graphSpt=0;

eBrem=1;

% ------------------------------------------------------------------------
% ECG record reading
[tmo,ecgo,fso,siginfo]=rdmat(recordName);
tmo=tmo'; recLen=tmo(end);

[nsamp,ns]=size(ecgo);

ip=1;
% ------------------------------------------------------------------------
% ---- Artifact canceling ----
% X=EcgImpArtCanc(ECG,fs,cName,graph,dbFlag,saveFig);
X=EcgImpArtCanc(ecgo,fso,recordName,graphS,0,saveGraphS);

% ---- detrending  ----
% Xd=EcgDetrFilt(X,fs,cName,graph,dbFlag,saveFig);
[Xd,pRad]=EcgDetrFilt3Hz_im(X,fso,recordName,graphS,0,saveGraphS);
% ------------------------------------------------------------------------
pRad=sum(pRad); 
ecgFeatNames(ip)={'pRad'}; ecgFeatVals(ip)=pRad; ip=ip+1;

% ------------------------------------------------------------------------
% ---- Power line interference removal by notch filtering ----
% Xf=EcgNotchFilt(Xd,fs,cName,graph,dbFlag,saveFig);
[Xf,pRpl]=EcgNotchFiltR(Xd,fso,recordName,graphS,0,saveGraphS);
if(debugFlag && graphSpt), figure; pwelch(Xf(:,1),[],[],[],fso); end
% pRpl=sum(pRpl);
% ecgFeatNames(ip)={'pRpl'}; ecgFeatVals(ip)=pRpl; ip=ip+1;
% ------------------------------------------------------------------------
for is=1:ns
    pRpre(is)=varsc(Xf(:,is)-ecgo(:,is),1,1)/varsc(X(:,is),1,1);
    if(pRpre(is)<=0), pRpre(is)=1e-9; end
    if(isnan(pRpre(is))), pRpre(is)=1; end
    pRpre(is)=log(pRpre(is));
end
pRpre=sum(pRpre);
ecgFeatNames(ip)={'pRpre'}; ecgFeatVals(ip)=pRpre; ip=ip+1;
%-------------------------------------------------------------------------
[Xl,pRhf]=EcgLowPass(Xf,fso,recordName,graphS,0,saveGraphS);
if(debugFlag && graphSpt), figure; pwelch(Xl(:,1),[],[],[],fso); end
pRhf=sum(pRhf);
ecgFeatNames(ip)={'pRhf'}; ecgFeatVals(ip)=pRhf; ip=ip+1;
% ------------------------------------------------------------------------
% 
[Xb46,pRb46]=Ecg4_6bandPass(Xl,fso,recordName,graphS,0,saveGraphS);
pRb46=sum(pRb46);
ecgFeatNames(ip)={'pRb46'}; ecgFeatVals(ip)=pRb46; ip=ip+1;
% ------------------------------------------------------------------------
ecgf=Xf;
% ------------------------------------------------------------------------
% ---- Signal interpolation
% [Xi,fs]=EcgInterp(X,fs,interpFact,cName,graph);
[Xl,fs]=EcgInterp(Xl,fso,4,recordName,0);
[ecg,fs]=EcgInterp(ecgf,fso,4,recordName,0);
[ecgDet,fs]=EcgInterp(ecgo,fso,4,recordName,0);
tm=linspace(tmo(1),tmo(end),length(ecg))';

%% QRS detection
pth=0.35;
[qrsMc,vadx]=QRSdetectorEcgM(ecg,fs,pth,-1);
qrsMs=qrsMc/fs;
qrsPc=qrsMc(:,1);
qrsPs=qrsPc/fs;
nQrs=numel(qrsPs);
RRs=diff(qrsPs);

%% signal feature extraction
% -------------------------------------------------------------------------
% QRS  canceling and extraction of spectral parameters from residual signal
if(nQrs/recLen > 0.2 && nQrs/recLen <4)
    % ---- QRS cancelling
    %Xr=EcgQRScanc(Xl,qrsP,fs,[],recordName,graphC,dbFlag,saveFigCm,figArtPar,fidLog);
    Xr=EcgQRScanc(Xl,qrsPc,fs,[],recordName,0,0);
    
    [FpPmax,mPdAF,mPdAFr]=Ecg_spectPeak(Xr(qrsPc(1):qrsPc(end)),fs,recordName,graphS,0,saveGraphS);
    
    
    [Xrb46,pRrb46]=Ecg4_6bandPass(Xr(qrsPc(1):qrsPc(end)),fs,recordName,graphS,0,saveGraphS);
    pRrb46=sum(pRrb46);
else
    FpPmax=NaN; mPdAF=NaN; mPdAFr=NaN; 
    Xrb46=NaN; pRrb46=NaN;
end
ecgFeatNames(ip)={'pRrb46'}; ecgFeatVals(ip)=pRrb46; ip=ip+1;
ecgFeatNames(ip)={'mPdAF'};  ecgFeatVals(ip)=mPdAF;  ip=ip+1;
ecgFeatNames(ip)={'mPdAFr'}; ecgFeatVals(ip)=mPdAFr; ip=ip+1;

% ------------------------------------------------------------------------

%% QRS extraction of morphology features
if(nQrs/recLen > 0.2 && nQrs/recLen <4)
    QRSwidthQ= (qrsMs(:,6)-qrsMs(:,2));
    
    [QRSwidthMeanQ,QRSwidthStdQ ]=meanstdsc(QRSwidthQ,2,2);
    
    [QRSmorphPar]=EcgQRSmorphPar(Xl,qrsPc,fs,[],recordName,0,0);
    
    QRSewidth=QRSmorphPar.QRSewidth;
    QRSppAmplMea=QRSmorphPar.QRSppAmplMea;
    QRSpprAmplMea=QRSmorphPar.QRSpprAmplMea;
    QRSpolMea=QRSmorphPar.QRSpolMea;
    
    QRSwidthMax=QRSmorphPar.QRSwidthMax;
    QRSwidthMin= QRSmorphPar.QRSwidthMin ;
    QRSwidthMea= QRSmorphPar.QRSwidthMea ;
    QRSwidthNmea= QRSmorphPar.QRSwidthNmea ;
    QRSwidthStd= QRSmorphPar.QRSwidthStd;
    pnWideQRS= QRSmorphPar.pnWideQRS;
    morphDiffr= QRSmorphPar.morphDiffr;
    eigDiff2= QRSmorphPar.eigDiff2;
    eigDiff= QRSmorphPar.eigDiff;
    
    clear atRR atQRS
    ii=2; ie=nQrs-isnan(QRSewidth(end));
    QRSwidthR=QRSewidth(ii:ie); RRsR=RRs(1:ie-1);
    RRmea=meansc(RRsR,20,5);
    i1=find(RRsR<RRmea);
    atRR(i1,1)=(RRmea-RRsR(i1))/RRmea;
    atQRS(i1,1)=QRSwidthR(i1) .*(1+atRR(i1));
    i2=find(RRsR>=RRmea);
    atRR(i2,1)=(RRsR(i2)-RRmea)/RRmea;
    atQRS(i2,1)=QRSwidthR(i2) .*(1+atRR(i2));
    
    pnatQRS=sum(atQRS-meansc(atQRS,0,30))/numel(atQRS);
    
else
    QRSwidthQ=[];
    QRSwidthMeanQ=NaN; QRSwidthStdQ=NaN;
    QRSewidth=[];
    QRSppAmplMea=NaN;
    QRSpprAmplMea=NaN;
    QRSpolMea=NaN;
    QRSwidthMax=NaN;
    QRSwidthMin=NaN;
    QRSwidthMea=NaN;
    QRSwidthNmea=NaN;
    QRSwidthStd=NaN;
    pnWideQRS=NaN;
    morphDiffr=NaN;
    eigDiff2=NaN;
    eigDiff=NaN;
    pnatQRS=NaN;
end
QRSmorphFeatNames={'QRSwidthMeanQ','QRSwidthStdQ','QRSppAmplMea','QRSpprAmplMea', ...
    'QRSpolMea', 'QRSwidthMax','QRSwidthMin','QRSwidthMea','QRSwidthNmea','QRSwidthStd',...
    'pnWideQRS','pnatQRS','morphDiffr','eigDiff2','eigDiff'};
QRSmorphFeatVals= [ QRSwidthMeanQ,  QRSwidthStdQ, QRSppAmplMea, QRSpprAmplMea, ...
    QRSpolMea, QRSwidthMax, QRSwidthMin, QRSwidthMea, QRSwidthNmea,  QRSwidthStd,...
    pnWideQRS,  pnatQRS,   morphDiffr,  eigDiff2,  eigDiff];

% -------------------------------------------------------------------------
%% RR series feature extraction
if(eBrem)
    % ectopic beats
    RRe=EctBeatRemM(RRs);

    RReSW=RRe~=RRs;
end
if(nQrs), RRatPrc=sum(RReSW)/nQrs; else RRatPrc=0; end
% -------------------------------------------------------------------------

[qrsPsc, qrsType, npQm]=selNNeAFqrsP(qrsPs, recordName);
if(nQrs)
    nAtypeP=(npQm.nLVC+npQm.nPVC+npQm.nFP+npQm.nFN)/nQrs;
    nPVCp=(npQm.nPVC)/nQrs;
else
    nAtypeP=NaN;
    nPVCp=NaN;
end
if(~isempty(QRSwidthQ))
    [AtypBeatPr,AtypBeatPrk1] = AtypBeatDetPr(qrsPs,QRSwidthQ, recLen);
else
    AtypBeatPr=NaN; AtypBeatPrk1=NaN;
end
if(~isempty(QRSewidth) & nQrs>1)
    if(isnan(QRSewidth(1))), QRSewidth(1)=QRSewidth(2); end
    if(isnan(QRSewidth(end))), QRSewidth(end)=QRSewidth(end-1); end
    [AtypBeatPre,AtypBeatPrk1e] = AtypBeatDetPr(qrsPs,QRSewidth, recLen);
else
    AtypBeatPre=NaN; AtypBeatPrk1e=NaN;
end
% -------------------------------------------------------------------------

[RRfeatNames,RRfeatVals] = RRfeatures(RRs);

% -------------------------------------------------------------------------
AtypBeatFeatNames={'RRatPrc','nAtypeP','nPVCp','AtypBeatPr','AtypBeatPrk1','AtypBeatPre','AtypBeatPrk1e'};
AtypBeatFeatVals=[RRatPrc,nAtypeP,nPVCp,AtypBeatPr,AtypBeatPrk1,AtypBeatPre,AtypBeatPrk1e];
% -------------------------------------------------------------------------
bigem = Bigem(RRs);  
trigem = Trigem(RRs);
ip=numel(AtypBeatFeatNames)+1;
AtypBeatFeatNames(ip)={'bigem'}; AtypBeatFeatVals(ip)=bigem; ip=ip+1;
AtypBeatFeatNames(ip)={'trigem'}; AtypBeatFeatVals(ip)=trigem; ip=ip+1;
% -------------------------------------------------------------------------

featNames=[ecgFeatNames,QRSmorphFeatNames,AtypBeatFeatNames,RRfeatNames];
featVals= [ecgFeatVals, QRSmorphFeatVals, AtypBeatFeatVals, RRfeatVals];
% -------------------------------------------------------------------------
% Log transform of some measures
featNamesTrasf={'QRSwidthStd','pnWideQRS','pnatQRS','RRatPrc',...
    'nAtypeP','nPVCp','RRbrady','RRtachy','KFD','Rmssdn','Mhlbw','IndPeC'};
xmin=[1.79e-04, 0, 2.2e-04, 0, 0, 0, 0, 0, 1.0, 0.0021, 0, -24.08];

for jp=1:numel(featNamesTrasf)
    ip=find(strcmp(featNames,featNamesTrasf(jp)));
    x=featVals(ip);
    featVals(ip)=log(max(x-xmin(jp),0)+.01);
end

% -------------------------------------------------------------------------
%% classification
% load classification model parameters
load('SVMmodel30_8525_Fe0831_Me0901_0.902_0.902_16.mat');

selFeatNames=model.features;
nSelFeat=numel(selFeatNames);
% fprintf('Seleted features: '); fprintf('%s, ',selFeatNames); fprintf('\n');
ips=zeros(nSelFeat,1);
for ip=1:nSelFeat
    ips(ip)=find(strcmp(featNames,selFeatNames{ip}));
end

xs=featVals(ips);

if(sum(isnan(xs)))
    classEst =4;
else
    classEst = simlssvm(model,xs);
    if(classEst<1 || classEst>4), classEst =4; end

end

classSymbols={'N','A','O','~'};

classifyResult=classSymbols{classEst};

