function [qrsM,varargout]=QRSdetectorEcgM(ecg,Fs,pth,icDet_in,icFp_in,dbFlag)
% -------------------------------------------------------------------------------------------------
% QRSdetectorEcg_FbGc.m: QRS detector - only one filter by convolution
%   - compute a filtered derivative
%   - take the absolute value and call QRSdetctorQ... routine
%
%   Input parameters:
%    ecg    : ECG signals
%    Fs     : sampling frequency
%    pth    : threshold on derivative
%    icDet  : channel indexes used in detection 
%             (0=all, n= channel n, -1 automatically chosen) 
%    icFp   : channel index for QRS fiducial point estimation
%             (n= channel n, -1 automatically chosen) 
%    dbFlag : flag enabling signal plot for tuning
%
%   Ouput parameters:
%   qrsM  : matrix (number_of_detected_QRS X 6) consisting of
%           a row of 6 time indeces for each detected QRS:
%             QRSref    (reference point, max signed derivative )
%             QRSonset  (derivative overcomes half threshold)
%             supThDer  (derivative overcomes threshold)
%             maxAbsDer (max absolute derivative)
%             infThDer  (derivative becomes lower than threshold)
%             QRSoffset (derivative decreses below half threshold)
%   vadx (optional) :  array of filtered absolute derivate values
%
%  Example: qrsM=QRSdetectorEcg_FbGc(ecg,Fs,0.35,0,1,0);
%  (Tested on MIT DB (1st channel): qrsD='eA2Th', pth=.35, bp(5-20) delay=0.60, d74, escl.207)
%     => Sens=0.9971, PPA=0.9983, F1=0.9977
%
% -------------------------------------------------------------------------------------------------
% Author: Maurizio Varanini, Clinical Physiology Institute, CNR, Pisa, Italy
% For any comment or bug report, please send e-mail to: maurizio.varanini@ifc.cnr.it
% -------------------------------------------------------------------------------------------------
nout = max(nargout,1)-1;
if(nargin <2), error('At least 2 parameters are required'); return; end
if(nargin <3), pth=0.4; end
if(nargin <4), icDet_in=0; end
if(nargin <5), icFp_in=icDet_in(1); end
if(nargin <6), dbFlag=0; end

% convert row to column if necessary
[nsampt,necg]=size(ecg);  if(nsampt<necg), ecg = ecg.'; necg=nsampt; end

if(icDet_in==-1), autoSel=1; icDet=1:necg; 
elseif(icDet_in==0), autoSel=0; icDet=1:necg;
else autoSel=0; icDet=icDet_in;
end
if(icFp_in==0), icFp=icDet(1); else  icFp=icFp_in;  end
if(icFp>0 && sum(icFp==icDet)~=1), fprintf('Error: channel selected for fiducial point should be one of the channel selected for detection\n'); pause; end

qrsDetFunction='QRSdetectorM';

RRts=0.86;  %  RR (seconds) typical of the animal
pmQT=1.1;    %1.0;   %  pmQT=0.7;   % fraction of QT length for QT mask

ecg=ecg(:,icDet);
icFp=find(icDet==icFp);
necg=size(ecg,2);
nsampt=size(ecg,1);

%---------------   Compute FIR bandpass filter coefficients   ------
%fmind = 5; fmaxd = 15;       % frequency band (Hz)
fmind = 5; fmaxd = 20;       % frequency band (Hz)
Wn = [fmind fmaxd]/(Fs/2);   % Nyquist frequency =1.0, (Fsn=2Hz)
delay = floor(0.28*Fs);      % delay=28ms  => filter window  width=56ms
% delay = floor(0.60*Fs);      % delay=60ms  => filter window  width=120ms
ord = delay*2;               % filter order: even => odd number of coefficients
b1 = fir1(ord,Wn);           % default -> Hamming,   length(b1)=ord+1

%--------------------
% ----    raw derivative filter coefficients
% ..ms before and after, ..ms in the middle
nu=ceil(0.0080*Fs); nz=floor(0.0050*Fs/2)*2 +1;  % nz=nearest odd value of (0.005*Fs)
%nu=ceil(0.0070*Fs); nz=floor(0.0040*Fs/2)*2 +1;  % nz=nearest odd value of (0.004*Fs)
b2=[ones(nu,1);zeros(nz,1);-ones(nu,1)];

b=conv(b1,b2)';   % total filter coefficients obtained by convolution
if(dbFlag)
  figure;  plot(b); 
  freqz(b,1,100,Fs);
end
delay=floor(length(b)/2);

% compute the derivative signal
ecgx=[repmat(ecg(1,:),delay,1);ecg;repmat(ecg(end,:),delay,1)];
decgr=filter(b,1,ecgx);   decgr= decgr(length(b):end,:); clear b ecgx;

%-------
adecg=abs(decgr); % absolute ECG derivative

if(necg>1 && autoSel==1)  %
    % -----------    chose the best ecg channel for detection
    wa=fix(8*Fs);             % wide window of 8 s
    we=fix(2*Fs);             % ecg typical window of 2 s
    wh=fix(0.1*Fs);           % short window of 0.2 s
    mDa=zeros(necg,1); mDe=zeros(necg,1); mDh=zeros(necg,1);
    for is=1:necg,
        % compute the average of maximum derivatives on wide windows (4-8s; 2% of maxima are discarted)
        mDa(is)=meanMaxSc(adecg(:,is), wa, 0,2);
        % compute the average of maximum derivatives on ecg typical windows (1-2s, 2% of maxima are discarted)
        mDe(is)=meanMaxSc(adecg(:,is), we, 0,2);
        % compute the average of maximum derivatives on short windows (0.1-0.2s; 5% of maxima are discarted)
        mDh(is)=meanMaxSc(adecg(:,is), wh, 0,5);
    end
%     qualFact=(mDe-mDh)./(mDe+mDh);
%     qualFact=(mDe-10*mDh)./(mDe+10*mDh);  % Sens=0.9563, PPA=0.9966, F1=0.9760
%     qualFact=(mDe-10*mDh-0.5*mDa)./(mDe+10*mDh+0.5*mDa);  % Sens=0.9770, PPA=0.9977, F1=0.9873
    qualFact=(mDe)./(mDh+0.5*mDa);  % Sens=0.9843, PPA=0.9985, F1=0.9913
    [maxQualFact,icDet]=max(qualFact);
    icDet=find(qualFact > 0.9*maxQualFact);
    fprintf('Selected channel(s) for detection: '); fprintf(' %d', icDet); fprintf('\n');
    %  La scelta automatica del canale peggiora la performance!!!  <===
end

if(necg>1 && icFp_in<=0)  %
    % -----------    chose the best ecg channel for signed derivative
    wa=fix(8*Fs);             % wide window of 8 s
    we=fix(2*Fs);             % ecg typical window of 2 s
    wh=fix(0.2*Fs);           % short window of 0.2 s
    mDa=zeros(necg,1); mDe=zeros(necg,1); mDh=zeros(necg,1);
    for is=1:necg,
        % compute the average of maximum derivatives on wide windows (4-8s; 2% of maxima are discarted)
        mDa(is)=meanMaxSc(adecg(:,is), wa, 0,5);
        % compute the average of maximum derivatives on ecg typical windows (1-2s, 2% of maxima are discarted)
        mDe(is)=meanMaxSc(adecg(:,is), we, 0,5);
        % compute the average of maximum derivatives on short windows (0.1-0.2s; 5% of maxima are discarted)
        mDh(is)=meanMaxSc(adecg(:,is), wh, 0,1);
    end
    qualFact=(mDe)./(mDh+0.5*mDa);
    [dummy,icFp]=max(qualFact);
    fprintf('Selected channel for fiducial point= %d\n', icFp);
    %  La scelta automatica del canale peggiora la performance!!!  <===
else
    icFp=1;
end

adecg=adecg(:,icDet);

vdx=decgr(:,icFp);

vadx=sum(adecg,2);

if(dbFlag)
    % plot signal and derivative
    nsecpag=10;     % number of seconds of signal to display
    iip=1; fip=iip + nsecpag*Fs-1;
    xtime=iip:fip; xtime=xtime'/Fs;
    figure;
    subplot(2,1,1), plot(xtime,ecg(iip:fip,icDet),'b');
    subplot(2,1,2), plot(xtime,vadx(iip:fip),'g');
end

if(dbFlag), fprintf('Executing %s detection routine\n', qrsDetFunction); end

%qrsP=eval([qrsDetFunction,'(vadx,vdx,Fs,pth,0.86,1.0)']);
qrsM=eval([qrsDetFunction,'(vadx,vdx,Fs,pth,RRts,pmQT)']);

if(nout==1), varargout{1} = vadx; end
