function [Xd,pRad]=EcgDetrFilt3Hz_im(X,fs,cName,graph,dbFlag,saveFig)
% -------------------------------------------------------------------------------------------
%   ECG detrending
%
%  A baseline signal is estimated applying a low pass Butterworth filter 
%  in forward and backward direction.  The filter cut frequency is 3.17Hz. 
%  Each detrended signal is obtained as difference between the original signal 
%  and the estimated baseline. 
%  In case of residual artifacts due to fast baseline movements, median filtering is used.
%  Median filtering is applied on each interval, not on the whole record.
%
%  function Xd=ecgDetrFilt(X,fs,cName,graph,dbFlag)
%
%  X      : signal matrix (one signal per column)
%  fs     : sampling frequency
%  cName  : record name
%  graf   : flag enabling figure drawing
%  dbFlag : flag enabling figure drawing for tuning and debugging
%
% -------------------------------------------------------------------------------------------
%   Maurizio Varanini, Clinical Physiology Institute, CNR, Pisa, Italy
%   For any comment or bug report, please send e-mail to: maurizio.varanini@ifc.cnr.it
% -------------------------------------------------------------------------------------------
% This program is free software; you can redistribute it and/or modify it under the terms
% of the GNU General Public License as published by the Free Software Foundation; either
% version 2 of the License, or (at your option) any later version.
%
% This program is distributed "as is" and "as available" in the hope that it will be useful,
% but WITHOUT ANY WARRANTY of any kind; without even the implied warranty of MERCHANTABILITY
% or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.
% -------------------------------------------------------------------------------------------

if(nargin<3), cName=''; end
if(nargin<4), graph=0; end
if(nargin<5), dbFlag=0; end
if(nargin<6), saveFig=0; end
graphD= graph &dbFlag ;
graphSpt= graph &dbFlag ;

if(dbFlag)
fprintf('\n --------------------------------------------------------- \n');
progname = mfilename;
fprintf('Program: %s,   record name: %s\n', progname, cName);
end
if(saveFig), figFmt='png';
    figPath=fullfile('../Figure/',progname);
    if(~exist(figPath,'dir')), mkdir(figPath); end
end
%-------------------------------------------------------------
% recording duration
[ndt, ns]=size(X);
vtime= [1:ndt]/fs;
%-------------------------------------------------------------------------------
fmaxd=5;        % --> 3.17 Hz = filtfilt 3dB cut frequency
fmaxn = fmaxd/(fs/2);
[b,a]= butter(1,fmaxn,'low');
Xb=filtfilt(b,a,X);  %  estimated baseline

%  remove baseline from original ecg
Xd=X-Xb;
%-------------------------------------------------------------------------------
RRpmean=1.;
ww=fix(RRpmean*1.5*fs);   % wide windows  (many contain at least one QRS)
for is=1:ns
    xd=Xd(:,is); n=length(xd);
    [xmeaMi,xmeaMa] = meanMiMaSc(xd,ww,8,8);
    thmima=1.5;
    xthmi=xmeaMi*thmima;    xthma=xmeaMa*thmima;
    if(graphD)
        figure, set(gcf,'Color','white');  hold on;
        plot(xd,'b.-');
        plot(xmeaMi*ones(n,1),'m');
        plot(xmeaMa*ones(n,1),'m');
        plot(xthmi*ones(n,1),'r');
        plot(xthma*ones(n,1),'r');
        title(['EcgBaseline_atest:',' xd(b), mi&ma (m), th (r)']);
    end
    
    iibadDetr=find(xd<xthmi | xthma<xd);
%    if(length(iibadDetr)>1.3*fs)
    if(~isempty(iibadDetr)) 
        x=X(:,is);  xb=Xb(:,is);
        wmh=fix(0.26*fs/2); wm=2*wmh+1;    %  window width
        xi=median(x(1:wm));
        xf=median(x(end-wm+1:end));
        xt=[xi+zeros(wmh,1); x; xf+zeros(wmh,1)];
        ib=1;
        isInt(ib)=iibadDetr(1);
        for k=1:length(iibadDetr),         % merge intervals too close
            if(iibadDetr(k)-isInt(ib)> 0.1*fs)
                ieInt(ib)=iibadDetr(k-1);
                ib=ib+1;
                isInt(ib)=iibadDetr(k);
            end
        end
        ieInt(ib)=iibadDetr(k);
        nb=ib;
        medFlag=0;
        for ib=1:nb,                  % select intervals longer than 0.3*fs
            if((ieInt(ib)-isInt(ib))< 0.3*fs ), continue; end
            medFlag=1;
            isMf=max(1,isInt(ib)-round(0.04*fs));
            ieMf=min(ieInt(ib)+round(0.04*fs),ndt);
            isMft=isMf;
            ieMft=ieMf+wmh+wmh;
            fprintf('Median filtering, is=%d, ib=%d\n', is,ib);
            xp = medfilt1mit(xt(isMft:ieMft),wm,1);   % median filtering
            xb(isMf:ieMf) = xp(wmh+1:end-wmh);
        end
        if(~medFlag), continue; end
        x=x-xb;         % remove baseline from original ECG
        %graphD=1;
        if(graphD)
            figure, set(gcf,'Color','white');
            subplot(2,1,1), hold on, plot(vtime,Xb(:,is),'r'), plot(vtime,xb,'b');
            wgmimaV=mimaxscG(Xb(:,is),0,0,.1);
            ylim(wgmimaV);
            % set(gca,'XTickLabel','');
            title('IIR estimated baseline (r) & median filter corrected (b)');
            subplot(2,1,2), hold on, plot(vtime,Xd(:,is),'r'), plot(vtime,x,'b');
            wgmimaV=mimaxscG(Xd(:,is),0,0,.1);
            ylim(wgmimaV);
            title('IIR filtered ECG (r) & median filter corrected (b)');
            shg
        end
        
        [xmeaMi,xmeaMa] = meanMiMaSc(x,ww,8,8);
        thmima=1.3;
        xthmi=xmeaMi*thmima;    xthma=xmeaMa*thmima;
        iinf=find(x<xthmi); x(iinf)=xthmi;
        isup=find(x>xthma); x(isup)=xthma;
        fmaxd=150;    fmaxn=fmaxd/(fs/2);  % Normalized cut-off frequency
        [b,a]= butter(1,fmaxn,'low');
        x=filtfilt(b,a,x);     %  low-pass filtered signal
        if(graphSpt)
            figure; freqz(conv(b,b(end:-1:1)),conv(a,a(end:-1:1)),[0:0.005:2],fs);
        end
        Xb(:,is)=xb;
        Xd(:,is)=x;
    end
%    end
    
%     % iibadDetr(find(diff(iibadDetr)>1*fs));
%     if(length(iibadDetr)>1.3*fs)
%         fprintf('Switching to median filtering, bads=%d\n', length(iibadDetr));
%         wm=fix(0.26*fs); wm=2*floor(wm/2)+1;    %  window width
%         Xb(:,is) = medfilt1mit(X(:,is),wm,1);
%         x=X(:,is)-Xb(:,is);         % remove baseline from original ECG
%         [xmeaMi,xmeaMa] = meanMiMaSc(x,ww,8,8);
%         thmima=1.3;
%         xthmi=xmeaMi*thmima;    xthma=xmeaMa*thmima;
%         iinf=find(x<xthmi); x(iinf)=xthmi;
%         isup=find(x>xthma); x(isup)=xthma;
%         fmaxd=150;    fmaxn=fmaxd/(fs/2);  % Normalized cut-off frequency
%         [b,a]= butter(1,fmaxn,'low');
%         x=filtfilt(b,a,x);     %  low-pass filtered signal
%         Xd(:,is)=x;
%         if(graphSpt)
%             figure; freqz(conv(b,b(end:-1:1)),conv(a,a(end:-1:1)),[0:0.005:2],fs);
%         end
%     end
end

for is=1:ns
    pRad(is)=varsc(Xd(:,is)-X(:,is),1,1)/varsc(X(:,is),1,1);
    if(pRad(is)<=0), pRad(is)=1e-9; end
    if(isnan(pRad(is))), pRad(is)=1; end
    pRad(is)=log(pRad(is));
end

if(graphSpt)
    for is=1:ns,
        figure, set(gcf,'Color','white');
        pwelch(X(:,is),[],[],[],fs);
        title('Original ECG Welch spectrum');
        shg
    end
end
if(graphSpt)
    for is=1:ns,
        figure, set(gcf,'Color','white');
        pwelch(Xd(:,is),[],[],[],fs);
        title('Detrended ECG Welch spectrum');
        shg
    end
end

if(graphD)
    figure, set(gcf,'Color','white');
    for is=1:ns,
        subplot(ns,1,is), hold on, plot(vtime,X(:,is),'r'), plot(vtime,Xb(:,is),'b');
        wgmimaV=mimaxscG(X(:,is),0,0,.1);
        ylim(wgmimaV);
        if(is~=ns), set(gca,'XTickLabel',''); end
        if(is==1), title('Original ECG (r) & estimated baseline(b)'); end
    end
    shg
end
if(graph)
    figure, set(gcf,'Color','white');
    for is=1:ns,
        subplot(ns,1,is), hold on, plot(vtime,X(:,is),'r'), plot(vtime,Xd(:,is),'b');
        wgmimaV=mimaxscG(Xd(:,is),0,0,.4);
        ylim(wgmimaV);
        if(is~=ns), set(gca,'XTickLabel',''); end
        if(is==1), title([cName,': original(r) & detrended(b) ECG']); end
    end
    if(saveFig)
        figName=fullfile(figPath,[cName,'_DetrFilt']);
        print(gcf, ['-d',figFmt],figName);
    end
end
%if(dbFlag & learning) PlotSgnMrkNc(X', QRSa, fs, cName); end

end %== function ================================================================
%

