function [eigenmodel, W, l, ProjectedData] = flda(Data, Labels)

% function [eigenmodel, W, l, ProjectedData] = flda(Data, Labels)
%
% Fisher's Linear Discriminant Analysis
%
% Applies Fisher's criterion to project data into a new sample space where 
% the between-class scatter matrix is maximized and the within-class 
% scatter matrix is minimized.
%
% Input_
%
%   Data  : original data matrix row: observation, column: variable
%   Labels: labels indicating the belonging class of each observation
%
% Ouptut_
%
%   eigenmodel : Structure of the LDA model. The structure contains Sb
%   (between-class scatter matrix), Sw (within-class scatter matrix), Swc
%   (class c specific scatter matrix), Nc (number of instances per class),
%   mu (mean), mc (mean per class), W (projection matrix), l (eigenvalues).
%   W : Optimized projection matrix that maximizes the between-class 
%   scatter matrix and minimizes the within-class scatter matrix.
%   l : associated eigenvalues for each column vector of W.
%   ProjectedData: projected data using transformation matrix W. 
%
% Contact: Salvador Tortajada, vesaltor (at) upv.es

[nsam, nvar] = size(Data);
nclas = max(size(unique(Labels)));

gsize = zeros(nclas, 1);

% Calculate mean per class (mc)
% within scatter matrix per class (Swc)
% and Within Scatter Matrix (Sw)

mc = zeros(nvar, nclas);
Swc = zeros(nvar,nvar,nclas);
Sw = zeros(nvar);
for c=1:nclas
    gsize(c) = size(find(Labels==c),1);
    Swc(:,:,c) = ((gsize(c)-1) * cov(Data(Labels==c,:)));
    Sw = Sw + Swc(:,:,c);
    mc(:,c) = (mean(Data(Labels==c,:)))';
end

% Calculate Total Scatter Matrix

St = (nsam - 1) * cov(Data);

% Calculate Between Scatter Matrix

Sb = St - Sw;

% Use Cholesky's decomposition

C = chol(Sw);
[X,l,W] = svd(C\(C'\Sb));

k = min(nvar, nclas - 1);       % compute maximum number of discriminants

% Using other optimization criteria (see Fukunaga 1990).

%[X,l,W] = svd(Sw\St); 
%[X,l,W] = svd(St\Sb);

% Using other optimization criteria using independent coefficients

%C = chol(Sw);
%[X,l,W] = svd(C'\Sb/C);
%V = C\W;
%S = diag(sqrt(diag(Sw)));
%D = S * V;
%B = sqrt(nsam - nclas) * S\D;
%BB = B(:,1:k);
%BB(end+1,:) = -mean(Data) * BB;
%ProjectedData = [Data ones(nsam,1)] * BB;

ProjectedData = Data * W(:,1:k);

eigenmodel.Sb = Sb;
eigenmodel.Sw = Sw;
eigenmodel.Swc = Swc;
eigenmodel.Nc = gsize;
eigenmodel.mu = (mean(Data))';
eigenmodel.mc = mc;
eigenmodel.W = W;
eigenmodel.l = l;


