%% Funcin que realiza el filtrado ptimo para analizar la onda P
% Lo que realiza es un filtrado de tipo butterworth paso banda con las
% frecuencias de corte que se le pasen y el orden que se le pase.
%
% Parmetros de entrada:
%   * ecg: registro de ecg de entrada a filtrar
%   * fs: frecuencia de muestreo del ecg
%   * fci: frecuencia de corte inferior del filtro paso banda (Hz)
%   * fcs: frecuencia de corte superior del filtro paso banda (Hz)
%   * orden [opcional]: Orden del filtro 
%   * res_plot [opcional]: Si vale 1 se visualiza la respuesta frecuencial 
%                          del filtro diseado. Si vale cero 0 no se 
%                          representa la respuesta.
%
% Parmetros de salida: 
%   * ecg_p: registro de ecg de salida al filtrar con lnea 
%                 isoelctrica sin correguir.
%   * ecg_p_corr: lnea isoelctrica corregida mediante la medianda. De 
%                 este modo se va a poder operar correctamente para
%                 clculos de amplitud de onda P y extraccin de
%                 caractersticas morfolgicas.


function [ecg_p, ecg_p_corr] = filtrado_Paso_Banda(ecg,fs,fci,fcs,orden,res_plot)
%% Configuracin de parmetros opcionales
if nargin < 5
    orden = 2;      % Orden por defecto (2 orden)
    res_plot = 0;   % No se representa la respuesta
elseif nargin < 6
    orden = orden;  % Orden introducido por el usuario
    res_plot = 0;   % No se representa la respuesta       
else 
    orden = orden;  % Orden introducido por el usuario
    res_plot = res_plot; % Mostrar la respuesta del filtro segn el usuario      
end

%% Filtrado Paso-Banda con respuesta Butterworth
Wi = fci/(fs/2);  % Freciencia de corte inferior normalizada
Ws = fcs/(fs/2);  % Freciencia de corte superior normalizada
Wn = [Wi Ws];     % Frecuencias paso banda normalizadas
[B,A] = butter(orden,Wn); 
ecg_p = filtfilt(B, A, ecg);

%% Correcin de la lnea isoelctrica del ecg filtrado (con mediana de seal sin filtrar)
ecg_p_corr = ecg_p - mode(ecg)*ones(1,length(ecg_p))';

%% Representacin de la respuesta del filtro diseado
if res_plot == 1
    figure, freqz(B,A,1e6,fs);   % Visualizar la respuesta en f del filtro
    figure, hold on,             % Visualizar los registros (sin fil, con fil y filt corregido
    plot(ecg), plot(ecg_p,'r'), plot(ecg_p_corr,'g'), plot(zeros(1,length(ecg)),'k')
end
end
