function classifyResult = challenge(recordName,Ann)  %Debug: Ann
%
% Sample entry for the 2017 PhysioNet/CinC Challenge.
%
% INPUTS:
% recordName: string specifying the record name to process
%
% OUTPUTS:
% classifyResult: integer value where
%                     N = normal rhythm
%                     A = AF
%                     O = other rhythm
%                     ~ = noisy recording (poor signal quality)
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% The challenge function requires that you have downloaded the challenge
% data 'training_set' in a subdirectory of the current directory.
%    http://physionet.org/physiobank/database/challenge/2017/
%
% This dataset is used by the generateValidationSet.m script to create
% the annotations on your training set that will be used to verify that
% your entry works properly in the PhysioNet testing environment.
%
%
% Version 1.0
%
%
% Written by: Chengyu Liu and Qiao Li January 20 2017
%             chengyu.liu@emory.edu  qiao.li@emory.edu
%-------------------------------------------------------------------------
% Version: 2017-08-29 (Official Phase - Entry 4)
% Last modified by:
% Ivaylo Christov, e-mail: <Ivaylo.Christov@biomed.bas.bg>
% Vessela Krasteva, e-mail: <vessika@biomed.bas.bg>
% Institute of Biophysics and Biomedical Enngineering, 
% Bulgarian Academy of Sciences 
%
%% -----------------------------------------------------------------------
% Initialization
if nargin<2
    Ann = 'NA';
end
fName = recordName(end-5:end);

classifyResult = 'N'; % default output normal rhythm
P_wave = 0;
RRstd = 5;
RRmean = 0;
QRS_amp_std_norm = 0;
ECG_range = 0;
meanCor = 0;  
perc25Cor = 0;
noise = 0;

%% -----------------------------------------------------------------------
% ECG Reading
[tm,ecg,fs,siginfo]=rdmat(recordName);

Hz=fs;     % sampling rate

% Drift filtration
T=1/Hz;					% [s] sampling period
Fc=0.64;				% [Hz]
X=ecg;
[Y]=driftFilter(X,Fc,T);
ecg=Y;	% Drift filtration

%-----------
% Initial QRS detection

[QRS,sign,en_thres] = qrs_detect2(ecg',0.25,0.6,fs);

Max=max(ecg)-min(ecg);
ECG_range=Max;
while isempty(QRS);
    ecg=ecg*2/Max;
    [QRS,sign,en_thres] = qrs_detect2(ecg',0.25,0.6,fs);
end

%-------------
% Noise Canselation
iQRSvalid=QRS;

% Noise detected as QRS
From_=0; To_=0;

if (length(ecg)/Hz)/length(QRS)>2.1;
    for ii=1:length(QRS)
        From_=QRS(ii)-300;
        if From_<1; From_=1; end
        To_=QRS(ii)+300;
        if To_>length(ecg); To=length(ecg); end
        ecg(From_:To_)=0;
        noise=noise+1;
    end
    [QRS,sign,en_thres] = qrs_detect2(ecg',0.25,0.6,fs);
    
    while isempty(QRS);
    ecg=ecg*2;
    [QRS,sign,en_thres] = qrs_detect2(ecg',0.25,0.6,fs);
    end

    
end %if (length(ecg)/Hz)/length(QRS)>1.8;

% Are number of QRSs in half1 and half2 close 
half=0;
half1 =length(find(QRS/Hz<(length(ecg)/Hz)/2));
half2 =length(find(QRS/Hz>(length(ecg)/Hz)/2));
F=find(QRS/Hz<(length(ecg)/Hz)/2);
QRS1=QRS(F);
F=find(QRS/Hz>(length(ecg)/Hz)/2);
QRS2=QRS(F);
if (half1/half2>2 || half1/half2<0.5)
    half=1; 
    % process 1st half
    ecg1=ecg(1:round(length(ecg)/2));
    [QRS,sign,en_thres] = qrs_detect2(ecg1',0.25,0.6,fs);
    if length(QRS1)<length(QRS); QRS1=QRS; end
    
    if (length(ecg1)/Hz)/length(QRS1)>2.1;
        for ii=1:length(QRS1)
            From_=QRS1(ii)-300;
            if From_<1; From_=1; end
            To_=QRS1(ii)+300;
            if To_>length(ecg1); To_=length(ecg1); end
            ecg1(From_:To_)=0;
        end
        [QRS,sign,en_thres] = qrs_detect2(ecg1',0.25,0.6,fs);
        QRS1=QRS;
    end
      
    % process 2nd half
    ecg2=ecg(round(length(ecg)/2)+1:length(ecg));
    [QRS,sign,en_thres] = qrs_detect2(ecg2',0.25,0.6,fs);
    if length(QRS2)<length(QRS); QRS2=QRS; end
    
    if (length(ecg2)/Hz)/length(QRS2)>2.1;
        for ii=1:length(QRS2)
            From_=QRS2(ii)-300;
            if From_<1; From_=1; end
            To_=QRS2(ii)+300;
            if To_>length(ecg2); To_=length(ecg2); end
            ecg2(From_:To_)=0;
        end
        [QRS,sign,en_thres] = qrs_detect2(ecg2',0.25,0.6,fs);
        QRS2=QRS+round(length(ecg)/2+1);
    else QRS2=QRS+round(length(ecg)/2+1);
    end
    QRS=[QRS1 QRS2];
end

%------------------------
% Validation of QRS
validQRS=[];
iQRSnoise=[];
validQRS(1:length(QRS)) = 1; % Default: QRS is valid
if(From_>0)||(To_>0)    
   iQRSnoise = find(QRS>max(1,From_-5) & QRS<max(1,To_-5));
   if(~isempty(iQRSnoise))
      validQRS(iQRSnoise) = 0;      
   end
end

%---------------------------
% RR intervals
%RRtotal = (QRS(2:end)-QRS(1:end-1))/Hz;
iR = 1;
RR = []; RR = 0;
for ii=2:1:length(QRS)
   if((validQRS(ii)==1)&&(validQRS(ii-1)==1))
       RRlast = (QRS(ii)-QRS(ii-1))/Hz;
       if(RRlast<2)
          RR(iR) = RRlast;
          iR = iR+1; 
       end
   end
end
RRmean = mean(RR);
if RRmean==0; RRmean=0.80; end;         % no ECG
RRmean = (round(RRmean*1000))/1000;
HR=60/RRmean;

RRstd = std(RR);
RRstd = (round(RRstd*1000))/1000;

featHRV = funcHRV(RR*Hz,Hz);

%---------
iQRSvalid = find(validQRS==1);
QRS = QRS(iQRSvalid);

% Amplitude normalization
Max=max(ecg); Min=min(ecg);
ecg_norm=ecg/(Max-Min);
QRS_amp_std_norm=std(ecg_norm(QRS));
QRS_amp_std_norm = (round(QRS_amp_std_norm*1000))/1000; 
QRSampstd_abs = ECG_range*QRS_amp_std_norm;
%---------------------------
% Fiducial point = at the max peak
left=140*Hz/1000;    %140 ms
right=140*Hz/1000;   %140 ms
j=1;
if QRS(j)<=left; j=2; end   % the record  starts with QRS ?

% Is QRS positive or negative ?
pos=length(find(ecg(QRS)>=0)); neg=length(find(ecg(QRS)<0));
clear FP;

if pos>neg;
    for j=j:length(QRS)-1;
        [M,m]=max(ecg(QRS(j)-left:QRS(j)+right));
        FP(j)=QRS(j)-left+m-1;
    end
else 
     for j=j:length(QRS)-1; 
        [M,m]=min(ecg(QRS(j)-left:QRS(j)+right));
        FP(j)=QRS(j)-left+m-1;
    end
end
if FP(1)==0; FP=FP(2:length(FP)); end

for j=1:length(FP)-2    %exclude doublets
    if FP(j)==FP(j+1);
        FP=[FP(1:j) FP(j+2:length(FP))];
    end
end
    

% Exclude extrasystole & noise from Average Pattern (amplitude <> mean+/-std
FPold=FP;
FPamp=ecg(FP);
FPamp_mean=mean(FPamp);
FPamp_std=std(FPamp);
up=FPamp_mean+FPamp_std;
down=FPamp_mean-FPamp_std;

FP_=find(FPamp<up & FPamp>down);
if(length(FPamp)==0)
   QRSrejRat=0;
else
    QRSrejRat=100*(length(FPamp)-length(FP_))/length(FPamp); %(%)
end

FP=FP(FP_);
%---------------------------
% Average Pattern
Left=300*Hz/1000;    %300 ms left
Right=600*Hz/1000;
if RRmean>1; Right=round((600*Hz/1000)*RRmean); end

    
if FP(1)==FP(2); FP=FP(2:length(FP)); end
On=FP-Left;
Off=FP+Right;
k=1;
if On(1)<=0; k=k+1; end
if On(2)<=0; k=k+1; end
if Off(length(Off))>length(ecg); l=2; else l=1; end
ecg_mean=ecg(On(k):Off(k)); k=k+1;
nmb=1;
for i1=k:length(FP)-l;
    ecg_mean=ecg_mean + ecg(FP(i1)-Left : FP(i1)+Right);
    nmb=nmb+1;
end
ecg_mean=ecg_mean/nmb;
[Max, QRSp]=max(abs(ecg_mean));

%---------------------------
% The QRSpeak is the fidutial point (FP)
QRSp=Left+1;
ecg_mean';

% Reduce length of Lead in dependance of RR
ecg_mean=ecg_mean(1:length(ecg_mean)-round(10/RRmean));
Lead=ecg_mean';

% Find the izoelectric point on Average ecg
Flat=20*Hz/1000;        % 20 ms
From=QRSp; To=From-80*Hz/1000;
peak_y=abs(ecg_mean(QRSp));


if To<10; To=10; end
Crit=0.02*peak_y;	% make Crit proportional to the QRSampl;
[Pnt,Crit1]=Flat_l(Lead,Crit,From,To,Flat);  
Iz=Pnt;

fragm=0;
% Second turn needed?
Max=max(Lead(QRSp-100*Hz/1000:QRSp+100*Hz/1000));
Min=min(Lead(QRSp-100*Hz/1000:QRSp+100*Hz/1000));
if abs(Lead(QRSp)-Lead(Iz))<(Max-Min)*30/100; fragm=1; 
    From=Iz-round(20*Hz/1000); To=From-round(80*Hz/1000);
    if To<10; To=10; end
    [Pnt,Crit]=Flat_l(Lead,Crit,From,To,Flat); 
    Iz=Pnt;
end
   
 
%---------------------------
%   Find the QRS-offset (J point)
From=QRSp;
To=From+100*Hz/1000;     % to 100 ms right ...
if RRmean<0.45; To=From+50*Hz/1000; end % to 80 ms right
[Pnt,Crit]=Flat_r(Lead,Crit,From,To,Flat);  
J=Pnt;

% Second pass - move J to the right
if J-QRSp<60*Hz/1000 && abs(Lead(J))>(max(Lead)-min(Lead))*25/100 && RRmean>0.6
    From=round(J+40*Hz/1000);
    To=round(From+70*Hz/1000);
    [Pnt,Crit]=Flat_r(Lead,Crit,From,To,Flat);  
    J=Pnt;
else 
    % Third pass - move J to the left
    if (J-QRSp)*1000/Hz>100;
        From=QRSp;
        To=From+50;
        if Lead(QRSp)>=0
            a=find(Lead(From:To)<=Lead(Iz));
        else
            a=find(Lead(From:To)>Lead(Iz));
        end
        J=QRSp+a(1);
    end
end
QRSwid = 1000*(J-Iz)/Hz; %ms

%---------------------------
% QRS Correlation
left_=QRSp-Iz;
right_=J-QRSp;
mean_QRS=Lead(QRSp-left_:QRSp+right_);
maxCor = [];
for ii=2:length(QRS)-1;
    ecg_QRS=ecg(QRS(ii)-left_:QRS(ii)+right_);
    corfunc=xcorr(mean_QRS, ecg_QRS,'coeff');
    maxCor(ii-1) = max(corfunc);
end


iNotNan = find(~isnan(maxCor)); 
if(isempty(iNotNan))
   meanCor = 0;
   medianCor = 0;
   perc25Cor = 0;
else
  meanCor = mean(maxCor(iNotNan));  
  medianCor = median(maxCor(iNotNan));
  perc25Cor = prctile(maxCor(iNotNan),25);
end

if RRmean<0.6; 
    left=200*Hz/1000;
    corr=Left-left;
    Lead=Lead(QRSp-left:QRSp+400*Hz/1000); %length(Lead);
    Iz=Iz-corr;
    J=J-corr;
    QRSp=QRSp-corr;
end


% Iz==0
Lead=Lead-Lead(Iz);
QRSamp=Lead(QRSp)-Lead(Iz);

%---------------------------
% Tend detection
% Searching for big peaks by 'wings' function
clear Wing;
s=30*Hz/1000;   % 40 ms    

Wing(J+s:length(Lead)-s)=...
    (Lead(J:length(Lead)-2*s)-Lead(J+s:length(Lead)-s)).*...
    (Lead(J+s:length(Lead)-s)-Lead(J+2*s:length(Lead)));
Wing=Wing(J+s:length(Wing));

if RRmean>0.8; To=300;
elseif RRmean<0.6; To=200;
else To=250;
end

[Mi2,mi2]=max(Wing(50:min([To*Hz/1000 length(Wing)])));
Tp2=J+mi2+s-1+50;

Lead=Lead-Lead(J);

[Mi1,mi1]=min(Wing(50:min([To*Hz/1000 length(Wing)])));
Tp1=J+mi1+s-1+50;

Tp=min([Tp1 Tp2]);

[Ma,ma]=max(abs(Lead(J+100*Hz/1000:length(Lead)-150*Hz/1000)));

Tp=J+100*Hz/1000+ma-1;

if Lead(Tp)>=0
    [Mi,mi]= min(Lead(Tp:length(Lead)-100*Hz/1000));
else
    [Mi,mi]= max(Lead(Tp:length(Lead)-100*Hz/1000));
end
T_end=Tp+mi-1;


 % Search fot inflection point
c=length(Lead)-round(50*Hz/1000);
Tfrom=Tp+round(80*Hz/1000);

a=Lead(Tfrom+6:c-6)-Lead(Tfrom+3:c-9);
b=Lead(Tfrom+6:c-6)-Lead(Tfrom+9:c-3);
infl=a.*b;
inf=find(infl>=0);

if isempty(inf)
    T_end=T_end; 
else
    infl=inf(1);
    T_end=Tfrom+infl+6;
end

%---------
 % Find P wave on Average ecg
Lead=Lead-Lead(Iz); % shift ECG to put Iz at zero

% Positive P-wave
[Max m1]=max(Lead(Iz:Iz+100*Hz/1000));
[Min m2]=min(Lead(Iz:Iz+100*Hz/1000)); Min=abs(Min);
beg=max([Iz-70, 10]);
prag2=0.0005;     % 0.5 uV threshold

if     Max>=2*Min; P_positive=1; 
elseif Min>=2*Max; P_positive=0; 
elseif m1>m2; P_positive=0;
else   P_positive=1;
end

if P_positive==1   %P-wave sign is the same as QRS sign
    [Max ,t1]=max(Lead(beg:Iz));
    P_y=Max; 
    P_x=t1+beg-1;
    if (P_x>50*Hz/1000) && (Iz-P_x>=30*Hz/1000)
        if (Lead(P_x)-Lead(P_x-10*Hz/1000)>prag2) && (Lead(P_x)-Lead(P_x+10*Hz/1000)>prag2) ...
                && (Lead(P_x)-Lead(P_x-20*Hz/1000)>4*prag2) && (Lead(P_x)-Lead(P_x+20*Hz/1000)>4*prag2)
            P_wave=1;
        end
    end
else
    [Min ,t1]=min(Lead(beg:Iz)); 
    P_y=Min; P_x=t1+beg-1;
    if (P_x>50*Hz/1000) && (Iz-P_x>=30*Hz/1000)
        if (Lead(P_x-10*Hz/1000)-Lead(P_x)>prag2) && (Lead(P_x+10*Hz/1000)-Lead(P_x)>prag2) ...
                && (Lead(P_x-20*Hz/1000)-Lead(P_x)>4*prag2) && (Lead(P_x+20*Hz/1000)-Lead(P_x)>4*prag2)
            P_wave=1;
        end
    end
end

Ppeak=P_x;

% Amplitude check of Ppeak
prag1=abs(Lead(QRSp)/16);
if Ppeak/Hz<0.1; prag1=prag1*1.3; end   %far from QRS?
if abs(Lead(Ppeak)-Lead(Iz))<prag1 %(mV)
   P_wave=0; 
end

if P_wave==1; 
    Pbeg=max([Ppeak-80*Hz/1000 1]);
    % Amplitude check of Pbeg
    Pamp=abs(Lead(Ppeak)-Lead(Pbeg));
    if Pamp<prag1*0.7
        P_wave=0;
    end
end

PQ = 0;
if(P_wave==1)
    PQ = 1000*(Iz-P_x)/Hz;
end

Lead=Lead-Lead(Iz);
QT=1000*(T_end-Iz)/Hz;
Tamp=Lead(Tp);
Pamp=Lead(P_x);

%[QT QRSamp Pamp Tamp]

%---------
% Do not consider ectopic beats & noise

Dev=std(ecg(FP));
while Dev>0.25;
    FPmean=mean(ecg(FP));
    [Max_min, m]=max(abs(ecg(FP)-FPmean));
    FP=[FP(1:m-1) FP(m+1:length(FP))];
    noise=noise+1;
    Dev=std(ecg(FP));
end
%---------
% Signal from extraction of TP-interval
% High HR and AFF can not be calculated
if HR<120; 
   
    % remove P-QRS-T interval
    left=QRSp-P_x; right=T_end-QRSp;
    ecg_null=ecg;
    j=1;
    k=1; if FP(k)-left<=0; k=2; end
    FPlast=length(FP); if FP(FPlast)+right>length(ecg); FPlast=FPlast-1; end
    for k=k:FPlast-1
        ecg_null(FP(k)-left:FP(k)+right)=0;
        T(j)=FP(k)+right; P(j)=FP(k+1)-left; j=j+1;
    end
   
    a=size(ecg);
    if a(2)==1; ecg=ecg'; end
    TP=ecg(T(1):P(1)); 
    for j=2:length(P)
        TP=[TP, ecg(T(j):P(j))];
    end
    
    % Differentiation in 20 ms
    Diff(1:length(TP)-5) = TP(6:length(TP)) - TP(1:length(TP)-5);
    Diff = abs(Diff); %Absolute value
    
    if  isempty(Diff)==1;
        Diff(1:1000)=0.02; % 0.02 by default
        ecg_null(1:length(ecg))=0;
    else
        % Cut big differences
        for j=1:length(Diff)
            if Diff(j)>0.04; 
                Diff(j)=0; %0; 
            end
        end
        
        % Filtering
        X=Diff; n=11; clear Y; 
        [Y]=smoothFilter(X,n);
        Diff=Y;
    end
    

else
    Diff(1:1000)=0.02; % 0.02 by default
    ecg_null(1:length(ecg))=0;
end

AFF=median(Diff); % Median level of signal 
%AFF=(round(AFF*1000))/1000;   

% Right Bubdle Branch Block
RBBB=0; 
[Ma,ma]=max(Lead(Iz:J));
[Mi,mi]=min(Lead(Iz:J));
if QRSwid>140   
    %[Lead(Tp),Ma,ma,Mi,mi]
    if Lead(Tp)>0 && mi>ma && abs(Mi)>Ma; RBBB=1;
    elseif Lead(Tp)<0 && mi<ma && abs(Mi)<Ma; RBBB=1;
    end
end

% Elevation of J
QRSampl=max(Lead(Iz:J))+abs(min(Lead(Iz:J)));
J_elev=abs(Lead(J))/QRSampl;
J_elev_abs=abs(Lead(J));

% Sign of QRS & T
dif_sign=0;
if (Lead(QRSp)>=0 && Lead(Tp)<=0);
    dif_sign=1;
end
if (Lead(QRSp)<=0 && Lead(Tp)>=0);
    dif_sign=1;
end
%-------------
% Curvature
Step = 5;
ampNorm = range(Lead(Iz:J));
[kQRS]=curvature(Lead(Iz:J+Step),Step,Hz,ampNorm); % (Lead(Iz:J),Hz);
[kT]=curvature(Lead(J:T_end),Step,Hz,ampNorm); % (Lead(Iz:J),Hz);
[kP]=curvature(Lead(1:Iz-Step),Step,Hz,ampNorm); % (Lead(Iz:J),Hz);
 

%------------------------------------------------------
% Classifier
DATA =   [featHRV.HRmean,... %1
            featHRV.RRmean,... %2
            featHRV.RRstd, ... %3
            featHRV.RRstdP, ...%4
            featHRV.RRmeand, ...%5
            featHRV.RRmeandP, ...%6
            featHRV.RRmedian, ...%7
            featHRV.RRratioMeanVsMedian,...%8
            featHRV.dRRmean, ...%9
            featHRV.dRRmeanP, ...%10
            featHRV.dRRstd, ...%11
            featHRV.dRRstdP, ...%12
            featHRV.dRRmeand, ...%13
            featHRV.dRRmeandP,...%14
            featHRV.PNN50, ...%15
            featHRV.PNN50P, ...%16
            featHRV.RMSSD, ...%17
            featHRV.RMSSDP, ...%18
            featHRV.HRVTriangularIndex128, ...%19
            featHRV.ratioSD1SD2, ...%20
            featHRV.CCM, ... %21
            featHRV.corRR,... %22
            meanCor, ...%23
            medianCor, ...%24
            perc25Cor, ...%25
            ECG_range, ...%26
            QRS_amp_std_norm,...%27
            QRSampstd_abs,...%28
            abs(QRSamp), ...%29
            P_wave, ...%30
            abs(Pamp),...%31
            abs(Tamp),...%32
            QRSwid, ...%33
            PQ, ...%34
            QT, ...%35
            noise,...%36
            AFF*10000, ... %37
            QRSrejRat,... %38  
            1000*J_elev, ... %39
            1000*J_elev_abs, ... %40
            fragm, ... %41 
            dif_sign ... %42
            half ... %43 %Different QRS detection in half1 and half2
            max(kQRS) ... %44
            max(kQRS)/max(kP) ...  %45
            max(kQRS)/max(kT) ... %46
            max(kT)/max(kP) ... %47
            RBBB]; %48
            %ent1 ... %48
            %ent2];   %49

[classifyResult,classifyNumber] = classifyDA(DATA); 
%classifyNumber = 1;

%------------------------------------------------------
% DEBUG:
%if(~strcmp(Ann,classifyResult))&&(strcmp(Ann,'O')) % Errors only

%if half~0
    %Ann== 'O'
%if(strcmp(recordName(end-5:end),'A00011'))
%if(RBBB==1)
%   DEBUG_VisualSave; %!!!! Comment in the Official Entry
%end
%------------------------------------------------------
A = 1;

end %function classifyResult = challenge(recordName)

%% *************************************************************************
function[Pnt,Crit1]=Flat_l(Lead,Crit,From,To,Flat)
%% 	Flat_l()
%	Searching a flat segment to the left
%	The amplitude criteria (uV - fixed) is addaptive
%	If a flat is not found in the interval -> Crit=Crit+1uV: start again
% Input:
%    Lead - a lead to work on
%    From - from this point
%    To - to this point
%    Crit - amplitude criteria
%    Flat - No of points where Up or Down criteria must be fullfilled
% Output:
%    Pnt - beg point where flat segment found
%     = 0 - no flat segment	
%--------------------------------
Crit1=abs(Crit);
Pnt=0; a=0; 
while Pnt==0;
    if a<2; a=a+1;			%increase a
    else  Crit1=Crit1+0.003; %increase Crit with 3 uV
        if Crit1>0.500; Pnt=floor((From-To)/2); end % Flat_l not found
    end
    
    for i=From:-1:To;
         d(1:Flat)=Lead(i:-1:i-Flat+1)-Lead(i-1:-1:i-Flat);
         if max(abs(d)) < Crit1 &&...
                 abs(Lead(i)-Lead(i-Flat-1)) < a*Crit1 &&...
                 abs(Lead(i)-Lead(i-floor(Flat/2))) < 1.4*a*Crit1 &&...	%1.4
                 abs(Lead(i-Flat)-Lead(i-floor(Flat/2))) < 1.4*a*Crit1;	%1.4
             Pnt=i; break
         end
    end
end
end % function Flat_l()

%% *************************************************************************
function[Pnt,Crit1]=Flat_r(Lead,Crit,From,To,Flat)
%% 	Flat_r()    
%	Searching a flat segment to the right
%	The amplitude criteria (uV - fixed) is addaptive
%	If a flat is not found in the interval -> Crit=Crit+1uV: start again
% Input:
%    Lead - a lead to work on
%    From - from this point
%    To - to this point
%    Crit - amplitude criteria
%    Flat - No of points where Up or Down criteria must be fullfilled
% Output:
%    Pnt - beg point where flat segment found
%     = 0 - no flat segment	
%--------------------------------
Crit1=abs(Crit);
Pnt=0; a=0; 
while Pnt==0;
    if a<2; a=a+1;			%increase a
    else Crit1=Crit1+0.003;	%increase Crit with 3 uV	
        if Crit1>0.500; Pnt=floor((To-From)/2); end % Flat_r not found
    end
           
	for i=From:To;
   	d(1:Flat)=Lead(i:i+Flat-1)-Lead(i+1:i+Flat);
      if max(abs(d)) < Crit1 &&...
            abs(Lead(i)-Lead(i+Flat)) < a*Crit1 &&...
            abs(Lead(i)-Lead(i+floor(Flat/2))) < 1.4*a*Crit1 &&...
            abs(Lead(i+Flat)-Lead(i+floor(Flat/2))) < 1.4*a*Crit1;...
            Pnt=i; break
      end
    end
        
end
end % function Flat_r()
%% *************************************************************************
function [Y]=driftFilter(X,Fc,T)
%	---- High-pass filtering ----
% Yn=C1(Xn-X(n-1))+C2Y(n-1)
% C1 = 1/[1+tan(Fc*pi*T)]
% C2 = [1-tan(Fc*pi*T)]/[1+tan(Fc*pi*T)]
% Inputs:	X - signal;
%				Fc - cut-off frequency [Hz]
%				T -  sampling period [s]
% Output:	Y - filtered signal

% make X as row
s=size(X);
if s(1)==1;
else X=X'; 
end
    
% Shift input signal to the first point
X=X-X(1);

% Filter coefficients
c1 = 1/(1+tan(Fc*pi*T));
c2 = (1-tan(Fc*pi*T))/(1+tan(Fc*pi*T));

b=[c1 -c1];	a=[1 -c2];
Y1=filter(b,a,X);
  Yf=fliplr(Y1);
  Yf=Yf-Yf(1);
Y=filter(b,a,Yf);
Y=fliplr(Y);
end % function driftFilter()

%% *************************************************************************
function[Y]=smoothFilter(X,n)
%	Smoothing by least-squares cubic approximation procedure (Savitsky and Golay)

% Inputs:	X - input signal
%			n - smoothing interval on each side

% Output	Y - smoothed signal

siz=size(X);		% if the input is vertical matrix make it horizontal
if siz(2)==1; 
    X=rot90(X);
end
   
i=-n:n;				% coeficient calculation
coef=((3.*n.*n+3.*n-1)-5.*i.*i);
coef=rot90(coef);
a=1;b=coef;
Y=filter(b,a,X);
Y=Y*3/((2*n+1)*(4*n*n+4*n-3));
Y(1:n)=Y(n+1);
Y(length(X)-n+1:length(X))=Y(length(X)-n);

end % function smoothFilter(X,n)
%% *************************************************************************
function [featHRV] = funcHRV(RR,Fs)
NbRR =  length(RR);
if(NbRR<=5)
    %--------------
    % RR-features
    featHRV.NbRR = NbRR;
    featHRV.HRmean = 0;
    featHRV.RRmean = 0;
    featHRV.RRstd = 0;
    featHRV.RRstdP = 0;
    featHRV.RRmeand = 0;
    featHRV.RRmeandP = 0;
    featHRV.RRmedian = 0;
    featHRV.RRratioMeanVsMedian = 0;
    % dRR-features
    featHRV.dRRmean = 0;
    featHRV.dRRmeanP = 0;
    featHRV.dRRstd = 0;
    featHRV.dRRstdP = 0;
    featHRV.dRRmeand = 0;
    featHRV.dRRmeandP = 0;
    featHRV.PNN50 = 0;
    featHRV.PNN50P = 0;
    featHRV.RMSSD = 0;
    featHRV.RMSSDP = 0;
    featHRV.HRVTriangularIndex128 = 0;
    featHRV.ratioSD1SD2 = 0;
    featHRV.CCM = 0;       
    featHRV.corRR = 0;
    %--------------
else
    %--------------
    % RR-TACHOGRAM
    RR = 1000*(RR/Fs);                     %[ms]
    RRmean = mean(RR);                     %[ms]
    RRstd  = std(RR);                      %[ms]
    RRstdP = round(100*RRstd/RRmean);      %[%]
    RRmeand = mad(RR);                     %[ms]
    RRmeandP = round(100*RRmeand/RRmean);  %[%]
    RRmedian  = median(RR(1));             %[ms]
    RRratioMeanVsMedian = RRmean/RRmedian; %[NA]

    HRmean = 60/(RRmean/1000);             %[BPM]
    %--------------
    %dRR-TACHOGRAM
    dRR = RR(2:end)-RR(1:end-1);           %[ms]
    dRRmean = mean(abs(dRR));              %[ms]
    dRRmeanP = round(100*dRRmean/RRmean);  %[%]
    dRRstd = std(abs(dRR));                %[ms]
    dRRstdP = round(100*dRRstd/RRmean);    %[%]
    dRRmeand = mad(abs(dRR));              %[ms]
    dRRmeandP = round(100*dRRmeand/RRmean);%[%]
    NN50 = length(find(abs(dRR)>50));
    PNN50P = (100*NN50/length(dRR));        %[%] round
    PNN50 = 1000*60*NN50/sum(RR); %[/min]  NN50PerMinute
    RMSSD = sqrt(mean(dRR.^2)); %rms(abs(dRR)); %[ms]
    RMSSDP = round(100*RMSSD/RRmean);      %[%]
    
    % Poincare Plot SD1 and SD2
    SD1 = sqrt((dRRstd^2)/2); 
    SD2 = sqrt(2*(RRstd^2) - (dRRstd^2)/2);
    ratioSD1SD2 = SD1/SD2; 
    RR1 = RR(1:end-1);
    RR2 = RR(2:end);
    CCM = 0;
    for i=3:1:length(RR1)
       CCM = CCM + polyarea([RR1(i-2:i) RR1(i-2)],[RR2(i-2:i) RR2(i-2)]);       
    end
    CCM = CCM/(pi*SD1*SD2);       
    corR = corrcoef(RR1,RR2); corRR = corR(1,2);
%     corR = corrcoef(RR1(1:length(RR3)),RR3); corR3 = corR(1,2)
%     corR = corrcoef(RR1(1:length(RR4)),RR4); corR4 = corR(1,2)
%     corR = corrcoef(RR1(1:length(RR5)),RR5); corR5 = corR(1,2)
%     sumcorR = (corR2+corR3+corR4+corR5)/4;
    %[R(1,2) RLO(1,2) RUP(1,2) P(1,2)] 
    %[R,P,RLO,RUP] = corrcoef(RR(1:end-2),RR(3:end))
    %[R,P,RLO,RUP] = corrcoef(RR(1:end-3),RR(4:end))
    
    
    %--------------
    %RR-HISTOGRAM
    StepBin = 1000/128;                    % Sampling of 128 Hz: Usually Used. Sampling Interval 7.8125 ms
    BinEdges = StepBin*(-64:1:63) + RRmean-StepBin/2;  
    RRHist = histc(RR,BinEdges);
    HRVTriangularIndex128 = NbRR/max(RRHist);   %HRV Triangular Index = totalNbRR/NbRR in the modal bin
    %From all variety of time domain and geometric methods available, the Task
    %Force of the ESC and the NASPE has recommended the use of four measures
    %for HRV assessment: SDNN, SDANN, RMSSD and the HRV triangular index.
    BinEdges = BinEdges+StepBin/2; %Correct the center to go to 0
    BinEdgesP = 100*(BinEdges/RRmean)-100;
    RRHistP = 100*RRHist/NbRR; 
    %--------------
    % RR-features
    featHRV.NbRR = NbRR;
    featHRV.HRmean = HRmean;
    featHRV.RRmean = RRmean;
    featHRV.RRstd = RRstd;
    featHRV.RRstdP = RRstdP;
    featHRV.RRmeand = RRmeand;
    featHRV.RRmeandP = RRmeandP;
    featHRV.RRmedian = RRmedian;
    featHRV.RRratioMeanVsMedian = RRratioMeanVsMedian;
    % dRR-features
    featHRV.dRRmean = dRRmean;
    featHRV.dRRmeanP = dRRmeanP;
    featHRV.dRRstd = dRRstd;
    featHRV.dRRstdP = dRRstdP;
    featHRV.dRRmeand = dRRmeand;
    featHRV.dRRmeandP = dRRmeandP;
    featHRV.PNN50 = PNN50;
    featHRV.PNN50P = PNN50P;
    featHRV.RMSSD = RMSSD;
    featHRV.RMSSDP = RMSSDP;
    featHRV.HRVTriangularIndex128 = HRVTriangularIndex128;
    featHRV.ratioSD1SD2 = ratioSD1SD2;
    featHRV.CCM = CCM;       
    featHRV.corRR = corRR;
    %--------------    
    %--------------
    % Plot of HRV
    %plotHRV
    %A = 1;
end

end  %funcHRV()
%% *************************************************************************
function [classifyResult,classifyNumber]=classifyDA(FeatMeas)

%% 	classifyDA() implements Linear Discriminant Analysis
%--------------------------------
% version 2017-08-29 (ENTRY4 Official-phase) 
%---------------------------------
DAType = 'linear';
Priorprob.group = [1 2 3 4];
%---------------------------------
fileDATA = 'DATA_TRAIN.mat';
load(fileDATA); %FeatAll, ClassValues 
[NbCases,NbFeatures] = size(FeatAll); 
HR = FeatAll(:,1);
HRmeas = FeatMeas(1,1);

if(HRmeas<=50)
     iIncludedCases = find(HR<=50);    
     Priorprob.prob = [0.25	0.05 0.3 0.4];
     FeatNumbers = [2  18  34  40  44  47  46  37  36  45  11  17  10  27  30  35  39  43  20  19  41  24  48  13  29  42   1  15  33  25  21   9  28]; %33 features
elseif(HRmeas>=100)
     iIncludedCases = find(HR>=100); 
     Priorprob.prob = [0.1 0.3 0.55 0.05];      
     FeatNumbers = [15  25  33  32  35  29   7   9   6  37   2  17  19   5  20  18  21  22   4   3  10  16  23  40  11  24  36  38  34   1   8  30  27]; % 33 features
else %HR=(50 100)
    iIncludedCases = find(HR>50 & HR<100);    
    Priorprob.prob = [0.6 0.05 0.3 0.05];
    FeatNumbers = [15  20  30  10  23  22  39  29  45  37  33  21  34  40  43  31  14  38  18  48  41   8  35  26  27  24  42  46   6  19  44   4  32  28  47  25  36   7   5   3   1  11  17   9  12]; %45 features
end
iTrain = 1:1:NbCases; % All cases for training
iTrain = intersect(iTrain,iIncludedCases);

FeatValues = [FeatAll(:,FeatNumbers); FeatMeas(FeatNumbers)];

[DA.class] = classify(FeatValues,FeatValues(iTrain,:),ClassValues(iTrain),DAType,Priorprob);
classifyNumber = DA.class(end);

if(classifyNumber(1)==1)
    classifyResult = 'N';
elseif(classifyNumber(1)==2)
    classifyResult = 'A';    
elseif(classifyNumber(1)==3)
    classifyResult = 'O';    
elseif(classifyNumber(1)==4)   
    classifyResult = '~';
end
    
end % function classifyDA()
%% *************************************************************************
function [k]=curvature(s,Step,Fs,ampNorm)
% Curvature evaluation during QRS
%A*z = 0, with z := [a,b,c,d]' in the (algebra) least-quares sense.
%Use svd of A to find z.

s=s';
t = (1:1:length(s))/Fs;

for i=1:length(s)-Step
    x=s(i:i+5); 
    y=t(i:i+5);
    x=x-mean(x);
    y=y-mean(y);
    
    A = [x.^2+y.^2,x,y,ones(size(x))]; % Set up least squares problem
    [U,S,V] = svd(A,0); % Use economy version sing. value decompos.
    a = V(1,4); b = V(2,4); % Choose eigenvector from V
    c = V(3,4); d = V(4,4); % with smallest eigenvalue
    xc = -b/(2*a); yc = -c/(2*a); % Find center and radius of the
    r = sqrt(xc^2+yc^2-d/a); % circle, a*(x^2+y^2)+b*x+c*y+d=0
    
    k(i)=1/r;    
end
k=100*k*10/ampNorm; %(%)
k=k-min(k);

% figure(500); 
% cla; hold on; grid on;
% plot(s);
% plot(k/100,'r');


end %function  [k]=curvature(x,Fs)
%% *************************************************************************
