/*
 * trig.c
 *
 *  Created on: Mar 31, 2015
 *      Author: hvaanane
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>
#include <limits.h>
#include <math.h>
#include <float.h>

#include "util.h"
#include "tests.h"
#include "types.h"
#include "matrix.h"
#include "test_types.h"
#include "trig.h"
#include "rhythm.h"
#include "conf.h"
#include "data.h"
#include "array.h"

#include "memory_manager.h"

#define TRIG_CHECK_CLUSTER_OFFSETS

struct trig_helper{
	int nch;
	double *threshold_values;
	size_t step_len;
	size_t n_steps;
	size_t n_events;
	float **ch_badness_for_events;
	char **badness_type_for_events;
};


int debug_break(void){
	int i=5*5;
	return(i);
}

int trig_free_helper(struct trig_helper *helper){
	if(helper->threshold_values){
		my_free(helper->threshold_values);
	}
	if(helper->ch_badness_for_events){
		matrix_free_f(helper->ch_badness_for_events);
	}
	if(helper->badness_type_for_events){
		matrix_free_c(helper->badness_type_for_events);
	}
	return(0);
}



int trig_ch_range_comparison(const void *event1,const void *event2){
	if(((struct test_range *)(event1))->start < ((struct test_range *)(event2))->start){
		return(-1);
	}
	else if(((struct test_range *)(event1))->start > ((struct test_range *)(event2))->start){
		return(1);
	}
	return(0);
}

int trig_check_ch_ranges(struct test_ranges *ch_ranges, struct data *data){
	int ch;
	size_t i;
	for(ch=0;ch<data->number_of_channels;ch++){
		for(i=0;i<ch_ranges[ch].n_ranges;i++){
			if(ch_ranges[ch].ranges[i].start>=ch_ranges[ch].ranges[i].end || ch_ranges[ch].ranges[i].end>data->samples_per_channel){  // anomaly?
				return(1.0);
			}
		}
	}
	return(0);
}

int trig_refine_ch_ranges(struct test_ranges *ch_ranges, struct data *data){
	int ch;
	size_t i,n;
	size_t max_cap;
	size_t last_anomaly_len;
	size_t total_anomaly;
//	if(trig_check_ch_ranges(ch_ranges, data)){
//		printf("*************Oops************\n");
//	}

	for(ch=0;ch<data->number_of_channels;ch++){
	  // sort
		if(ch_ranges[ch].n_ranges>1){
			qsort(ch_ranges[ch].ranges,ch_ranges[ch].n_ranges,sizeof(struct test_range),trig_ch_range_comparison);

			// combine
			total_anomaly=0.0;
			for(n=0,i=1;i<ch_ranges[ch].n_ranges;i++){
				last_anomaly_len=ch_ranges[ch].ranges[n].end-ch_ranges[ch].ranges[n].start;
				switch(ch_ranges[ch].ranges[n].type){
				case TEST_SHORT_OUT_OF_RANGE:
					max_cap=0.1*data->sample_freq;
					break;
				case TEST_LOW_SNR:
				case TEST_SIGNAL_OUT_OF_RANGE:
					max_cap=MIN(0.5*data->sample_freq,last_anomaly_len);
					break;
				case TEST_COMBINED_ANOMALY:
				case TEST_NO_SIGNAL:
					if(last_anomaly_len>5.0*data->sample_freq){ // constant error in lead
						max_cap=6.0*data->sample_freq;
					}
					max_cap=MIN(2.0*data->sample_freq,last_anomaly_len);
					break;
				default:
					max_cap=MIN(1.0*data->sample_freq,last_anomaly_len);
					break;
				}
				if(ch_ranges[ch].ranges[n].end-ch_ranges[ch].ranges[n].start<max_cap){
					max_cap=ch_ranges[ch].ranges[n].end-ch_ranges[ch].ranges[n].start;
				}
				if(ch_ranges[ch].ranges[i].type==TEST_SHORT_OUT_OF_RANGE){ // only small cap is allowed BEFORE SHORT_OUT_OF_RANGEs
					max_cap=0.1*data->sample_freq;
				}

				if(ch_ranges[ch].ranges[i].start<ch_ranges[ch].ranges[n].end+max_cap){ // do combine?
					ch_ranges[ch].ranges[n].end=MAX(ch_ranges[ch].ranges[i].end,ch_ranges[ch].ranges[n].end);
					if(ch_ranges[ch].ranges[i].type!=ch_ranges[ch].ranges[n].type){
						ch_ranges[ch].ranges[n].type=TEST_COMBINED_ANOMALY;
					}
				}
				else{
					total_anomaly+=ch_ranges[ch].ranges[n].end-ch_ranges[ch].ranges[n].start;
					n++;
					if(i>n){
						ch_ranges[ch].ranges[n]=ch_ranges[ch].ranges[i];
					}
				}
			}
			total_anomaly+=ch_ranges[ch].ranges[n].end-ch_ranges[ch].ranges[n].start;
			n++;
			if(ch_ranges[ch].n_ranges>n){
				printf("Combined %lu ch_ranges (from %lu to %lu) (total %lf s, %lf %%) in ch %d\n",ch_ranges[ch].n_ranges-n,ch_ranges[ch].n_ranges,n,(double)(total_anomaly)/data->sample_freq, 100.0*(double)(total_anomaly)/(double)(data->samples_per_channel),ch);
			}
			else{
				printf("Total ch_ranges %lf s (%lf %%) in ch %d\n",(double)(total_anomaly)/data->sample_freq, 100.0*(double)(total_anomaly)/(double)(data->samples_per_channel),ch);
			}
			ch_ranges[ch].n_ranges=n;

		}
	}
	return(0);
}


static int trig_estimate_ch_goodness(struct data *data, struct test_events *events,  struct test_ch_info *ch_info){
	int ch;
	if(ch_info->ch_goodness==NULL){
		ch_info->ch_goodness=my_malloc(data->number_of_channels*sizeof(double));
		for(ch=0;ch<data->number_of_channels;ch++){
			ch_info->ch_goodness[ch]=1.0;
		}
	}
	// SKIP REST: CinC2017 version is simplified for one ch only -- no support for this
	return(0);
}


static int trig_create_slopes_signal(struct data const *data, struct data *energy){
	struct conf_trig *conf=conf_trig_get();
	size_t i,j;
	int ch;
	double *tmp_data=NULL,*tmp_data2=NULL;
	size_t half_win=(int)(conf->win_len*data->sample_freq/2.0+0.5),step_len=(int)(conf->step_len*data->sample_freq+0.5);
	size_t half_step=step_len/2;
	size_t ave_end;
	size_t j_max;
	double ave_len;
	double min,max;
	if(data->samples_per_channel<=2*half_win+1){
		printf("Window is longer than data -> Can't do anything...\n");
		return(1);
	}
	data_initiate_and_alloc_based_on_template(energy,data);
//	tmp_data=my_malloc(sizeof(double)*data->samples_per_channel+1);
	tmp_data2=my_malloc(sizeof(double)*data->samples_per_channel+1);


	// loop all the channels
	for(ch=0;ch<data->number_of_channels;ch++){
		tmp_data = energy->ch[ch].raw; // dirty trick to save one alloc
		// 1. define slope signal to tmp_data
		for(i=0;i<step_len;i++){
			tmp_data[i]=data->ch[ch].raw[i]-data->ch[ch].raw[0];
			tmp_data[data->samples_per_channel-1-i]=data->ch[ch].raw[data->samples_per_channel-1]-data->ch[ch].raw[data->samples_per_channel-1-i];
		}
		for(;i<data->samples_per_channel;i++){
			tmp_data[i-half_step]=data->ch[ch].raw[i]-data->ch[ch].raw[i-step_len];
		}
//		printf("%d max - min\n",ch+1);
		// 2. calculate the difference of the max and min values of the slope inside moving window
		if(conf->mode==MINMAX_DIFFERENCE){
			for(i=0;i<data->samples_per_channel;i++){
				j=MIN_ZERO_SUBS(i,half_win);
				j_max=MIN(i+half_win,data->samples_per_channel-1);
				min=max=tmp_data[j++];
				for(;j<=j_max;j++){
					if(tmp_data[j]>max)
						max=tmp_data[j];
					else if(tmp_data[j]<min)
						min=tmp_data[j];
				}
				tmp_data2[i]=max-min;
			}
		}
		else if(conf->mode==MINWEIGHTED_DIFFERENCE){
			for(i=0;i<data->samples_per_channel;i++){
				j=MIN_ZERO_SUBS(i,half_win);
				j_max=MIN(i+half_win,data->samples_per_channel-1);
				min=max=tmp_data[j++];
				for(;j<=j_max;j++){
					if(tmp_data[j]>max)
						max=tmp_data[j];
					else if(tmp_data[j]<min)
						min=tmp_data[j];
				}
				tmp_data2[i]=max-min+MIN(max,-min);
			}
		}
		else{
			for(i=0;i<data->samples_per_channel;i++){
				j=MIN_ZERO_SUBS(i,half_win);
				j_max=MIN(i+half_win,data->samples_per_channel-1);
				min=max=tmp_data[j++];
				for(;j<=j_max;j++){
					if(tmp_data[j]>max)
						max=tmp_data[j];
					else if(tmp_data[j]<min)
						min=tmp_data[j];
				}
				tmp_data2[i]=MIN(max,-min);
			}
		}

		// 3. take a moving average of the signal for smoothing / low-pass filtering, and making it easier to define 'weighed mid point'
		for(i=0;i<data->samples_per_channel;i++){
			j=MIN_ZERO_SUBS(i,half_win);
			energy->ch[ch].raw[i]=0.0;
			ave_end=MIN(i+half_win+1,data->samples_per_channel);
			ave_len=(double)(ave_end-j);
			for(;j<ave_end;j++){
				energy->ch[ch].raw[i]+=tmp_data2[j];
			}
			energy->ch[ch].raw[i]/=ave_len;
		}

	}

//	my_free(tmp_data);
	my_free(tmp_data2);

	return (0);
}

static int trig_create_std_signal(struct data const *data, struct data *energy){
	size_t i;
	int ch;
//	size_t win_len=MAX(4,0.050*data->sample_freq); // TODO use parameters from conf, if conf is available (!=NULL) xxxx
	size_t win_len=MAX(4,0.100*data->sample_freq); // TODO use parameters from conf, if conf is available (!=NULL)
	size_t half_win=win_len/2;
	double inv_win_len=1.0/(double)(win_len);
	double ave,sum,sum2;

	data_initiate_and_alloc_based_on_template(energy,data);

	// loop all the channels
	for(ch=0;ch<data->number_of_channels;ch++){
		sum=sum2=0.0;
		for(i=0;i<win_len;i++){
			sum+=data->ch[ch].raw[i];
			sum2+=POW2(data->ch[ch].raw[i]);
			ave=sum/(double)(i+1);
			if(i>=half_win){
				energy->ch[ch].raw[i-half_win]=sqrt(sum2/(double)(i+1)-POW2(ave));
			}
		}
		for(;i<data->samples_per_channel;i++){
			sum+=data->ch[ch].raw[i]-data->ch[ch].raw[i-win_len];
			sum2+=POW2(data->ch[ch].raw[i])-POW2(data->ch[ch].raw[i-win_len]);
			ave=sum*inv_win_len;
			energy->ch[ch].raw[i-half_win]=sqrt(sum2*inv_win_len-POW2(ave));
		}
		for(;i<data->samples_per_channel+half_win;i++){
			sum-=data->ch[ch].raw[i-win_len];
			sum2-=POW2(data->ch[ch].raw[i-win_len]);
			ave=sum/(double)(win_len-(i-data->samples_per_channel));
			energy->ch[ch].raw[i-half_win]=sqrt(sum2/(double)(win_len-(i-data->samples_per_channel))-POW2(ave));
		}
	}


	return (0);

}



static int trig_define_threshold_values(struct data *energy, struct test_ch_info *ch_info, struct trig_helper *helper){
	struct conf_trig *conf=conf_trig_get();
	size_t i,j,k,n;
	int ch;
	double min_max_value,max_max_value;
	size_t half_min_win=(size_t)(conf->min_rr*energy->sample_freq/2.0+0.5),half_max_win=(size_t)(conf->max_rr*energy->sample_freq/2.0+0.5);
	size_t total_low_snr,n_total_low_snr;
	int not_good,good_now;

	helper->n_steps=ceil((double)(energy->samples_per_channel)/(double)(half_max_win));
	helper->step_len=half_max_win;

	// alloc data
	double *max_data=my_malloc(energy->samples_per_channel*sizeof(double));
	double *max_max_data=my_malloc(helper->n_steps*sizeof(double));
	double *min_max_data=my_malloc(helper->n_steps*sizeof(double));
	double *just_max_max_data=my_malloc(helper->n_steps*sizeof(double));
	double *just_min_max_data=my_malloc(helper->n_steps*sizeof(double));

	// loop all the channels

	for(ch=0;ch<energy->number_of_channels;ch++){
		total_low_snr=0;
		n_total_low_snr=0;

		// 1. define max values in moving half min_rr long windows
		for(i=0;i<energy->samples_per_channel;i++){
			j=MIN_ZERO_SUBS(i,half_min_win);
			max_data[i]=energy->ch[ch].raw[j++];
			for(;j<=i;j++){
				if(max_data[i]<energy->ch[ch].raw[j]){
					max_data[i]=energy->ch[ch].raw[j];
				}
			}
		}
		// 2. define min and max values of the previously defined max values during max_rr windows, and in half max_rr steps

		for(n=0,i=0,k=0;i<helper->n_steps;i++){
			j=(i<=1 ? 0 : (i-1)*half_max_win);
			for(;k<ch_info->ch_ranges[ch].n_ranges && (ch_info->ch_ranges[ch].ranges[k].end<j || tests_is_bad_range(ch_info->ch_ranges[ch].ranges[k].type));k++);
			if(k<ch_info->ch_ranges[ch].n_ranges && ch_info->ch_ranges[ch].ranges[k].start<(i+1)*half_max_win){ // bad range? -> skip
				continue;
			}
			max_max_data[i]=min_max_data[i]=max_data[j++];
			for(;j<=MIN(energy->samples_per_channel-1,(i+1)*half_max_win);j++){
				if(max_max_data[i]<max_data[j]){
					max_max_data[i]=max_data[j];
				}
				else if(min_max_data[i]>max_data[j]){
					min_max_data[i]=max_data[j];
				}
			}
			just_max_max_data[n]=max_max_data[i];
			just_min_max_data[n]=min_max_data[i];
			n++;
		}

		// 3. define threshold values

		max_max_value=array_kth_biggest(just_max_max_data,n,n*0.1);
		min_max_value=array_kth_biggest(just_min_max_data,n,n*0.9);

		helper->threshold_values[ch]=(conf->threshold_high_weight*max_max_value+(1.0-conf->threshold_high_weight)*min_max_value); // constant threshold value over the measurement

		// 4. check the energy SNR

		not_good=0;
		for(i=0,k=0;i<helper->n_steps;i++){
			j=(i<=1 ? 0 : (i-1)*half_max_win);
			for(;k<ch_info->ch_ranges[ch].n_ranges && ch_info->ch_ranges[ch].ranges[k].end<j;k++);
			if(k<ch_info->ch_ranges[ch].n_ranges && ch_info->ch_ranges[ch].ranges[k].start<(i+1)*half_max_win){ // already bad range? -> skip
				continue;
			}
			good_now=min_max_data[i] < helper->threshold_values[ch] && max_max_data[i] > helper->threshold_values[ch]; // binary goodness value (WIP)
			if(good_now && not_good){ // should we end the TEST_LOW_SNR range?
				not_good=0;
				ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].end=(i)*half_max_win;
				total_low_snr+=ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].end-ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].start;
				ch_info->ch_ranges[ch].n_ranges++;
				n_total_low_snr++;
			}
			else if(!good_now && !not_good){ // should we start new TEST_LOW_SNR range?
				not_good=1;
				ch_info->ch_ranges[ch].ranges=my_realloc(ch_info->ch_ranges[ch].ranges,(ch_info->ch_ranges[ch].n_ranges+1)*sizeof(struct test_range));
				ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].start=j;
				ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].type=TEST_LOW_SNR;
			}
		}
		if(not_good){ // should we (finally) end the TEST_LOW_SNR range?
			not_good=0;
			ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].end=energy->samples_per_channel-1;
			total_low_snr+=ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].end-ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].start;
			ch_info->ch_ranges[ch].n_ranges++;
			n_total_low_snr++;
		}
		printf("Found %lu LOW_SNR ranges (%.3lf %%) from ch %d\n",n_total_low_snr,100.0*(double)(total_low_snr)/(double)(energy->samples_per_channel),ch+1);
	}

	my_free(just_max_max_data);
	my_free(just_min_max_data);
	my_free(max_data);
	my_free(max_max_data);
	my_free(min_max_data);

	return(0);
}

static int trig_event_probability(struct data *energy_data,struct test_ch_info *ch_info, struct trig_helper *helper, struct test_events *events, int only_undefined){
	// SKIP REST: CinC2017 version is simplified for one ch only -- no support for this
	return(0);
}


#define EVENT_ALLOC_STEP 10000
#define TRIG_SKIP_THIS_RANGE(type) (type==TEST_SHORT_OUT_OF_RANGE ? 1 : 0)   // Switch ranges to include when defining events



// TODO: analyze one ch at time + analyze goodness of results -- don't return anything if it is very unlikely to be correct..
static inline size_t trig_find_one_event(struct data *energy_data,size_t start, size_t end, struct test_ch_info *ch_info, struct trig_helper *helper){
	size_t i;
	size_t max_i=start;
	int ch;
	double sum;
	double max=0.0;

	for(i=start;i<end;i++){
		sum=0.0;
		for(ch=0;ch<energy_data->number_of_channels;ch++){
			sum+=energy_data->ch[ch].raw[i];
		}
		if(sum>max){
			max=sum;
			max_i=i;
		}
	}
	if(max_i==start || max_i==end-1){  // is max on the border? we don't want that..
		return(0);  // return 'nothing found'
	}

	for(ch=0;ch<energy_data->number_of_channels;ch++){
		if(energy_data->ch[ch].raw[max_i]>0.5*helper->threshold_values[ch]){ // is there any signal? is even one of the channels above half of the threshold level..?
			return(max_i);
		}
	}
	return(0);  // nothing found
}

static int trig_define_cinc2017_initial_events(struct data const *active_data,struct data *energy_data,struct test_ch_info *ch_info, struct trig_helper *helper, struct test_events *events,struct test_event_infos *event_infos){
	struct conf_trig *trig_conf=conf_trig_get();
	int n_alloced=300;
	int range_ind=0;
	int next_bad=0;
	int i;
	int max_i;
	int ch=0; // we analyze only ch == 0!!!
	double *data=energy_data->ch[ch].raw;
	int nsamp = energy_data->samples_per_channel;
	struct test_ranges *ranges = ch_info->ch_ranges+ch;

	double prev_data;
	double threshold_value=helper->threshold_values[ch];
	double min;
	int n_under_threshold, n_under_threshold_limit = trig_conf->max_below_threshold*energy_data->sample_freq+0.5;
	int min_rr = trig_conf->min_rr*energy_data->sample_freq+0.5;
	int max_qrs_width =MAX(trig_conf->win_len,(trig_conf->min_rr-0.5*trig_conf->win_len))*energy_data->sample_freq+0.5;
	events->events=my_malloc(n_alloced*sizeof(struct test_event));
	events->n_events=0;
	events->n_ranges=0;
	events->ranges=NULL;

	min=FLT_MAX;

	max_i=-min_rr;

	if(range_ind<ranges->n_ranges){
		next_bad=ranges->ranges[range_ind].start;
	}
	else{
		next_bad=nsamp+1;
	}

	for(i=1;i<nsamp;i++){
		prev_data=data[i-1];
		if(next_bad <= i){
			i=ranges->ranges[range_ind].end; // jump to the end of the 'bad segment'
			for(;range_ind<ranges->n_ranges && (ranges->ranges[range_ind].end<=i || TRIG_SKIP_THIS_RANGE(ranges->ranges[range_ind].type));range_ind++);
			if(range_ind<ranges->n_ranges){
				next_bad=ranges->ranges[range_ind].start;
			}
			else{
				next_bad=nsamp+1;
			}
			continue;
		}

		if(data[i]<min){
			min=data[i];
		}
		if(i-max_i<min_rr){
			continue;
		}
		if(data[i]>threshold_value && data[i]-min>0.25*threshold_value && data[i]>prev_data){ // we found one
			// look for the max value
			min=data[i];
			n_under_threshold=0;
			for(max_i=i,i++;i<nsamp;i++){  // look for local maximum
				prev_data=data[i-1];
				if(data[i]>data[max_i]){
					max_i=i;
					min=data[max_i]; // we reset the min value, since want to get the min value AFTER the max
				}
				else if(data[i]<min){
					min=data[i];
				}
				if(data[i]<threshold_value)
					n_under_threshold++;
				else
					n_under_threshold=0;
				if(n_under_threshold>=n_under_threshold_limit) // locality / seek time limit for data to be under the threshold
					break;
				if(i-max_i>=max_qrs_width && prev_data<data[i]){  // locality / seek time limit for maximum distance from the maximum
					break;
				}
				prev_data=data[i];
			}

			// add event
			if(events->n_events>=n_alloced){
				n_alloced+=300;
				events->events=my_realloc(events->events,n_alloced*sizeof(struct test_event));
			}
			events->events[events->n_events].sample=max_i;
			events->events[events->n_events].sub_sample=0.0;
			events->events[events->n_events].probability=1.0;
			events->events[events->n_events++].type=TEST_BEAT_UNDEFINED;
			i=max_i+1; // backstep for setting new seek to start right after the last one..
		}
	}
	return(0);
}


int trig_test_events(struct test_events *events){
	size_t i;
	for(i=0;i<events->n_events-1;i++){
		if(events->events[i].sample>=events->events[i+1].sample){
			printf("Events are corrupted from index %lu (%lu)\n",i,events->events[i].sample);
			return(-1);
		}
	}
	return(0);
}

int trig_test_event_ranges(struct test_range *ranges, int n_ranges){
	ssize_t i;
	for(i=0;i<n_ranges;i++){
		if(ranges[i].start>=ranges[i].end){
			printf("Events are corrupted from index %lu (%lu - %lu)\n",i,ranges[i].start, ranges[i].end);
			return(-1);
		}
	}
	for(i=0;i<n_ranges-1;i++){
		if(ranges[i].end>=ranges[i+1].start){
			printf("Events are corrupted from index %lu (%lu - %lu, next: %lu - %lu)\n",i,ranges[i].start, ranges[i].end,ranges[i+1].start, ranges[i+1].end);
			return(-1);
		}
	}
	return(0);
}


//#define SHORT_FIR_FILTER
#ifdef SHORT_FIR_FILTER
/*

FIR filter designed with
http://t-filter.appspot.com

sampling frequency: 250 Hz

* 5 Hz - 25 Hz
  gain = 1
  desired ripple = 5 dB
  actual ripple = 3.758695500089128 dB

* 45 Hz - 125 Hz
  gain = 0
  desired attenuation = -30 dB
  actual attenuation = -30.88337239822076 dB

*/

#define FILTER_TAP_NUM 13

static double filter_taps[FILTER_TAP_NUM] = {
  -0.02634061470015919,
  -0.017667570935711537,
  0.014480192258281717,
  0.08288651564658531,
  0.17154245042745822,
  0.24740490603099094,
  0.27731883285466746,
  0.24740490603099094,
  0.17154245042745822,
  0.08288651564658531,
  0.014480192258281717,
  -0.017667570935711523,
  -0.02634061470015919
};
#else
/*
sampling frequency: 250 Hz

* 0 Hz - 0.5 Hz
  gain = 0
  desired attenuation = -30 dB
  actual attenuation = -30.34018653103896 dB

* 3 Hz - 40 Hz
  gain = 1
  desired ripple = 6 dB
  actual ripple = 4.963950824424056 dB

* 48 Hz - 125 Hz
  gain = 0
  desired attenuation = -30 dB
  actual attenuation = -30.34018653103896 dB

*/

/*#define FILTER_TAP_NUM 55
static double filter_taps[FILTER_TAP_NUM] = {
  -0.034037603033608754,
  -0.044344768982980926,
  -0.049883495287146716,
  -0.04088633450302861,
  -0.023916306790530303,
  -0.008965432572309712,
  -0.00174728648051977,
  -0.0018717529778548567,
  -0.006666731947075155,
  -0.014371833084313152,
  -0.022362344438400912,
  -0.02514465648786348,
  -0.017817652770843132,
  -0.0038460422850733118,
  0.0036766548767816526,
  -0.006522259521124267,
  -0.02907088149563562,
  -0.04170884927334606,
  -0.02613980218485189,
  0.008238436327178175,
  0.025718292165067966,
  -0.0023331114189566664,
  -0.06049941662739721,
  -0.08893978040733472,
  -0.03025580706149239,
  0.11326206752522426,
  0.2665872262660048,
  0.33244652293252913,
  0.2665872262660048,
  0.11326206752522426,
  -0.03025580706149239,
  -0.08893978040733472,
  -0.06049941662739721,
  -0.002333111418956664,
  0.025718292165067966,
  0.008238436327178175,
  -0.02613980218485189,
  -0.04170884927334605,
  -0.02907088149563562,
  -0.0065222595211242625,
  0.003676654876781661,
  -0.0038460422850733118,
  -0.017817652770843132,
  -0.02514465648786348,
  -0.022362344438400912,
  -0.014371833084313161,
  -0.006666731947075155,
  -0.0018717529778548567,
  -0.00174728648051977,
  -0.008965432572309712,
  -0.023916306790530303,
  -0.04088633450302861,
  -0.049883495287146716,
  -0.044344768982980926,
  -0.034037603033608754
};*/


#define FILTER_TAP_NUM 129
static double filter_taps[FILTER_TAP_NUM] = {-0.000000,
-0.000000,
-0.000177,
-0.000000,
-0.000423,
-0.000000,
-0.000747,
-0.000000,
-0.001158,
0.000000,
-0.001661,
0.000000,
-0.002261,
-0.000000,
-0.002965,
-0.000000,
-0.003773,
0.000000,
-0.004686,
-0.000000,
-0.005704,
-0.000000,
-0.006824,
-0.000000,
-0.008040,
-0.000000,
-0.009346,
-0.000000,
-0.010731,
-0.000000,
-0.012184,
0.000000,
-0.013693,
0.000000,
-0.015242,
-0.000000,
-0.016814,
0.000000,
-0.018394,
0.000000,
-0.019961,
0.000000,
-0.021496,
-0.000000,
-0.022981,
0.000000,
-0.024395,
0.000000,
-0.025720,
0.000000,
-0.026938,
0.000000,
-0.028031,
0.000000,
-0.028984,
0.000000,
-0.029784,
0.000000,
-0.030418,
-0.000000,
-0.030878,
-0.000000,
-0.031157,
-0.000000,
0.968750,
-0.000000,
-0.031157,
-0.000000,
-0.030878,
-0.000000,
-0.030418,
0.000000,
-0.029784,
0.000000,
-0.028984,
0.000000,
-0.028031,
0.000000,
-0.026938,
0.000000,
-0.025720,
0.000000,
-0.024395,
0.000000,
-0.022981,
-0.000000,
-0.021496,
0.000000,
-0.019961,
0.000000,
-0.018394,
0.000000,
-0.016814,
-0.000000,
-0.015242,
0.000000,
-0.013693,
0.000000,
-0.012184,
-0.000000,
-0.010731,
-0.000000,
-0.009346,
-0.000000,
-0.008040,
-0.000000,
-0.006824,
-0.000000,
-0.005704,
-0.000000,
-0.004686,
0.000000,
-0.003773,
-0.000000,
-0.002965,
-0.000000,
-0.002261,
0.000000,
-0.001661,
0.000000,
-0.001158,
-0.000000,
-0.000747,
-0.000000,
-0.000423,
-0.000000,
-0.000177,
-0.000000,
-0.000000};
#endif

static int do_bp_filter(struct bp_filter *bp_filter,struct data *filtered_data,struct data const *original_data){
	size_t i,j;
	int ch;



	if(bp_filter->type==IIR){
		data_initiate_based_on_template(filtered_data,original_data);
//		double *tmp_data=my_malloc( original_data->samples_per_channel*sizeof(double));
		for(ch=0;ch<original_data->number_of_channels;ch++){
/*			for(i=0;i< original_data->samples_per_channel;i++){
				tmp_data[i]=original_data->ch[ch].raw[original_data->samples_per_channel-i-1];
			}

			dsp_chebyshev_high_pass(&(filtered_data->ch[ch].raw), tmp_data, original_data->samples_per_channel,
						0.050, 0.5,
						2,  original_data->sample_freq);
			for(i=0;i< original_data->samples_per_channel;i++){
				filtered_data->ch[ch].raw[i]=filtered_data->ch[ch].raw[original_data->samples_per_channel-i-1];
			}
*/
	//		dsp_bandpass(&(filtered_data->ch[ch].raw), original_data->ch[ch].raw, original_data->samples_per_channel, bp_filter->low,bp_filter->high, original_data->sample_freq);
		}
		//my_free(tmp_data);
	}
	else if(bp_filter->type==NONLINEAR){
		double middle_val,max_step;
		data_initiate_and_alloc_based_on_template(filtered_data,original_data);

		max_step=1000.0/original_data->sample_freq;
		for(ch=0;ch<original_data->number_of_channels;ch++){
			middle_val=0.0;
			for(i=0;i<original_data->samples_per_channel;i++){
				if(fabs(middle_val-filtered_data->ch[ch].raw[i])<max_step){
					middle_val=filtered_data->ch[ch].raw[i];
				}
				else if(middle_val>filtered_data->ch[ch].raw[i]){
					middle_val-=max_step;
				}
				else{
					middle_val+=max_step;
				}
				filtered_data->ch[ch].raw[i]-=middle_val;
			}
		}
	}
	else{ // default
//	if(bp_filter->type==FIR){ // FIR? We have only hardcoded option available..
		int half_win=FILTER_TAP_NUM/2;
		data_initiate_and_alloc_based_on_template(filtered_data,original_data);

		// FIR
		for(ch=0;ch<original_data->number_of_channels;ch++){
			for(i=0;i<original_data->samples_per_channel;i++){
				filtered_data->ch[ch].raw[i]=0.0;
				for(j=0;j<FILTER_TAP_NUM;j++){
					if(i+j<half_win || i+j>original_data->samples_per_channel+half_win){
						continue;
					}
					filtered_data->ch[ch].raw[i]+=original_data->ch[ch].raw[i+j-half_win]*filter_taps[j];
				}
			}
		}
	}




	return(0);
}

// Detect and filter out the impulse artifacts -- the big one sample amplitude deviations. These can be found at least from AHA data (for example from case 4002 around 613500/250 s from start)
int trig_filter_impulse_artifacts(struct data *data, double impulse_limit_in_mv){
	int ch;
	size_t i;
	double *signal;
	double impulse_limit;
	size_t count=0;
	for(ch=0;ch<data->number_of_channels;ch++){
		signal=data->ch[ch].raw;
		impulse_limit=impulse_limit_in_mv*data->ch[ch].samples_per_mv;
		for(i=1;i<data->samples_per_channel-1;i++){
			if((signal[i]-signal[i-1]>impulse_limit && signal[i]-signal[i+1]>impulse_limit) || (signal[i-1]-signal[i]>impulse_limit && signal[i+1]-signal[i]>impulse_limit)){
				signal[i]=(signal[i-1]+signal[i+1])/2;     // TODO: implement sinc-interpolation (for impulses not too close to each other)
				count++;
			}
		}
	}
	if(count){
		printf("%lu impulse artifacts interpolated from signal\n",count);
	}
	return(0);
}

int trig_define_notch_frequencies(struct data *data, struct data *notch_data, struct test_ranges *ch_ranges){
// ******* conf *********
	int grey_frequency=6,bad_frequency=10; // TODO: should be configurable
//	int grey_frequency=7,bad_frequency=13; // TODO: should be configurable

	struct conf_trig *conf=conf_trig_get();
	int ch;
	double notch_amplitude_limit;
	int win_len;
	int half_len;
	int above_grey,above_limit;
	size_t *notches=my_malloc(data->samples_per_channel*sizeof(size_t));
	int direction;
	int frequency;
	size_t i,j,k,n;
	size_t max_i,min_i;
	size_t grey_start;
	size_t total_bad;
	size_t n_ranges_before;
	double *minmax=NULL;

	win_len=2.0*conf->min_rr*data->sample_freq;
	half_len=win_len/2;

	if(notch_data){
		data_initiate_and_alloc_based_on_template(notch_data,data);
	}

	for(ch=0;ch<data->number_of_channels;ch++){
		n_ranges_before=ch_ranges[ch].n_ranges;
		if(conf->notch_rel_limit>0){  // define the notch limit value as relative to data values
			double min,max;
			int step_len=conf->max_rr*data->sample_freq;
			int nsteps=data->samples_per_channel/step_len;

			if(minmax==NULL){
				minmax=my_malloc((nsteps)*sizeof(double));
			}
			for(i=0;i<nsteps;i++){
				j=i*step_len;
				min=max=data->ch[ch].raw[j];
				for(++j;j<(i+1)*step_len;j++){
					if(min>data->ch[ch].raw[j]){
						min=data->ch[ch].raw[j];
					}
					else if(max<data->ch[ch].raw[j]){
						max=data->ch[ch].raw[j];
					}
				}
				minmax[i]=max-min;
			}
			notch_amplitude_limit=conf->notch_abs_limit+conf->notch_rel_limit*array_kth_biggest(minmax,nsteps,0.25*nsteps);
		}
		else{
			notch_amplitude_limit=conf->notch_abs_limit;
		}

		// find the notches
		max_i=min_i=0;
		direction=0;
		n=0;
		for(i=0;i<data->samples_per_channel;i++){
			if(direction>=0){ // setting positive (max) notch?
				if(data->ch[ch].raw[i]>data->ch[ch].raw[max_i]){  // update max value?
					max_i=i;
				}
				else if((data->ch[ch].raw[i]<data->ch[ch].raw[max_i]-notch_amplitude_limit)){ // set positive notch and start looking for next negative one
					notches[n++]=max_i;
					min_i=i;
					direction=-1;
				}
			}
			if(direction<=0){ // setting negative (min) notch?
				if(data->ch[ch].raw[i]<data->ch[ch].raw[min_i]){  // update max value?
					min_i=i;
				}
				else if((data->ch[ch].raw[i]>data->ch[ch].raw[min_i]+notch_amplitude_limit)){ // set negative notch and start looking for next positive one
					notches[n++]=min_i;
					max_i=i;
					direction=1;
				}
			}
		}

		if(notch_data){
			// define notch frequencies
			frequency=0;
			for(j=0,k=0,i=0;i<data->samples_per_channel+half_len;i++){
				if(j<n && notches[j]==i){ // one more notch to window
					j++;
					frequency++;
				}
				if(i>win_len && k<n && notches[k]==i-win_len){ // one notch out
					k++;
					frequency--;
				}
				if(i>=half_len){
					notch_data->ch[ch].raw[i-half_len]=100.0*frequency;
				}
			}
		}
		if(ch_ranges){
			// define bad ranges
			frequency=0;
			above_limit=above_grey=0;
			total_bad=0;
			for(j=0,k=0;k<n;){
				if(j<n && (notches[j]<win_len || notches[j]<notches[k]+win_len)){ // new event in window -> frequency increases
					frequency++;
					j++;
				}
				else{ // frequency decreases
					frequency--;
					k++;
				}

				if(above_limit){ // we are already in 'bad range' and looking for the end of the range
					if(frequency<grey_frequency){ // stop bad range?
						ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].end=MIN(notches[k-1]+win_len/2,data->samples_per_channel-1);
						total_bad+=ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].end-ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start;
						ch_ranges[ch].n_ranges++;
						above_limit=0;
						above_grey=0;
					}
				}
				else{
					if(frequency>=grey_frequency){ // Are we in 'grey area'?
						if(!above_grey){
							if(notches[j-1]<win_len/2){
								grey_start=0;
							}
							else{
								grey_start=notches[j-1]-win_len/2;
							}
							above_grey=1;
						}
					}
					else{
						above_grey=0;
					}
					if(frequency>=bad_frequency){ // Is this bad, is the notch frequency too high?
						above_limit=1;
						ch_ranges[ch].ranges=my_realloc(ch_ranges[ch].ranges,(ch_ranges[ch].n_ranges+1)*sizeof(struct test_range));
						ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start=grey_start;
						ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].type=TEST_NOISY_NOTCHES;
					}
				}
			}
			if(above_limit){ // Are we still above the limit?
				ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].end=data->samples_per_channel-1;
				total_bad+=ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].end-ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start;
				ch_ranges[ch].n_ranges++;
			}
			printf("Found %lu bad notch frequency ranges (%.3lf %%) from ch %d\n",ch_ranges[ch].n_ranges-n_ranges_before,100.0*(double)(total_bad)/(double)(data->samples_per_channel),ch+1);
		}
	}
	my_free(notches);
	FREE(minmax);


	return(0);
}


static inline void trig_update_sorted(double old_val,double new_val, double *sorted,int len){
	int i;
	if(new_val==old_val){
		return;
	}

	if(new_val>old_val){
		for(i=0;i<len-1 && old_val>sorted[i];i++);
		for(;i<len-1 && new_val>sorted[i+1];i++){
			sorted[i]=sorted[i+1];
		}
		sorted[i]=new_val;
	}
	else{
		for(i=len-1;i>0 && old_val<sorted[i];i--);
		for(;i>0 && new_val<sorted[i-1];i--){
			sorted[i]=sorted[i-1];
		}
		sorted[i]=new_val;
	}
}

int trig_detect_big_median_level_changes(struct data *data, struct test_ranges *ch_ranges){
	int ch;
	int median_len=0.400*data->sample_freq;
	int median_ind=median_len/2;
//	double lvl_change_limit_in_mv=0.85;
	double lvl_change_limit_in_mv=1.0;
	double lvl_change_limit;
	size_t i;
	size_t nbad;
	double *median_data=my_malloc(data->samples_per_channel*sizeof(double));
	double *sorted=my_malloc(median_len*sizeof(double));
	for(ch=0;ch<data->number_of_channels;ch++){
		lvl_change_limit=lvl_change_limit_in_mv*data->ch[ch].samples_per_mv;
		nbad=0;
		// create median signal
		memcpy(sorted,data->ch[ch].raw,median_len*sizeof(double));
		array_dsort(sorted,median_len);
		median_data[0]=sorted[median_ind];
		for(i=1;i<data->samples_per_channel-median_len;i++){
			trig_update_sorted(data->ch[ch].raw[i-1],data->ch[ch].raw[i+median_len-1], sorted,median_len);
			median_data[i]=sorted[median_ind];
		}

		// check too big lvl changes
		for(i=0;i<data->samples_per_channel-2*median_len;i++){
			if(fabs(median_data[i]-median_data[i+median_ind])>lvl_change_limit){
				ch_ranges[ch].ranges=my_realloc(ch_ranges[ch].ranges,(ch_ranges[ch].n_ranges+1)*sizeof(struct test_range));
				ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start=i;
				ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].end=i+2*median_len;
				ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].type=TEST_SIGNAL_LEVEL_CHANGE;
				nbad++;
				ch_ranges[ch].n_ranges++;
				i+=median_len;
			}
		}
		if(nbad){
			printf("Marked %lu big signal level changes in ch %d\n",nbad,ch+1);
		}
	}

	my_free(sorted);
	my_free(median_data);
	return(0);
}

// Detect returning amplitude drifts -- typical artifact in AHA dataset
// TODO: MEDIAN lvl drift version might be better for not-so-constant lvl changes (see 2102.txt around 1332051 for example!)
int trig_detect_returning_amplitude_drifts(struct data *data, struct test_ranges *ch_ranges){
	size_t i,j,k;
	size_t step_start,saturation_end,step_peak;
	int ch;
	double *signal;
	int step=0.05*data->sample_freq;
	int min_len=0.15*data->sample_freq;
	int ave_len=0.39*data->sample_freq;
	int step_detected=0,return_step_detected;;
	double step_limit,step_limit_in_mv=0.6;
	double saturation_limit,saturation_limit_in_mv=0.1;
	double saturation_level,ave;
	size_t nbad,bad_len;
	for(ch=0;ch<data->number_of_channels;ch++){
		nbad=0;
		bad_len=0;
		step_limit=step_limit_in_mv*data->ch[ch].samples_per_mv;
		saturation_limit=saturation_limit_in_mv*data->ch[ch].samples_per_mv;
		signal=data->ch[ch].raw;
		for(i=step+ave_len;i<data->samples_per_channel-step-min_len;i++){
			if(signal[i]-signal[i-step]>step_limit){  // positive step?
				step_detected=1;
				for(j=i+1;signal[j]>signal[j-1];j++);  // continue to the peak of the step
			}
			else if(signal[i]-signal[i-step]<-step_limit){ // negative step?
				step_detected=-1;
				for(j=i+1;signal[j]<signal[j-1];j++); // continue to the peak of the step
			}
			if(step_detected){  						// *** 1/4 Is there a big enough step?
				step_start=i-step;
				step_peak=j-1;
				for(;j<step_peak+step && j<data->samples_per_channel && fabs(signal[j]-signal[step_peak])<saturation_limit;j++);
				if(j==step_peak+step){
					for(;j<data->samples_per_channel-1 && (fabs(signal[j]-signal[j-step])<saturation_limit || fabs(signal[j+1]-signal[j-step])<saturation_limit);j++);
					saturation_end=j;
					if(j-step_peak>min_len && signal[j]-signal[j-step]*step_detected<0){ // *** 2/4 Is there saturation after the step, and is the return from the saturation to the correct direction?
						ave=0.0;
						for(k=step_start-ave_len;k<step_start;k++){
							ave+=signal[k];
						}
						ave/=(double)(ave_len);
						if((nbad && ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].end+ave_len>step_start) ? fabs(saturation_level-signal[step_peak])<2.0*saturation_limit : fabs(ave-signal[step_peak])>step_limit){// *** 3/4 Did the 'base signal level' before the step differ enough from the saturated signal, or did the last drift just end?
							return_step_detected=0;
							if(step_detected>0){
								for(;signal[j]<signal[j-1];j++){  // continue to the end of the "returning step"
									if(signal[j-step]-signal[j]>step_limit){
										return_step_detected=1;
									}
								}
							}
							else{
								for(;signal[j]>signal[j-1];j++){  // continue to the end of the "returning step"
									if(signal[j]-signal[j-step]>step_limit){
										return_step_detected=1;
									}
								}
							}
							if(return_step_detected){  // *** 4/4  was the return to the 'base signal level' big enough?
								saturation_level=signal[saturation_end-step];
								ch_ranges[ch].ranges=my_realloc(ch_ranges[ch].ranges,(ch_ranges[ch].n_ranges+1)*sizeof(struct test_range));
								ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start=step_start;
								ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].end=j+step;
								ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].type=TEST_RETURNING_AMPLITUDE_DRIFT;
								bad_len+=ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].end-ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start;
								nbad++;
								ch_ranges[ch].n_ranges++;

								i=j+step;
							}
							else{
	//							printf("No return step (ch: %d, i: %lu)\n",ch,step_peak);
								i=step_peak;
							}
						}
					}
				}
				step_detected=0;
			}
		}
		if(nbad){
			printf("Marked %lu returning drifts (%.3lf %%) from ch %d\n",nbad,100.0*((double)(bad_len)/(double)(data->samples_per_channel)),ch+1);
		}

	}

	return(0);
}

// Detect amplitude drifts -- typical artifact in MIT dataset
int trig_detect_amplitude_drifts(struct data *data, struct test_ranges *ch_ranges){
	ssize_t prev,i;
	ssize_t ave_i;
	int ch;
	int step_len=0.4*data->sample_freq;
	int ave_len=0.3*data->sample_freq;
	int half_ave=ave_len/2;
	int bad,prev_bad;
	size_t nbad,nbad_ave_samples, nbad_samples;
	ssize_t bad_range_starts_at;
	double ave_len_d=ave_len;
	double ave_limit=3000.0,limit=5000.0;
	double *ave=my_malloc(data->samples_per_channel*sizeof(double));

	if(data->samples_per_channel<ave_len+step_len){
		printf("Oops! Too short data for amplitude drift detection\n");
		return(0);
	}

	// Loop all the chs
	for(ch=0;ch<data->number_of_channels;ch++){

		// create moving ave
		ave[0]=data->ch[ch].raw[0];
		for(i=1;i<ave_len;i++){
			ave[0]+=data->ch[ch].raw[i];
		}
		for(i=1;i<data->samples_per_channel-ave_len;i++){
			prev=i-1;
			ave[i]=ave[prev]-data->ch[ch].raw[prev]+data->ch[ch].raw[prev+ave_len];
			ave[prev]/=ave_len_d;
		}
		prev=i-1;
		ave[prev]/=ave_len_d;

		// check the limit breaks, and create the bad ranges
		nbad=nbad_ave_samples=nbad_samples=0;
		for(bad=0,prev_bad=0,i=step_len;i<data->samples_per_channel-step_len;i++){
			ave_i=i-half_ave;
			if(fabs(data->ch[ch].raw[i]-data->ch[ch].raw[i-step_len])>limit || fabs(data->ch[ch].raw[i]-data->ch[ch].raw[i+step_len])>limit){
				bad=1;
				nbad_samples++;
			}
			else if((ave_i>=step_len && fabs(ave[ave_i]-ave[ave_i-step_len])>ave_limit) || (ave_i+step_len+half_ave+1<data->samples_per_channel && fabs(ave[ave_i]-ave[ave_i+step_len])>ave_limit)){
				bad=1;
				nbad_ave_samples++;
			}
			else{
				bad=0;
			}
			if(bad && !prev_bad){ // start bad range?
				bad_range_starts_at=i-step_len;
				if(nbad==0 || ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].end<bad_range_starts_at){ // start new range?
					ch_ranges[ch].ranges=my_realloc(ch_ranges[ch].ranges,(ch_ranges[ch].n_ranges+1)*sizeof(struct test_range));
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start=bad_range_starts_at;
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].type=TEST_SIGNAL_AMPLITUDE_DRIFT;
					nbad++;
				}
				else{ // ..continue prev range
					ch_ranges[ch].n_ranges--;
				}
			}
			else if(!bad && prev_bad){ // end bad range?
				ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].end=i+step_len;
				ch_ranges[ch].n_ranges++;
			}
			prev_bad=bad;
		}
		if(prev_bad){ // still bad at the signal end?
			ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].end=data->samples_per_channel;
			ch_ranges[ch].n_ranges++;
		}
		if(nbad){
			printf("Marked %lu amplitude drifts (%.3lf+%.3lf %%)from ch %d\n",nbad,100.0*(double)nbad_samples/(double)(data->samples_per_channel-2*step_len),100.0*(double)nbad_ave_samples/(double)(data->samples_per_channel-2*step_len-ave_len),ch+1);
		}
	}
	my_free(ave);


	return(0);
}

int trig_detect_saturated_segments(struct data *data, struct test_ranges *ch_ranges){
	struct conf_trig *conf=conf_trig_get();
	double range_min=0.0,range_max=0.0;
	size_t n_min_saturated=0,n_max_saturated=0;
	size_t i,n_not_saturated;
	size_t n_ranges_before,total_bad,total_n_saturated;
	int ch;
	int min_n_saturated=2;
	int max_non_saturated_between=0.100*data->sample_freq;
	int ignore_less_than_saturated=0.05*data->sample_freq;

	if(conf->data_max>conf->data_min){ // Do we have given range limits?
		range_min=conf->data_min;
		range_max=conf->data_max;
		n_min_saturated=min_n_saturated;
		n_max_saturated=min_n_saturated;
	}
	else{ // specify limits from data
		for(ch=0;ch<data->number_of_channels;ch++){
			for(i=0;i<data->samples_per_channel;i++){
				if(range_max<data->ch[ch].raw[i]){
					range_max=data->ch[ch].raw[i];
				}
				else if(range_min>data->ch[ch].raw[i]){
						range_min=data->ch[ch].raw[i];
				}
			}
		}
	// Check if the limits look ok -- is there at least one segment of saturated data?
		for(ch=0;ch<data->number_of_channels;ch++){
			for(i=0;i<data->samples_per_channel;i++){
				if(range_max==data->ch[ch].raw[i] && n_max_saturated<min_n_saturated){
					for(n_max_saturated=1;range_max<=data->ch[ch].raw[i+n_max_saturated];n_max_saturated++);
				}
				else if(range_min==data->ch[ch].raw[i] && n_min_saturated<min_n_saturated){
					for(n_min_saturated=1;range_min>=data->ch[ch].raw[i+n_min_saturated];n_min_saturated++);
				}
			}
		}
		if(n_max_saturated<min_n_saturated && n_min_saturated<min_n_saturated){ // was there any?
			return(0);
		}

	}
	if(n_max_saturated>=min_n_saturated && n_min_saturated>=min_n_saturated){ // Do we have both amplitude range limits defined?
		printf("Saturation limits: %lf & %lf\n",range_min,range_max);
		for(ch=0;ch<data->number_of_channels;ch++){
			n_ranges_before=ch_ranges[ch].n_ranges;
			total_bad=0;
			total_n_saturated=0;
			for(i=0;i<data->samples_per_channel;i++){
				if(range_max<=data->ch[ch].raw[i] || range_min>=data->ch[ch].raw[i]){
					ch_ranges[ch].ranges=my_realloc(ch_ranges[ch].ranges,(ch_ranges[ch].n_ranges+1)*sizeof(struct test_range));
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start=i;
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].type=TEST_SIGNAL_OUT_OF_RANGE;
					for(n_not_saturated=0,i++;i<data->samples_per_channel && n_not_saturated<max_non_saturated_between;i++){
						if(range_max<=data->ch[ch].raw[i] || range_min>=data->ch[ch].raw[i]){
							n_not_saturated=0;
						}
						else{
							n_not_saturated++;
						}
					}
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges++].end=i-n_not_saturated;
					if(i-n_not_saturated-ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].start<ignore_less_than_saturated){
						ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].type=TEST_SHORT_OUT_OF_RANGE;
					}
					else{
						total_bad+=ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].end-ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].start;
						total_n_saturated++;
					}
				}
			}
			printf("Marked %lu/%lu saturated ranges (%.3lf %%) from ch %d\n",total_n_saturated,ch_ranges[ch].n_ranges-n_ranges_before, 100.0*(double)(total_bad)/(double)(data->samples_per_channel),ch+1);
		}
	}
	else if(n_max_saturated>=min_n_saturated){ // Do we have just the maximum amplitude limit defined?
		printf("Saturation max limit: %lf\n",range_max);
		for(ch=0;ch<data->number_of_channels;ch++){
			n_ranges_before=ch_ranges[ch].n_ranges;
			total_bad=0;
			total_n_saturated=0;
			for(i=0;i<data->samples_per_channel;i++){
				if(range_max<=data->ch[ch].raw[i]){
					ch_ranges[ch].ranges=my_realloc(ch_ranges[ch].ranges,(ch_ranges[ch].n_ranges+1)*sizeof(struct test_range));
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start=i;
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].type=TEST_SIGNAL_OUT_OF_RANGE;
					for(n_not_saturated=0,i++;i<data->samples_per_channel && n_not_saturated<max_non_saturated_between;i++){
						if(range_max<=data->ch[ch].raw[i]){
							n_not_saturated=0;
						}
						else{
							n_not_saturated++;
						}
					}
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges++].end=i-n_not_saturated;
					if(i-n_not_saturated-ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].start<ignore_less_than_saturated){
						ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].type=TEST_SHORT_OUT_OF_RANGE;
					}
					else{
						total_bad+=ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].end-ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].start;
						total_n_saturated++;
					}
				}
			}
			printf("Marked %lu/%lu saturated ranges (%.3lf %%) from ch %d\n",total_n_saturated,ch_ranges[ch].n_ranges-n_ranges_before,100.0*(double)(total_bad)/(double)(data->samples_per_channel),ch+1);
		}
	}
	else if(n_min_saturated>=min_n_saturated){ // Do we have just the maximum amplitude limit defined?
		printf("Saturation min limit: %lf\n",range_min);
		for(ch=0;ch<data->number_of_channels;ch++){
			n_ranges_before=ch_ranges[ch].n_ranges;
			total_bad=0;
			total_n_saturated=0;
			for(i=0;i<data->samples_per_channel;i++){
				if(range_min>=data->ch[ch].raw[i]){
					ch_ranges[ch].ranges=my_realloc(ch_ranges[ch].ranges,(ch_ranges[ch].n_ranges+1)*sizeof(struct test_range));
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].start=i;
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges].type=TEST_SIGNAL_OUT_OF_RANGE;
					for(n_not_saturated=0,i++;i<data->samples_per_channel && n_not_saturated<max_non_saturated_between;i++){
						if(range_min>=data->ch[ch].raw[i]){
							n_not_saturated=0;
						}
						else{
							n_not_saturated++;
						}
					}
					ch_ranges[ch].ranges[ch_ranges[ch].n_ranges++].end=i-n_not_saturated;
					if(i-n_not_saturated-ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].start>ignore_less_than_saturated){
						ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].type=TEST_SHORT_OUT_OF_RANGE;
					}
					else{
						total_bad+=ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].end-ch_ranges[ch].ranges[ch_ranges[ch].n_ranges-1].start;
						total_n_saturated++;
					}
				}
			}
			printf("Marked %lu/%lu saturated ranges (%.3lf %%) from ch %d\n",total_n_saturated,ch_ranges[ch].n_ranges-n_ranges_before,100.0*(double)(total_bad)/(double)(data->samples_per_channel),ch+1);
		}
	}
	return(0);
}

static int trig_analyze_ch_and_detect_no_signal_segments(struct data *data,  struct test_ch_info *ch_info, struct trig_helper *helper){
	struct conf_trig *conf=conf_trig_get();
	size_t i,j,k,n;
	int ch;
	size_t half_min_win=(size_t)(conf->min_rr*data->sample_freq/2.0+0.5),half_max_win=(size_t)(conf->max_rr*data->sample_freq/2.0+0.5);
	size_t total_no_signal,n_total_no_signal;
	struct data std_data,*energy;
	int not_good;
//	double min_snr=2.0, min_abs_signal = 20.0;
	double min_snr=2.0, min_abs_signal = 20.0;
	double noise;

	helper->n_steps=ceil((double)(data->samples_per_channel)/(double)(half_max_win));
	helper->step_len=half_max_win;

	// alloc data
	double *max_data=my_malloc(data->samples_per_channel*sizeof(double));
	double *max_max_data=my_malloc(helper->n_steps*sizeof(double));
	double *min_max_data=my_malloc(helper->n_steps*sizeof(double));
	double *snr=my_malloc(helper->n_steps*sizeof(double));
//	double *just_max_max_data=my_malloc(helper->n_steps*sizeof(double));
//	double *just_min_max_data=my_malloc(helper->n_steps*sizeof(double));

	trig_create_std_signal(data, &std_data);
	energy=&std_data;

	// loop all the channels

	for(ch=0;ch<energy->number_of_channels;ch++){
		total_no_signal=0;
		n_total_no_signal=0;

		// 1. define max values in moving half min_rr long windows
		for(i=0;i<energy->samples_per_channel;i++){
			j=MIN_ZERO_SUBS(i,half_min_win);
			max_data[i]=energy->ch[ch].raw[j++];
			for(;j<=i;j++){
				if(max_data[i]<energy->ch[ch].raw[j]){
					max_data[i]=energy->ch[ch].raw[j];
				}
			}
		}
		// 2. define min and max values of the previously defined max values during max_rr windows, and in half max_rr steps
		if(ch_info->ch_ranges==NULL){
			for(i=0,n=0,k=0;i<helper->n_steps;i++){
				j=(i<=1 ? 0 : (i-1)*half_max_win);
				max_max_data[i]=min_max_data[i]=max_data[j++];
				for(;j<=MIN(energy->samples_per_channel-1,(i+1)*half_max_win);j++){
					if(max_max_data[i]<max_data[j]){
						max_max_data[i]=max_data[j];
					}
					else if(min_max_data[i]>max_data[j]){
						min_max_data[i]=max_data[j];
					}
				}
			}
			n=helper->n_steps;
		}
		else{
			for(i=0,k=0;i<helper->n_steps;i++){
				j=(i<=1 ? 0 : (i-1)*half_max_win);
				for(;k<ch_info->ch_ranges[ch].n_ranges && ch_info->ch_ranges[ch].ranges[k].end<j;k++);
				if(k<ch_info->ch_ranges[ch].n_ranges && ch_info->ch_ranges[ch].ranges[k].start<(i+1)*half_max_win){ // bad range? -> skip
					continue;
				}
				max_max_data[i]=min_max_data[i]=max_data[j++];
				for(;j<=MIN(energy->samples_per_channel-1,(i+1)*half_max_win);j++){
					if(max_max_data[i]<max_data[j]){
						max_max_data[i]=max_data[j];
					}
					else if(min_max_data[i]>max_data[j]){
						min_max_data[i]=max_data[j];
					}
				}
			}
		}

		// 3. Detect and mark "No signal" segments
		not_good=0;
		for(i=0,n=0,k=0;i<helper->n_steps;i++){
			j=(i<=1 ? 0 : (i-1)*half_max_win);
			for(;k<ch_info->ch_ranges[ch].n_ranges && ch_info->ch_ranges[ch].ranges[k].end<j;k++);
			if(k<ch_info->ch_ranges[ch].n_ranges && ch_info->ch_ranges[ch].ranges[k].start<(i+1)*half_max_win){ // bad range? -> skip
				continue;
			}

			if(max_max_data[i]>min_snr*min_max_data[i] && max_max_data[i]-min_max_data[i]>min_abs_signal){ // is signal ok?
				if(not_good){ // should we end the TEST_NO_SIGNAL range?
					not_good=0;
					ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].end=(i)*half_max_win;
					total_no_signal+=ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].end-ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].start;
					ch_info->ch_ranges[ch].n_ranges++;
					n_total_no_signal++;
				}
//				just_max_max_data[n]=max_max_data[i];
//				just_min_max_data[n]=min_max_data[i];
				n++;
			}
			else if(!not_good){ // should we start new TEST_NO_SIGNAL range?
				not_good=1;
				ch_info->ch_ranges[ch].ranges=my_realloc(ch_info->ch_ranges[ch].ranges,(ch_info->ch_ranges[ch].n_ranges+1)*sizeof(struct test_range));
				ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].start=j;
				ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].type=TEST_NO_SIGNAL;
			}
		}
		if(not_good){ // should we (finally) end the TEST_NO_SIGNAL range?
			not_good=0;
			ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].end=energy->samples_per_channel-1;
			total_no_signal+=ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].end-ch_info->ch_ranges[ch].ranges[ch_info->ch_ranges[ch].n_ranges].start;
			ch_info->ch_ranges[ch].n_ranges++;
			n_total_no_signal++;
		}
/*		if(ch_info->ch_ranges[ch].n_ranges>1){
			qsort(ch_info->ch_ranges[ch].ranges,ch_info->ch_ranges[ch].n_ranges,sizeof(struct test_range),trig_ch_range_comparison);
		}*/
		printf("Found %lu NO_SIGNAL (%.3lf %%) from ch %d\n",n_total_no_signal,100.0*(double)(total_no_signal)/(double)(energy->samples_per_channel),ch+1);

		// 4. Estimate signal amplitude and signal-to-noise ratio

		if(!ch_info->amplitude_normalization){
			ch_info->amplitude_normalization=my_malloc(energy->number_of_channels*sizeof(double));
		}
		if(!ch_info->ch_goodness){
			ch_info->ch_goodness=my_malloc(energy->number_of_channels*sizeof(double));
		}

		if(helper->n_steps==1){
			if(min_max_data[0]>0){
				ch_info->ch_goodness[ch]=max_max_data[0]/min_max_data[0];
			}
			else{
				ch_info->ch_goodness[ch]=FLT_MAX;
			}
			ch_info->amplitude_normalization[ch]=1.0/max_max_data[0];
		}
		else{
			for(n=0;n<helper->n_steps-1;n++){
				noise=(min_max_data[n]+min_max_data[n+1]);
				if(noise>0.0){
					snr[n]=(max_max_data[n]+max_max_data[n+1])/noise; // Notice: we store SNR estimate temporarily to ch_goodness. Final ch_goodness value will be defined later!
				}
				else{
					snr[n]=FLT_MAX;
				}
			}
			ch_info->ch_goodness[ch]=array_kth_biggest(snr,helper->n_steps-1,0.5*(helper->n_steps-1)); // Notice: we store SNR estimate temporarily to ch_goodness. Final ch_goodness value will be defined later!
			if(ch_info->ch_goodness[ch]>1.01){
				ch_info->ch_goodness[ch]=log2(ch_info->ch_goodness[ch]);
			}
			else{
				ch_info->ch_goodness[ch]=log2(1.01);
			}
			ch_info->amplitude_normalization[ch]=1.0/array_kth_biggest(max_max_data,helper->n_steps-1,0.5*(helper->n_steps-1));
		}
	}

	free_data(energy);
	my_free(max_data);
	my_free(max_max_data);
	my_free(min_max_data);
	my_free(snr);
//	my_free(just_max_max_data);
//	my_free(just_min_max_data);
	return(0);
}

int trig_define_amplitude_normalization(struct test_ch_info *ch_info, struct test_events *events, struct test_clusters *cluster_set, struct data *data){
	int ch;
	double *amplitudes=my_malloc(events->n_events*sizeof(double));
	size_t i,j,k,n;
	ssize_t start,end;
	int half_win=0.075*data->sample_freq;
	double min,max;

	if(!ch_info->amplitude_normalization){
		ch_info->amplitude_normalization=my_malloc(data->number_of_channels*sizeof(double));
	}

	for(ch=0;ch<data->number_of_channels;ch++){
		n=0;
		for(k=0;k<cluster_set->n_clusters;k++){
			if(cluster_set->clusters[k].type!=TEST_NORMAL_BEAT){
				continue;
			}
			for(i=0;i<cluster_set->clusters[k].n_beats;i++){
				start=(ssize_t)(events->events[cluster_set->clusters[k].inds[i]].sample)-half_win;
				if(start<0){
					start=0;
				}
				end=(ssize_t)(events->events[cluster_set->clusters[k].inds[i]].sample)+half_win;
				if(end>=data->samples_per_channel){
					end=data->samples_per_channel-1;
				}
				min=max=data->ch[ch].raw[start];
				for(j=start+1;j<end;j++){
					if(max<data->ch[ch].raw[j]){
						max=data->ch[ch].raw[j];
					}
					else if(min>data->ch[ch].raw[j]){
						min=data->ch[ch].raw[j];
					}
				}
				amplitudes[n++]=max-min;
			}
		}
		if(n==0){
			printf("No N templates for amplitude normalization\n");
			ch_info->amplitude_normalization[ch]=FLT_MAX;
		}
		ch_info->amplitude_normalization[ch]=1.0/array_kth_biggest(amplitudes,n,0.5*n);
	}

	my_free(amplitudes);
	return(0);
}

int trig_define_ok_chs_for_events(struct test_ch_info *ch_info, struct test_events *events,struct trig_helper *helper, int event_len_after, int event_len_before, double freq){

	helper->ch_badness_for_events=NULL;
	// SKIP REST: CinC2017 version is simplified for one ch only -- no support for this
	return(0);
}



int trig_sort_cluster_inds(struct test_clusters *cluster_set){
	int k;
	for(k=0;k<cluster_set->n_clusters;k++){
		array_ssort(cluster_set->clusters[k].inds,cluster_set->clusters[k].n_beats);
	}

	return(0);
}

int trig_create_event_clusters(struct data *data, double ***dataset, struct test_events *events, struct test_clusters *cluster_set,  struct test_ch_info *ch_info, struct trig_helper *helper){
	// just process all in one run &
	// SKIP REST: CinC2017 version is simplified for one ch only
	size_t i;
	size_t *process_these=my_malloc(sizeof(size_t)*events->n_events);
	for(i=0;i<events->n_events;i++){
		process_these[i]=i;
	}
	create_cinc2017_event_clusters(data, ch_info->ch_goodness, events, dataset , process_these,events->n_events, cluster_set );
	my_free(process_these);
	return(0);
}


int trig_identify_main_normal_clusters(struct test_events *events, struct test_ch_info *ch_info, struct test_clusters *cluster_set, double **correlations_mat, double **short_correlations_mat, double freq){
	struct conf_cluster *cluster_conf=conf_cluster_get();
	int i;
	int normal_main=0;
	size_t n_similar,n_similar2,n_total;
	int *cluster_offsets=my_malloc(cluster_set->n_clusters*sizeof(int)),*cluster_offsets2;

	// 1. Identify main clusters of normal beats

	// 1.a. start from the biggest group and count the similar and not-similar clusters

	correlations_mat[0][0]=1.0;
	cluster_offsets[0]=0;
	tests_get_cluster_correlations(ch_info->ch_goodness, 0,1, cluster_set, cluster_offsets, correlations_mat[0], 0);
	n_similar=n_total=cluster_set->clusters[0].n_beats;
	for(i=1;i<cluster_set->n_clusters;i++){
		n_total+=cluster_set->clusters[i].n_beats;
		if(correlations_mat[0][i]>=cluster_conf->similarity_limit_for_clusters){
			n_similar+=cluster_set->clusters[i].n_beats;
		}
	}
	if(2*n_similar>n_total){  // Did we find so dominant event that we can label it as "N" right away?
		normal_main=0;
	}
	else{ // 1.b. ..if not, then we check also the biggest not similar cluster
		cluster_offsets2=my_malloc(cluster_set->n_clusters*sizeof(int));
		for(normal_main=1;correlations_mat[0][normal_main]>=cluster_conf->similarity_limit_for_clusters;normal_main++); // select the biggest 'not similar' cluster
		cluster_offsets2[0]=0;
		tests_get_cluster_correlations(ch_info->ch_goodness, normal_main,1, cluster_set, cluster_offsets2, correlations_mat[normal_main], 0);
		n_similar2=0;
		for(i=1;i<cluster_set->n_clusters;i++){
		  if(correlations_mat[normal_main][i]>cluster_conf->similarity_limit_for_clusters){
				n_similar2+=cluster_set->clusters[i].n_beats;
			}
		}

		if(2*n_similar2>n_total){  // Did we find so dominant event now that we can label it as "N" right away?

		}
		else{ // 1.c. .. then we have to do some (simplified heavily for cinc2017
			if(n_similar>1.25*n_similar2 || cluster_set->clusters[0].markers[0].qrs_width<cluster_set->clusters[normal_main].markers[0].qrs_width){
				normal_main=0;
			}
			printf("No clear main N cluster. Selecting cluster %d\n",normal_main);
		}
		if(normal_main>0){
			my_free(cluster_offsets);
			cluster_offsets=cluster_offsets2;
		}
		else{
			my_free(cluster_offsets2);
		}
	}

	// 2. Set cluster_offsets of the correlating (correlation > 'corr_limit_for_cluster_offset') clusters

	for(i=0;i<cluster_set->n_clusters;i++){
		if(correlations_mat[normal_main][i]>cluster_conf->similarity_limit_for_cluster_offsets){
			cluster_set->clusters[i].cluster_offset=cluster_offsets[i];
		}
	}

	// 3. Set the cluster type labels for all cluster close enough to just selected 'clusters[normal_main]'

	for(i=0;i<cluster_set->n_clusters;i++){
		if(correlations_mat[normal_main][i]>cluster_conf->similarity_limit_for_clusters){
			cluster_set->clusters[i].type=TEST_NORMAL_BEAT;
		}
	}

	// SKIP THIS in short one ch CINC2017 measurements
	//4. Go through rest of the clusters and mark those that have more than half of the neighboring beats similar, and that haven't have too high heart rate (to exclude VTs)
	/*for(i=0;i<cluster_set->n_clusters;i++){
		if(cluster_set->clusters[i].type!=TEST_BEAT_UNDEFINED){
			continue;
		}
		tests_get_cluster_tachyratio(events, cluster_set->clusters+i, cluster_set->original_freq, &(cluster_set->clusters[i].annotation_markers.tachy_ratio),&(cluster_set->clusters[i].annotation_markers.extreme_tachy_ratio));
		if(cluster_set->clusters[i].annotation_markers.close_by_similarity>0.5 && cluster_set->clusters[i].annotation_markers.tachy_ratio<0.5){
			cluster_set->clusters[i].type=TEST_NORMAL_BEAT;
		}
	}*/

	// 5. Set the cluster type labels for all cluster close enough to just selected 'clusters[normal_main]',and that haven't have too high heart rate (to exclude VTs)

	tests_get_one_ch_cluster_correlations(normal_main, 0,cluster_set, cluster_offsets, short_correlations_mat[normal_main], 1, 0.050, 0.050, 0.100);
	for(i=0;i<cluster_set->n_clusters;i++){
		if(cluster_set->clusters[i].type!=TEST_BEAT_UNDEFINED){
			continue;
		}
		if(short_correlations_mat[normal_main][i]>cluster_conf->similarity_limit_for_clusters){
			tests_get_cluster_tachyratio(events, cluster_set->clusters+i, cluster_set->original_freq, &(cluster_set->clusters[i].annotation_markers.tachy_ratio),&(cluster_set->clusters[i].annotation_markers.extreme_tachy_ratio));
			if(cluster_set->clusters[i].annotation_markers.tachy_ratio<0.5){
				cluster_set->clusters[i].type=TEST_NORMAL_BEAT;
			}
		}
	}

	// free
	my_free(cluster_offsets);
	return(0);
}

int trig_identify_evident_artifact_clusters(struct test_clusters *cluster_set){
// ***** conf *****
	double slope_limit=400.0; // TODO: limit values should be defined somewhere else..
	double initial_slope_limit=0.200;
	int ch=0; // CinC2017 -- modified for one ch only!!!

	int i,k;
	double max;

	for(i=0;i<cluster_set->n_clusters;i++){
		for(k=0;k<cluster_set->clusters[i].n_templates;k++){
			if(cluster_set->clusters[i].type==TEST_NORMAL_BEAT && cluster_set->clusters[i].markers[k].max_qrs_slope[ch]<0.5*slope_limit){
				slope_limit = cluster_set->clusters[i].markers[k].max_qrs_slope[ch];		// special limit scale down for low amplitude data
			}
		}
	}

	for(i=0;i<cluster_set->n_clusters;i++){
		if(cluster_set->clusters[i].type!=TEST_BEAT_UNDEFINED){
			continue;
		}
		// low max QRS slope?
		max=0.0;
		for(k=0;k<cluster_set->clusters[i].n_templates;k++){
			if(max<cluster_set->clusters[i].markers[k].max_qrs_slope[ch]){
				max=cluster_set->clusters[i].markers[k].max_qrs_slope[ch];
			}
		}
		if(max<slope_limit){
			cluster_set->clusters[i].type=TEST_ARTIFACT;
			continue;
		}

		// too long initial QRS raise time (initial QRS slope length)?
		max=0.0;
		for(k=0;k<cluster_set->clusters[i].n_templates;k++){
			if((cluster_set->clusters[i].bad_chs[k]==NULL || cluster_set->clusters[i].bad_chs[k][ch]<TEST_EVENT_TOTALLY_BAD) && max<cluster_set->clusters[i].markers[k].initial_qrs_slope_length[ch]){
				max=cluster_set->clusters[i].markers[k].initial_qrs_slope_length[ch];
			}
		}
		if(max>initial_slope_limit){
			cluster_set->clusters[i].type=TEST_ARTIFACT;
			continue;
		}
	}
	return(0);
}

static inline void trig_set_event_types_from_cluster(struct test_cluster *cluster, struct test_events *events){
	struct conf_cluster *cluster_conf=conf_cluster_get();
	size_t i,ind;
//	double offset;
//	int move_i;
	for(i=0;i<cluster->n_beats;i++){
		ind=cluster->inds[i];
		events->events[ind].type=cluster->type;
		events->events[ind].sub_sample=(double)(cluster->cluster_offset+cluster->sample_offsets[i])/(double)(cluster_conf->frequency_mult);
		// TODO: use sample offset info!!!!
//		move_i=(int)(offset<0 ? offset-0.5 : offset+0.5)
//		events->events[ind].sample-=(int)(offset<0 ? offset-0.5 : offset+0.5);
//		cluster->sample_offsets[i]-=(int)(offset<0 ? offset-0.5 : offset+0.5)*cluster_conf->frequency_mult;
//		events->events[ind].sample_fine_tune=(float)(cluster_set->cluster->sample_offsets%frequency_mult)/(float)(frequency_mult);
	}
}


static inline void trig_set_event_types_from_cluster_set(struct test_clusters *cluster_set, enum test_trigger_type  type, struct test_events *events){
	int k;
	for(k=0;k<cluster_set->n_clusters;k++){
		if(cluster_set->clusters[k].type==type){
			trig_set_event_types_from_cluster(cluster_set->clusters+k,events);
		}
	}
	return;
}


/*int trig_define_marker_ranges_for_clusters(struct test_clusters *cluster_set, struct data *active_data, struct test_events *events, struct test_cluster_conf *cluster_conf, struct test_cluster_markers n_range[2], struct test_cluster_markers v_range[2]){
	size_t i,j,main_n;
	int ch;
	for(i=0;i<cluster_set->n_clusters && cluster_set->clusters[i].type!=TEST_NORMAL_BEAT;i++);
	if(i==cluster_set->n_clusters){ // No normal beats found. Return error (there has to be at least one normal cluster defined before calling this function)
		return(1);
	}
	main_n=i;
	// alloc and initialize the n_ranges with the values from clusters[main_n]
	for(i=0;i<2;i++){
		n_range[i].qrs_width=cluster_set->clusters[main_n].markers[0].qrs_width;
		n_range[i].initial_qrs_slope_length=my_malloc(cluster_set->nch*sizeof(double));
		memcpy(n_range[i].initial_qrs_slope_length,cluster_set->clusters[main_n].markers[0].initial_qrs_slope_length,cluster_set->nch*sizeof(double));
		n_range[i].max_qrs_slope=my_malloc(cluster_set->nch*sizeof(double));
		memcpy(n_range[i].max_qrs_slope,cluster_set->clusters[main_n].markers[0].max_qrs_slope,cluster_set->nch*sizeof(double));
		n_range[i].qrs_power=my_malloc(cluster_set->nch*sizeof(double));
		memcpy(n_range[i].qrs_power,cluster_set->clusters[main_n].markers[0].qrs_power,cluster_set->nch*sizeof(double));
		n_range[i].n_qrs_waves=my_malloc(cluster_set->nch*sizeof(int));
		memcpy(n_range[i].n_qrs_waves,cluster_set->clusters[main_n].markers[0].n_qrs_waves,cluster_set->nch*sizeof(int));

		v_range[i].initial_qrs_slope_length=my_malloc(cluster_set->nch*sizeof(double));
		v_range[i].max_qrs_slope=my_malloc(cluster_set->nch*sizeof(double));
		v_range[i].qrs_power=my_malloc(cluster_set->nch*sizeof(double));
		v_range[i].n_qrs_waves=my_malloc(cluster_set->nch*sizeof(double));
	}

	// find the existing range of values in the TEST_NORMAL_BEAT type clusters
	for(i=main_n;i<cluster_set->n_clusters;i++){
		for(j=0;j<cluster_set->clusters[i].n_templates;j++){
			if(n_range[0].qrs_width>cluster_set->clusters[i].markers[j].qrs_width){
				n_range[0].qrs_width=cluster_set->clusters[i].markers[j].qrs_width;
			}
			else if(n_range[1].qrs_width<cluster_set->clusters[i].markers[j].qrs_width){
				n_range[1].qrs_width=cluster_set->clusters[i].markers[j].qrs_width;
			}
			for(ch=0;ch<cluster_set->nch;ch++){
				if(cluster_set->clusters[i].ok_chs[j][ch]){


				}
			}
		}
	}
	return(0);
}
*/
/*
	    Features for "V":
	     * High energy waveform on multiple channels
	     * Followed by T-wave
	     * preceding + next RR close to median RR
	     * Several similar complexes
	     * If RR >0.8 next RR and P template match before the wave -> "F"
	     * if timing inside estimated T-wave of the preceding beat -> V on T
		Features for multiple "V" (VT):
		 * Multiple succeeding waveforms in high bpm
		 * waveform duration > 50 ms
		Features for A:
		 * Other channel(s) ok, but event still visible only on one channel
		 * Huge baseline drift in original signal: signal doesn't return to original lvl
		 * Long first slope -> T-wave
		 * Similar to T-template -> T-wave
		 * Loner / semi-loner with abnormal duration:
		 ** 	Very short (<50 ms)
		 ** 	very long / no clear difference to surrounding noise

	*/


int trig_classify_cinc2017_clusters_and_events(struct test_clusters *cluster_set, struct test_events *events, ssize_t *event_cluster_ids, double *ratio_to_nn,double *median_nn,double **correlations_mat, double **short_correlations_mat){
	int i,j,prev,prev2,next;
	double prev_rr,rr,next_rr;
	int base_normal=0;
//	int ch=0;
	int n_classified=0;
	int modified=0;
	struct conf_cluster *cluster_conf=conf_cluster_get();

	for(j=0;j<cluster_set->n_clusters;j++){
		if(cluster_set->clusters[j].type==TEST_NORMAL_BEAT){
			base_normal=j;
			break;
		}
	}

	// 1. first all the clusters..
	for(j=0;j<cluster_set->n_clusters;j++){
		if(cluster_set->clusters[j].type==TEST_BEAT_UNDEFINED){ // Classify only the UNDEFINED (all the others have been classified already..)
		// 1.1. quite similar to NORMAL, and "not early" -> NORMAL
			if(cluster_set->clusters[j].annotation_markers.similarity_to_normals>cluster_conf->minimum_similarity_for_normals &&
					cluster_set->clusters[j].annotation_markers.event_timing > 0.85){
				cluster_set->clusters[j].type=TEST_NORMAL_BEAT;
				for(i=0;i<cluster_set->clusters[j].n_beats;i++,n_classified++){
					events->events[cluster_set->clusters[j].inds[i]].type=TEST_NORMAL_BEAT;
				}
			}

			// 1.2. not more than 500 ms from the previous beat, and similar to template T-wave -> "T wave" ARTIFACT

			//rhythm_normality_for_events(events,ratio_to_nn, median_nn, 0.850*active_data->sample_freq,1); // update the ratio_to_nn and median_nn values


			// 1.3. More than one similar, wider (QRS duration big / bigger than NORMAL) and early (RR shorter than expected) -> VENTRICULAR / wide (aberrant supraventricular) ectopic beat
			if(cluster_set->clusters[j].n_beats>1 && cluster_set->clusters[j].markers[0].qrs_width>0.100 &&
					cluster_set->clusters[j].markers[0].qrs_width > 1.05 * cluster_set->clusters[base_normal].markers[0].qrs_width &&
					cluster_set->clusters[j].annotation_markers.event_timing < 0.9){
				cluster_set->clusters[j].type=TEST_VENTRICULAR;
				for(i=0;i<cluster_set->clusters[j].n_beats;i++,n_classified++){
					events->events[cluster_set->clusters[j].inds[i]].type=TEST_VENTRICULAR;
				}
			}
		}
	}

	// Then the rest of the beats beat-by-beat
	modified=0;
	for(i=0;i<events->n_events;i++){
		if(events->events[i].type==TEST_BEAT_UNDEFINED){
			// 4. Unique and too close to the following NORMAL beat (too short succeeding RR time)-> ARTIFACT
			if(event_cluster_ids[i]<0){
				for(next=i+1;next<events->n_events && events->events[next].type==TEST_ARTIFACT;next++);
				if(next<events->n_events && events->events[next].type==TEST_NORMAL_BEAT && events->events[next].sample-events->events[i].sample<0.300*cluster_set->original_freq){
					events->events[i].type=TEST_ARTIFACT;
					n_classified++;
					modified++;
					if(event_cluster_ids[i]>=0){
						if(cluster_set->clusters[event_cluster_ids[i]].n_beats<2){
							cluster_set->clusters[event_cluster_ids[i]].type=TEST_ARTIFACT;
						}
					}
				}
			}
		}
	}
	if(modified){
		rhythm_normality_for_events(events,ratio_to_nn, median_nn, 0.850*cluster_set->original_freq,1); // update the ratio_to_nn and median_nn values
	}

	// 5. Early beat after NORMAL beat -> SUPRAVENTRICULAR or VENTRICULAR ectopic beat
	for(j=0,i=2;i<events->n_events-1;i++){
		if(events->events[i].type==TEST_VENTRICULAR || ratio_to_nn[i]>0.85){
			continue;
		}
		for(prev=i-1;prev>=0 && events->events[prev].type==TEST_ARTIFACT;prev--);
		for(prev2=prev-1;prev2>0 && events->events[prev2].type==TEST_ARTIFACT;prev2--);
		for(next=i+1;next<events->n_events && events->events[next].type==TEST_ARTIFACT;next++);
		if(next>=events->n_events || prev<0 || events->events[prev].type!=TEST_NORMAL_BEAT){
			continue;
		}
		for(;j<events->n_ranges && (events->ranges[j].type!=TEST_UNREADABLE || events->ranges[j].end<events->events[prev2].sample);j++);
		if(j<events->n_ranges && events->ranges[j].start<events->events[next].sample){ // too close to UNREADABLE / SHUTDOWN -- can't define all the RR values
			continue;
		}
		prev_rr=(events->events[prev].sample-events->events[prev].sub_sample)-(events->events[prev2].sample-events->events[prev2].sub_sample);
		rr=(events->events[i].sample-events->events[i].sub_sample)-(events->events[prev].sample-events->events[prev].sub_sample);
		next_rr=(events->events[next].sample-events->events[next].sub_sample)-(events->events[i].sample-events->events[i].sub_sample);
		if(rr<0.7*prev_rr || (rr<0.8*prev_rr && rr>1.2*next_rr)){ // early?
			if(event_cluster_ids[i]>=0){
				if(cluster_set->clusters[event_cluster_ids[i]].annotation_markers.similarity_to_normals>cluster_conf->minimum_similarity_for_normals){
					events->events[i].type=TEST_SUPRAVENTRICULAR;
					n_classified++;
				}
				else if(cluster_set->clusters[event_cluster_ids[i]].markers[0].qrs_width>0.090){// if(cluster_set->clusters[event_cluster_ids[i]].markers[0].qrs_width > 1.05 * cluster_set->clusters[base_normal].markers[0].qrs_width){
					events->events[i].type=TEST_VENTRICULAR;
					cluster_set->clusters[event_cluster_ids[i]].type=TEST_VENTRICULAR;
					n_classified++;
				}
			}
		}
	}

	return(n_classified);
}


static inline int trig_ignore_supraventriculars_in_range(enum test_range_type type, int ignore_in_af_range){
	switch(type){
	case TEST_UNREADABLE:
		return(1);
	case TEST_AF:
	case TEST_AFL:
	case TEST_AVJ:
		return(ignore_in_af_range);
	default:
		return(0);
	}
}


int trig_pick_supraventriculars_from_normals(struct test_events *events, double sfreq){
// Very simple version.. TODO: use median_nn value too..
	size_t i,j,n,k=0;
	size_t last[2];  // the last two not TEST_ARTIFACT events
	size_t rr;
	double *ratio_to_nn=my_malloc(events->n_events*sizeof(double));
	double	*median_nn=my_malloc(events->n_events*sizeof(double));
	struct conf_rhythm *rhythm_conf=conf_rhythm_get();
	double tachy_rr_limit=60.0/rhythm_conf->tachy_limit*sfreq;
	double extreme_tachy_rr_limit=60.0/rhythm_conf->extreme_tachy_limit*sfreq;
	int ignore_in_af_range=!(rhythm_conf->set_supraventriculars_during_af);

	rhythm_normality_for_events(events,ratio_to_nn, median_nn, 0.850*sfreq,1); // TODO: we should use previously calculated numbers rather than recalculating them again..

	for(i=0,n=0;i<events->n_events && n<2;i++){  // continue until we have two beats before so we can estimate the rr time change
		if(events->events[i].type!=TEST_ARTIFACT){
			last[n++]=i;
		}
	}
	for(;i<events->n_events;i++){
		for(;k<events->n_ranges && (!trig_ignore_supraventriculars_in_range(TEST_UNREADABLE,ignore_in_af_range)  || events->ranges[k].end<events->events[i].sample);k++);
		if(k<events->n_ranges && events->ranges[k].start<events->events[i].sample){ // skip bad ranges
			for(;i<events->n_events && events->ranges[k].end>events->events[i].sample;i++);
			for(n=0;i<events->n_events && n<2;i++){
				if(events->events[i].type!=TEST_ARTIFACT){
					last[n++]=i;
				}
			}
		}
		if(events->events[i].type==TEST_ARTIFACT){
			continue;
		}
		if(events->events[i].type==TEST_NORMAL_BEAT){
			rr=events->events[i].sample-events->events[last[1]].sample;
			if(events->events[last[0]].type==TEST_NORMAL_BEAT && events->events[last[1]].type==TEST_NORMAL_BEAT){  // do we have three normal beats in row?
				if(ratio_to_nn[i]<0.8){ // are we at least 20 % below the average (median) heart rate?
					if(rr<0.7*(events->events[last[1]].sample-events->events[last[0]].sample)){  // did we have over 30 % drop in RR?
						events->events[i].type=TEST_SUPRAVENTRICULAR;
					}
					else{
						for(j=i+1;j<events->n_events && events->events[j].type==TEST_ARTIFACT;j++);
						if(j<events->n_events && rr<0.80*(events->events[last[1]].sample-events->events[last[0]].sample) && rr<0.80*(events->events[j].sample-events->events[i].sample)){ // did we have at least 20 % drop in current AND will we have at least 20 % increase in next RR?
							events->events[i].type=TEST_SUPRAVENTRICULAR;
						}
					}
				}
			}
			else if(events->events[last[1]].type==TEST_SUPRAVENTRICULAR){  // was the last beat supraventricular?
				if(rr<extreme_tachy_rr_limit || (rr<1.05*(events->events[last[1]].sample-events->events[last[0]].sample) && rr<tachy_rr_limit)){  // is it extreme tachy or tachy with the RR the same (5 % margin) or less than the prev RR?
					events->events[i].type=TEST_SUPRAVENTRICULAR;
				}
			}
		}
		last[0]=last[1];
		last[1]=i;
	}
	my_free(median_nn);
	my_free(ratio_to_nn);
	return(0);
}

int trig_drop_artifact_events(struct test_events *events){
	size_t i,n;
	for(n=0,i=0;i<events->n_events;i++){
		if(events->events[i].type==TEST_ARTIFACT){
			continue;
		}
		if(n==i){
			n++;
		}
		else{
			events->events[n++]=events->events[i];
		}
	}
	events->n_events=n;
	return(0);
}

int trig_copy_events(struct test_events *to, struct test_events *from){
	to->n_events=from->n_events;
	if(from->n_events>0){
		to->events=my_malloc(from->n_events*sizeof(struct test_event));
		memcpy(to->events,from->events,from->n_events*sizeof(struct test_event));
	}
	else{
		to->events=NULL;
	}
	to->n_ranges=from->n_ranges;
	if(from->n_ranges>0){
		to->ranges=my_malloc(from->n_ranges*sizeof(struct test_range));
		memcpy(to->ranges,from->ranges,from->n_ranges*sizeof(struct test_range));
	}
	else{
		to->ranges=NULL;
	}
	return(0);
}

int trig_copy_ok_events(struct test_events *to, struct test_events *from){
	size_t i,n;
	for(n=0,i=0;i<from->n_events;i++){
		if(from->events[i].type==TEST_ARTIFACT){
			continue;
		}
		else{
			to->events[n++]=from->events[i];
		}
		to->n_events=n;
	}
	if(from->n_ranges>0){
		to->ranges=my_malloc(from->n_ranges*sizeof(struct test_range));
		memcpy(to->ranges,from->ranges,from->n_ranges*sizeof(struct test_range));
	}
	else{
		to->ranges=NULL;
	}
	return(0);
}

//int trig_close_by_similarity(struct test_clusters *cluster_set, )

int trig_estimate_event_rate_normality(struct test_events *events, double **normality_){
	// TODO define 'typical RR range' and check for 30 % deviations from that
	size_t last_event,i,k;
	double *nn=my_malloc(events->n_events*sizeof(double));
	double *normality=my_malloc(events->n_events*sizeof(double));
	double min_max_allowed_dev=0.60,max_dev;
	double min_increase=1.10;
	double rate_adaptation=3.0;
	double update_step=0.10;
	double ave_diff;
	double diff_rate;

	// create nn timeseries for all the events
	nn[0]=-1; // not defined
	for(last_event=0,k=0,i=1;i<events->n_events;i++){
		for(;k<events->n_ranges && (events->ranges[k].type!=TEST_UNREADABLE || events->ranges[k].end<events->events[last_event].sample);k++);
		if((k<events->n_ranges && events->ranges[k].start<events->events[i].sample) || events->events[i].type==TEST_ARTIFACT){ // is there bad range after the last event, or is the current event TEST_ARTIFACT, which should be skipped?
			nn[i]=-1; // nn undefined
		}
		else if(events->events[last_event].type!=TEST_NORMAL_BEAT || events->events[i].type!=TEST_NORMAL_BEAT){ // is either previous or the next event 'not normal'?
			nn[i]=nn[last_event]; // keep the last nn
		}
		else{
			nn[i]=events->events[i].sample-events->events[last_event].sample;
		}
		if(events->events[i].type!=TEST_ARTIFACT){ // update current i to the next last event index
			last_event=i;
		}
	}
	ave_diff=0.0;
	for(last_event=0,i=1;i<events->n_events-1;i++){

		if(events->events[last_event].type==TEST_NORMAL_BEAT && events->events[i].type==TEST_NORMAL_BEAT){
			ave_diff=(1.0-update_step)*ave_diff+update_step*fabs(nn[i]-nn[last_event]); // TODO: abs_diff to abs_diff over several beats
			diff_rate=(2.0*ave_diff)/(nn[i]+nn[last_event]);
			max_dev=MAX(min_max_allowed_dev,(1.0-rate_adaptation*diff_rate));
			if(nn[i]>min_increase*nn[last_event] && nn[i]<max_dev*nn[i+1]){
				events->events[i].type=TEST_SUPRAVENTRICULAR;
			}
		}
		if(events->events[i].type!=TEST_ARTIFACT){ // update current i to the next_last event index (we ignore the TEST_ARTIFACTs)
			last_event=i;
		}
	}
	free(nn);
	*normality_=normality;
	return(0);
}



// partial trig_do for getting just goodness value when reanalyzing clusters
int trig_define_ch_goodness(struct data *original_data, char *conf_file, struct test_ch_info *ch_info){
	struct conf_trig *trig_conf=conf_trig_get();
	struct data filtered_data;
	struct data *active_data;
	struct trig_helper helper;

//	size_t ind_offset=0;

//	return(0);

	helper.nch=original_data->number_of_channels;
	helper.ch_badness_for_events=NULL;

	// 1. Do the band pass filtering (optional)
	if(trig_conf->bp_filter.type!=NOFILTER){
		do_bp_filter(&(trig_conf->bp_filter),&filtered_data,original_data);
		active_data=&filtered_data;
	}
	else{
		active_data=original_data;
	}

	// 3. Detect / estimate artifacts and noise


	// 3.1. Detect saturated signals
	trig_detect_saturated_segments(original_data,ch_info->ch_ranges);

	// 3.2. Estimate 'no signal' segments

//		trig_detect_no_signal_segments(original_data,ch_ranges,&trig_conf);
	trig_analyze_ch_and_detect_no_signal_segments(original_data, ch_info, &helper);

	// 3.3. Estimate 'the noisy' segments

	trig_define_notch_frequencies(active_data,NULL,ch_info->ch_ranges);


	// 3.3. Sort and combine noise / artifact ranges based on their time
	trig_refine_ch_ranges(ch_info->ch_ranges,original_data);

	// 3.4. Define goodness values for channels
	trig_estimate_ch_goodness(active_data, NULL, ch_info);

	trig_free_helper(&helper);
	if(trig_conf->bp_filter.type!=NOFILTER){
		free_data(&filtered_data);
	}

	return(0);
}

int trig_create_clusters(struct data *active_data, struct test_events *events, struct test_ch_info *ch_info, struct trig_helper *helper, struct test_clusters *cluster_set){
	double ***dataset=NULL;
	struct conf_cluster *cluster_conf=conf_cluster_get();
	int template_move=cluster_conf->move*active_data->sample_freq+0.5;
	int	beat_len_before=(cluster_conf->win_before*active_data->sample_freq+0.5);
	int	beat_len_after=cluster_conf->win_after*active_data->sample_freq+0.5;

	if(cluster_set->n_clusters<0){ // initialize clusters?
		cluster_set->n_clusters=0;
		cluster_set->clusters=NULL;
		cluster_set->original_freq=active_data->sample_freq;
		cluster_set->frequency_mult=cluster_conf->frequency_mult;
		cluster_set->nch=active_data->number_of_channels;
		cluster_set->template_offset=((int)(cluster_conf->ave_win_before*active_data->sample_freq+0.5)+template_move/2)*cluster_conf->frequency_mult;
		cluster_set->template_len=((int)(cluster_conf->ave_win_before*active_data->sample_freq+0.5)+(int)(cluster_conf->ave_win_after*active_data->sample_freq+0.5)+template_move+1)*cluster_conf->frequency_mult;
		cluster_set->template_in_sets_len=((int)(cluster_conf->win_before*active_data->sample_freq+0.5)+(int)(cluster_conf->win_after*active_data->sample_freq+0.5)+1);
		cluster_set->n_move=template_move*cluster_conf->frequency_mult;
	}

		// 5.2. Create (and alloc) the whitened data for all the events
	tests_create_whitened_dataset2(active_data, events, beat_len_before, beat_len_after, &dataset);

		// 5.3. Create the clusters
	trig_create_event_clusters(active_data, dataset, events, cluster_set, ch_info,  helper);

	matrix_free_d3(dataset);
	return(0);
}

int trig_create_cinc2017_clusters(struct data *active_data, struct test_events *events, struct test_ch_info *ch_info, struct trig_helper *helper, struct test_clusters *cluster_set){
	return(0);
}

//#define TRIG_PRINT_NEW_EVENTS
static int trig_redefine_events(struct trig_missing_events *missing_events, struct data *active_data,struct data *energy_data,struct test_ch_info *ch_info, struct trig_helper *helper, struct test_events *events,struct test_event_infos *event_infos,struct test_clusters *cluster_set, double ***correlations_mat,double ***short_correlations_mat){
	struct conf_trig *trig_conf=conf_trig_get();
	struct conf_cluster *cluster_conf=conf_cluster_get();

	struct test_events new_events,not_clustered_events;
	int *cluster_n;
	ssize_t i,j,n;
	size_t start,end;
	size_t found;
	size_t n_alloced;
	int min_n_beats=5;
	int corr_win_before=cluster_conf->win_before * active_data->sample_freq;
	int margin=trig_conf->min_rr * active_data->sample_freq;
	int artifact_range = 0.050 * active_data->sample_freq; // TODO: setup as trig parameter
	double corr;

//	trig_test_events(events);

	// 1. Initialize
	n_alloced=missing_events->n_missing;
	new_events.events=my_malloc(n_alloced*sizeof(struct test_event));
	new_events.n_events=0;
	not_clustered_events.events=my_malloc(n_alloced*sizeof(struct test_event));
	not_clustered_events.n_events=0;
	cluster_n=my_malloc(n_alloced*sizeof(int));

	// 2. Find new events
	for(j=0,i=0;i<missing_events->n_missing;i++){
		//  2.1. Process the range of missing events in expected_rr long segments
		start=missing_events->missing_range[i].start;
		while(start<missing_events->missing_range[i].end){
			found=0;
			// 2.1.1. Define the range (start & end) for looking for new event
			if(start+2.3*missing_events->missing_range[i].expected_rr<missing_events->missing_range[i].end){ // Do we expect there to be more than one event to find?
				end=start+1.5*missing_events->missing_range[i].expected_rr;
			}
			else{
				end=missing_events->missing_range[i].end-margin;
			}
			// 2.1.2 add some margins, and check that we stay in allowed range
			start+=margin;
			if(start<corr_win_before){
				start=corr_win_before;
			}
			if(end>=active_data->samples_per_channel-cluster_set->template_in_sets_len){
				end=active_data->samples_per_channel-cluster_set->template_in_sets_len;
			}

			if(new_events.n_events>=n_alloced){
				n_alloced*=2;
				new_events.events=my_realloc(new_events.events, n_alloced*sizeof(struct test_event));
				not_clustered_events.events=my_realloc(not_clustered_events.events, n_alloced*sizeof(struct test_event));
				cluster_n=my_realloc(cluster_n,n_alloced*sizeof(int));

			}

			// 2.22. Check if there is normal (multichannel) or one channel free (not restricted by prev amplitude based trig) match to existing cluster templates. If there is, add new event!

			tests_free_cluster_match(active_data, start-corr_win_before, end-corr_win_before, cluster_set, ch_info->ch_goodness, 0/*some_min_beats*/, &(new_events.events[new_events.n_events].sample), &(new_events.events[new_events.n_events].sub_sample), cluster_n+new_events.n_events, &corr);
			if(corr<cluster_conf->similarity_limit){ // Did we find match? XXX CHECK THE CRITERIA!!!!
				tests_one_ch_cluster_match(active_data, start-corr_win_before, end-corr_win_before, cluster_set, min_n_beats, &(new_events.events[new_events.n_events].sample), &(new_events.events[new_events.n_events].sub_sample), cluster_n+new_events.n_events, &corr);
			}
			if(corr>=cluster_conf->similarity_limit){ // Did we find match?
				// 2.3.1. Check that we didn't just find something that has already been marked as artifact.. TODO: ignore TEST_ARTIFACTs from checking range before hand rather than just rejecting them here -> it makes it possible to find new real events close to TEST_ARTIFACTs
				new_events.events[new_events.n_events].sample+=corr_win_before;
				for(;j<events->n_events && events->events[j].sample+artifact_range<new_events.events[new_events.n_events].sample;j++);
				if(j<events->n_events && events->events[j].sample<new_events.events[new_events.n_events].sample+artifact_range){ // Is there (TEST_ARTIFACT) event already?
					found=0;
					if(j>0){
						j--;
					}
#ifdef TRIG_PRINT_NEW_EVENTS
					printf("SKipping event at (%lu + %.1lf, %lf %% corr with cluster %d)\n",new_events.events[new_events.n_events].sample,new_events.events[new_events.n_events].sub_sample,corr*100.0, cluster_n[new_events.n_events]);
#endif
				}
				else{
					found=new_events.events[new_events.n_events].sample;
#ifdef TRIG_PRINT_NEW_EVENTS
					printf("Found new event (%lu + %.1lf, %lf %% corr with cluster %d)\n",found,new_events.events[new_events.n_events].sub_sample,corr*100.0, cluster_n[new_events.n_events]);
#endif
					new_events.n_events++;
				}
			}
			else{  // if we are still looking for..
			// 2.3. Check if we can find an energy maximum
				found=trig_find_one_event(energy_data,start, end, ch_info, helper);
				if(found){ // Did we find something?
					// 2.3.1. Check that we didn't just find something that has already been marked as artifact.. TODO: ignore TEST_ARTIFACTs from checking range before hand rather than just rejecting them here -> it makes it possible to find new real events close to TEST_ARTIFACTs
					for(;j<events->n_events && events->events[j].sample+artifact_range<found;j++);
					if(j<events->n_events && events->events[j].sample<found+artifact_range){ // Is there (TEST_ARTIFACT) event already?
						found=0;
#ifdef TRIG_PRINT_NEW_EVENTS
						printf("Skipping event at %lu\n",found);
#endif
						if(j>0){
							j--;
						}
					}
					else{ // we really did find something..
						cluster_n[new_events.n_events]=-1; // no cluster (yet)
						new_events.events[new_events.n_events].sample=found; // add to the new events..
						new_events.events[new_events.n_events++].sub_sample=0;
						not_clustered_events.events[not_clustered_events.n_events].sample=found; // add to the not clustered..
						not_clustered_events.events[not_clustered_events.n_events++].sub_sample=0;
#ifdef TRIG_PRINT_NEW_EVENTS
						printf("Found new event (%lu)\n",found);
#endif
					}
				}
			}

			if(found){ // Did we find new event?
				if(found+1.5*missing_events->missing_range[i].expected_rr<missing_events->missing_range[i].end){ // Do we expect there to be more to find?
					start=found;
				}
				else{
					start=missing_events->missing_range[i].end;
				}
			}
			else{
				if(end+margin<missing_events->missing_range[i].end){ // Do we want to try again?
					start=end-margin-1;
				}
				else{
					start=missing_events->missing_range[i].end;
				}
			}
		}
	}

	if(new_events.n_events>0){ // add new events?

		// 3. Combine new events to the old
		struct conf_cluster *cluster_conf=conf_cluster_get();
		int template_move=cluster_conf->move*active_data->sample_freq+0.5;
		int	beat_len_before=(cluster_conf->win_before*active_data->sample_freq+0.5);
		int	beat_len_after=cluster_conf->win_after*active_data->sample_freq+0.5;
		size_t *move_index=my_calloc(events->n_events,sizeof(size_t));
		size_t n_move;
		size_t j;
		struct trig_helper new_helper;
		struct test_clusters new_cluster_set;
		new_cluster_set.n_clusters=0;

		// 3.1. cluster the not clustered..

		if(not_clustered_events.n_events>0){
			// 3.1.1. define ok & goodness values for new events
			new_helper.nch=active_data->number_of_channels;
			new_helper.badness_type_for_events=NULL;
			new_helper.ch_badness_for_events=NULL;
			trig_define_ok_chs_for_events(ch_info, &not_clustered_events,&new_helper, template_move+beat_len_before,template_move+beat_len_after,active_data->sample_freq);

			// 3.1.2. create the clusters
			trig_create_clusters(active_data, &not_clustered_events, ch_info, &new_helper, &new_cluster_set);

			// 3.1.3 copy fine tuned samples and sub_samples to new events
			for(i=0,n=0;n<not_clustered_events.n_events;n++,i++){
				for(;cluster_n[i]>=0;i++); // find next not_clustered new event
				new_events.events[i].sample=not_clustered_events.events[n].sample;
				new_events.events[i].sub_sample=not_clustered_events.events[n].sub_sample;

			}
		}

		// 3.2. Combine events

//		trig_test_events(&new_events);
		// 3.2.1. add new to old events
		events->events=my_realloc(events->events,(events->n_events+new_events.n_events)*sizeof(struct test_event));
		for(i=events->n_events-1,n=new_events.n_events-1;n>=0;n--){
			n_move=n+1;
			for(;i>=0 && events->events[i].sample>new_events.events[n].sample;i--){
				move_index[i]=n_move;
				events->events[i+n_move]=events->events[i];
			}

			events->events[i+n_move].sample=new_events.events[n].sample;
			events->events[i+n_move].sub_sample=new_events.events[n].sub_sample/cluster_conf->frequency_mult;
			if(cluster_n[n]>=0){
				events->events[i+n_move].type=cluster_set->clusters[cluster_n[n]].type;
			}
			else{
				events->events[i+n_move].type=TEST_BEAT_UNDEFINED;
				not_clustered_events.events[--not_clustered_events.n_events].sample=i+n_move; 	// store the updated index value temporarily to 'sample' (will be used below)
			}
			new_events.events[n].sample=i+n_move; 	// store the updated index value temporarily to 'sample' (will be used below)
		}
		events->n_events+=new_events.n_events;

//		trig_test_events(events);

		// 3.2.2. Correct indexes in clusters to match updated events

		// 3.2.2.1 old clusters
		for(n=0;n<cluster_set->n_clusters;n++){
			for(i=0;i<cluster_set->clusters[n].n_beats;i++){
				cluster_set->clusters[n].inds[i]+=move_index[cluster_set->clusters[n].inds[i]];
			}
		}

		// 3.2.2.2. new clusters
		for(n=0;n<new_cluster_set.n_clusters;n++){
			for(i=0;i<new_cluster_set.clusters[n].n_beats;i++){
				new_cluster_set.clusters[n].inds[i]=not_clustered_events.events[new_cluster_set.clusters[n].inds[i]].sample; // notice we stored index temporarily to 'sample' (see above!)
			}
		}

		// 3.2.3 Add new one_ch matched events to clusters
		for(i=0;i<new_events.n_events;i++){
			if(cluster_n[i]>=0){
				cluster_set->clusters[cluster_n[i]].inds=my_realloc(cluster_set->clusters[cluster_n[i]].inds,(cluster_set->clusters[cluster_n[i]].n_beats+1)*sizeof(size_t));
				cluster_set->clusters[cluster_n[i]].inds[cluster_set->clusters[cluster_n[i]].n_beats]=new_events.events[i].sample; // notice we stored index temporarily to 'sample'  (see above!)
				cluster_set->clusters[cluster_n[i]].sample_offsets=my_realloc(cluster_set->clusters[cluster_n[i]].sample_offsets,(cluster_set->clusters[cluster_n[i]].n_beats+1)*sizeof(int));
				cluster_set->clusters[cluster_n[i]].sample_offsets[cluster_set->clusters[cluster_n[i]].n_beats]=(int)(new_events.events[i].sub_sample+0.5); // notice we stored offset temporarily here  (see above!)
				cluster_set->clusters[cluster_n[i]].n_beats++;

			}
		}
		my_free(move_index);

		// 3.3. Define cluster markers and combine clusters
		if(new_cluster_set.n_clusters>0){
			tests_define_cluster_markers(&new_cluster_set,ch_info);

			cluster_set->clusters=my_realloc(cluster_set->clusters,(cluster_set->n_clusters+new_cluster_set.n_clusters)*sizeof(struct test_cluster));
			memcpy(cluster_set->clusters+cluster_set->n_clusters, new_cluster_set.clusters,new_cluster_set.n_clusters*sizeof(struct test_cluster));
			cluster_set->n_clusters+=new_cluster_set.n_clusters;

			// 3.4. Reconfigure correlations matrix -- TODO: save previous correlation values
			matrix_free_d(*correlations_mat);
			matrix_free_d(*short_correlations_mat);
			(*correlations_mat)=matrix_malloc_d(cluster_set->n_clusters,cluster_set->n_clusters);
			(*short_correlations_mat)=matrix_malloc_d(cluster_set->n_clusters,cluster_set->n_clusters);
			for(i=0;i<cluster_set->n_clusters;i++){
				for(j=0;j<i;j++){
					(*correlations_mat)[j][i]=-2.0;
					(*short_correlations_mat)[j][i]=-2.0;
				}
				(*correlations_mat)[i][i]=1.0;
				(*short_correlations_mat)[i][i]=1.0;
			}
		}

	}

	my_free(new_events.events);
	my_free(not_clustered_events.events);
	my_free(cluster_n);

//	trig_test_events(events);

	return(new_events.n_events);
}

int trig_set_cluster_main_groups(struct test_clusters *cluster_set,struct test_ch_info *ch_info, double **correlations_mat){
	size_t i,j,n,tot_ind=0;
	int *cluster_offsets=my_malloc(cluster_set->n_clusters*sizeof(int));
	int type;
	for(type=TEST_NORMAL_BEAT;type<N_REAL_TEST_LABELS;type++){
		for(n=0,i=0;i<cluster_set->n_clusters;i++){
			if(cluster_set->clusters[i].type==type && cluster_set->clusters[i].main_type_group<0){
				cluster_set->clusters[i].main_type_group=n;
				for(j=i+1;j<cluster_set->n_clusters;j++){
					if(correlations_mat[i][j]<-1){ // do we have to calculate the correlation values?
						tests_get_cluster_correlations(ch_info->ch_goodness, i,j, cluster_set, cluster_offsets, correlations_mat[i], 0);
					}
					if(correlations_mat[i][j]>0.92/*cluster_conf->similarity_limit_for_clusters*/){
						cluster_set->clusters[j].main_type_group=n;
					}
				}
				n++;
				tot_ind++;
				//if(n>=TRIG_MAX_ANNOTATION_GROUPS){
				//	break;
				//}
			}
		}
	}
	my_free(cluster_offsets);
	return(0);
}

int trig_cinc2017_set_shutdown_ranges(struct test_events *events, struct test_ranges *ch_ranges){
	int i;
	if(events->n_ranges>0){
		my_free(events->ranges);
		events->ranges=NULL;
	}
	events->n_ranges=0;
	if(ch_ranges->n_ranges>0){
		events->ranges=my_malloc(ch_ranges->n_ranges*sizeof(struct test_range));
	}
	for(i=0;i<ch_ranges->n_ranges;i++){
		if(!TRIG_SKIP_THIS_RANGE(ch_ranges->ranges[i].type)){
			events->ranges[events->n_ranges].type=TEST_UNREADABLE;
			events->ranges[events->n_ranges].start=ch_ranges->ranges[i].start;
			events->ranges[events->n_ranges].end=ch_ranges->ranges[i].end;
			events->n_ranges++;
		}
	}
	return(0);
}

//#define PRINT_FILTERED_DATA
//#define PRINT_NOTCH_FREQUENCY

#define PRINT_QRS_ANALYSIS_ENERGY
int trig_do(struct data *original_data, struct data *energy_data, struct data *filtered_data_, struct test_events *events,struct test_event_infos *event_infos, struct test_ch_info *ch_info, char *conf_file, struct test_clusters *cluster_set){
	struct conf_trig *trig_conf=conf_trig_get();
	struct conf_cluster *cluster_conf=conf_cluster_get();
	struct data filtered_data;
	struct data *active_data;
	struct trig_helper helper;

	ssize_t *event_cluster_ids;
	double **correlations_mat=NULL;
	double **short_correlations_mat=NULL;

	int template_move;
	int beat_len_before,beat_len_after;

	int loop_redefine=0;
	double *ratio_to_nn;
	double *median_nn;


	size_t i,j;

//	size_t ind_offset=0;

	helper.nch=original_data->number_of_channels;
	helper.badness_type_for_events=NULL;
	helper.ch_badness_for_events=NULL;
	helper.threshold_values=my_malloc(sizeof(double)*(original_data->number_of_channels));


	// 1. Read & set the configuration info
	template_move=cluster_conf->move*original_data->sample_freq+0.5;
	beat_len_before=(cluster_conf->win_before*original_data->sample_freq+0.5);
	beat_len_after=cluster_conf->win_after*original_data->sample_freq+0.5;

	// 2. Filter signal

	// 2.1. Do the impulse filtering (optional, needed at least in some AHA cases)
	trig_filter_impulse_artifacts(original_data, MAX(0.75,0.75*250.0/original_data->sample_freq));  // impulse limit is set to 0.75 mV for data sampled at 250 Hz or above, and scaled up for lower data with lower sample rates


	// 2.2 Do the band pass filtering (optional)
	if(trig_conf->bp_filter.type!=NOFILTER){
		do_bp_filter(&(trig_conf->bp_filter),&filtered_data,original_data);
		active_data=&filtered_data;
	}
	else{
		active_data=original_data;
	}

	// 3. Detect / estimate artifacts and noise

	// 3.1 Detect amplitude drifts (needed especially for MIT data)
	if(trig_conf->reject_amplitude_drifts){
		trig_detect_amplitude_drifts(original_data,ch_info->ch_ranges);
	}

	// 3.2. Detect returning amplitude drifts (needed especially for AHA data)
	trig_detect_returning_amplitude_drifts(original_data, ch_info->ch_ranges);

	// 3.3 Detect big signal level (median) changes
	trig_detect_big_median_level_changes(original_data, ch_info->ch_ranges);

	// 3.2. Detect saturated signals
	if(trig_conf->reject_saturated){
		trig_detect_saturated_segments(original_data,ch_info->ch_ranges);
	}

	// 3.3. Estimate 'no signal' segments
	if(trig_conf->reject_no_signal){
		trig_analyze_ch_and_detect_no_signal_segments( original_data,ch_info, &helper);
	}

	// 3.4. Estimate 'the noisy' segments
	if(trig_conf->reject_notchy_signal){
		trig_define_notch_frequencies(active_data,NULL,ch_info->ch_ranges);
	}

	// 3.5. Sort and combine noise / artifact ranges based on their time
	trig_refine_ch_ranges(ch_info->ch_ranges,original_data);

	// 3.6. Define goodness values for channels
	trig_estimate_ch_goodness(active_data, NULL, ch_info);

	// 4. Trig / detect beats

		// 4.1. Define positive valued 'energy signal'

	//switch(trig_conf->type) // SKIP REST: CinC2017 version is simplified for one ch only -- no support for this
	trig_create_slopes_signal(active_data,energy_data);

		// 4.2. Define the threshold values based on the range of 'energy values'
	trig_define_threshold_values(energy_data,ch_info, &helper);
		// 4.2.2 Sort and combine (again) noise / artifact ranges based on their time
	trig_refine_ch_ranges(ch_info->ch_ranges,original_data);


//	trig_define_events(active_data,energy_data,ch_info, &helper,events,event_infos);
	trig_define_cinc2017_initial_events(active_data,energy_data,ch_info, &helper,events,event_infos);

	trig_cinc2017_set_shutdown_ranges(events, ch_info->ch_ranges);

	if(events->n_events<3){ // Skip! Less than three beats is just too little -- this is just too bad / noisy
		trig_free_helper(&helper);
		if(filtered_data_){
			*filtered_data_=*active_data;
		}
		else if(trig_conf->bp_filter.type!=NOFILTER){
			free_data(active_data);
		}
		return(0);
	}
		//4.4. Define ok channels (helper->ch_badness values) for all the events

	trig_define_ok_chs_for_events(ch_info, events,&helper, template_move+beat_len_before,template_move+beat_len_after,active_data->sample_freq);

	// 4.5. Estimate the propability that each of the event is real event! TODO: should be implemented as part of the trig_define_events and trig_redefine_events
	trig_event_probability(energy_data,ch_info, &helper, events, 0);

	// 5. Cluster / group similar beats / events and fine-tune trig times

	trig_create_clusters(active_data, events, ch_info, &helper, cluster_set);

	if(cluster_set->n_clusters==0){ // Skip! If we can't find any decent cluster this is just too bad / noisy
		trig_free_helper(&helper);
		if(filtered_data_){
			*filtered_data_=*active_data;
		}
		else if(trig_conf->bp_filter.type!=NOFILTER){
			free_data(active_data);
		}
		return(0);
	}

	// 6. Analyze clusters

		// 6.1. Regroup clusters -- find similar clusters
	//tests_combine_clusters(active_data, cluster_set, &cluster_conf);

	// 6.2 Define direct access from events to clusters
	event_cluster_ids=my_malloc(events->n_events*sizeof(ssize_t));
	tests_define_event_cluster_ids(events, event_cluster_ids, cluster_set);

	// 6.3. Define set of morphological markers for the clusters
	tests_define_cluster_markers(cluster_set,ch_info);

	// 7. Annotations, round 1a: identify the evident clusters:

	// 7.0 initialize cluster cluster correlation matrix
	correlations_mat=matrix_malloc_d(cluster_set->n_clusters,cluster_set->n_clusters);
	short_correlations_mat=matrix_malloc_d(cluster_set->n_clusters,cluster_set->n_clusters);
	for(i=0;i<cluster_set->n_clusters;i++){
		for(j=0;j<i;j++){
			correlations_mat[j][i]=-2.0;
			short_correlations_mat[j][i]=-2.0;
		}
		correlations_mat[i][i]=1.0;
		short_correlations_mat[i][i]=1.0;
	}

	// 7.1. Set the biggest cluster / group of clusters as "N"s
//	tests_define_cluster_annotation_markers(events,cluster_set,ch_info,event_cluster_ids,correlations_mat); // TODO: rewrite this. We need only close_by_similarity for the main_normal (next)

	trig_identify_main_normal_clusters(events, ch_info, cluster_set, correlations_mat,short_correlations_mat,original_data->sample_freq);

	trig_define_amplitude_normalization(ch_info,events, cluster_set, active_data);

	// 7.2 Identify the evident "A"s -- yes there are some morphologies (like some of the T-wave trigs) that can already be identified well enough
	trig_identify_evident_artifact_clusters(cluster_set);

	// 7.3. Set 'N' and 'A' values to events
	trig_set_event_types_from_cluster_set(cluster_set, TEST_NORMAL_BEAT, events);
	trig_set_event_types_from_cluster_set(cluster_set, TEST_ARTIFACT, events);

	// 8. Estimate the annotation &/ heart rate dependent cluster markers, 1st round
	ratio_to_nn=my_malloc(events->n_events*sizeof(double));
	median_nn=my_malloc(events->n_events*sizeof(double));
	tests_define_cluster_event_timings(events,cluster_set,0,0, ratio_to_nn, median_nn);
	tests_define_cluster_annotation_markers(events,cluster_set,ch_info,event_cluster_ids,correlations_mat);

	// 8.2. Beat-to-beat analysis
		  // 8.2.1. Is there T-wave?
		  // 8.2.2. Is there P-wave?
		  // 8.2.3. HR analysis -- is the beat premature?

	// 8.3 Create P and T templates and define 'N' morphological parameters and their range for the 'N' beats with prev RR > 80 % next RR
	//	trig_get_P_and_T_templates(active_data, @cluster_set, events, &trig_conf, &p_templates, &t_templates);

	// 8.4. Define marker ranges for normal and ventricular event clusters
	//	trig_define_marker_ranges_for_clusters(&cluster_set,active_data, events, &cluster_conf);

	// 9.0 Annotations, round 1b:

	// 9.1. Read the classification setups and define and set types to the rest of the clusters and events -- unknown type is still accepted

	trig_classify_cinc2017_clusters_and_events(cluster_set,events,event_cluster_ids,ratio_to_nn,median_nn,correlations_mat,short_correlations_mat);
	my_free(median_nn);
	my_free(ratio_to_nn);
//	trig_identify_clusters_and_set_event_types(cluster_set,events, ch_info, classification_confidence);

	// 9.2. Identify and change the supraventricular types to events with 'normal morphological' type
//	trig_pick_supraventriculars_from_normals(events,original_data->sample_freq);

//	abnormal_rhythm=rhythm_reset_af_ranges(events, active_data->sample_freq);
	// 10. Re trig for missing beats / events
	//   AND
	// 11. Add new events to clusters and -- if needed --  define more clusters and analyze their markers
	if(loop_redefine){ // todo: set as trig_parameter -> current setup can take crazy calculation time..
		struct trig_missing_events missing_events;
		double *ratio_to_nn;
		double *median_nn;
		size_t new_events_added=events->n_events;

//		trig_test_event_ranges(events->ranges,events->n_ranges);

		while(new_events_added){
			printf("Reanalyzing %lu events\n",new_events_added);
			new_events_added=0;

			ratio_to_nn=my_malloc(events->n_events*sizeof(double));
			median_nn=my_malloc(events->n_events*sizeof(double));

			missing_events.missing_range=NULL;
			missing_events.n_missing=0;
			rhythm_normality_for_events(events,ratio_to_nn, median_nn, 0.850*active_data->sample_freq,1);
			rhythm_identify_missing(events,ratio_to_nn, median_nn, active_data->sample_freq,&missing_events);

			if(missing_events.n_missing>0){
				new_events_added=trig_redefine_events(&missing_events, active_data,energy_data,ch_info, &helper, events,event_infos,cluster_set,&correlations_mat,&short_correlations_mat);
				if(new_events_added){  // Did we add new (potential) events? TODO: just update event_cluster_ids, don't recalculate everything..
					event_cluster_ids=my_realloc(event_cluster_ids,events->n_events*sizeof(ssize_t));
					tests_define_event_cluster_ids(events, event_cluster_ids, cluster_set);
					trig_define_ok_chs_for_events(ch_info, events,&helper, template_move+beat_len_before,template_move+beat_len_after,active_data->sample_freq); // TODO: modify so that only new beats are recalculated
					trig_event_probability(energy_data,ch_info, &helper, events, 1);
				}
			}

			if(missing_events.missing_range){
				my_free(missing_events.missing_range);
			}

			tests_define_cluster_event_timings(events,cluster_set,0,0, NULL,NULL);
			tests_define_cluster_annotation_markers(events,cluster_set,ch_info,event_cluster_ids,correlations_mat); // TODO: Don't recalculate the values for old clusters

			// Drop the confidence requirement, and re-idetify / re-annotate
	//		classification_confidence*=classification_confidence;
//			new_events_added=trig_identify_clusters_and_set_event_types(cluster_set,events, ch_info, classification_confidence);
			new_events_added+=trig_classify_cinc2017_clusters_and_events(cluster_set,events,event_cluster_ids,ratio_to_nn,median_nn,correlations_mat,short_correlations_mat);
			my_free(median_nn);
			my_free(ratio_to_nn);

		//	abnormal_rhythm=rhythm_reset_af_ranges(events, active_data->sample_freq);

		}

	}


	//  trig_add_to_event_clusters(active_data, dataset, new_event_ids, cluster_set , new_cluster_set, &cluster_conf, ch_info,  &helper);
	//	tests_define_cluster_markers(new_cluster_set);

	// 12.Annotations, round 2: Repeat the points 8. & 9. -- estimate (again) the annotation &/ heart rate dependent cluster markers, and try to annotate as many of hte missing annotations as possible
//	tests_define_cluster_annotation_markers(events,cluster_set,ch_info,event_cluster_ids,correlations_mat);
	// Is there T-wave?
	// Is there P-wave?
	// HR analysis -- is the beat premature?

	//	trig_get_P_and_T_templates(active_data, @cluster_set, events, &trig_conf, &p_templates, &t_templates);

	//	trig_define_marker_ranges_for_clusters(&cluster_set,active_data, events, &cluster_conf);


	// trig_identify_clusters_and_set_event_types(cluster_set,&cluster_conf,events, active_data, ch_info);

	// trig_pick_supraventriculars_from_normals(cluster_set,&cluster_conf,events, active_data);



	/*
    Features for "V":
     * High energy waveform on multiple channels
     * Followed by T-wave
     * preceding + next RR close to median RR
     * Several similar complexes
     * If RR >0.8 next RR and P template match before the wave -> "F"
     * if timing inside estimated T-wave of the preceding beat -> V on T, or B if waveform has "L/RBBB features"
	Features for multiple "V" (VT):
	 * Multiple succeeding waveforms in high bpm
	 * waveform duration > 50 ms
	Features for A:
	 * Other channel(s) ok, but event still visible only on one channel
	 * Huge baseline drift in original signal: signal doesn't return to original lvl
	 * Long first slope -> T-wave
	 * Similar to T-template -> T-wave
	 * Loner / semi-loner with abnormal duration:
	 ** 	Very short (<50 ms)
	 ** 	very long / no clear difference to surrounding noise

*/



	// 13. set main_type_goups for clusters
	trig_set_cluster_main_groups(cluster_set,ch_info,correlations_mat);

	// We don't use rhythms atm in CinC2017 REMEMBER TO TAKE BACK INTO USE IF RHYTHM ARE USED AGAIN IN THE FUTURE!!!!!! TODO XXX
	//	rhythm_reset_all(events, active_data->sample_freq);

	// 14. Collect the results for 'export structs'

	trig_sort_cluster_inds(cluster_set);

	// 15. Free


/*	{
		int i,ch;
		for(i=0;i<events->n_events;i++){
			for(ch=0;ch<active_data->number_of_channels;ch++){
				my_free(dataset[i][ch]);
			}
			my_free(dataset[i]);
		}
		my_free(dataset);
	}*/

	trig_free_helper(&helper);

//	trig_drop_artifact_events(events);

#ifdef PRINT_NOTCH_FREQUENCY
	free_data(energy_data);
	trig_define_notch_frequencies(active_data, energy_data,NULL);
#endif

	if(filtered_data_){
		*filtered_data_=*active_data;
//		memcpy(filtered_data_->ch,active_data->ch,sizeof(struct data));
//		filtered_data_->ch=active_data->ch;
	}
	else if(trig_conf->bp_filter.type!=NOFILTER){
		free_data(active_data);
	}
//	free_data(&energy_data);
	matrix_free_d(correlations_mat);
	matrix_free_d(short_correlations_mat);
	my_free(event_cluster_ids);

	for(i=0;i<cluster_set->n_clusters;i++){
		if(cluster_set->clusters[i].cluster_offset>200 || cluster_set->clusters[i].cluster_offset<-200){
			printf("Something wrong with cluster %lu\n",i);
		}
	}

	return(0);
}



size_t trig_get_n_events(enum test_trigger_type type,struct test_events *events){
	size_t i,n=0;
	for(i=0;i<events->n_events;i++){
		if(events->events[i].type==type){
			n++;
		}
	}
	return(n);
}


int trig_get_main_cluster_info(enum test_trigger_type type, struct test_clusters *cluster_set, size_t *n_main_clusters,double *main_cluster_rate){
	size_t i;
	size_t max_group=0;
	size_t *n_in_group=my_calloc(cluster_set->n_clusters,sizeof(size_t));
	size_t n_all=0;
	*n_main_clusters=0;
	for(i=0;i<cluster_set->n_clusters;i++){
		if(cluster_set->clusters[i].type==type){
			n_all+=cluster_set->clusters[i].n_beats;
			if(cluster_set->clusters[i].main_type_group>=0){
				n_in_group[cluster_set->clusters[i].main_type_group]+=cluster_set->clusters[i].n_beats;
			}
		}
	}
	if(n_all>0){
		for(i=0;i<cluster_set->n_clusters;i++){
			if(n_in_group[i]>0){
				(*n_main_clusters)++;
				if(n_in_group[i]>max_group){
					if(*n_main_clusters>=TRIG_MAX_ANNOTATION_GROUPS){
						max_group=0;
					}
					max_group=n_in_group[i];
				}
			}
		}
		*main_cluster_rate=(double)(max_group)/(double)(n_all);
	}
	my_free(n_in_group);
	return(0);
}

int trig_get_qrs_dur_range(enum test_trigger_type type, struct test_clusters *cluster_set, double *qrs_dur_min,double *qrs_dur_max){
  int k;
  int found=0;
  size_t main_size=0;
  size_t n_limit=2;
  double relative_limit=0.05;
  double qrs_dur;
  *qrs_dur_min=FLT_MAX;
  *qrs_dur_max=0.0;
  for(k=0;k<cluster_set->n_clusters;k++){
    if(cluster_set->clusters[k].type==type){
      found=1;
      qrs_dur=cluster_set->clusters[k].markers[0].qrs_offset-cluster_set->clusters[k].markers[0].qrs_onset;
      if(qrs_dur<*qrs_dur_min){
	*qrs_dur_min=qrs_dur;
      }
    if(qrs_dur>*qrs_dur_max){
	*qrs_dur_max=qrs_dur;
      }
    }
    if(found && (cluster_set->clusters[k].n_beats<relative_limit*main_size || cluster_set->clusters[k].n_beats<n_limit))
      break;
  }
  return(!found);
}




