/*

 * tests.c
 *
 *  Created on: Mar 18, 2015
 *      Author: hvaanane
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>
#include <limits.h>
#include <math.h>
#include <float.h>


#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>


//#define matrix_malloc_d3(_a, _b, _c) matrix_malloc_d3((_a), (_b), (_c), __FILE__, __LINE__)

#include "types.h"
#include "test_types.h"
#include "matrix.h"
#include "tests.h"
#include "util.h"
#include "trig.h"
#include "rhythm.h"
#include "tests.h"
#include "conf.h"
#include "mit.h"
#include "array.h"
#include "data.h"

#include "memory_manager.h"



#define TEST_FILE_VERSION_NO 1
#define VERSION_NO 0.800


struct annotation_statistics {
	size_t n_ok;			// number of correct annotations
	size_t tot_n_test;      // total number of test annotations
	size_t tot_n_ref;       // total number of reference annotations
	double pp;              // positive predictive value
	double sens;            // sensitivity
	double bias; // expectation value of reference_sample - test_sample in correct annotations
	double sd; // standard deviation of reference_sample - test_sample in correct annotations
};

/*
 * 	create set of configuration files based on rules in param_file. Files (<setup_description>.conf) are created to the same folder than the param_file
 */

int tests_free_events(struct test_events *events) {
	if (events->n_events)
		my_free(events->events);
	if (events->n_ranges)
		my_free(events->ranges);
	return (0);
}

int tests_free_event_infos(struct test_event_infos *event_infos) {
	size_t i;
	for (i = 0; i < event_infos->n_infos; i++) {
		my_free(event_infos->event_infos[i].info_txt);
	}
	my_free(event_infos->event_infos);
	event_infos->n_infos = 0;
	event_infos->event_infos = NULL;
	return (0);
}

int tests_free_event_markers(struct test_event_markers *markers, int n_markers) {
	int n;
	for (n = 0; n < n_markers; n++) {
		if (markers[n].n_values > 0 && markers[n].values) {
			my_free(markers[n].values);
		}
	}
	if (markers) {
		my_free(markers);
	}
	return (0);
}

int tests_free_marker_events(struct test_marker_events *marker_events,
		int n_marker_events) {
	int n;
	for (n = 0; n < n_marker_events; n++) {
		if (marker_events[n].n_samples > 0 && marker_events[n].samples) {
			my_free(marker_events[n].samples);
		}
	}
	if (marker_events) {
		my_free(marker_events);
	}
	return (0);
}

int tests_change_filename_ending(char *filename,char  *  parameter_filename, char *ending) {
	char *p,*p2;
	int res = 0;
	strcpy(filename, parameter_filename);
	p = strrchr(filename, '.');
	p2 = strrchr(filename, '/');
	if (p == NULL || (p2 && p<p2)) {
		res = 1;
		p = filename + strlen(filename);
		*p = '.';
	}
	p++;
	strcpy(p, ending);
	return (res);
}

char tests_trigger_type_to_c(enum test_trigger_type i) {
	switch (i) {
	case TEST_BEAT_UNDEFINED:
		return ('U');
	case TEST_NORMAL_BEAT:
		return ('N');
	case TEST_SUPRAVENTRICULAR:
		return ('S');
	case TEST_VENTRICULAR:
		return ('V');
	case TEST_FUSION:
		return ('F');
	case TEST_OTHER:
		return ('T');
	case TEST_OTHER_PSEUDOBEAT:
		return ('O');
	case TEST_UNREADABLE_PSEUDOBEAT:
		return ('X');
	case TEST_REFERENCE_BEAT_UNDEFINED:
		return ('u');
	case TEST_REFERENCE_NORMAL_BEAT:
		return ('n');
	case TEST_REFERENCE_SUPRAVENTRICULAR:
		return ('s');
	case TEST_REFERENCE_VENTRICULAR:
		return ('v');
	case TEST_REFERENCE_FUSION:
		return ('f');
	case TEST_REFERENCE_OTHER:
		return ('t');
	case TEST_REFERENCE_OTHER_PSEUDOBEAT:
		return ('o');
	case TEST_REFERENCE_UNREADABLE_PSEUDOBEAT:
		return ('x');
	case TEST_ARTIFACT:
		return ('A');
	case TEST_REFERENCE_AF_START:
		return ('1');
	case TEST_REFERENCE_AF_END:
		return ('2');
	case TEST_REFERENCE_AF_ON:
		return ('3');
	default:
		return ('-');
	}
}

char  *tests_trigger_type_to_string(enum test_trigger_type i) {
	switch (i) {
	case TEST_BEAT_UNDEFINED:
		return ("Undefined");
	case TEST_NORMAL_BEAT:
		return ("Normal");
	case TEST_SUPRAVENTRICULAR:
		return ("Supraventricular");
	case TEST_VENTRICULAR:
		return ("Ventricular");
	case TEST_FUSION:
		return ("Fusion");
	case TEST_OTHER:
		return ("Other");
	case TEST_OTHER_PSEUDOBEAT:
		return ("Pseudobeat");
	case TEST_UNREADABLE_PSEUDOBEAT:
		return ("Unreadable_pseudobeat");
	case TEST_REFERENCE_BEAT_UNDEFINED:
		return ("ref_undefined");
	case TEST_REFERENCE_NORMAL_BEAT:
		return ("ref_normal");
	case TEST_REFERENCE_SUPRAVENTRICULAR:
		return ("ref_supraventricular");
	case TEST_REFERENCE_VENTRICULAR:
		return ("ref_ventricular");
	case TEST_REFERENCE_FUSION:
		return ("ref_fusion");
	case TEST_REFERENCE_OTHER:
		return ("ref_other");
	case TEST_REFERENCE_OTHER_PSEUDOBEAT:
		return ("ref_pseudobeat");
	case TEST_REFERENCE_UNREADABLE_PSEUDOBEAT:
		return ("ref_unreadable_pseudo");
	case TEST_ARTIFACT:
		return ("Artifact");
	case TEST_REFERENCE_AF_START:
		return ("Ref_AF_start");
	case TEST_REFERENCE_AF_END:
		return ("Ref_AF_end");
	case TEST_REFERENCE_AF_ON:
		return ("Ref_AF_on");
	default:
		return ("-?-");
	}
}

char  *tests_range_name(enum test_range_type type) {
	switch(type){
	case TEST_RHYTHM_UNDEFINED:
//		return("undefined");
		return("undef");
	case TEST_UNREADABLE:
//		return("unreadable");
		return("bad");
	case TEST_ABNORMAL_RHYTHM:
	//		return("unreadable");
		return("abnorm");
	case REGION_OF_INTEREST:
		return("ROI");
	case TEST_SIGNAL_AMPLITUDE_DRIFT:
		return("signal_amplitude_drift");
	case TEST_NOISY_NOTCHES:
		return("noisy_notches");
	case TEST_SHORT_OUT_OF_RANGE:
		return("short_out_of_range");
	case TEST_SIGNAL_OUT_OF_RANGE:
		return("out_of_range");
	case TEST_LOW_SNR:
		return("low_snr");
	case TEST_NO_SIGNAL:
		return("no_signal");
	case TEST_NO_ENERGY_SIGNAL:
		return("no_energy_signal");
	case TEST_RETURNING_AMPLITUDE_DRIFT:
		return("returning_amplitude_drift");
	case TEST_SIGNAL_LEVEL_CHANGE:
		return("signal_level_change");
	case TEST_COMBINED_ANOMALY:
		return("combined_anomaly");
	case TEST_NORMAL_RHYTHM:
		return("normal");
	case TEST_VF:
		return("VF");
	case TEST_AF:
		return("AF");
	case TEST_AFL:
		return("AFL");
	case TEST_VT:
		return("VT");
	case TEST_SVT:
		return("SVT");
	case TEST_AVJ:
//		return("AV_junctional");
		return("AV_j");
	case TEST_V_PAIR:
		return("V_PAIR");
	case TEST_V_TRIPLET:
		return("V_TRIP");
	case TEST_V_RUN:
		return("V_RUN");
	case TEST_BIGEMINY:
		return("BIGEM");
	case TEST_TRIGEMINY:
		return("TRIGEM");
	case TEST_QUADRIGEMINY:
		return("QUADR");


	case TEST_REFERENCE_UNREADABLE:
		//return("reference_unreadable");
		return("ref_bad");
	case TEST_REFERENCE_REGION_OF_INTEREST:
		return("ref_ROI");
	case TEST_REFERENCE_VF:
		return("ref_VF");
	case TEST_REFERENCE_AF:
		return("ref_AF");
	default:
		return("unknown");
	}

}

#define PARAMETER_NAME_MAX 100
#define CONFIG_LINE_MAX 150
#define N_SETUP_FILES_MAX 100000

struct parameter_test_range {
	char parameter_name[PARAMETER_NAME_MAX];
	double min;
	double step;
	int n;
	double value;
};


static int inline tests_are_stds_similar(double std1, double std2){
	struct conf_cluster *conf=conf_cluster_get();
	if(std2>(1.0+conf->maximum_std_difference[0])*std1+conf->maximum_std_difference[1]){ // too big?
		return(0);
	}
	if(std1>(1.0+conf->maximum_std_difference[0])*std2+conf->maximum_std_difference[1]){ // too small?
		return(0);
	}
	return(1);
}

static int inline tests_are_one_ch_template_stds_similar(double **template_set1, double **template_set2, int len){
	struct conf_cluster *conf=conf_cluster_get();
	int ch=0;
	if(template_set2[ch][len]>(1.0+conf->maximum_std_difference[0])*template_set1[ch][len]+conf->maximum_std_difference[1]){ // too big?
		return(0);
	}
	if(template_set1[ch][len]>(1.0+conf->maximum_std_difference[0])*template_set2[ch][len]+conf->maximum_std_difference[1]){ // too small?

		return(0);
	}
	return(1);
}



int tests_trigger_type_from_trigger_type(enum test_trigger_type *to_type,enum trigger_type from_type) {
	switch (from_type) {
	case TRIGGERED_BEAT:
		*to_type = TEST_BEAT_UNDEFINED;
		break;
	case NORMAL_BEAT:
		*to_type = TEST_NORMAL_BEAT;
		break;
	case ARTIFACT:
		*to_type = TEST_ARTIFACT;
		break;
	case ASYSTOLE:
		*to_type = TEST_FUSION;
		break;
	case VENTRICULAR:
		*to_type = TEST_VENTRICULAR;
		break;
	default:
		*to_type = TEST_SUPRAVENTRICULAR;
	}
	return (0);
}


int tests_trigger_type_from_char(enum test_trigger_type *to_type,char from_type) {
	switch (from_type) {
	case 'T':
		*to_type = TEST_BEAT_UNDEFINED;
		break;
	case 'N':
		*to_type = TEST_NORMAL_BEAT;
		break;
	case 'A':
		*to_type = TEST_ARTIFACT;
		break;
	case 'V':
		*to_type = TEST_VENTRICULAR;
		break;
	case 'S':
		*to_type = TEST_SUPRAVENTRICULAR;
		break;
	default:
		*to_type = TEST_OTHER;
	}
	return (0);
}

int tests_event_comparison( void *event1,  void *event2) {
	if (((struct test_event *) (event1))->sample
			< ((struct test_event *) (event2))->sample) {
		return (-1);
	} else if (((struct test_event *) (event1))->sample
			> ((struct test_event *) (event2))->sample) {
		return (1);
	}
	return (0);
}





int tests_is_bad_range(enum test_range_type type){
	switch(type){
	case TEST_UNREADABLE:
	case TEST_SIGNAL_AMPLITUDE_DRIFT:
	case TEST_NOISY_NOTCHES:
	case TEST_SHORT_OUT_OF_RANGE:
	case TEST_SIGNAL_OUT_OF_RANGE:
	case TEST_LOW_SNR:
	case TEST_NO_SIGNAL:
	case TEST_NO_ENERGY_SIGNAL:
	case TEST_COMBINED_ANOMALY:
	case TEST_REFERENCE_UNREADABLE:
		return(1);
	default:
		return(0);
	}
}

int tests_is_rhythm_range(enum test_range_type type){
	switch(type){
	case TEST_VF:
	case TEST_AF:
	case TEST_AFL:
	case TEST_VT:
	case TEST_SVT:
	case TEST_REFERENCE_VF:
	case TEST_REFERENCE_AF:
		return(1);
	default:
		return(0);
	}
}

int tests_is_rhythm_af(enum test_range_type type){
	switch(type){
	case TEST_AF:
	case TEST_AFL:
	case TEST_REFERENCE_AF:
		return(1);
	default:
		return(0);
	}
}

// define common base value for the plots -- TODO alpha trimmed mean instead of basic arithmetic mean
double tests_get_base_val(double *data, size_t nsamp){
	size_t i;
	double base_val=0;
	for(i=0;i<nsamp;i++){
		base_val+=data[i];
	}
	return(base_val/(double)(nsamp));
}

static inline int tests_create_energy2(double **data, int len, int win_len, char *bad_chs, int nch, double *energy) {
	int i, j;
	int ch;
	double min, max;
	for (i = 0; i < len-win_len; i++) {
		energy[i] = 0.0;
		for (ch = 0; ch < nch; ch++) {
			if (bad_chs==NULL || bad_chs[ch]<TEST_EVENT_TOTALLY_BAD) {
				min = max = data[ch][i];
				for (j = i + 1; j < i + win_len; j++) {
					if (min > data[ch][j]) {
						min = data[ch][j];
					} else if (max < data[ch][j]) {
						max = data[ch][j];
					}
				}
				energy[i] += max - min;
			}
		}
	}
	return (0);
}



static int tests_max_slopes(double *data, int len, int step, double *min,double *max) {
	int i;
	double slope;
	*max = *min = data[step] - data[0];
	for (i = 1; i < len; i++) {
		slope = data[step + i] - data[i];
		if (slope > *max) {
			*max = slope;
		} else if (slope < *min) {
			*min = slope;
		}
	}
	return (0);
}

int tests_get_max_slopes(double **data, int trig_offset, int len, int nch,double freq, double *max_slope) {
	int ch;
	int start_i = MAX(trig_offset - 0.080 * freq, 0);  	// TODO: set as conf parameter (prev val 0.050)
	int end_i = MIN(trig_offset + 0.080 * freq, len);	// TODO: set as conf parameter
	int step = MAX(0.0125 * freq, 1);					// TODO: set as conf parameter (prev val 0.018)
	double min, max;
	for (ch = 0; ch < nch; ch++) {
		tests_max_slopes(data[ch] + start_i, end_i - start_i, step, &min, &max);
		max_slope[ch] = max - min;
	}
	return (0);
}

static inline void tests_raising_slope_duration(double **data, char *bad_chs, int trig_offset,int len, int nch, double freq, double *duration) {
	int ch, i, max_i, max_slope_i;
	int start_i = MAX(trig_offset - 0.120 * freq, 0);
	int end_i2 = MIN(trig_offset + 0.120 * freq, len);
	int end_i = MIN(trig_offset + 0.000 * freq, len);
	int step = MAX(0.020 * freq, 1);
	double base, max;
	double dt = 1.0 / freq;
	double abs_slope_limit = 1;
	double rel_slope_limit = 0.01;
	double slope_limit;
	double slope;

	// TODO: redesign algorithm to really check the 'initial raising part'
	for (ch = 0; ch < nch; ch++) {
		//base = (data[ch][start_i]+data[ch][end_i])/2.0;
		base = 0.0;
		max = fabs(data[ch][start_i] - base);
		max_i = start_i;
		for (i = start_i + 1; i < end_i; i++) {
			if (fabs(data[ch][i] - base) > max) {
				max = fabs(data[ch][i] - base);
				max_i = i;
			}
		}
		if(max_i==end_i){ // if max was on the border, continue until the peak is found
			for (; i < end_i2 && fabs(data[ch][i] - base) > max; i++) {
				max = fabs(data[ch][i] - base);
				max_i = i;
			}
		}
		duration[ch] = 0.0;

		if(bad_chs[ch]==TEST_EVENT_OK_BUT_SATURATED){ // special case where the peak of the signal is saturated? -- peak is defined as the middle point inside the saturated range
			int saturation_start,saturation_end;
			double saturation_tolerance=0.75; // we need a tolerance range below the actual peak value due to signal morphology changes during filtering and averaging;
			if (data[ch][max_i] > base) { // positive maximum?
				for(saturation_start=max_i-1;saturation_start>start_i && data[ch][saturation_start]>saturation_tolerance*data[ch][max_i];saturation_start--);
				for(saturation_end=max_i-1;saturation_end<end_i2 && data[ch][saturation_end]>saturation_tolerance*data[ch][max_i];saturation_end++);
			}
			else{ // negative maximum
				for(saturation_start=max_i-1;saturation_start>start_i && data[ch][saturation_start]<saturation_tolerance*data[ch][max_i];saturation_start--);
				for(saturation_end=max_i-1;saturation_end<end_i2 && data[ch][saturation_end]<saturation_tolerance*data[ch][max_i];saturation_end++);
			}
			duration[ch] =(double)((saturation_end-1)-(saturation_start+1))/(2.0*freq);
			max_i=saturation_start+1;
		}

		max=0.0;
		max_slope_i=max_i;
		if (data[ch][max_i] > base) { // positive maximum?
			for (i = max_i; i > step;i--){
				slope=data[ch][i] - data[ch][i - step];
				if(slope>max){
					max=slope;
					max_slope_i=i;
				}
				if(slope<=0){
					break;
				}
			}
			slope_limit=abs_slope_limit+rel_slope_limit*max;
			duration[ch]=(double)(max_i-max_slope_i)/freq; //
			for (i = max_slope_i-1; i > step && data[ch][i] - data[ch][i - step] > slope_limit;i--, duration[ch] += dt); // add duration until the slope (data[i-step]-data[i]) sign changes
		} else {
			for (i = max_i; i > step;i--){
				slope=data[ch][i - step] - data[ch][i];
				if(slope>max){
					max=slope;
					max_slope_i=i;
				}
				if(slope<=0){
					break;
				}
			}
			slope_limit=abs_slope_limit+rel_slope_limit*max;
			duration[ch]=(double)(max_i-max_slope_i)/freq; //
			for (i = max_slope_i-1; i > step && data[ch][i - step] - data[ch][i] > slope_limit;i--, duration[ch] += dt); // add duration until the slope (data[i-step]-data[i]) sign changes
		}
	}
}

static inline void tests_get_powers(double **data, int start_i, int end_i, int nch, double freq, double *powers) {
	int ch;
	int i;
	double b_k, b_x0, b_x1;

	for (ch = 0; ch < nch; ch++) {
		// define 'baseline' as line from average of two samples before to average of two samples after the QRS
		b_x0 = (data[ch][start_i - 2] + data[ch][start_i - 1]) / 2.0;
		b_x1 = (data[ch][end_i + 2] + data[ch][end_i + 1]) / 2.0;
		b_k = (b_x1 - b_x0) / (end_i - start_i + 3);

		// define the power
		powers[ch] = 0.0;
		for (i = start_i; i <= end_i; i++) {
			powers[ch] += POW2(data[ch][i] - (b_x0 + b_k * (i + 1.5)));
		}
//		powers[ch] = sqrt(powers[ch]/(1.0 + end_i - start_i));
		powers[ch] = sqrt(powers[ch]);
	}
}

static inline void tests_get_n_waves(double **data, int start_i, int end_i,int nch, double freq, int *n_waves) {
	int ch,i,n;
	int min_i,max_i;
	double min_wave_amp=50.0;
	double min_wave_time=0.06*freq;
	double min,max;
	int looking_for_min,looking_for_max;

//	size_t i;
//	double b_k,b_x0,b_x1;

	for (ch = 0; ch < nch; ch++) {
		min=max=data[ch][start_i-1];
		looking_for_min=looking_for_max=1;
		for(min_i=max_i=start_i-1,n=0,i=start_i;i<end_i;i++){
			if(min>data[ch][i]){ // new min value?
				min=data[ch][i]; // update min value
				if(!looking_for_min){
					max=data[ch][i]; // reset max value
					min_i=i;		// update last peak index
				}
				else if(max-min>min_wave_amp && i-max_i>min_wave_time){ // Did we found new wave?
					looking_for_min=0;  // stop looking for min
					looking_for_max=1;  // look for max
					min_i=i;
					n++;
				}
			}
			else if(max<data[ch][i]){ // new max value?
				max=data[ch][i];      // update max value
				if(!looking_for_max){
					min=data[ch][i]; // reset min value
					max_i=i;
				}
				else if(max-min>min_wave_amp && i-min_i>min_wave_time){ // Did we found new wave?
					looking_for_max=0;  // stop looking for max
					looking_for_min=1;  // look for min
					max_i=i;
					n++;
				}
			}
		}
		n_waves[ch] = n;
	}
}

// resamples data segment (length len) to prealloced resampled array (len*resample_mult).
// Notice that data has to have at least TESTS_LANCZOS_LEN samples before and after the actual len long segment
#define TESTS_LANCZOS_LEN 4
#ifndef M_PI
#define M_PI		3.14159265358979323846	/* pi */
#endif
static int tests_resample_segment(double *data, double *resampled, int len, int new_resample_mult) {
	static double **lanczos_window = NULL;
	static int resample_mult = 0;
	size_t i, j;
	int k;
	double x;
	if (new_resample_mult != resample_mult && lanczos_window) { // free dynamically alloced static data?
		for (i = 1; i < resample_mult; i++) {
			my_free(lanczos_window[i] - TESTS_LANCZOS_LEN);
		}
		my_free(lanczos_window);
		lanczos_window = NULL;
		resample_mult = 0;
	}

	if (new_resample_mult < 1) { // did we only want to free the data..?
		return (0);
	}
	if (new_resample_mult == 1) { // special case where we just copy the data
		memcpy(resampled, data, len * sizeof(double));
		return (0);
	}
	if (!lanczos_window) { // are the lanczos windows already defined, or should we do it now?
		resample_mult = new_resample_mult;
		lanczos_window = my_malloc((resample_mult) * sizeof(double *));
		for (j = 1; j < resample_mult; j++) {
			lanczos_window[j] = (double *) (my_malloc(
					(2 * TESTS_LANCZOS_LEN) * sizeof(double)))
					+ TESTS_LANCZOS_LEN; // notice that we set some offset to the actual array pointer to allow the use of negative indexes in math below
			for (k = -TESTS_LANCZOS_LEN; k < TESTS_LANCZOS_LEN; k++) {
				x = (double) (k + 1) - (double) (j) / (double) (resample_mult); // the actual datapoint where the lanczos_window is defined
				lanczos_window[j][k] = (double) (TESTS_LANCZOS_LEN)
						* sin(M_PI * x)
						* sin(M_PI * x / (double) (TESTS_LANCZOS_LEN))
						/ (POW2(M_PI) * POW2(x));
			}
		}

	}

	// The actual resampling
	for (i = 0; i < len; i++) {
		resampled[i * resample_mult] = data[i]; // lanczos_window[0] is just an impulse array
		for (j = 1; j < resample_mult; j++) {
			resampled[i * resample_mult + j] = 0.0;
			for (k = -TESTS_LANCZOS_LEN; k < TESTS_LANCZOS_LEN; k++) {
				resampled[i * resample_mult + j] += lanczos_window[j][k]
						* data[i + k + 1];
			}
		}
	}
	return (0);
}

// remove the average, and scale the data so its standard deviation is one
static void inline tests_whiten_data(double *data, int length, double *ave_,double *sd_, double *output_data) {
	double ave = 0.0, ave2 = 0.0, inv_sd;
	int i;
	double *whitened = output_data ? output_data : data;
	for (i = 0; i < length; i++) {
		ave += data[i];
		ave2 += POW2(data[i]);
	}
	ave /= (double) (length);
	if (ave_) {
		*ave_ = ave;
	}
	ave2 /= (double) (length);
	if (ave2 == POW2(ave)) { // Is the signal constant? Constant signal can't be whitened...
		if (sd_) {
			*sd_ = 0.0;
		}
		for (i = 0; i < length; i++) {
			whitened[i] = 0.0;
		}
		return;
	}
	if (sd_) {
		*sd_ = sqrt(ave2 - POW2(ave));
		inv_sd = 1.0 / (*sd_);
	} else {
		inv_sd = 1.0 / (sqrt(ave2 - POW2(ave)));
	}
	for (i = 0; i < length; i++) {
		whitened[i] = (data[i] - ave) * inv_sd;
	}
}


// remove the average, and scale the data so its standard deviation is one. Save the SD value to the end of the data vector (dirty trick to save some allocation requests)
static void inline tests_whiten_and_downsample_data2(double *data, int length,int mult, double *output_data) {
	double ave = 0.0, ave2 = 0.0, inv_sd;
	int i;
	double *whitened = output_data ? output_data : data;
	for (i = 0; i < length; i++) {
		ave += data[i * mult];
		ave2 += POW2(data[i * mult]);
	}
	ave /= (double) (length);
	ave2 /= (double) (length);
	whitened[length] = ave2 - POW2(ave);  // Dirty trick to be realized during allocation of the data!!!
	if(whitened[length]>0.000001){ // notice that due to rounding errors, even negative values are possible
		whitened[length]=sqrt(whitened[length]);
		inv_sd = 1.0 / whitened[length];

		for (i = 0; i < length; i++) {
			whitened[i] = (data[i * mult] - ave) * inv_sd;
		}
	}
	else{ // no signal -- just line
		for (i = 0; i < length; i++) {
			whitened[i] = 0.0;
		}
		whitened[i] = 1.0;
	}
}

static inline void tests_sumproduct(double *data1, double *data2, int len,
		double *sumproduct) {
	int i;
	*sumproduct = 0.0;
	for (i = 0; i < len; i++) {
		*sumproduct += data1[i] * data2[i];
	}
}

// data1 hasn't been whitened
static inline void tests_sumproduct_s1(double *data1, double *data2, int len,double *sumproduct) {
	double ave = 0.0, ave2 = 0.0, sd;
	int i;
	for (i = 0; i < len; i++) {
		ave += data1[i];
		ave2 += POW2(data1[i]);
	}
	ave /= (double) (len);
	ave2 /= (double) (len);
	*sumproduct=0.0;
	sd = sqrt(ave2 - POW2(ave));
	if(sd>0.0){
		*sumproduct = 0.0;
		for (i = 0; i < len; i++) {
			*sumproduct += (data1[i] - ave) * data2[i];
		}
		*sumproduct /= sd;
	}
}

// data1 hasn't been whitened
static inline void tests_sumproduct_and_sd(double *data1, double *data2, int len,double *sumproduct,double *sd) {
	double ave = 0.0, ave2 = 0.0;
	int i;
	for (i = 0; i < len; i++) {
		ave += data1[i];
		ave2 += POW2(data1[i]);
	}
	ave /= (double) (len);
	ave2 /= (double) (len);
	*sumproduct = 0.0;
	*sd = sqrt(ave2 - POW2(ave));
	if(*sd>0.0){
		for (i = 0; i < len; i++) {
			*sumproduct += (data1[i] - ave) * data2[i];
		}
		*sumproduct /= *sd;
	}
}

int tests_create_whitened_dataset(struct data *data, struct test_events *events,int win_before, int win_after, double ****dataset, double **std) {
	int len = win_before + 1 + win_after;
	int ch;
	size_t n_events;
	double one_std;
	double inv_nch = 1.0 / (double) (data->number_of_channels);
	size_t i;

	// alloc
	(*dataset) = matrix_calloc_d3(events->n_events,data->number_of_channels,len);
/*	(*dataset) = my_malloc(events->n_events * sizeof(double **));
	for (i = 0; i < events->n_events; i++) {
		(*dataset)[i] = my_malloc(data->number_of_channels * sizeof(double *));
		for (ch = 0; ch < data->number_of_channels; ch++) {
			(*dataset)[i][ch] = my_calloc(len, sizeof(double));
		}
	}*/

	// Check the beat range. Mark the 'skipped events' with std[i]==0.0;
	(*std) = my_calloc(events->n_events, sizeof(double));
	for (i = 0; events->events[i].sample < win_before; i++) { // skip the first beats if their windows don't fit the data
		*std[i] = 0.0;
	}
	for (n_events = events->n_events;events->events[n_events - 1].sample + win_after + 1>= data->samples_per_channel; n_events--) { // skip the last beats if their windows don't fit the data
		*std[i] = 0.0;
	}

	// loop and whiten all the samples
	for (; i < n_events; i++) {
		for (ch = 0; ch < data->number_of_channels; ch++) {
			tests_whiten_data(data->ch[ch].raw + events->events[i].sample - win_before,len, NULL, &one_std, (*dataset)[i][ch]);
			(*std)[i] += one_std;
		}
		(*std)[i] *= inv_nch;
	}
	return (0);
}

int tests_create_whitened_dataset2(struct data *data, struct test_events *events,int win_before, int win_after, double ****dataset) {
	int len = win_before + 1 + win_after;
	int ch;
	size_t n_events;
	double one_std;
	size_t i;

	// alloc
	(*dataset) = matrix_calloc_d3(events->n_events,data->number_of_channels,len+1);
/*	(*dataset) = my_malloc(events->n_events * sizeof(double **));
	for (i = 0; i < events->n_events; i++) {
		(*dataset)[i] = my_malloc(data->number_of_channels * sizeof(double *));
		for (ch = 0; ch < data->number_of_channels; ch++) {
			(*dataset)[i][ch] = my_calloc(len+1, sizeof(double)); // dirty trick to save the std value to the end of data
		}
	}*/

	// Check the beat range. Mark the 'skipped events' with std[i]==0.0;
	for (i = 0; events->events[i].sample < win_before; i++) { // skip the first beats if their windows don't fit the data
		for (ch = 0; ch < data->number_of_channels; ch++) {
			(*dataset)[i][ch][len] = 0.0;
		}
	}
	for (n_events = events->n_events;n_events>0 && events->events[n_events - 1].sample + win_after + 1>= data->samples_per_channel; n_events--) { // skip the last beats if their windows don't fit the data
		for (ch = 0; ch < data->number_of_channels; ch++) {
			(*dataset)[n_events -1][ch][len] = 0.0;
		}
	}

	// loop and whiten all the samples
	for (; i < n_events; i++) {
		for (ch = 0; ch < data->number_of_channels; ch++) {
			tests_whiten_data(data->ch[ch].raw + events->events[i].sample - win_before,len, NULL, &one_std, (*dataset)[i][ch]);
			(*dataset)[i][ch][len] = one_std;
		}
	}
	return (0);
}


static inline void tests_create_template_set2(struct data *data, size_t set, int len_, int move_, int mult_, double ****templates_) {
	size_t i;
	int ch;
	static int len = 0;
	static int move = 0;
	static int mult = 0;
	static int nch = 0;
	static double *resampled;
	static double ***templates;
	static int alloced = 0;

	if (!data || len != len_ || move != move_ || mult != mult_ || nch != data->number_of_channels) { // free?
		if (alloced) {
			my_free(resampled);
			matrix_free_d3(templates);
			alloced = 0;
		}
		if(!data){
			return;
		}
	}

	if (!alloced) { // alloc?
		len = len_;
		move = move_;
		mult = mult_;
		nch = data->number_of_channels;
		resampled = my_malloc((len + move) * mult * sizeof(double));
		templates = matrix_malloc_d3(move * mult,nch,len +1);
		*templates_ = templates;
		alloced = 1;
	}

	// create the templates

	for (ch = 0; ch < data->number_of_channels; ch++) {
		tests_resample_segment(data->ch[ch].raw + set, resampled, len + move,mult);
		for (i = 0; i < move * mult; i++) {
			tests_whiten_and_downsample_data2(resampled + i, len, mult, templates[i][ch]);
		}
	}
	*templates_ = templates;

}



static inline void tests_alloc_template_set2(int nch, int len, int move,int mult, double ****templates_) {

	*templates_=matrix_malloc_d3(move * mult,nch,len+1);
	return;

}




static inline void tests_templates_from_average2(double **average, double ***templates, int nch, int len, int offset, int move, int mult) {
	int ch;
	int i;
	for (ch = 0; ch < nch; ch++) {
		for (i = 0; i < move * mult; i++) {
			tests_whiten_and_downsample_data2(average[ch] + i + offset, len, mult, templates[i][ch]);
		}
	}
	return;
}

static inline void tests_add_to_average(struct data *data, int set, int len_, int mult_, int resampled_set_offset, double **average) {
	static double *resampled = NULL;
	static int len = 0;
	static int mult = 0;
	static int alloced = 0;
	int i;
	int ch;

	if (!data || len != len_ || mult != mult_) { // free?
		if (alloced) {
			my_free(resampled);
			len = 0;
			mult = 0;
			alloced = 0;
		}
		if(!data){
			return;
		}
	}
	if (!alloced) { // alloc?
		len = len_;
		mult = mult_;
		resampled = my_malloc((len + 1) * mult * sizeof(double));
		alloced = 1;
	}

	if (resampled_set_offset < 0) { // change resampled_set_offset to positive
		set--;
		resampled_set_offset += mult;
	}
	for (ch = 0; ch < data->number_of_channels; ch++) {
		tests_resample_segment(data->ch[ch].raw + set, resampled, len + 1,
				mult);
		for (i = 0; i < len * mult; i++) {
			average[ch][i] += resampled[i + resampled_set_offset];
		}
	}
	return;
}

static inline void tests_get_downsampled_sumproduct(double *data1,
		double *data2, int len, int mult, double *sumproduct) {
	int i;
	double ave1 = 0.0, ave1_2 = 0.0, sd1;
	double ave2 = 0.0, ave2_2 = 0.0, sd2;
	for (i = 0; i < len; i++) {
		ave1 += data1[i * mult];
		ave1_2 += POW2(data1[i * mult]);
		ave2 += data2[i];
		ave2_2 += POW2(data2[i]);
	}
	ave1 /= (double) (len);
	ave2 /= (double) (len);
	ave1_2 /= (double) (len);
	ave2_2 /= (double) (len);
	sd1 = sqrt(ave1_2 - POW2(ave1));
	sd2 = sqrt(ave2_2 - POW2(ave2));
	*sumproduct = 0.0;
	for (i = 0; i < len; i++) {
		*sumproduct += (data1[i * mult] - ave1) * (data2[i] - ave2);
	}
	*sumproduct /= (sd1 * sd2) * (double) (len);
}

static inline void tests_get_max_sumproduct(double *data1, double *data2,
		int len, int max_move, int *min_ind, double *max_sumproduct) {
	double sumproduct;
	int i;
	tests_sumproduct_s1(data1, data2, len, max_sumproduct);
	*min_ind = 0;
	for (i = 1; i < max_move; i++) {
		tests_sumproduct_s1(data1 + i, data2, len, &sumproduct);
		if (sumproduct > *max_sumproduct) {
			*max_sumproduct = sumproduct;
			*min_ind = i;
		}
	}
	*max_sumproduct /= (double) (len);
}


int tests_get_max_correlations(double ***templates, int n_templates,double ***samples, size_t n_samples, int len, int nch, int *inds,double *correlations) {
	size_t i;
	double correlation, one_correlation;
	int j;
	int ch;
	double per_nch_and_len = 1.0 / (double) (nch * len);
	for (i = 0; i < n_samples; i++) {
		correlations[i] = -1.0;
		for (j = 0; j < n_templates; j++) {
			correlation = 0.0;
			for (ch = 0; ch < nch; ch++) {
				tests_sumproduct(templates[j][ch], samples[i][ch], len,
						&one_correlation);
				correlation += one_correlation;
			}
			if (correlation > correlations[i]) {
				correlations[i] = correlation;
				inds[i] = j;
			}
		}
		correlations[i] *= per_nch_and_len;
	}

	return (0);
}

int tests_get_max_correlations2(double ***templates, int n_templates,
		double ***samples, size_t n_samples, int len, int nch, int initial_step,
		int *inds, double *correlations) {
	size_t i;
	double correlation, one_correlation;
	int j;
	int ch;
	int step;
	double per_nch_and_len = 1.0 / (double) (nch * len);
	for (i = 0; i < n_samples; i++) {
		correlations[i] = -1.0;
		for (j = 0; j < n_templates; j += initial_step) {
			correlation = 0.0;
			for (ch = 0; ch < nch; ch++) {
				tests_sumproduct(templates[j][ch], samples[i][ch], len,
						&one_correlation);
				correlation += one_correlation;
			}
			if (correlation > correlations[i]) {
				correlations[i] = correlation;
				inds[i] = j;
			}
		}
		for (step = initial_step / 2; step > 0; step /= 2) {
			if (inds[i] > step) {
				correlation = 0.0;
				for (ch = 0; ch < nch; ch++) {
					tests_sumproduct(templates[inds[i] - step][ch],
							samples[i][ch], len, &one_correlation);
					correlation += one_correlation;
				}
				if (correlation > correlations[i]) {
					correlations[i] = correlation;
					inds[i] -= step;
					continue;
				}
			}
			if (inds[i] + step < n_templates) {
				correlation = 0.0;
				for (ch = 0; ch < nch; ch++) {
					tests_sumproduct(templates[inds[i] + step][ch],
							samples[i][ch], len, &one_correlation);
					correlation += one_correlation;
				}
				if (correlation > correlations[i]) {
					correlations[i] = correlation;
					inds[i] += step;
					continue;
				}
			}
		}
		correlations[i] *= per_nch_and_len;
	}

	return (0);
}

int tests_get_max_correlations3(double ***templates, int n_templates,double ***samples, size_t n_samples, char *skip_these, int len, int nch,int initial_step, int *inds, double *correlations) {
	size_t i;
	double correlation, one_correlation;
	int j;
	int ch;
	int step;
	double per_nch_and_len = 1.0 / (double) (nch * len);
	for (i = 0; i < n_samples; i++) {
		correlations[i] = -99999999.9;
		if (skip_these && skip_these[i])
			continue;
		for (j = 0; j < n_templates; j += initial_step) {
			correlation = 0.0;
			for (ch = 0; ch < nch; ch++) {
				tests_sumproduct(templates[j][ch], samples[i][ch], len,
						&one_correlation);
				correlation += one_correlation;
			}
			if (correlation > correlations[i]) {
				correlations[i] = correlation;
				inds[i] = j;
			}
		}
		for (step = initial_step / 2; step > 0; step /= 2) {
			if (inds[i] > step) {
				correlation = 0.0;
				for (ch = 0; ch < nch; ch++) {
					tests_sumproduct(templates[inds[i] - step][ch],
							samples[i][ch], len, &one_correlation);
					correlation += one_correlation;
				}
				if (correlation > correlations[i]) {
					correlations[i] = correlation;
					inds[i] -= step;
					continue;
				}
			}
			if (inds[i] + step < n_templates) {
				correlation = 0.0;
				for (ch = 0; ch < nch; ch++) {
					tests_sumproduct(templates[inds[i] + step][ch],
							samples[i][ch], len, &one_correlation);
					correlation += one_correlation;
				}
				if (correlation > correlations[i]) {
					correlations[i] = correlation;
					inds[i] += step;
					continue;
				}
			}
		}
		correlations[i] *= per_nch_and_len;
	}

	return (0);
}

int tests_get_max_correlations4(double ***templates, int n_templates,char *ok_template_channels, double ***samples, size_t n_samples,char **ok_sample_channels, char *skip_these, int len, int nch_,int initial_step, int *inds, double *correlations) {
	size_t i;
	double correlation, one_correlation;
	int j;
	int ch;
	double nok;
	double per_nch_and_len;
	int step;
	static char *okch = NULL;
	static int nch = 0;

	if (nch_ != nch) {
		nch = nch_;
		if (nch_ <= 0) { // free
			if (okch) {
				my_free(okch);
				okch = NULL;
			}
			return (0);
		} else { // alloc
			okch = my_realloc(okch, nch * sizeof(char));
		}
	}
	if (ok_template_channels == NULL && ok_sample_channels == NULL) { // Is there bad / rejected channels --  should we use algorithm version that is optimized for 'all ok' setup?
		return (tests_get_max_correlations3(templates, n_templates, samples,
				n_samples, skip_these, len, nch, initial_step, inds,
				correlations));
	}
	for (i = 0; i < n_samples; i++) {
		correlations[i] = -99999999.9;
		if (skip_these && skip_these[i])
			continue;

		// check the ok channels
		for (nok = 0, ch = 0; ch < nch; ch++) {
			if ((ok_template_channels == NULL || ok_template_channels[ch]) && (ok_sample_channels == NULL	|| ok_sample_channels[i] == NULL || ok_sample_channels[i][ch])) {
				okch[ch] = 1;
				nok++;
			} else {
				okch[ch] = 0;
			}
		}
		if (nok == 0) {
			inds[i] = n_templates / 2;
			continue;
		}
		per_nch_and_len = 1.0 / (double) (nok * len);

		// find the max correlation in 'initial_step' resolution
		for (j = 0; j < n_templates; j += initial_step) {
			correlation = 0.0;
			for (ch = 0; ch < nch; ch++) {
				if (okch[ch]) {
					tests_sumproduct(templates[j][ch], samples[i][ch], len, &one_correlation);
					correlation += one_correlation;
				}
			}
			correlation *= per_nch_and_len;
			if (correlation > correlations[i]) {
				correlations[i] = correlation;
				inds[i] = j;
			}
		}
		// look for the actual max value around the initial max value..
		for (step = initial_step / 2; step > 0; step /= 2) {
			if (inds[i] > step) {
				correlation = 0.0;
				for (ch = 0; ch < nch; ch++) {
					if (okch[ch]) {
						tests_sumproduct(templates[inds[i] - step][ch],
								samples[i][ch], len, &one_correlation);
						correlation += one_correlation;
					}
				}
				correlation *= per_nch_and_len;
				if (correlation > correlations[i]) {
					correlations[i] = correlation;
					inds[i] -= step;
					continue;
				}
			}
			if (inds[i] + step < n_templates) {
				correlation = 0.0;
				for (ch = 0; ch < nch; ch++) {
					if (okch[ch]) {
						tests_sumproduct(templates[inds[i] + step][ch],
								samples[i][ch], len, &one_correlation);
						correlation += one_correlation;
					}
				}
				correlation *= per_nch_and_len;
				if (correlation > correlations[i]) {
					correlations[i] = correlation;
					inds[i] += step;
					continue;
				}
			}
		}
	}

	return (0);
}

#define TESTS_CORR_WEIGHT(weight,std1,std2) (std1*std2)



int tests_get_max_correlations5(double *ch_goodness, double ***templates, int n_templates,double ***samples, size_t *process_these, size_t n_to_process, int len, int nch,int initial_step, int *inds, double *correlations) {
	size_t i;
	double correlation, one_correlation;
	int j;
	int ch;
	int step;
	size_t ind;
	double weight,total_weight;
	for (i = 0; i < n_to_process; i++) {
		correlations[i] = -99999999.9;
		ind=process_these ? process_these[i] : i;
		for (j = 0; j < n_templates; j += initial_step) {
			correlation = 0.0;
			total_weight=0.0;
			for (ch = 0; ch < nch; ch++) {
				tests_sumproduct(templates[j][ch], samples[ind][ch], len,&one_correlation);
				weight=ch_goodness[ch]*TESTS_CORR_WEIGHT(ch_weight,templates[j][ch][len],samples[ind][ch][len]); // define weight based on original STDs (stored in templates[j][ch][len] and in samples[ind][ch][len])
				total_weight += weight;
				correlation += weight * one_correlation;
			}
			if (correlation > total_weight * correlations[i]) {
				correlations[i] = correlation/total_weight;
				inds[i] = j;
			}
		}
		if(correlations[i]<0){ // too low.. stop here
			inds[i]=n_templates/2;
			continue;
		}
		for (step = initial_step / 2; step > 0; step /= 2) {
			if (inds[i] > step) {
				correlation = 0.0;
				total_weight=0.0;
				for (ch = 0; ch < nch; ch++) {
					tests_sumproduct(templates[inds[i] - step][ch],samples[ind][ch], len, &one_correlation);
					weight=ch_goodness[ch]*TESTS_CORR_WEIGHT(ch_weight,templates[inds[i] - step][ch][len],samples[ind][ch][len]); // define weight based on original STDs (stored in templates[j][ch][len] and in samples[ind][ch][len])
					correlation += weight * one_correlation;
					total_weight += weight;
				}
				if (correlation > total_weight * correlations[i]) {
					correlations[i] = correlation/total_weight;
					inds[i] -= step;
					continue;
				}
			}
			if (inds[i] + step < n_templates) {
				correlation = 0.0;
				total_weight=0.0;
				for (ch = 0; ch < nch; ch++) {
					tests_sumproduct(templates[inds[i] + step][ch],samples[ind][ch], len, &one_correlation);
					weight=ch_goodness[ch]*TESTS_CORR_WEIGHT(ch_weight,templates[inds[i] + step][ch][len],samples[ind][ch][len]); // define weight based on original STDs (stored in templates[j][ch][len] and in samples[ind][ch][len])
					correlation += weight * one_correlation;
					total_weight += weight;
				}
				if (correlation > total_weight * correlations[i]) {
					correlations[i] = correlation / total_weight;
					inds[i] += step;
					continue;
				}
			}
		}
		correlations[i] /= (double)(len);
	}

	return (0);
}

#define TEST_WEIGHT_FOR_OK_BUT_SATURATED 0.5 /// xxx not tested yet!!!!
int tests_get_max_correlations6(double *ch_goodness_, double ***templates, int n_templates,char *bad_template_channels, float *template_ch_badness, double ***samples,char **bad_sample_channels, float **sample_ch_badness, size_t *process_these, size_t n_to_process, int len, int nch_,int initial_step, int require_same_good_ch, int *inds, double *correlations) {
	size_t i;
	double correlation, one_correlation;
	int j;
	int ch;
	double nok;
	double weight,total_weight;
	int step;
	size_t ind;
	static double *ch_goodness=NULL;
	static char *okch = NULL;
	static int nch = 0;


	if (nch_ != nch) {
		nch = nch_;
		if (nch_ <= 0) { // free
			if (okch) {
				my_free(okch);
				okch = NULL;
				my_free(ch_goodness);
				ch_goodness=NULL;
			}
			return (0);
		} else { // alloc
			okch = my_realloc(okch, nch * sizeof(char));
			ch_goodness = my_realloc(ch_goodness, nch * sizeof(double));
		}
	}
	if (bad_template_channels == NULL && bad_sample_channels == NULL) { // Is there bad / rejected channels --  should we use algorithm version that is optimized for 'all ok' setup?
		return (tests_get_max_correlations5(ch_goodness_, templates, n_templates, samples,process_these, n_to_process, len, nch, initial_step, inds,correlations));
	}
	for (i = 0; i < n_to_process; i++) {
		correlations[i] = -99999999.9;
		ind=process_these ? process_these[i] : i;
		// check the ok channels
		for (nok = 0, ch = 0; ch < nch; ch++) {
			if ((bad_sample_channels == NULL || bad_sample_channels[ind] == NULL || bad_sample_channels[ind][ch]<TEST_EVENT_TOTALLY_BAD) && sample_ch_badness[ind][ch]<1.0) { // event ch ok?
				if ((bad_template_channels == NULL || bad_template_channels[ch]<TEST_EVENT_TOTALLY_BAD) && template_ch_badness[ch]<1.0) {				// template ch ok?
					ch_goodness[ch]=ch_goodness_[ch]*(1.0-MAX(template_ch_badness[ch],sample_ch_badness[ind][ch]));
					okch[ch] = 1;
					nok++;
				}
				else{
					okch[ch] = 0;
					if(require_same_good_ch){
						nok=0;
						break;
					}
				}
			} else {
				okch[ch] = 0;
			}
		}
		if (nok == 0) {
			inds[i] = n_templates / 2;
			continue;
		}

		// find the max correlation in 'initial_step' resolution
		for (j = 0; j < n_templates; j += initial_step) {
			correlation = 0.0;
			total_weight = 0.0;
			for (ch = 0; ch < nch; ch++) {
				if (okch[ch]) {
					tests_sumproduct(templates[j][ch], samples[ind][ch], len, &one_correlation);
					weight=ch_goodness[ch]*TESTS_CORR_WEIGHT(ch_weight,templates[j][ch][len],samples[ind][ch][len]); // define weight based on original STDs (stored in templates[j][ch][len] and in samples[ind][ch][len])
					total_weight += weight;
					correlation += weight * one_correlation;
				}
			}
			correlation /= total_weight*len;
			if (correlation > correlations[i]) {
				correlations[i] = correlation;
				inds[i] = j;
			}
		}
		// look for the actual max value around the initial max value..
		for (step = initial_step / 2; step > 0; step /= 2) {
			if (inds[i] > step) {
				correlation = 0.0;
				total_weight = 0.0;
				for (ch = 0; ch < nch; ch++) {
					if (okch[ch]) {
						tests_sumproduct(templates[inds[i] - step][ch],samples[ind][ch], len, &one_correlation);
						weight=ch_goodness[ch]*TESTS_CORR_WEIGHT(ch_weight,templates[inds[i] - step][ch][len],samples[ind][ch][len]); // define weight based on original STDs (stored in templates[j][ch][len] and in samples[ind][ch][len])
						total_weight += weight;
						correlation += weight * one_correlation;
					}
				}
				correlation /= total_weight*len;
				if (correlation > correlations[i]) {
					correlations[i] = correlation;
					inds[i] -= step;
					continue;
				}
			}
			if (inds[i] + step < n_templates) {
				correlation = 0.0;
				total_weight = 0.0;
				for (ch = 0; ch < nch; ch++) {
					if (okch[ch]) {
						tests_sumproduct(templates[inds[i] + step][ch],samples[ind][ch], len, &one_correlation);
						weight=ch_goodness[ch]*TESTS_CORR_WEIGHT(ch_weight,templates[inds[i] + step][ch][len],samples[ind][ch][len]); // define weight based on original STDs (stored in templates[j][ch][len] and in samples[ind][ch][len])
						total_weight += weight;
						correlation += weight * one_correlation;
					}
				}
				correlation /= total_weight*len;
				if (correlation > correlations[i]) {
					correlations[i] = correlation;
					inds[i] += step;
				}
			}
		}
	}

	return (0);
}

int tests_get_max_one_ch_correlations(double ***templates, int n_templates,double ***samples,size_t *process_these, size_t n_to_process, int len, int nch,int initial_step, int *inds, double *correlations) {
	size_t i;
	double correlation;
	int j;
	int ch,max_ch;
	int step;
	size_t ind;

	for (i = 0; i < n_to_process; i++) {
		correlations[i] = -1.0;
		ind=process_these ? process_these[i] : i;

		// find the max correlation in 'initial_step' resolution
		for (j = 0; j < n_templates; j += initial_step) {
			for (ch = 0; ch < nch; ch++) {
				tests_sumproduct(templates[j][ch], samples[ind][ch], len, &correlation);
				correlation/=len;

				if (correlation > correlations[i] && tests_are_stds_similar(templates[j][ch][len], samples[ind][ch][len])) {
					correlations[i] = correlation;
					inds[i] = j;
					max_ch=ch;
				}
			}
		}

		if(correlations[i]>0.0){
			// look for the actual max value around the initial max value..
			for (step = initial_step / 2; step > 0; step /= 2) {
				if (inds[i] > step) {
					tests_sumproduct(templates[inds[i] - step][max_ch],samples[ind][max_ch], len, &correlation);
					correlation /= len;
					if (correlation > correlations[i] && tests_are_stds_similar(templates[inds[i] - step][max_ch][len], samples[ind][max_ch][len])) {
						correlations[i] = correlation;
						inds[i] -= step;
						continue;
					}
				}
				if (inds[i] + step < n_templates) {
					tests_sumproduct(templates[inds[i] + step][max_ch],samples[ind][max_ch], len, &correlation);
					correlation /= len;
					if (correlation > correlations[i]  && tests_are_stds_similar(templates[inds[i] + step][max_ch][len], samples[ind][max_ch][len])) {
						correlations[i] = correlation;
						inds[i] += step;
					}
				}
			}
		}
	}

	return (0);
}

// interpolate the parabolic peak location (index) around the array peak value at x[0]. Notice that the peak can't be on the edge of the array -- x[-1] and x[1] has to be allowed
inline double tests_parabolic_max_ind(double *x) {
	return ((x[-1] - x[1]) / (2.0 * (x[-1] + 2.0 * x[0] + x[1])));
}

// interpolate the parabolic peak value around the array peak value at x[0]. Notice that the peak can't be on the edge of the array -- x[-1] and x[1] has to be allowed
inline double tests_parabolic_max_val(double *x) {
	return (x[0] - POW2(x[-1]-x[1]) / (8 * (x[-1] - 2.0 * x[0] + x[1])));
}


int tests_free_cluster(struct test_cluster *cluster){
	int k;
	FREE(cluster->sample_offsets);
	FREE(cluster->inds);

	for(k=0;k<cluster->n_templates;k++){
		if(cluster->bad_chs){
			FREE(cluster->bad_chs[k]);
		}
		FREE(cluster->ch_badness[k]);
		if(cluster->template_sets && cluster->template_sets[k]){
			matrix_free_d3(cluster->template_sets[k]);
			/*for (j = 0; j < cluster_set->n_move; j++) {
				for (ch = 0; ch < cluster_set->nch; ch++) {
					FREE(cluster->template_sets[k][j][ch]);
				}
				my_free(cluster->template_sets[k][j]);
			}
			my_free(cluster->template_sets[k]);*/
		}
		if(cluster->templates && cluster->templates[k]){
			matrix_free_d(cluster->templates[k]);
			/*for (ch = 0; ch < cluster_set->nch; ch++) {
				FREE(cluster->templates[k][ch]);
			}
			my_free(cluster->templates[k]);*/
		}
		if(cluster->markers){
			FREE(cluster->markers[k].initial_qrs_slope_length);
			FREE(cluster->markers[k].max_qrs_slope);
			FREE(cluster->markers[k].n_qrs_waves);
			FREE(cluster->markers[k].qrs_power);
		}
	}
	FREE(cluster->bad_chs);
	FREE(cluster->ch_badness);
	FREE(cluster->template_sets);
	FREE(cluster->templates);
	FREE(cluster->markers);

	return(0);
}


int tests_free_clusters(struct test_clusters *cluster_set){
	int k,n;
	if(cluster_set->n_clusters<=0){
		return(0);
	}
//	for(n=0;n<cluster_set->n_clusters;n++){
	for(n=cluster_set->n_clusters-1;n>=0;n--){ // memory_manager is optimized for FIFO
		FREE(cluster_set->clusters[n].sample_offsets);
		FREE(cluster_set->clusters[n].inds);

		for(k=0;k<cluster_set->clusters[n].n_templates;k++){
			if(cluster_set->clusters[n].bad_chs){
				FREE(cluster_set->clusters[n].bad_chs[k]);
			}
			FREE(cluster_set->clusters[n].ch_badness[k]);
			if(cluster_set->clusters[n].template_sets && cluster_set->clusters[n].template_sets[k]){
				matrix_free_d3(cluster_set->clusters[n].template_sets[k]);
				/*for (j = 0; j < cluster_set->n_move; j++) {
					for (ch = 0; ch < cluster_set->nch; ch++) {
						FREE(cluster_set->clusters[n].template_sets[k][j][ch]);
					}
					my_free(cluster_set->clusters[n].template_sets[k][j]);
				}
				my_free(cluster_set->clusters[n].template_sets[k]);*/
			}
			if(cluster_set->clusters[n].templates && cluster_set->clusters[n].templates[k]){
				matrix_free_d(cluster_set->clusters[n].templates[k]);
				/*for (ch = 0; ch < cluster_set->nch; ch++) {
					FREE(cluster_set->clusters[n].templates[k][ch]);
				}
				my_free(cluster_set->clusters[n].templates[k]);*/
			}
			if(cluster_set->clusters[n].markers){
				FREE(cluster_set->clusters[n].markers[k].initial_qrs_slope_length);
				FREE(cluster_set->clusters[n].markers[k].max_qrs_slope);
				FREE(cluster_set->clusters[n].markers[k].n_qrs_waves);
				FREE(cluster_set->clusters[n].markers[k].qrs_power);
			}
		}
		FREE(cluster_set->clusters[n].bad_chs);
		FREE(cluster_set->clusters[n].ch_badness);
		FREE(cluster_set->clusters[n].template_sets);
		FREE(cluster_set->clusters[n].templates);
		FREE(cluster_set->clusters[n].markers);
	}
	FREE(cluster_set->clusters);
	cluster_set->clusters=NULL;
	cluster_set->n_clusters=0;
	return(0);
}



int tests_print_partial_channel_cluster_ids(struct test_clusters *cluster_set,struct test_events *events, size_t min_sample, size_t max_sample){
	int i,j,k,n;
	int ch;
	int in_range=1;
	printf("Clusters with bad channels: ");
	for(n=0,i=0;i<cluster_set->n_clusters;i++){
		if(events){
			for(in_range=0,j=0;j<cluster_set->clusters[i].n_beats;j++){
				if(events->events[cluster_set->clusters[i].inds[j]].sample>min_sample && events->events[cluster_set->clusters[i].inds[j]].sample<max_sample){
					in_range=1;
					break;
				}
			}
		}
		if(in_range){
			for(k=0;k<cluster_set->clusters[i].n_templates;k++){
				if(cluster_set->clusters[i].bad_chs[k]){
					for(ch=0;ch<cluster_set->nch;ch++){
						if(cluster_set->clusters[i].bad_chs[k][ch]==TEST_EVENT_TOTALLY_BAD){
							printf("%d/%d, ",i,k);
							n++;
							break;
						}
					}
				}
			}
		}
	}
	printf("\nBad channels in %d/%d clusters\n",n,cluster_set->n_clusters);
	return(0);
}

int tests_cluster_comparison(const void *cluster1, const void *cluster2) {
	if (((struct test_cluster *) (cluster1))->n_beats
			> ((struct test_cluster *) (cluster2))->n_beats) {
		return (-1);
	} else if (((struct test_cluster *) (cluster1))->n_beats
			< ((struct test_cluster *) (cluster2))->n_beats) {
		return (1);
	}
	return (0);
}


int tests_recluster(struct data *data, double *ch_goodness, struct test_events *events,double ***dataset, char **bad_chs_in_dataset,float **badness_in_dataset, struct test_clusters *cluster_set) {
	struct conf_cluster *conf=conf_cluster_get();
	double *correlations;
	double tmp_corr;
	int tmp_ind;
	int *inds;
	size_t i, j, k,  max_j;
	int ch;
	int move = conf->move * data->sample_freq + 0.5;
	int win_before = conf->win_before * data->sample_freq + 0.5;
	int win_after = conf->win_after * data->sample_freq + 0.5;
	int ave_win_before = conf->ave_win_before * data->sample_freq + 0.5;
	int ave_win_after = conf->ave_win_after * data->sample_freq + 0.5;
	struct test_cluster *clusters = cluster_set->clusters;
	int n_clusters = cluster_set->n_clusters;


	// alloc
	correlations = my_malloc(events->n_events * sizeof(double));
	inds = my_malloc(events->n_events * sizeof(int));


	// find matches to the already defined cluster templates
	for (i = 0;i < events->n_events && events->events[i].sample < ave_win_before;i++);
	for(;i<events->n_events && events->events[i].sample + ave_win_after < data->samples_per_channel;i++){

		correlations[i] = -2.0;
		for (j = 0;j < n_clusters; j++) {


			for (k = 0; k < clusters[j].n_templates; k++) {

				// check that the template has enough ok channels
				if(clusters[j].bad_chs[k]){
					for (ch = 0; ch < data->number_of_channels; ch++) {
						if(!bad_chs_in_dataset || !bad_chs_in_dataset[i] || bad_chs_in_dataset[i][ch]<TEST_EVENT_TOTALLY_BAD){  // ch ok in event, or at least not totally bad?
							if(clusters[j].bad_chs[k][ch]>=TEST_EVENT_TOTALLY_BAD){    // ch in template totally bad?
								break;
							}
						}
					}
					if(ch < data->number_of_channels){
						continue;
					}
				}

				tests_get_max_correlations6(ch_goodness, clusters[j].template_sets[k],move * conf->frequency_mult, clusters[j].bad_chs[k],clusters[j].ch_badness[k], dataset+i, bad_chs_in_dataset+i,badness_in_dataset+i,NULL,1, win_before + 1 + win_after,data->number_of_channels, conf->frequency_mult,1,&tmp_ind, &tmp_corr);
				if (tmp_corr > correlations[i] && tests_are_one_ch_template_stds_similar(clusters[j].template_sets[k][tmp_ind], dataset[i],  win_before + 1 + win_after)) {
					max_j = j;
					correlations[i] = tmp_corr;
					inds[i] = tmp_ind;
				}
			}
		}
		if (correlations[i] > -1.0) {
			clusters[max_j].n_beats++;
			clusters[max_j].inds = my_realloc(clusters[max_j].inds,clusters[max_j].n_beats * sizeof(size_t));
			clusters[max_j].sample_offsets = my_realloc(clusters[max_j].sample_offsets,clusters[max_j].n_beats * sizeof(int));
			clusters[max_j].sample_offsets[clusters[max_j].n_beats - 1] = inds[i] - move / 2 * conf->frequency_mult;
			clusters[max_j].inds[clusters[max_j].n_beats - 1] = i;
		}
		else{
			printf("Oops! Didn't found any cluster match for %lu/%lu\n",i,events->events[i].sample);
		}
	}
	// Sort the clusters based on their beat count
	qsort(clusters, n_clusters, sizeof(struct test_cluster),tests_cluster_comparison);

	// Check if there are empty clusters to drop & free
	for(j=0;j<n_clusters && clusters[j].n_beats>0;j++);
	if(j<n_clusters){
		cluster_set->n_clusters=j;
		for(;j<n_clusters;j++){
			tests_free_cluster(clusters+j);
		}
		cluster_set->clusters=my_realloc(clusters,cluster_set->n_clusters * sizeof(struct test_cluster));
	}

	my_free(correlations);
	my_free(inds);

	return(0);
}

size_t tests_free_cluster_match(struct data *data, size_t start, size_t end, struct test_clusters *cluster_set, double *ch_goodness, size_t min_n_beats, size_t *sample, double *sample_offset, int *best_n, double *corr){
	struct conf_cluster *cluster_conf=conf_cluster_get();
	size_t i,k,n;
	size_t best_k;
	int j;
	int ch;
	int template_ind;
	double plen=1.0/(double)(cluster_set->template_in_sets_len);
	double one_corr;
	double sd;
	double correlation;
	double weight;
	double total_weight;
	*sample = 0;
	*corr=-2.0;
	*sample_offset=0.0;

	// Find the best one ch correlation
	for(n=0;n<cluster_set->n_clusters /*&& cluster_set->clusters[n].n_beats>=min_n_beats*/ && *corr<cluster_conf->similarity_limit;n++){
		if(cluster_set->clusters[n].type==TEST_ARTIFACT || cluster_set->clusters[n].n_beats<min_n_beats/* || cluster_set->clusters[n].type==TEST_BEAT_UNDEFINED*/){  // xxx TODO: use ordered clusters and trust that they clusters are ordered
			continue;
		}
		//			template_ind=cluster_set->n_move/2-cluster_set->clusters[n].cluster_offset; // TODO: use cluster_offste and check - vs + ...! Notice that is this is used, it should be 'readjusted' again from *sample timing, since we want / assume times relative to template
		template_ind=cluster_set->n_move/2;
		for(k=0;k<cluster_set->clusters[n].n_templates;k++){
			for(i=start;i<end;i++){
				for(ch=0;ch<data->number_of_channels;ch++){
					correlation=0.0;
					total_weight=0.0;
					if(cluster_set->clusters[n].bad_chs[k]==NULL || cluster_set->clusters[n].bad_chs[k][ch]==0){
						tests_sumproduct_and_sd(data->ch[ch].raw+i,cluster_set->clusters[n].template_sets[k][template_ind][ch],cluster_set->template_in_sets_len,&one_corr, &sd);
						weight=ch_goodness[ch]*(1.0-cluster_set->clusters[n].ch_badness[k][ch])*TESTS_CORR_WEIGHT(1.0,cluster_set->clusters[n].template_sets[k][template_ind][ch][cluster_set->template_in_sets_len],sd);
						total_weight+=weight;
						correlation+=weight*one_corr;
					}
				}
				if(total_weight){
					correlation*=plen/total_weight; // XXX
					if(correlation>*corr){
						*corr=correlation;
						*sample=i;
						*best_n=n;
						best_k=k;
					}
				}
			}
		}
	}
	// finetune the correlation
	if(*corr>0.0){
		for(j=template_ind-cluster_conf->frequency_mult+1;j<template_ind+cluster_conf->frequency_mult-1;j++){
			if(j==template_ind){
				continue;
			}
			for(ch=0;ch<data->number_of_channels;ch++){
				correlation=0.0;
				total_weight=0.0;
				if(cluster_set->clusters[*best_n].bad_chs[best_k]==NULL || cluster_set->clusters[*best_n].bad_chs[best_k][ch]==0){
					tests_sumproduct_and_sd(data->ch[ch].raw+*sample,cluster_set->clusters[*best_n].template_sets[best_k][j][ch],cluster_set->template_in_sets_len,&one_corr, &sd);
					weight=ch_goodness[ch]*(1.0-cluster_set->clusters[*best_n].ch_badness[best_k][ch])*TESTS_CORR_WEIGHT(ch_weight,cluster_set->clusters[*best_n].template_sets[best_k][j][ch][cluster_set->template_in_sets_len],sd);
					total_weight+=weight;
					correlation+=weight*one_corr;
				}
			}
			if(total_weight){
				correlation*=plen/total_weight;
				if(correlation>*corr){
					*corr=correlation;
					*sample_offset=(double)(j-template_ind);
				}
			}
		}
	}

	return(0);
}

size_t tests_one_ch_cluster_match(struct data *data, size_t start, size_t end, struct test_clusters *cluster_set, size_t min_n_beats, size_t *sample, double *sample_offset, int *best_n, double *corr){
	struct conf_cluster *cluster_conf=conf_cluster_get();
	size_t i,k,n;
	size_t best_k;
	int j;
	int best_ch=0,ch;
	int template_ind;
	double plen=1.0/(double)(cluster_set->template_in_sets_len);
	double one_corr;
	*sample = 0;
	*corr=-2.0;
	*sample_offset=0.0;

	// Find the best one ch correlation
	for(n=0;n<cluster_set->n_clusters /*&& cluster_set->clusters[n].n_beats>=min_n_beats*/ && *corr<cluster_conf->similarity_limit;n++){
		if(cluster_set->clusters[n].type==TEST_ARTIFACT || cluster_set->clusters[n].n_beats<min_n_beats/* || cluster_set->clusters[n].type==TEST_BEAT_UNDEFINED*/){  // xxx TODO: use ordered clusters and trust that they clusters are ordered
			continue;
		}
		//			template_ind=cluster_set->n_move/2-cluster_set->clusters[n].cluster_offset; // TODO: check - vs + ...!!!
		template_ind=cluster_set->n_move/2;
		for(k=0;k<cluster_set->clusters[n].n_templates;k++){
			for(ch=0;ch<data->number_of_channels;ch++){
				if(cluster_set->clusters[n].bad_chs[k]==NULL || cluster_set->clusters[n].bad_chs[k][ch]==0){
					for(i=start;i<end/*-cluster_set->template_in_sets_len*/;i++){
						tests_sumproduct_s1(data->ch[ch].raw+i,cluster_set->clusters[n].template_sets[k][template_ind][ch],cluster_set->template_in_sets_len,&one_corr);
						one_corr*=plen;
						if(one_corr>*corr){
							*corr=one_corr;
							*sample=i;
							best_ch=ch;
							*best_n=n;
							best_k=k;
						}
					}
				}
			}
		}
	}
	// finetune the correlation
	if(*corr>0.0){
		for(j=template_ind-cluster_conf->frequency_mult+1;j<template_ind+cluster_conf->frequency_mult-1;j++){
			if(j==template_ind){
				continue;
			}
			tests_sumproduct_s1(data->ch[best_ch].raw+*sample,cluster_set->clusters[*best_n].template_sets[best_k][j][best_ch],cluster_set->template_in_sets_len,&one_corr);
			one_corr*=plen;
			if(one_corr>*corr){
				*corr=one_corr;
				*sample_offset=(double)(j-template_ind);
			}

		}
	}

	return(0);
}

//#define TESTS_NAN_DATA_CHECK

int create_cinc2017_event_clusters(struct data *data,  double *ch_goodness, struct test_events *events,double ***dataset, size_t *process_these, size_t n_to_process, struct test_clusters *cluster_set) {
	struct conf_cluster *conf=conf_cluster_get();
	double ***templates, ***ave_templates;
	double **average;
//	double *std, *templ_std;
	double *correlations;
	double tmp_corr, tmp_f;
	int tmp_ind;
	int *inds;
	size_t template_ind = 5;
	size_t i, j, k, n_ave, max_j;
	int ch;
	size_t n_unprocessed = n_to_process;
	int clusters_alloced = cluster_set->n_clusters;
	int move = conf->move * data->sample_freq + 0.5;
	int win_before = conf->win_before * data->sample_freq + 0.5;
	int win_after = conf->win_after * data->sample_freq + 0.5;
	int ave_win_before = conf->ave_win_before * data->sample_freq + 0.5;
	int ave_win_after = conf->ave_win_after * data->sample_freq + 0.5;
	int ave_offset = (ave_win_before - win_before) * conf->frequency_mult;
	struct test_cluster *clusters = cluster_set->clusters;
	int n_clusters = cluster_set->n_clusters;
	int initial_tries=0;
	int max_n_initial_tries=20;
	int minimum_main_n_ave=MIN(conf->max_n_ave/2,n_to_process/3);
	double template_event_probability;

	srand(17); // same 'random numbers' on every run -- good for testing effects on small parameter modifications
//	srand((unsigned int)(time(NULL)));   // 'real random numbers' -- good for testing the effect of the initial template pick

	// alloc
	correlations = my_malloc(n_to_process * sizeof(double));
	inds = my_malloc(n_to_process * sizeof(int));

	// Skip the first and the last beats that don't fit the window
	for (i = n_unprocessed - 1;i > 0 && events->events[process_these[i]].sample + ave_win_after + move >= data->samples_per_channel; i--) {
			n_unprocessed--;
		}
	for (i = 0;i < n_unprocessed && events->events[process_these[i]].sample < ave_win_before + move;i++) {
		n_unprocessed--;
		process_these[i] = process_these[n_unprocessed];
	}


	// start by finding matches to the existing clusters
	if (n_clusters) {
//		int max_k;
		for (i = 0; i < n_unprocessed; ) {
			correlations[i] = -1.0;
			for (j = 0;j < n_clusters && (correlations[i] < conf->similarity_limit || clusters[j].n_beats > 1); j++) {
				for (k = 0; k < clusters[j].n_templates; k++) {
					tests_get_max_correlations5(ch_goodness, clusters[j].template_sets[k],move * conf->frequency_mult, dataset,process_these+i,1, win_before + 1 + win_after,data->number_of_channels, conf->frequency_mult,&tmp_ind, &tmp_corr);
					if (tmp_corr > correlations[i] && tests_are_one_ch_template_stds_similar(clusters[j].template_sets[k][tmp_ind], dataset[process_these[i]],  win_before + 1 + win_after)) {
						max_j = j;
//						max_k=k;
						correlations[i] = tmp_corr;
						inds[i] = tmp_ind;
					}
				}
			}
			if (correlations[i] > conf->similarity_limit) {
				clusters[max_j].n_beats++;
				clusters[max_j].inds = my_realloc(clusters[max_j].inds,clusters[max_j].n_beats * sizeof(size_t));
				clusters[max_j].sample_offsets = my_realloc(clusters[max_j].sample_offsets,clusters[max_j].n_beats * sizeof(int));
				clusters[max_j].sample_offsets[clusters[max_j].n_beats - 1] = inds[i] - move / 2 * conf->frequency_mult;
				clusters[max_j].inds[clusters[max_j].n_beats - 1] = process_these[i];
				n_unprocessed--;
				process_these[i] = process_these[n_unprocessed]; // replace the processed with the last of the unprocessed
			}
			else{
				i++; // go to next
			}
		}
	}

	// loop until all the events belong to some cluster
	while (n_unprocessed > 0) { // are there any beats that don't belong to any clusters left?
		// Select unprocessed good (based on predefined event probability) random beat as template
		template_ind = n_unprocessed * (double) (rand()-1) / RAND_MAX;
		template_event_probability=events->events[process_these[template_ind]].probability;
		for(i=1;template_event_probability<1.0 && i<MIN(n_unprocessed,max_n_initial_tries);i++){
			j = n_unprocessed * (double) (rand()-1) / RAND_MAX;
			if(events->events[process_these[j]].probability>template_event_probability){
				template_ind=j;
				template_event_probability=events->events[process_these[template_ind]].probability;
			}
		}

		// Initialize cluster
		if (n_clusters >= clusters_alloced) {
			clusters_alloced += 100;
			clusters = my_realloc(clusters,clusters_alloced * sizeof(struct test_cluster));
		}
		clusters[n_clusters].type = TEST_BEAT_UNDEFINED;
		clusters[n_clusters].main_type_group=-1;
		clusters[n_clusters].n_beats = 0;
		clusters[n_clusters].cluster_offset = 0;
		clusters[n_clusters].sample_offsets = my_malloc(n_unprocessed * sizeof(int));
		clusters[n_clusters].inds = my_malloc(n_unprocessed * sizeof(size_t));
		clusters[n_clusters].n_templates = 1;
		clusters[n_clusters].templates = my_malloc(sizeof(double **));
		clusters[n_clusters].template_sets = my_malloc(sizeof(double ***));
		clusters[n_clusters].bad_chs = my_malloc(sizeof(char *));
		clusters[n_clusters].ch_badness = my_malloc(sizeof(float *));
		clusters[n_clusters].markers = NULL;
		clusters[n_clusters].bad_chs[0] = my_calloc(data->number_of_channels, sizeof(char));
		clusters[n_clusters].ch_badness[0] = my_calloc(data->number_of_channels, sizeof(float));


		// Find the highest correlation values with the unprocessed beats
		printf("Cluster %d: Selected %lu %lu as template\r", n_clusters + 1,process_these[template_ind], events->events[process_these[template_ind]].sample);
		tests_create_template_set2(data,events->events[process_these[template_ind]].sample - win_before - move / 2,win_before + 1 + win_after, move, conf->frequency_mult,&templates);
		tests_get_max_correlations5(ch_goodness, templates, move * conf->frequency_mult, dataset, process_these,n_unprocessed, win_before + 1 + win_after, data->number_of_channels,conf->frequency_mult, inds, correlations);
		// create average

		average = matrix_calloc_d(data->number_of_channels, (ave_win_before + 1 + ave_win_after + move) * conf->frequency_mult);
	/*	average = my_malloc(data->number_of_channels * sizeof(double *));
		for (ch = 0; ch < data->number_of_channels; ch++) {
			average[ch] = my_calloc((ave_win_before + 1 + ave_win_after + move) * conf->frequency_mult, sizeof(double));
		}*/
		for (n_ave = 0, j = 0; j < n_unprocessed && n_ave < conf->max_n_ave;j++) {
			i = (template_ind + j) % n_unprocessed; // start averaging from the original template beat
			if (correlations[i] > conf->similarity_limit_for_averaging && tests_are_one_ch_template_stds_similar(templates[inds[i]], dataset[process_these[i]],  win_before + 1 + win_after)) {
				n_ave++;
				tests_add_to_average(data, events->events[process_these[i]].sample - ave_win_before - inds[i] / conf->frequency_mult, ave_win_before + 1 + ave_win_after + move, conf->frequency_mult, -(inds[i] % conf->frequency_mult), average);
			/*	if(bad_chs_in_dataset && bad_chs_in_dataset[process_these[i]]){ // Additional check for saving the 'not totally ok' info. TODO: remove or redo the saving and checking part. This is only tmp implementation for testing
					if (clusters[n_clusters].bad_chs[0]==NULL) {
						clusters[n_clusters].bad_chs[0] = my_malloc(data->number_of_channels * sizeof(char));
						memcpy(clusters[n_clusters].bad_chs[0],bad_chs_in_dataset[process_these[i]],data->number_of_channels * sizeof(char));
						clusters[n_clusters].ch_badness[0] = my_malloc(data->number_of_channels * sizeof(float));
						memcpy(clusters[n_clusters].ch_badness[0],badness_in_dataset[process_these[i]],data->number_of_channels * sizeof(float));
					} else {
						for (ch = 0; ch < data->number_of_channels; ch++) {
							clusters[n_clusters].bad_chs[0][ch] = MIN(clusters[n_clusters].bad_chs[0][ch],bad_chs_in_dataset[process_these[i]][ch]);
							same for the badness
						}
					}
				}*/
			}

		}
		if(n_ave==0){
			printf("No/corrupted ave\n");
		}
		if(n_clusters==0 && n_ave<minimum_main_n_ave && initial_tries<max_n_initial_tries){
			matrix_free_d(average); // TODO: skip free and alloc during next round
			initial_tries++;
			continue;
		}

		// divide by n_ave for the markers and/or print
		tmp_f = 1.0 / (double) (n_ave);
		for (ch = 0; ch < data->number_of_channels; ch++) {
			for (j = 0; j < (ave_win_before + 1 + ave_win_after + move) * conf->frequency_mult; j++) {
				average[ch][j] *= tmp_f;
			}
		}

		// Redo the look for the max correlations with the averaged template and create the cluster
		tests_alloc_template_set2(data->number_of_channels, win_before + 1 + win_after, move, conf->frequency_mult,&ave_templates);
		tests_templates_from_average2(average, ave_templates,data->number_of_channels, win_before + 1 + win_after,ave_offset, move, conf->frequency_mult);
		tests_get_max_correlations5(ch_goodness, ave_templates, move * conf->frequency_mult,dataset, process_these,n_unprocessed,win_before + 1 + win_after, data->number_of_channels,conf->frequency_mult, inds, correlations);


		// Create the cluster
		for (i = 0; i < n_unprocessed;) {
			if (correlations[i] > conf->similarity_limit && tests_are_one_ch_template_stds_similar(ave_templates[inds[i]], dataset[process_these[i]],  win_before + 1 + win_after)) {
				clusters[n_clusters].inds[clusters[n_clusters].n_beats] = process_these[i];
				clusters[n_clusters].sample_offsets[clusters[n_clusters].n_beats] = inds[i] - move / 2 * conf->frequency_mult;
				clusters[n_clusters].n_beats++;
				n_unprocessed--;
				process_these[i] = process_these[n_unprocessed]; // replace the processed with the last of the unprocessed
				correlations[i] = correlations[n_unprocessed];
				inds[i] = inds[n_unprocessed];
			}
			else{
				i++;
			}
		}
		if (clusters[n_clusters].n_beats == 0) { // Did the average differ so much that even the original beat didn't match in the end -- do we have to recreate one beat 'average' and cluster?
			clusters[n_clusters].inds[clusters[n_clusters].n_beats] = process_these[template_ind];
			clusters[n_clusters].sample_offsets[clusters[n_clusters].n_beats] = 0;
			clusters[n_clusters].n_beats++;
			for (ch = 0; ch < data->number_of_channels; ch++) {
				memset(average[ch], 0, (win_before + 1 + win_after + move) * conf->frequency_mult * sizeof(double));
			}
			n_ave = 1;
			tests_add_to_average(data,events->events[process_these[template_ind]].sample - ave_win_before - move / 2, ave_win_before + 1 + ave_win_after + move, conf->frequency_mult, 0, average);
			// TODO: check!!! tests_templates_from_average2(average, templates,data->number_of_channels, win_before + 1 + win_after,ave_offset, move, conf->frequency_mult);
			tests_templates_from_average2(average, ave_templates,data->number_of_channels, win_before + 1 + win_after,ave_offset, move, conf->frequency_mult);

			n_unprocessed--;
			process_these[template_ind] = process_these[n_unprocessed]; // replace the processed with the last of the unprocessed
			correlations[template_ind] = correlations[n_unprocessed];
			inds[template_ind] = inds[n_unprocessed];
		/*	process_these[i] = process_these[n_unprocessed]; // replace the processed with the last of the unprocessed
			correlations[i] = correlations[n_unprocessed];
			inds[i] = inds[n_unprocessed];*/
		}

		clusters[n_clusters].template_sets[0] = ave_templates;
		clusters[n_clusters].templates[0] = average;
		clusters[n_clusters].n_ave=n_ave;

		printf("and found %lu beats for the cluster. %lu beats still unprocessed.\r",clusters[n_clusters].n_beats, n_unprocessed);
		clusters[n_clusters].sample_offsets = my_realloc(clusters[n_clusters].sample_offsets,clusters[n_clusters].n_beats * sizeof(int));
		clusters[n_clusters].inds = my_realloc(clusters[n_clusters].inds,clusters[n_clusters].n_beats * sizeof(size_t));

		n_clusters++;

/*		{
			cluster_set->clusters = clusters;
			cluster_set->n_clusters = n_clusters;
			tests_plot_cluster_templates("/var/www/RAecg_annotation/cache", "debug_pics", cluster_set, events, data);
			tests_print_first_cluster_beats("/var/www/RAecg_annotation/cache/debug_pics/first_cluster_beats.txt", data,events, cluster_set,50);
		}*/
	}
	printf("\n");
	//	printf("Checking if the clusters could be combined\n");
	// Sort the clusters based on their beat count
	qsort(clusters, n_clusters, sizeof(struct test_cluster),tests_cluster_comparison);

	printf("%d clusters in total\n", n_clusters);
	my_free(correlations);
	my_free(inds);
	cluster_set->clusters = clusters;
	cluster_set->n_clusters = n_clusters;
	return (0);
}





int tests_add_unprocessed_to_event_clusters(struct data *data, double *ch_goodness, struct test_events *events,double ***dataset, char **bad_chs_in_dataset,float **badness_in_dataset, size_t *process_these, size_t n_to_process, struct test_clusters *cluster_set) {
	struct conf_cluster *conf=conf_cluster_get();
	double ***templates, ***ave_templates;
	double **average;
//	double *std, *templ_std;
	double *correlations;
	double tmp_corr, tmp_f;
	int tmp_ind;
	int *inds;
	size_t template_ind = 5;
	size_t i, j, k, n_ave, max_j;
	int ch;
	size_t n_unprocessed = n_to_process;
	int clusters_alloced = cluster_set->n_clusters;
	int move = conf->move * data->sample_freq + 0.5;
	int win_before = conf->win_before * data->sample_freq + 0.5;
	int win_after = conf->win_after * data->sample_freq + 0.5;
	int ave_win_before = conf->ave_win_before * data->sample_freq + 0.5;
	int ave_win_after = conf->ave_win_after * data->sample_freq + 0.5;
	int ave_offset = (ave_win_before - win_before) * conf->frequency_mult;
	struct test_cluster *clusters = cluster_set->clusters;
	int n_clusters = cluster_set->n_clusters;

	srand(17); // same 'random numbers' on every run -- good for testing effects on small parameter modifications
//	srand((unsigned int)(time(NULL)));   // 'real random numbers' -- good for testing the effect of the initial template pick

	// alloc
	correlations = my_malloc(n_to_process * sizeof(double));
	inds = my_malloc(n_to_process * sizeof(int));

	// Skip the first and the last beats that don't fit the window
	for (i = n_unprocessed - 1;i > 0 && events->events[process_these[i]].sample + ave_win_after + move >= data->samples_per_channel; i--) {
			n_unprocessed--;
		}
	for (i = 0;i < n_unprocessed && events->events[process_these[i]].sample < ave_win_before + move;i++) {
		n_unprocessed--;
		process_these[i] = process_these[n_unprocessed];
	}


	// start by finding matches to the existing clusters
	if (n_clusters) {
//		int max_k;
		for (i = 0; i < n_unprocessed; ) {

			correlations[i] = -1.0;
			for (j = 0;j < n_clusters && (correlations[i] < conf->similarity_limit || clusters[j].n_beats > 1); j++) {
				for (k = 0; k < clusters[j].n_templates; k++) {
//					tests_get_max_correlations4(clusters[j].template_sets[k],move * conf->frequency_mult, clusters[j].ok_chs[k],dataset + process_these[i], 1,ok_chs_in_dataset ? ok_chs_in_dataset + process_these[i] : NULL,NULL, win_before + 1 + win_after,data->number_of_channels, conf->frequency_mult,&tmp_ind, &tmp_corr);
					tests_get_max_correlations5(ch_goodness, clusters[j].template_sets[k],move * conf->frequency_mult,  dataset, process_these+i,1, win_before + 1 + win_after,data->number_of_channels, conf->frequency_mult,&tmp_ind, &tmp_corr);
					if (tmp_corr > correlations[i] && tests_are_one_ch_template_stds_similar(clusters[j].template_sets[k][tmp_ind], dataset[process_these[i]],  win_before + 1 + win_after)) {
						max_j = j;
//						max_k=k;
						correlations[i] = tmp_corr;
						inds[i] = tmp_ind;
					}
				}
			}
			if (correlations[i] > conf->similarity_limit) {
				clusters[max_j].n_beats++;
				clusters[max_j].inds = my_realloc(clusters[max_j].inds,clusters[max_j].n_beats * sizeof(size_t));
				clusters[max_j].sample_offsets = my_realloc(clusters[max_j].sample_offsets,clusters[max_j].n_beats * sizeof(int));
				clusters[max_j].sample_offsets[clusters[max_j].n_beats - 1] = inds[i] - move / 2 * conf->frequency_mult;
				clusters[max_j].inds[clusters[max_j].n_beats - 1] = process_these[i];
				n_unprocessed--;
				process_these[i] = process_these[n_unprocessed]; // replace the processed with the last of the unprocessed
			}
			else{
				i++; // go to next
			}
		}
	}

	// loop until all the events belong to some cluster
	while (n_unprocessed > 0) { // are there any beats that don't belong to any clusters left?
		// Select unprocessed random beat -> template
		template_ind = n_unprocessed * (double) (rand()-1) / RAND_MAX;

		// Initialize cluster
		if (n_clusters >= clusters_alloced) {
			clusters_alloced += 100;
			clusters = my_realloc(clusters,clusters_alloced * sizeof(struct test_cluster));
		}
		clusters[n_clusters].type = TEST_BEAT_UNDEFINED;
		clusters[n_clusters].main_type_group=-1;
		clusters[n_clusters].n_beats = 0;
		clusters[n_clusters].cluster_offset = 0;
		clusters[n_clusters].sample_offsets = my_malloc(n_unprocessed * sizeof(int));
		clusters[n_clusters].inds = my_malloc(n_unprocessed * sizeof(size_t));
		clusters[n_clusters].n_templates = 1;
		clusters[n_clusters].templates = my_malloc(sizeof(double **));
		clusters[n_clusters].template_sets = my_malloc(sizeof(double ***));
		clusters[n_clusters].bad_chs = my_malloc(sizeof(char *));
		clusters[n_clusters].ch_badness = my_malloc(sizeof(float *));
		clusters[n_clusters].markers = NULL;
		if (bad_chs_in_dataset && bad_chs_in_dataset[process_these[template_ind]]) {
			clusters[n_clusters].bad_chs[0] = my_malloc(data->number_of_channels * sizeof(char));
			memcpy(clusters[n_clusters].bad_chs[0],bad_chs_in_dataset[process_these[template_ind]],data->number_of_channels * sizeof(char));
			clusters[n_clusters].ch_badness[0] = my_malloc(data->number_of_channels * sizeof(float));
			memcpy(clusters[n_clusters].ch_badness[0],bad_chs_in_dataset[process_these[template_ind]],data->number_of_channels * sizeof(float));
		} else {
			clusters[n_clusters].bad_chs[0] = NULL;
			clusters[n_clusters].ch_badness[0] = NULL;
		}

		// Find the highest correlation values with the unprocessed beats
		printf("Cluster %d: Selected %lu %lu as template\r", n_clusters + 1,process_these[template_ind], events->events[process_these[template_ind]].sample);
		tests_create_template_set2(data,events->events[process_these[template_ind]].sample - win_before - move / 2,win_before + 1 + win_after, move, conf->frequency_mult,&templates);
		tests_get_max_correlations5(ch_goodness, templates, move * conf->frequency_mult, dataset, process_these,n_unprocessed, win_before + 1 + win_after, data->number_of_channels,conf->frequency_mult, inds, correlations);
		// create average

		average = matrix_calloc_d(data->number_of_channels, (ave_win_before + 1 + ave_win_after + move) * conf->frequency_mult);
	/*	average = my_malloc(data->number_of_channels * sizeof(double *));
		for (ch = 0; ch < data->number_of_channels; ch++) {
			average[ch] = my_calloc((ave_win_before + 1 + ave_win_after + move) * conf->frequency_mult, sizeof(double));
		}*/
		for (n_ave = 0, j = 0; j < n_unprocessed && n_ave < conf->max_n_ave;j++) {
			i = (template_ind + j) % n_unprocessed; // start averaging from the original template beat
			if (correlations[i] > conf->similarity_limit_for_averaging && tests_are_one_ch_template_stds_similar(templates[inds[i]], dataset[process_these[i]],  win_before + 1 + win_after)) {
				if(bad_chs_in_dataset && bad_chs_in_dataset[process_these[i]]){ // check that all the same channels are ok that are ok in template
					for (ch = 0; ch < data->number_of_channels; ch++) {
						if(!bad_chs_in_dataset || !bad_chs_in_dataset[process_these[template_ind]]){
							if(bad_chs_in_dataset[process_these[i]][ch]>TEST_EVENT_TOTALLY_OK){ // is the ch ok in template, but not in sample i?)
								break;
							}
							else if(bad_chs_in_dataset[process_these[template_ind]][ch]<bad_chs_in_dataset[process_these[i]][ch]){ // is the sample worse than the template?
								break;
							}
						}
					}
					if(ch < data->number_of_channels){
						continue;
					}
				}
				n_ave++;
				tests_add_to_average(data, events->events[process_these[i]].sample - ave_win_before - inds[i] / conf->frequency_mult, ave_win_before + 1 + ave_win_after + move, conf->frequency_mult, -(inds[i] % conf->frequency_mult), average);
			/*	if(bad_chs_in_dataset && bad_chs_in_dataset[process_these[i]]){ // Additional check for saving the 'not totally ok' info. TODO: remove or redo the saving and checking part. This is only tmp implementation for testing
					if (clusters[n_clusters].bad_chs[0]==NULL) {
						clusters[n_clusters].bad_chs[0] = my_malloc(data->number_of_channels * sizeof(char));
						memcpy(clusters[n_clusters].bad_chs[0],bad_chs_in_dataset[process_these[i]],data->number_of_channels * sizeof(char));
						clusters[n_clusters].ch_badness[0] = my_malloc(data->number_of_channels * sizeof(float));
						memcpy(clusters[n_clusters].ch_badness[0],badness_in_dataset[process_these[i]],data->number_of_channels * sizeof(float));
					} else {
						for (ch = 0; ch < data->number_of_channels; ch++) {
							clusters[n_clusters].bad_chs[0][ch] = MIN(clusters[n_clusters].bad_chs[0][ch],bad_chs_in_dataset[process_these[i]][ch]);
							same for the badness
						}
					}
				}*/
			}

		}
		if(n_ave==0){
			printf("No/corrupted ave\n");
		}
		// divide by n_ave for the markers and/or print
		tmp_f = 1.0 / (double) (n_ave);
		for (ch = 0; ch < data->number_of_channels; ch++) {
			for (j = 0; j < (ave_win_before + 1 + ave_win_after + move) * conf->frequency_mult; j++) {
				average[ch][j] *= tmp_f;
			}
		}

		// Redo the look for the max correlations with the averaged template and create the cluster
		tests_alloc_template_set2(data->number_of_channels, win_before + 1 + win_after, move, conf->frequency_mult,&ave_templates);
		tests_templates_from_average2(average, ave_templates,data->number_of_channels, win_before + 1 + win_after,ave_offset, move, conf->frequency_mult);
		tests_get_max_correlations5(ch_goodness, ave_templates, move * conf->frequency_mult,dataset, process_these,n_unprocessed,win_before + 1 + win_after, data->number_of_channels,conf->frequency_mult, inds, correlations);


		// Create the cluster
		for (i = 0; i < n_unprocessed;) {
			if (correlations[i] > conf->similarity_limit && tests_are_one_ch_template_stds_similar(ave_templates[inds[i]], dataset[process_these[i]],  win_before + 1 + win_after)) {
				clusters[n_clusters].inds[clusters[n_clusters].n_beats] = process_these[i];
				clusters[n_clusters].sample_offsets[clusters[n_clusters].n_beats] = inds[i] - move / 2 * conf->frequency_mult;
				clusters[n_clusters].n_beats++;
				n_unprocessed--;
				process_these[i] = process_these[n_unprocessed]; // replace the processed with the last of the unprocessed
				correlations[i] = correlations[n_unprocessed];
				inds[i] = inds[n_unprocessed];
			}
			else{
				i++;
			}
		}
		if (clusters[n_clusters].n_beats == 0) { // Did the average differ so much that even the original beat didn't match in the end -- do we have to recreate one beat 'average' and cluster?
			clusters[n_clusters].inds[clusters[n_clusters].n_beats] = process_these[template_ind];
			clusters[n_clusters].sample_offsets[clusters[n_clusters].n_beats] = 0;
			clusters[n_clusters].n_beats++;
			for (ch = 0; ch < data->number_of_channels; ch++) {
				memset(average[ch], 0, (win_before + 1 + win_after + move) * conf->frequency_mult * sizeof(double));
			}
			n_ave = 1;
			tests_add_to_average(data,events->events[process_these[template_ind]].sample - ave_win_before - move / 2, ave_win_before + 1 + ave_win_after + move, conf->frequency_mult, 0, average);
			// TODO: check!!! tests_templates_from_average2(average, templates,data->number_of_channels, win_before + 1 + win_after,ave_offset, move, conf->frequency_mult);
			tests_templates_from_average2(average, ave_templates,data->number_of_channels, win_before + 1 + win_after,ave_offset, move, conf->frequency_mult);

			n_unprocessed--;
			process_these[template_ind] = process_these[n_unprocessed]; // replace the processed with the last of the unprocessed
			correlations[template_ind] = correlations[n_unprocessed];
			inds[template_ind] = inds[n_unprocessed];
		/*	process_these[i] = process_these[n_unprocessed]; // replace the processed with the last of the unprocessed
			correlations[i] = correlations[n_unprocessed];
			inds[i] = inds[n_unprocessed];*/
		}

		clusters[n_clusters].template_sets[0] = ave_templates;
		clusters[n_clusters].templates[0] = average;
		clusters[n_clusters].n_ave=n_ave;

#ifdef TESTS_NAN_DATA_CHECK
		{
			int n_,ch_,i_;
			for(n_=0;n_<move*conf->frequency_mult;n_++){
				for(ch_=0;ch_<data->number_of_channels;ch_++){
					for(i_=0;i_<cluster_set->template_len;i_++){
						if(isnan(clusters[n_clusters].templates[0][ch_][i_])){
							printf("Corrupted  data\n");
						}
					}
				}
			}
		}
#endif

		printf("and found %lu beats for the cluster. %lu beats still unprocessed.\r",clusters[n_clusters].n_beats, n_unprocessed);
		clusters[n_clusters].sample_offsets = my_realloc(clusters[n_clusters].sample_offsets,clusters[n_clusters].n_beats * sizeof(int));
		clusters[n_clusters].inds = my_realloc(clusters[n_clusters].inds,clusters[n_clusters].n_beats * sizeof(size_t));

		n_clusters++;

/*		{
			cluster_set->clusters = clusters;
			cluster_set->n_clusters = n_clusters;
			tests_plot_cluster_templates("/var/www/RAecg_annotation/cache", "debug_pics", cluster_set, events, data);
			tests_print_first_cluster_beats("/var/www/RAecg_annotation/cache/debug_pics/first_cluster_beats.txt", data,events, cluster_set,50);
		}*/
	}
	printf("\n");
	//	printf("Checking if the clusters could be combined\n");
	// Sort the clusters based on their beat count
	qsort(clusters, n_clusters, sizeof(struct test_cluster),tests_cluster_comparison);

	printf("%d clusters in total\n", n_clusters);
	my_free(correlations);
	my_free(inds);
	cluster_set->clusters = clusters;
	cluster_set->n_clusters = n_clusters;
	return (0);
}


static inline double template_correlation(double *template,double ave_t,double std_t, double *data, int len,double max_std_diff_mult){
	double ave_d=0.0,std_d=0.0;
	double corr=0.0;
	double tmp_d;
	int i;
	for(i=0;i<len;i++){
		ave_d+=data[i];
	}
	ave_d/=(double)(len);
	for(i=0;i<len;i++){
		tmp_d=data[i]-ave_d;
		corr+=(template[i]-ave_t)*(tmp_d);
		std_d+=tmp_d*tmp_d;
	}
	if(std_t>std_d){
		if(std_d*max_std_diff_mult<std_t){
			corr/=std_t;
		}
		else{
			corr/=sqrt(std_t*std_d);
		}
	}
	else{
		if(std_t*max_std_diff_mult<std_d){
			corr/=std_d;
		}
		else{
			corr/=sqrt(std_t*std_d);
		}
	}
	return(corr);
}

/*static inline double resampled_template_correlation(double *template,double ave_t,double sdp_t, int index_mult, double *data, int len,double max_sdp_diff_mult){
	double ave_d=0.0,sdp_d=0.0;
	double corr=0.0;
	double tmp_d;
	int i;
	for(i=0;i<len;i++){
		ave_d+=data[i];
	}
	ave_d/=(double)(len);
	for(i=0;i<len;i++){
		tmp_d=data[i]-ave_d;
		corr+=(template[index_mult*i]-ave_t)*(tmp_d);
		sdp_d+=tmp_d*tmp_d;
	}
	if(sdp_t>sdp_d){
		if(sdp_d*max_sdp_diff_mult<sdp_t){
			corr/=sdp_t;
		}
		else{
			corr/=sqrt(sdp_t*sdp_d);
		}
	}
	else{
		if(sdp_t*max_sdp_diff_mult<sdp_d){
			corr/=sdp_d;
		}
		else{
			corr/=sqrt(sdp_t*sdp_d);
		}
	}
	return(corr);
}*/

double max_sliding_template_correlation(double *data1, double *ave1, double *sdp1, double *data2, int len, int n_slide, double max_std_diff_mult, int *max_ind){
	int i;
	double tmpd;
	double max_corr=-2.0;
	if(*sdp1<0.0){ // undefined
		*ave1=*sdp1=0.0;
		for(i=0;i<len;i++){
			*ave1+=data1[i];
		}
		*ave1/=(double)(len);
		for(i=0;i<len;i++){
			tmpd=data1[i]-*ave1;
			*sdp1+=tmpd*tmpd;
		}
	}
	for(i=0;i<n_slide;i++){
		tmpd=template_correlation(data1,*ave1,*sdp1, data2+i, len,max_std_diff_mult);
		if(tmpd>max_corr){
			max_corr=tmpd;
			*max_ind=i;
		}
	}
	return(max_corr);
}

int tests_get_cluster_correlations(double *ch_goodness, int compare_to, int start_from,struct test_clusters *cluster_set, int *cluster_offsets,double *cluster_corrs,int only_undefined) {
	struct conf_cluster *conf=conf_cluster_get();
	int i, k;
	int tmp_offset;
	int move = conf->move * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int win_before = conf->win_before * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int win_after = conf->win_after * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int ch=0;
	int template_offset=cluster_set->template_offset-win_before;
	int compare_offset=cluster_set->template_offset-win_before-move/2;
	double template_ave=0.0;
	double template_sdp=-1.0;
	double tmp_corr;
	for (i = start_from; i < cluster_set->n_clusters; i++) {
		if (!only_undefined || cluster_set->clusters[i].type == TEST_BEAT_UNDEFINED) {
			cluster_corrs[i] = -1.0;

			for (k = 0; k < cluster_set->clusters[i].n_templates; k++) {
				tmp_corr=max_sliding_template_correlation(cluster_set->clusters[compare_to].templates[0][ch]+template_offset, &template_ave,&template_sdp,
										cluster_set->clusters[i].templates[k][ch] + compare_offset, win_before + win_after, move, conf->allowed_amplitude_variation_mult, &tmp_offset);
				if (tmp_corr > cluster_corrs[i]){
					cluster_corrs[i] = tmp_corr;
				}
				if (k == 0) {
					cluster_offsets[i] = -tmp_offset + template_offset - compare_offset;
				}
			}
		}
	}
	return (0);
}



int tests_get_max_cluster_correlation(double *ch_goodness, int this,enum test_trigger_type this_type, int min_n_beats, struct test_clusters *cluster_set,int *cluster_offset, double *cluster_corr) {
	struct conf_cluster *conf=conf_cluster_get();
	int i;
	int tmp_offset;
	int move = conf->move * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int win_before = conf->win_before * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int win_after = conf->win_after * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int ch=0,k=0;
	int template_offset=cluster_set->template_offset-win_before;
	int compare_offset=cluster_set->template_offset-win_before-move/2;
	double template_ave=0.0;
	double template_sdp=-1.0;
	double tmp_corr;
	*cluster_corr = -1.0;
	for (i = 0; i < cluster_set->n_clusters; i++) {
		if (cluster_set->clusters[i].type == this_type && cluster_set->clusters[i].n_beats>=min_n_beats) {
			tmp_corr=max_sliding_template_correlation(cluster_set->clusters[this].templates[0][ch]+template_offset, &template_ave,&template_sdp,
					cluster_set->clusters[i].templates[k][ch] + compare_offset, win_before + win_after, move, conf->allowed_amplitude_variation_mult, &tmp_offset);
			if (tmp_corr > *cluster_corr) {
				*cluster_corr = tmp_corr;
				*cluster_offset = -(-tmp_offset + template_offset - compare_offset);
			}
		}
	}
	return (0);
}



int tests_get_n_corr(int this, int min_n_beats, int max_comparisons, double *ch_goodness, struct test_clusters *cluster_set, double *cluster_corr, double *std_diff) {
	struct conf_cluster *conf=conf_cluster_get();
	int i, k, n;
	int n_compared=0;
	int tmp_offset;
	int ch=0;
	int move = conf->move * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int win_before = conf->win_before * cluster_set->original_freq  * cluster_set->frequency_mult + 0.5;
	int win_after = conf->win_after * cluster_set->original_freq  * cluster_set->frequency_mult + 0.5;
	int template_offset=cluster_set->template_offset-win_before;
	int compare_offset=cluster_set->template_offset-win_before-move/2;
	double template_ave=0.0;
	double template_sdp=-1.0;
	double tmp_corr;
	*cluster_corr = -2.0;
	*std_diff=-1.0;

	for(n=0;n<cluster_set->n_clusters && n_compared<max_comparisons;n++){
		if(cluster_set->clusters[n].type == TEST_NORMAL_BEAT && (cluster_set->clusters[n].n_beats>=min_n_beats || !n_compared)){
			if(this==n){ // perfect match!
				*cluster_corr=1.0;
				*std_diff=1.0;
				return(1.0);
			}
			n_compared++;
			for (i = 0; i < cluster_set->clusters[n].n_templates; i++) {
				for (k = 0; k < cluster_set->clusters[this].n_templates; k++) {
					tmp_corr=max_sliding_template_correlation(cluster_set->clusters[this].templates[0][ch]+template_offset, &template_ave,&template_sdp,
										cluster_set->clusters[i].templates[k][ch] + compare_offset, win_before + win_after, move, conf->allowed_amplitude_variation_mult, &tmp_offset);
					if (tmp_corr > *cluster_corr) {
						*cluster_corr = tmp_corr;
					}
				}
			}
		}
	}
	return (0);
}

void sliding_template_correlation(double *data1, double *ave1, double *sdp1, double *data2, int len, int n_slide, double max_std_diff_mult, double *sliding_corr){
	int i;
	double tmpd;
	if(*sdp1<0.0){ // undefined
		*ave1=*sdp1=0.0;
		for(i=0;i<len;i++){
			*ave1+=data1[i];
		}
		for(i=0;i<len;i++){
			tmpd=data1[i]-*ave1;
			*sdp1+=tmpd*tmpd;
		}
	}
	for(i=0;i<n_slide;i++){
		sliding_corr[i]=template_correlation(data1,*ave1,*sdp1, data2, len,max_std_diff_mult);
	}
}




int tests_get_one_ch_cluster_correlations(int compare_to, int start_from,struct test_clusters *cluster_set, int *cluster_offsets,double *cluster_corrs,int only_undefined, double win_before_s, double win_after_s, double fine_tune) {
	struct conf_cluster *conf=conf_cluster_get();
	int i, k;
	int ch=0;
	int tmp_offset;
	int move = fine_tune * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int win_before =win_before_s * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int win_after = win_after_s * cluster_set->original_freq * cluster_set->frequency_mult + 0.5;
	int template_offset=cluster_set->template_offset-win_before;
	int compare_offset=cluster_set->template_offset-win_before-move/2;
	double template_ave=0.0;
	double template_sdp=-1.0;
	double tmp_corr;
	if(template_offset<0 || template_offset + move + win_after >= cluster_set->template_len){
		printf("Error! Correlation window goes outside the data\n");
		return(-1);
	}
	for (i = start_from; i < cluster_set->n_clusters; i++) {
		if (!only_undefined || cluster_set->clusters[i].type == TEST_BEAT_UNDEFINED) {
			cluster_corrs[i] = -1.0;
			cluster_offsets[i] = 0;
			if(i==compare_to){
				cluster_corrs[i] = 1.0;
				continue;
			}

			for (k = 0; k < cluster_set->clusters[i].n_templates; k++) {
				tmp_corr=max_sliding_template_correlation(cluster_set->clusters[compare_to].templates[0][ch]+template_offset, &template_ave,&template_sdp,
						cluster_set->clusters[i].templates[k][ch] + compare_offset, win_before + win_after, move, conf->allowed_amplitude_variation_mult, &tmp_offset);
				if (tmp_corr > cluster_corrs[i]){
					cluster_corrs[i] = tmp_corr;
					cluster_offsets[i] = -tmp_offset + template_offset - compare_offset;
				}
			}
		}
	}
	return (0);
}

int tests_get_one_ch_n_corr(int this, int min_n_beats, int max_comparisons, struct test_clusters *cluster_set, double *cluster_corr) {
	struct conf_cluster *conf=conf_cluster_get();
	int i, k, n;
	int n_compared=0;
	int tmp_offset;
	int move = conf->move * cluster_set->original_freq + 0.5;
//	int win_before = conf->win_before * cluster_set->original_freq + 0.5;
//	int win_after = conf->win_after * cluster_set->original_freq + 0.5;
	int win_before = 0.050 * cluster_set->original_freq + 0.5;
	int win_after = 0.050 * cluster_set->original_freq + 0.5;
	double tmp_corr;
	*cluster_corr = -2.0;

	for(n=0;n<cluster_set->n_clusters && n_compared<max_comparisons;n++){
		if(cluster_set->clusters[n].type == TEST_NORMAL_BEAT && cluster_set->clusters[n].n_beats>=min_n_beats){
			if(this==n){ // perfect match!
				*cluster_corr=1.0;
				return(1.0);
			}
			n_compared++;
			for (i = 0; i < cluster_set->clusters[n].n_templates; i++) {
				for (k = 0; k < cluster_set->clusters[this].n_templates; k++) {
					tests_get_max_one_ch_correlations(cluster_set->clusters[this].template_sets[k],move * conf->frequency_mult,cluster_set->clusters[n].template_sets[i]+ move / 2 * conf->frequency_mult,NULL,1,win_before + 1 + win_after, cluster_set->nch,conf->frequency_mult, &tmp_offset, &tmp_corr);
					if (tmp_corr > *cluster_corr) {
						*cluster_corr = tmp_corr;
					}
				}
			}
		}
	}
	return (0);
}


/*
// define the correlation to the first (normal) cluster. I
int tests_get_n_corr(int this,double *ch_goodness, struct test_clusters *cluster_set, struct test_cluster_conf *conf, double *cluster_corr, double *std_diff) {
	int i, k;
	int tmp_offset;
	int move = conf->move * cluster_set->original_freq + 0.5;
	int win_before = conf->win_before * cluster_set->original_freq + 0.5;
	int win_after = conf->win_after * cluster_set->original_freq + 0.5;
	double tmp_corr;
	*cluster_corr = -1.0;
	*std_diff=-1.0;
	if(cluster_set->clusters[0].type!=TEST_NORMAL_BEAT){
		*cluster_corr=-2.0;
		*std_diff=-1.0;
		return(-1.0);
	}
	if(this==0){
		*cluster_corr=1.0;
		*std_diff=1.0;
		return(1.0);
	}

	for (i = 0; i < cluster_set->clusters[0].n_templates; i++) {
		for (k = 0; k < cluster_set->clusters[this].n_templates; k++) {
			tests_get_max_correlations6(ch_goodness, cluster_set->clusters[this].template_sets[k],move * conf->frequency_mult,cluster_set->clusters[this].bad_chs[k],cluster_set->clusters[this].ch_badness[k],cluster_set->clusters[0].template_sets[i]+ move / 2 * conf->frequency_mult, cluster_set->clusters[0].bad_chs+i,cluster_set->clusters[0].ch_badness+i, NULL,1,win_before + 1 + win_after, cluster_set->nch,conf->frequency_mult, &tmp_offset, &tmp_corr);
			if (tmp_corr > *cluster_corr) {
				*cluster_corr = tmp_corr;
				*std_diff=tests_get_max_std_diff(cluster_set->clusters[i].template_sets[k][tmp_offset], cluster_set->clusters[this].template_sets[0][move / 2 * conf->frequency_mult], cluster_set->clusters[i].ch_badness[k], cluster_set->clusters[this].ch_badness[0], win_before + 1 + win_after,cluster_set->nch);
			}
		}
	}
	return (0);
}*/


/*int tests_get_max_event_cluster_correlation(size_t ind,struct test_events *events, struct data *data, enum test_trigger_type this_type, int min_n_beats, struct test_clusters *cluster_set,int *cluster_offset, double *cluster_corr,struct test_cluster_conf *conf) {
	int ch, i, k;
	int tmp_offset;
	int move = conf->move * cluster_set->original_freq + 0.5;
	int win_before = conf->win_before * cluster_set->original_freq + 0.5;
	int win_after = conf->win_after * cluster_set->original_freq + 0.5;
	double tmp_corr;
	*cluster_corr = -1.0;
	for (i = 0; i < cluster_set->n_clusters; i++) {
		if (cluster_set->clusters[i].type == this_type && cluster_set->clusters[i].n_beats>=min_n_beats) {
			for (k = 0; k < cluster_set->clusters[i].n_templates; k++) {
				for(ch=0;ch<cluster_set->nch && (cluster_set->clusters[i].ok_chs[k]==NULL || cluster_set->clusters[i].ok_chs[k][ch]>0 || (cluster_set->clusters[this].ok_chs[0] && cluster_set->clusters[this].ok_chs[0][ch]==0));ch++);
				if(ch<cluster_set->nch){ // Was there some channel bad in 'i' that is ok in 'this'? -- we want to compare only to clusters with same or more ok channels.
					continue;
				}
				tests_get_max_correlations6(cluster_set->clusters[i].template_sets[k],move * conf->frequency_mult,cluster_set->clusters[i].ok_chs[k],cluster_set->clusters[this].template_sets[0]+ move / 2 * conf->frequency_mult, cluster_set->clusters[this].ok_chs, NULL,1,win_before + 1 + win_after, cluster_set->nch,conf->frequency_mult, &tmp_offset, &tmp_corr);
				if (tmp_corr > *cluster_corr && tests_are_template_stds_similar(cluster_set->clusters[i].template_sets[k][tmp_offset], cluster_set->clusters[this].template_sets[0][move / 2 * conf->frequency_mult], cluster_set->clusters[i].ok_chs[k], cluster_set->clusters[this].ok_chs[0], win_before + 1 + win_after,cluster_set->nch, conf)) {
					*cluster_corr = tmp_corr;
					*cluster_offset = -(tmp_offset - move / 2 * conf->frequency_mult);
				}show
			}
		}
	}
	return (0);
}xxx*/

// Combine similar clusters and drop the extra data from the clusters
int tests_combine_clusters(struct data *data, struct test_clusters *cluster_set,struct test_ch_info *ch_info) {
	struct conf_cluster *conf=conf_cluster_get();
	struct test_cluster *clusters = cluster_set->clusters;
	int n_clusters = cluster_set->n_clusters;
	size_t i, j;
	int k, ch;
	int move = conf->move * data->sample_freq + 0.5;
	int win_after = conf->win_after * data->sample_freq + 0.5;
	int win_before = conf->win_before * data->sample_freq + 0.5;
	int cluster_ind, tmp_ind, *cluster_inds;
	double cluster_corr, tmp_corr, *cluster_ind_corrs;

	cluster_inds = my_calloc(n_clusters, sizeof(int));
	cluster_ind_corrs = my_malloc(n_clusters * sizeof(double));
	for (i = 0; i < n_clusters; i++) {
		cluster_ind_corrs[i] = 0.7; // minimum correlation for adjusting relative correlation index
	}
	for (i = 0; i < n_clusters; i++) {
		if (clusters[i].n_beats == 0) {
			continue;
		}
		// Finalize the average template and add the cluster_inds to inds
		for (ch = 0; ch < data->number_of_channels; ch++) {
			for (k = 0; k < (win_before + 1 + win_after) * conf->frequency_mult;
					k++) {
				clusters[i].templates[0][ch][k] = clusters[i].templates[0][ch][k
						+ move / 2 * conf->frequency_mult - cluster_inds[i]];
			}
			clusters[i].templates[0][ch] = my_realloc(clusters[i].templates[0][ch],
					(win_before + 1 + win_after) * conf->frequency_mult
							* sizeof(double));
		}
		if (cluster_inds[i]) {
			for (k = 0; k < clusters[i].n_beats; k++) {
				clusters[i].sample_offsets[k] -= cluster_inds[i]; // correct the offset against cluster[0]
			}
		}
		for (j = i + 1; j < n_clusters; j++) {
			if (clusters[j].n_beats == 0) {
				continue;
			}
			cluster_corr = -1.0;
			for (k = 0; k < clusters[i].n_templates; k++) {
				tests_get_max_correlations6(ch_info->ch_goodness, cluster_set->clusters[i].template_sets[k],move * conf->frequency_mult,cluster_set->clusters[i].bad_chs[k],cluster_set->clusters[i].ch_badness[k],cluster_set->clusters[0].template_sets[j]+ move / 2 * conf->frequency_mult, cluster_set->clusters[0].bad_chs+j,cluster_set->clusters[0].ch_badness+j, NULL,1,win_before + 1 + win_after, cluster_set->nch,conf->frequency_mult, 0,&tmp_ind, &tmp_corr);
/*				tests_get_max_correlations4(
						clusters[i].template_sets[k] + MAX(-cluster_inds[i], 0),
						move * conf->frequency_mult - abs(cluster_inds[i]),
						clusters[i].ok_chs[k],
						clusters[j].template_sets[0]
								+ move / 2 * conf->frequency_mult, 1,
						clusters[j].ok_chs, NULL, win_before + 1 + win_after,
						data->number_of_channels, conf->frequency_mult,
						&tmp_ind, &tmp_corr);*/
				if (tmp_corr > cluster_corr) {
					cluster_corr = tmp_corr;
				}
				if (k == 0) {
					cluster_ind =
							tmp_ind
									- move / 2
											* conf->frequency_mult+MAX(-cluster_inds[i],0);
				}
			}
			if (cluster_corr > conf->similarity_limit_for_clusters) { // combine the cluster[j] to cluster[i]?
				clusters[i].sample_offsets = my_realloc(clusters[i].sample_offsets,
						(clusters[i].n_beats + clusters[j].n_beats)
								* sizeof(int));
				clusters[i].inds = my_realloc(clusters[i].inds,
						(clusters[i].n_beats + clusters[j].n_beats)
								* sizeof(size_t));
				for (k = 0; k < clusters[j].n_beats; k++) {
					clusters[i].inds[k + clusters[i].n_beats] =
							clusters[j].inds[k];
					clusters[i].sample_offsets[k + clusters[i].n_beats] =
							clusters[j].sample_offsets[k] - cluster_inds[i]
									- cluster_ind; // correct the offsets between the cluster[i] and cluster[0] and between the cluster[i] and cluster[j]
				}
				clusters[i].n_beats += clusters[j].n_beats;
				my_free(clusters[j].inds);
				my_free(clusters[j].sample_offsets);
				clusters[i].bad_chs = my_realloc(clusters[i].bad_chs,(clusters[i].n_templates + 1) * sizeof(char *));
				clusters[i].bad_chs[clusters[i].n_templates] = clusters[j].bad_chs[0];
				clusters[i].ch_badness = my_realloc(clusters[i].ch_badness,(clusters[i].n_templates + 1) * sizeof(float *));
				clusters[i].ch_badness[clusters[i].n_templates] = clusters[j].ch_badness[0];
				clusters[i].templates = my_realloc(clusters[i].templates,(clusters[i].n_templates + 1) * sizeof(double **));
				clusters[i].templates[clusters[i].n_templates] = clusters[j].templates[0]; // copy / move the average template
				for (ch = 0; ch < data->number_of_channels; ch++) {
					for (k = 0; k < (win_before + 1 + win_after)* conf->frequency_mult; k++) {
						clusters[i].templates[clusters[i].n_templates][ch][k] = clusters[i].templates[clusters[i].n_templates][ch][k+ move / 2 * conf->frequency_mult - cluster_inds[i] - cluster_ind];
					}
					clusters[i].templates[clusters[i].n_templates][ch] = my_realloc(clusters[i].templates[clusters[i].n_templates][ch],(win_before + 1 + win_after)* conf->frequency_mult * sizeof(double));
				}
				clusters[i].template_sets = my_realloc(clusters[i].template_sets,(clusters[i].n_templates + 1) * sizeof(double ***));
				clusters[i].template_sets[clusters[i].n_templates] = clusters[j].template_sets[0];
				clusters[i].n_templates++;
				my_free(clusters[j].bad_chs);
				my_free(clusters[j].ch_badness);
				my_free(clusters[j].templates);
				my_free(clusters[j].template_sets);
				clusters[j].n_beats = 0; // mark the cluster as 'processed', by setting its n_beats field to zero
			} else if (cluster_corr > cluster_ind_corrs[j]) { // set the index against the first set of cluster who correlate at least with 0.7 correlation
				cluster_ind_corrs[j] = cluster_corr;
				cluster_inds[j] = cluster_ind + cluster_inds[i];
			}
		}
		/*for(k=0;k<clusters[i].n_templates;k++){// Free the template_sets, we don't need it anymore
		 tests_free_template_set(data->number_of_channels,win_before+1+win_after,move,conf->frequency_mult, clusters[i].template_sets[k]);
		 }
		 my_free(clusters[i].template_sets);
		 clusters[i].template_sets=NULL;*/
	}
	// Drop the 'empty clusters'
	cluster_set->n_clusters = 0;
	for (i = 0; i < n_clusters; i++) {
		if (clusters[i].n_beats == 0) {
			continue;
		}
		if (cluster_set->n_clusters < i) {
			clusters[cluster_set->n_clusters] = clusters[i];
		}
		cluster_set->n_clusters++;
	}
	my_free(cluster_inds);
	my_free(cluster_ind_corrs);
	return (0);
}

#define TEST_CORR_MAX_MOVE 10
#define TEST_CORR_MAX_MULT 8
#define TEST_CORR_HALFLEN 25
#define TEST_CORR_N_TEST 100
//#define PRINT_ECG_SAMPLES
//#define COMPARE_CUMPRODUCT_SCALING

int tests_print_max_qrs_correlations(struct data *data,
		struct test_events *events) {

	size_t i;
	size_t offset = 1;
	size_t max_ind = MIN(offset+TEST_CORR_N_TEST, data->samples_per_channel);
	int ch = 0;
	int ind;
	int n_mult;
	double data1[(2 * TEST_CORR_HALFLEN + 1 + 2 * TEST_CORR_MAX_MOVE + 1)
			* TEST_CORR_MAX_MULT];
	double data2[(2 * TEST_CORR_HALFLEN + 1) * TEST_CORR_MAX_MULT];
	int mult = 1;
	double **results = NULL;
	double **ind_results = NULL;
	double *extra_results = NULL;
	time_t start_time = time(NULL);
#ifdef COMPARE_CUMPRODUCT_SCALING
	FILE *fp_c1=NULL,*fp_c2=NULL;
#endif

#ifdef PRINT_ECG_SAMPLES
	int j;
	FILE *fp=NULL,*fp2=NULL,*fp3=NULL;
	fp=fopen("original_output.txt","w");
	fp2=fopen("corrected_output.txt","w");
	fp3=fopen("resampled_output.txt","w");
#endif

#ifdef COMPARE_CUMPRODUCT_SCALING
	fp_c1=fopen("scaled_sumproducts.txt","w");
	fp_c2=fopen("unscaled_sumproducts.txt","w");
#endif

	for (n_mult = 0, mult = 1; mult <= TEST_CORR_MAX_MULT; mult *= 2, n_mult++)
		;
	results = my_malloc(n_mult * sizeof(double *));
	ind_results = my_malloc(n_mult * sizeof(double *));
	extra_results = my_malloc((max_ind - offset) * sizeof(double));

	// define
	for (n_mult = 0, mult = 1; mult <= TEST_CORR_MAX_MULT;
			mult *= 2, n_mult++) {
		start_time = time(NULL);
		printf("mult: %d, time: ", mult);
		results[n_mult] = my_malloc((max_ind - offset) * sizeof(double));
		ind_results[n_mult] = my_malloc((max_ind - offset) * sizeof(double));

		tests_resample_segment(
				data->ch[ch].raw
						+ events->events[offset].sample-TEST_CORR_HALFLEN-TEST_CORR_MAX_MOVE,
				data1, 2 * TEST_CORR_HALFLEN + 1 + 2 * TEST_CORR_MAX_MOVE + 1,
				mult);
		//tests_whiten_data(data1,(2*TEST_CORR_HALFLEN+1+2*TEST_CORR_MAX_MOVE)*mult,NULL,NULL);
		for (i = offset; i < max_ind; i++) {
#ifdef PRINT_ECG_SAMPLES
			if(mult==1) {
				for(j=0;j<=2*TEST_CORR_HALFLEN;j++) {
					fprintf(fp,"%lg\t",data->ch[ch].raw[events->events[i].sample-TEST_CORR_HALFLEN+j]);
				}
				fprintf(fp,"\n");
			}
#endif
			tests_resample_segment(
					data->ch[ch].raw
							+ events->events[i].sample-TEST_CORR_HALFLEN, data2,
					2 * TEST_CORR_HALFLEN + 1, mult);
			tests_whiten_data(data2, (2 * TEST_CORR_HALFLEN + 1) * mult, NULL,
			NULL, NULL);
			tests_get_max_sumproduct(data1, data2,
					(2 * TEST_CORR_HALFLEN + 1) * mult,
					(2 * TEST_CORR_MAX_MOVE) * mult, &ind,
					results[n_mult] + i - offset);
			ind_results[n_mult][i - offset] = (double) (ind) / (double) (mult)
					- (double) (TEST_CORR_MAX_MOVE);
			if (mult == TEST_CORR_MAX_MULT) {
				tests_get_downsampled_sumproduct(data1 + ind,
						data->ch[ch].raw
								+ events->events[i].sample-TEST_CORR_HALFLEN,
						2 * TEST_CORR_HALFLEN + 1, mult,
						extra_results + i - offset);
			}
#ifdef COMPARE_CUMPRODUCT_SCALING
			tests_compare_sumproducts_scaling(fp_c1,fp_c2,data1,data2, (2*TEST_CORR_HALFLEN+1)*mult, (2*TEST_CORR_MAX_MOVE)*mult);
#endif

#ifdef PRINT_ECG_SAMPLES
			ind-=TEST_CORR_MAX_MOVE*mult;
			if(mult==1) {
				for(j=0;j<ind;j++) {
					fprintf(fp2,"0.0\t");
				}
				for(;j<=(2*TEST_CORR_HALFLEN)*mult+(ind<0 ? ind : 0);j++) {
					fprintf(fp2,"%lg\t",data2[-ind+j]);
				}
				fprintf(fp2,"\n");
			}
			if(mult==TEST_CORR_MAX_MULT) {
				for(j=0;j<ind;j++) {
					fprintf(fp3,"0.0\t");
				}
				for(;j<=(2*TEST_CORR_HALFLEN)*mult+(ind<0 ? ind : 0);j++) {
					fprintf(fp3,"%lg\t",data2[-ind+j]);
				}
				fprintf(fp3,"\n");
			}
#endif

		}
		printf("%lu \n", time(NULL) - start_time);
	}

	// print
	printf("Offset: %lu, n: %d, max_move: %d , max_mult: %d\n", offset,
	TEST_CORR_N_TEST, TEST_CORR_MAX_MOVE, TEST_CORR_MAX_MULT);
	for (i = offset; i < max_ind; i++) {
		printf("%lu. %d %lu:", i - offset + 1, events->events[i].type,
				events->events[i].sample);
		for (n_mult = 0, mult = 1; mult <= TEST_CORR_MAX_MULT;
				mult *= 2, n_mult++) {
			printf("\t%lf", results[n_mult][i - offset]);
		}
		printf("\t%lf", extra_results[i - offset]);
		for (n_mult = 0, mult = 1; mult <= TEST_CORR_MAX_MULT;
				mult *= 2, n_mult++) {
			printf("\t%lf", ind_results[n_mult][i - offset]);
		}
		printf("\n");
	}
	for (n_mult = 0, mult = 1; mult <= TEST_CORR_MAX_MULT;
			mult *= 2, n_mult++) {
		my_free(results[n_mult]);
		my_free(ind_results[n_mult]);
	}
	my_free(results);
	my_free(ind_results);
	my_free(extra_results);
#ifdef COMPARE_CUMPRODUCT_SCALING
	fclose(fp_c1);
	fclose(fp_c2);
#endif

#ifdef PRINT_ECG_SAMPLES
	fclose(fp);
	fclose(fp2);
	fclose(fp3);
#endif
	return (0);
}

int tests_print_correlation_histograms(struct data *data,struct test_events *events) {

	size_t i, j, k;
	int ch = 0;
	double data1[(2 * TEST_CORR_HALFLEN + 1 + 2 * TEST_CORR_MAX_MOVE + 1)
			* TEST_CORR_MAX_MULT];
	double data2[(2 * TEST_CORR_HALFLEN + 1) * TEST_CORR_MAX_MULT];
	int mult = 8;
	static size_t *histograms[N_REAL_TEST_LABELS][N_REAL_TEST_LABELS];
	static double min[N_REAL_TEST_LABELS][N_REAL_TEST_LABELS];
	static double max[N_REAL_TEST_LABELS][N_REAL_TEST_LABELS];
	static double ave[N_REAL_TEST_LABELS][N_REAL_TEST_LABELS];
	static size_t n[N_REAL_TEST_LABELS][N_REAL_TEST_LABELS];
	static int initialized = 0;
	double correlation;
	int ind;
	char filename[FILENAME_MAX];
	FILE *fp;
	if (data == NULL) { // print results and free static data
		if (!initialized) {
			printf("Trying to print uninitialized correlation histograms\n");
			return (1);
		}

		for (i = 0; i < N_REAL_TEST_LABELS; i++) {
			sprintf(filename, "correlation_histograms_%lu.txt", i);
			fp = fopen(filename, "w");
			for (j = 0; j < N_REAL_TEST_LABELS; j++) {
				fprintf(fp, "%lu\t%lg\t%lg\t%lg", n[i][j], min[i][j],
						ave[i][j] / (double) (n[i][j]), max[i][j]);
				for (k = 0; k < 21; k++) {
					fprintf(fp, "\t%lu", histograms[i][j][k]);
				}
				fprintf(fp, "\n");
			}
			fclose(fp);
		}

		for (i = 0; i < N_REAL_TEST_LABELS; i++) {
			for (j = 0; j < N_REAL_TEST_LABELS; j++) {
				my_free(histograms[i][j]);
			}
		}
		initialized = 0;
		return (0);
	}
	if (!initialized) { // initialize?
		for (i = 0; i < N_REAL_TEST_LABELS; i++) {
			for (j = 0; j < N_REAL_TEST_LABELS; j++) {
				histograms[i][j] = my_calloc(21, sizeof(size_t));
				min[i][j] = 1.0;
				max[i][j] = -1.0;
				ave[i][j] = 0.0;
				n[i][j] = 0;
			}
		}
		initialized = 1;
	}

	// define
	for (i = 1; i < events->n_events - 1; i++) {
		tests_resample_segment(
				data->ch[ch].raw
						+ events->events[i].sample-TEST_CORR_HALFLEN-TEST_CORR_MAX_MOVE,
				data1, 2 * TEST_CORR_HALFLEN + 1 + 2 * TEST_CORR_MAX_MOVE + 1,
				mult);
		//tests_whiten_data(data1,(2*TEST_CORR_HALFLEN+1+2*TEST_CORR_MAX_MOVE)*mult,NULL,NULL,NULL);
		for (j = i + 1; j < events->n_events - 1; j++) {
			if (n[(int) (events->events[i].type)][(int) (events->events[j].type)]
					> 100000) {
				continue;
			}
			tests_resample_segment(
					data->ch[ch].raw
							+ events->events[j].sample-TEST_CORR_HALFLEN, data2,
					2 * TEST_CORR_HALFLEN + 1, mult);
			tests_whiten_data(data2, (2 * TEST_CORR_HALFLEN + 1) * mult, NULL,
			NULL, NULL);
			tests_get_max_sumproduct(data1, data2,
					(2 * TEST_CORR_HALFLEN + 1) * mult,
					(2 * TEST_CORR_MAX_MOVE) * mult, &ind, &correlation);
			n[(int) (events->events[i].type)][(int) (events->events[j].type)]++;
			ave[(int) (events->events[i].type)][(int) (events->events[j].type)] +=
					correlation;
			if (min[(int) (events->events[i].type)][(int) (events->events[j].type)]
					> correlation) {
				min[(int) (events->events[i].type)][(int) (events->events[j].type)] =
						correlation;
			}
			if (max[(int) (events->events[i].type)][(int) (events->events[j].type)]
					< correlation) {
				max[(int) (events->events[i].type)][(int) (events->events[j].type)] =
						correlation;
			}
			if (correlation < 0.80) {
				histograms[(int) (events->events[i].type)][(int) (events->events[j].type)][20]++;
				if (events->events[i].type == events->events[j].type
						&& events->events[i].type == TEST_NORMAL_BEAT) {
					printf(
							"Low correlation (%lg) between %lu %lu and %lu %lu\n",
							correlation, i, events->events[i].sample, j,
							events->events[j].sample);
				}
			} else {
				histograms[(int) (events->events[i].type)][(int) (events->events[j].type)][(int) ((1.0
						- correlation) * 100.0)]++;
			}
			if (events->events[i].type != events->events[j].type) {
				n[(int) (events->events[j].type)][(int) (events->events[i].type)]++;
				ave[(int) (events->events[j].type)][(int) (events->events[i].type)] +=
						correlation;
				if (min[(int) (events->events[j].type)][(int) (events->events[i].type)]
						> correlation) {
					min[(int) (events->events[j].type)][(int) (events->events[i].type)] =
							correlation;
				}
				if (max[(int) (events->events[j].type)][(int) (events->events[i].type)]
						< correlation) {
					max[(int) (events->events[j].type)][(int) (events->events[i].type)] =
							correlation;
				}
				if (correlation < 0.80) {
					histograms[(int) (events->events[j].type)][(int) (events->events[i].type)][20]++;
				} else {
					histograms[(int) (events->events[j].type)][(int) (events->events[i].type)][(int) ((1.0
							- correlation) * 100.0)]++;
				}
			}
		}
	}

	return (0);
}



int tests_qrs_onset_and_offset(double **data, char *bad_chs, int trig_offset,int len, int nch, double freq, double *onset_, double *offset_) {
	int i, j, ch, nok_ch;
	int onset, offset;
	double energy_win_len = 0.030, onset_win_start = -0.150, onset_win_end = -0.050, offset_win_start = 0.050, offset_win_end = 0.180, offset_win_seek_start = 0.150;
	int energy_win_len_i = MAX(freq * energy_win_len, 4);
	int half_energy_win_i=energy_win_len_i/2;
	double max_energy, min_energy, limit, slope_limit;
	int start_i, end_i;
	double abs_threshold_limit = 10.0;
	double drop_time = 0.100;
	static int alloced_len=-1;
	static double *energy=NULL;

	if(len<0){ // free
		if(alloced_len>0){
			alloced_len=-1;
			my_free(energy);
			energy=NULL;
		}
		return(0);
	}
	if(len>alloced_len){
		alloced_len=len;
		energy=my_malloc(alloced_len*sizeof(double));
	}

	for (nok_ch = 0, ch = 0; ch < nch; ch++) {
		if (bad_chs[ch]<TEST_EVENT_TOTALLY_BAD) {
			nok_ch++;
		}
	}
	if (nok_ch == 0) {
		*onset_ = onset_win_end * freq + trig_offset;
		*offset_ = offset_win_start * freq + trig_offset;
		return (0);
	}

	// 1. create energy signal
	tests_create_energy2(data, len, energy_win_len_i,bad_chs, nch, energy+half_energy_win_i);

	// 1. estimate the QRS energy level
	start_i = (onset_win_start + onset_win_end) * 0.5 * freq + trig_offset;
	end_i = (offset_win_start + offset_win_end) * 0.5 * freq + trig_offset;

	max_energy=energy[start_i];
	for(i=start_i+1;i<=end_i;i++){
		if(energy[i]>max_energy){
			max_energy=energy[i];
		}
	}

	// 2. QRS onset

	// 2a. Define window for noise / baseline estimate
	start_i = onset_win_start * freq + trig_offset;
	end_i = onset_win_end * freq + trig_offset;

	// 2b. Estimate the 'baseline level'
	min_energy=energy[start_i];
	for(i=start_i+1;i<=end_i;i++){
		if(energy[i]<min_energy){
			min_energy=energy[i];
		}
	}

	// 2c. Find the sample where the energy drops below the higher threshold limit. Start checking from [end_i]

	limit = (4.0 * min_energy + max_energy) / 5.0 + abs_threshold_limit * nok_ch;
	j=end_i; // start from end_u
	if (energy[j] < limit) { // energy already below limit? -> move forward in time
		for (j++; j < trig_offset && energy[j]<limit; j++);
	} else { // else (energy is above) move backward in time
		for (j--; j > start_i && energy[j]>=limit; j--);
	}

	// 2d. Move backwards in time as long as the energy keeps dropping fast enough
	slope_limit=(limit-min_energy)/(drop_time*freq);
	onset=j+1;
	for(;j>start_i;j--){
		if(energy[j]<energy[onset] && (energy[onset]-energy[j])>slope_limit*(onset-j)){
			onset=j;
		}
	}
	onset+=half_energy_win_i/2; // final adjustment for removing a bit of the energy_filter delay

	// 5. QRS offset

	// 5a. Define window for noise / baseline estimate
	start_i = offset_win_start * freq + trig_offset;
	end_i = offset_win_end * freq + trig_offset;

	// 5b. Estimate the 'baseline level'
	min_energy=energy[start_i];
	for(i=start_i+1;i<=end_i;i++){
		if(energy[i]<min_energy){
			min_energy=energy[i];
		}
	}

	// 5c. Return from the end of the window until amplitudes reaches the limit value
	limit = (4.0 * min_energy + max_energy) / 5.0 + abs_threshold_limit * nok_ch;
	j=(start_i+end_i)/2;
	for (j = offset_win_seek_start * freq + trig_offset; j > trig_offset && energy[j]<=limit; j--);

	// 5d. Move back (forward) in time as long as the energy keeps dropping fast enough
	slope_limit=(limit-min_energy)/(drop_time*freq);
	offset=j-1;
	for(;j<end_i;j++){
		if(energy[j]<energy[offset] && (energy[offset]-energy[j])>slope_limit*(j-offset)){
			offset=j;
		}
	}
	offset-=half_energy_win_i/2; // final adjustment for removing a bit of the energy_filter delay

	*onset_ = onset;
	*offset_ = offset;
	return (0);
}



int tests_print_cluster_templates(char *filename,struct test_clusters *cluster_set, struct test_events *events, struct test_range *range) {
	FILE *fp = fopen(filename, "w");
	int i, j, k, ch;

	if(range){ // select only templates with beats inside REGION_OF_INTEREST range?
		size_t ind,min_ind,max_ind;
		int *in_roi=my_calloc(cluster_set->n_clusters,sizeof(int));
		// define the index limits for events
		for (min_ind = 0; min_ind<events->n_events && events->events[min_ind].sample<range->start;min_ind++);
		for (max_ind = min_ind; max_ind<events->n_events && events->events[max_ind].sample<range->end;max_ind++);
		max_ind--;

		// header
		for (j = 0; j < cluster_set->n_clusters; j++) {
#ifdef TESTS_SORTED_CLUSTER_INDS
			for(ind=0;ind<cluster_set->clusters[j].n_beats && cluster_set->clusters[j].inds[ind]<max_ind;ind++){
				if(cluster_set->clusters[j].inds[ind]>min_ind){
					in_roi[j]=1;
					break;
				}
			}
#endif
			for(ind=0;ind<cluster_set->clusters[j].n_beats;ind++){
				if(cluster_set->clusters[j].inds[ind]>min_ind && cluster_set->clusters[j].inds[ind]<max_ind){
					in_roi[j]=1;
					break;
				}
			}
			if(in_roi[j]){
				for (k = 0; k < cluster_set->clusters[j].n_templates; k++) {
					for (ch = 0; ch < cluster_set->nch; ch++) {
						fprintf(fp, "%d_%d_%d\t", j, k, ch);
					}
				}
			}
		}
		fprintf(fp, "\n");

		// data
		for (i = 0; i < cluster_set->template_len; i++) {
			for (j = 0; j < cluster_set->n_clusters; j++) {
				if(in_roi[j]){
					for (k = 0; k < cluster_set->clusters[j].n_templates; k++) {
						for (ch = 0; ch < cluster_set->nch; ch++) {
							fprintf(fp, "%lf\t",cluster_set->clusters[j].templates[k][ch][i]);
						}
					}
				}
			}
			fprintf(fp, "\n");
		}
		my_free(in_roi);
	}
	else{ // ... or print all
		// header
		for (j = 0; j < cluster_set->n_clusters; j++) {
			for (k = 0; k < cluster_set->clusters[j].n_templates; k++) {
				for (ch = 0; ch < cluster_set->nch; ch++) {
					fprintf(fp, "%d_%d_%d\t", j, k, ch);
				}
			}
		}
		fprintf(fp, "\n");

		// data
		for (i = 0; i < cluster_set->template_len; i++) {
			for (j = 0; j < cluster_set->n_clusters; j++) {
				for (k = 0; k < cluster_set->clusters[j].n_templates; k++) {
					for (ch = 0; ch < cluster_set->nch; ch++) {
						fprintf(fp, "%lf\t",cluster_set->clusters[j].templates[k][ch][i]);
					}
				}
			}
			fprintf(fp, "\n");
		}
	}
	fclose(fp);
	return (0);
}

int tests_print_first_cluster_beats(char *filename, struct data *data,struct test_events *events, struct test_clusters *cluster_set,int n_beats) {
	FILE *fp;
	int i, j, k, ch;
	int offset;
	char c_filename[FILENAME_MAX];

	for (j = 0; j < cluster_set->n_clusters; j++) {
		sprintf(c_filename, "%s.%d.txt", filename, j);
		fp = fopen(c_filename, "w");
		offset = cluster_set->template_offset
				+ cluster_set->clusters[j].cluster_offset;
		// header
		fprintf(fp, "time\t");
		for (ch = 0; ch < cluster_set->nch; ch++) {
			fprintf(fp, "%dc_%d\t", j, ch);
			for (k = 0; k < MIN(cluster_set->clusters[j].n_beats, n_beats);
					k++) {
				fprintf(fp, "%d_%d\t", k, ch);
			}
		}
		fprintf(fp, "\n");

		// data
		for (i = 0; i < cluster_set->template_len; i++) {
			for (ch = 0; ch < cluster_set->nch; ch++) {
				fprintf(fp, "%lf\t%lf\t",(i - offset) / (cluster_set->original_freq * cluster_set->frequency_mult),cluster_set->clusters[j].templates[0][ch][i]);
				for (k = 0; k < MIN(cluster_set->clusters[j].n_beats, n_beats);
						k++) {
					if ((-cluster_set->clusters[j].sample_offsets[k] + i - offset) % cluster_set->frequency_mult == 0) {
						fprintf(fp, "%lf\t", data->ch[ch].raw[events->events[cluster_set->clusters[j].inds[k]].sample + (-cluster_set->clusters[j].sample_offsets[k] + i - offset) / cluster_set->frequency_mult]);
					} else {
						fprintf(fp, "\t");
					}
				}
			}
			fprintf(fp, "\n");
		}
		fclose(fp);
	}
	return (0);
}

// create relation from events to clusters (TODO: this info should be attached directly to events)
int tests_define_event_cluster_ids(struct test_events *events, ssize_t *event_cluster_ids, struct test_clusters *cluster_set){
	size_t i;
	int k;

	for(i=0;i<events->n_events;i++){
		event_cluster_ids[i]=-1;
	}
	for(k=0;k<cluster_set->n_clusters;k++){
		for(i=0;i<cluster_set->clusters[k].n_beats;i++){
			event_cluster_ids[cluster_set->clusters[k].inds[i]]=k;
		}
	}
	return(0);
}


double tests_define_close_by_normal_correlations(int this, struct test_clusters *cluster_set, struct test_events *events, ssize_t *event_clusters, int half_win_len, double *ch_goodness,  double **correlations_mat){
	struct conf_cluster *conf=conf_cluster_get();
	int k1,k2;
	size_t i,j,ind,n_corr=0;
	ssize_t c1,c2;
	int tmp_offset;
	int move = conf->move * cluster_set->original_freq + 0.5;
	int win_before = conf->win_before * cluster_set->original_freq + 0.5;
	int win_after = conf->win_after * cluster_set->original_freq + 0.5;
	double tmp_corr;
	double corr_sum=0.0;
	for(i=0;i<cluster_set->clusters[this].n_beats;i++){
		ind=cluster_set->clusters[this].inds[i];
		for(j=half_win_len>ind ? 0 : ind-half_win_len;j<MIN(ind+half_win_len,events->n_events);j++){
			if(j==ind){
				continue;
			}
			if(event_clusters[j]<0){
				continue;
			}
			if(cluster_set->clusters[event_clusters[j]].type==TEST_ARTIFACT){
				continue;
			}
			if(this<event_clusters[j]){
				c1=this;
				c2=event_clusters[j];
			}
			else{
				c1=event_clusters[j];
				c2=this;
			}
			if(c1==c2){ // Same cluster?
				corr_sum+=1.0;
				n_corr++;
				continue;
			}
			if(correlations_mat[c1][c2] < -1.0){ // Is the correlation value undefined?
				// define correlation value
				correlations_mat[c1][c2]=0.0; // 0.0 correlation is set as minimum
				for (k1 = 0; k1 < cluster_set->clusters[c1].n_templates; k1++){
					for (k2 = 0; k2 < cluster_set->clusters[c2].n_templates; k2++){
						tests_get_max_correlations6(ch_goodness, cluster_set->clusters[c2].template_sets[k2],move * conf->frequency_mult,cluster_set->clusters[c2].bad_chs[k2],cluster_set->clusters[c2].ch_badness[k2],cluster_set->clusters[c1].template_sets[k1] + move / 2 * conf->frequency_mult, cluster_set->clusters[c2].bad_chs+k2,cluster_set->clusters[c2].ch_badness+k2, NULL, 1, win_before + 1 + win_after, cluster_set->nch,conf->frequency_mult, 0,&tmp_offset, &tmp_corr);
						if (tmp_corr > correlations_mat[c1][c2] && tests_are_one_ch_template_stds_similar(cluster_set->clusters[c1].template_sets[k1][tmp_offset], cluster_set->clusters[c2].template_sets[k2][move / 2 * conf->frequency_mult], win_before + 1 + win_after)){
							correlations_mat[c1][c2] = tmp_corr;
						}
					}
				}
			}
			corr_sum+=correlations_mat[c1][c2];
			n_corr++;
		}
	}
	if(n_corr==0){
		return(0.0);
	}
	return(corr_sum/(double)(n_corr));
}

double tests_define_close_by_similarity(int this, struct test_clusters *cluster_set, struct test_events *events, ssize_t *event_clusters, int half_win_len, double *ch_goodness,  double **correlations_mat){
	struct conf_cluster *conf=conf_cluster_get();
	int k1,k2;
	size_t i,j,ind;
	ssize_t c1,c2;
	int tmp_offset;
	int move = conf->move * cluster_set->original_freq + 0.5;
	int win_before = conf->win_before * cluster_set->original_freq + 0.5;
	int win_after = conf->win_after * cluster_set->original_freq + 0.5;
	double tmp_corr;
	int n_close_by=0,n_similar=0;
	double limit=conf->similarity_limit_for_clusters;
	for(i=0;i<cluster_set->clusters[this].n_beats;i++){
		ind=cluster_set->clusters[this].inds[i];
		for(j=half_win_len>ind ? 0 : ind-half_win_len;j<MIN(ind+half_win_len,events->n_events);j++){
			if(j==ind){
				continue;
			}
			if(events->events[event_clusters[j]].type==TEST_ARTIFACT){
				continue;
			}
			if(this<event_clusters[j]){
				c1=this;
				c2=event_clusters[j];
			}
			else{
				c1=event_clusters[j];
				c2=this;
			}
			if(c1<0){
				continue;
			}
			else if(c1==c2){ // Same cluster?
				n_close_by++;
				n_similar++;
				continue;
			}
			if(correlations_mat[c1][c2] < -1.0){ // Is the correlation value undefined?
				// define correlation value
				correlations_mat[c1][c2]=0.0; // 0.0 correlation is set as minimum
				for (k1 = 0; k1 < cluster_set->clusters[c1].n_templates; k1++){
					for (k2 = 0; k2 < cluster_set->clusters[c2].n_templates; k2++){
						tests_get_max_correlations5(ch_goodness, cluster_set->clusters[c2].template_sets[k2],move * conf->frequency_mult,cluster_set->clusters[c1].template_sets[k1] + move / 2 * conf->frequency_mult, NULL, 1, win_before + 1 + win_after, cluster_set->nch,conf->frequency_mult, &tmp_offset, &tmp_corr);
						if (tmp_corr > correlations_mat[c1][c2] && tests_are_one_ch_template_stds_similar(cluster_set->clusters[c1].template_sets[k1][tmp_offset], cluster_set->clusters[c2].template_sets[k2][move / 2 * conf->frequency_mult], win_before + 1 + win_after)){
							correlations_mat[c1][c2] = tmp_corr;
						}
					}
				}
			}
			if(correlations_mat[c1][c2]>limit){
				n_similar++;
			}
			n_close_by++;
		}
	}
	if(n_close_by==0){
		return(0.0);
	}
	return((double)(n_similar)/(double)(n_close_by));
}

//#define CORRECT_AMPLITUDE_SCALING_IN_TEMPLATES
int tests_define_cluster_markers(struct test_clusters *cluster_set, struct test_ch_info *ch_info) {
	size_t i, j;
	double qrs_onset, qrs_offset;
	double **data;
	int trig_offset;
	int len;
	int ch,nch;
	double freq;
	char *bad_chs;
	char *no_bad_chs = my_calloc(cluster_set->nch,sizeof(char));
	double *ave_max_qrs_slope=my_calloc(cluster_set->nch,sizeof(double));
	double *ave_qrs_power=my_calloc(cluster_set->nch,sizeof(double));
	size_t *total_nok=my_calloc(cluster_set->nch,sizeof(size_t));

	for (i = 0; i < cluster_set->n_clusters; i++) {
		cluster_set->clusters[i].markers = my_realloc(cluster_set->clusters[i].markers,cluster_set->clusters[i].n_templates * sizeof(struct test_cluster_markers));
		len = cluster_set->template_len;
		nch = cluster_set->nch;
		freq = cluster_set->original_freq * cluster_set->frequency_mult;
		trig_offset = cluster_set->template_offset - cluster_set->clusters[i].cluster_offset;

		for (j = 0; j < cluster_set->clusters[i].n_templates; j++) {
			data = cluster_set->clusters[i].templates[j];
			if (cluster_set->clusters[i].bad_chs[j]) {
				bad_chs = cluster_set->clusters[i].bad_chs[j];
			} else {
				bad_chs = no_bad_chs;
			}

#ifdef CORRECT_AMPLITUDE_SCALING_IN_TEMPLATES
			{
				size_t i;
				int ch;
				for(ch=0;ch<cluster_set->nch;ch++){
					for(i=0;i<len;i++){
						data[ch][i]*=4.8828125;
					}
				}

			}
#endif

			tests_qrs_onset_and_offset(data, bad_chs, trig_offset, len, nch, freq, &qrs_onset, &qrs_offset);
			cluster_set->clusters[i].markers[j].qrs_onset=qrs_onset / freq;
			cluster_set->clusters[i].markers[j].qrs_offset=qrs_offset / freq;
			cluster_set->clusters[i].markers[j].qrs_width = (qrs_offset - qrs_onset) / freq;

			cluster_set->clusters[i].markers[j].initial_qrs_slope_length = my_malloc(cluster_set->nch * sizeof(double));
			tests_raising_slope_duration(data, bad_chs, trig_offset, len, nch, freq,cluster_set->clusters[i].markers[j].initial_qrs_slope_length);

			cluster_set->clusters[i].markers[j].max_qrs_slope = my_malloc(cluster_set->nch * sizeof(double));
			tests_get_max_slopes(data, trig_offset, len, nch, freq,cluster_set->clusters[i].markers[j].max_qrs_slope);

			cluster_set->clusters[i].markers[j].qrs_power = my_malloc(cluster_set->nch * sizeof(double));
			tests_get_powers(data, MAX(1, qrs_onset), MIN(qrs_offset, len - 2),nch, freq, cluster_set->clusters[i].markers[j].qrs_power);

			cluster_set->clusters[i].markers[j].n_qrs_waves = my_malloc(cluster_set->nch * sizeof(int));
			tests_get_n_waves(data, trig_offset, len, nch, freq,cluster_set->clusters[i].markers[j].n_qrs_waves);

			for(ch=0;ch<nch;ch++){
				if(!bad_chs[ch]){
					total_nok[ch]+=cluster_set->clusters[i].n_beats;
					ave_max_qrs_slope[ch]+=cluster_set->clusters[i].markers[j].max_qrs_slope[ch]*cluster_set->clusters[i].n_beats;
					ave_qrs_power[ch]+=cluster_set->clusters[i].markers[j].qrs_power[ch]*cluster_set->clusters[i].n_beats;
				}
			}

#ifdef TESTS_SAVE_TEMPLATES
//			printf("Cluster %lu/%lu: QRS [%.1lf,%.1lf], trig_offset: %d\n", i,j, qrs_onset, qrs_offset, trig_offset);
#endif
		}
	}

/*	// define average values
	for(ch=0;ch<nch;ch++){
		if(total_nok[ch]>0){
			ave_max_qrs_slope[ch]/=(double)(total_nok[ch]);
			ave_qrs_power[ch]/=(double)(total_nok[ch]);
		}
	}

	// define relative amplitude values 1
	for (i = 0; i < cluster_set->n_clusters; i++) {
		for (j = 0; j < cluster_set->clusters[i].n_templates; j++) {
			for(ch=0;ch<nch;ch++){
				if(ave_max_qrs_slope[ch]>0.0){
					cluster_set->clusters[i].markers[j].max_qrs_slope[ch]/=ave_max_qrs_slope[ch];
				}
				if(ave_qrs_power[ch]){
					cluster_set->clusters[i].markers[j].qrs_power[ch]/=ave_qrs_power[ch];
				}
			}
		}
	}*/

	// define relative amplitude values 2
	/*for (i = 0; i < cluster_set->n_clusters; i++) {
		for (j = 0; j < cluster_set->clusters[i].n_templates; j++) {
			for(ch=0;ch<nch;ch++){
				if(ave_max_qrs_slope[ch]>0.0){
					cluster_set->clusters[i].markers[j].max_qrs_slope[ch]*=ch_info->amplitude_normalization[ch];
				}
				if(ave_qrs_power[ch]){
					cluster_set->clusters[i].markers[j].qrs_power[ch]*=ch_info->amplitude_normalization[ch];
				}
			}
		}
	}*/

	my_free(no_bad_chs);
	my_free(total_nok);
	my_free(ave_max_qrs_slope);
	my_free(ave_qrs_power);
	return (0);

}

size_t tests_event_ind_from_sample(struct test_events *events, size_t sample){
	size_t ind=events->n_events/2;
	if(events->events[ind].sample>sample){
		for(;ind>0 && events->events[ind].sample>sample;ind--);
		if(events->events[ind].sample!=sample && sample-events->events[ind].sample>events->events[ind+1].sample-sample){
			ind++;
		}
	}
	else{
		for(;ind<events->n_events-1 && events->events[ind].sample<sample;ind++);
		if(events->events[ind].sample!=sample && sample-events->events[ind-1].sample<events->events[ind].sample-sample){
			ind--;
		}
	}
	return(ind);
}

int tests_cluster_from_event_ind(struct test_clusters *cluster_set, size_t ind, int *move){
	int k;
	size_t i;
	for(k=0;k<cluster_set->n_clusters;k++){
		for(i=0;i<cluster_set->clusters[k].n_beats;i++){
			if(cluster_set->clusters[k].inds[i]==ind){
				if(move){
					*move=cluster_set->clusters[k].sample_offsets[i]+cluster_set->clusters[k].cluster_offset;
				}
				return(k);
			}
		}
	}
	if(move){
		*move=0;
	}
	return(-1);
}



struct tests_marker_statistics {
	double ave[N_REAL_TEST_LABELS];
	double sd[N_REAL_TEST_LABELS];
	size_t count[N_REAL_TEST_LABELS];
};

struct tests_marker_accuracy {
	enum test_trigger_type type1;
	enum test_trigger_type type2;
	size_t n_correct1;
	size_t n_correct2;
	size_t n_total1;
	size_t n_total2;
};

#define N_MARKER_BINS 40

static int tests_define_marker_histogram_statistics(
		struct tests_marker_statistics *statistics,
		size_t bins[N_REAL_TEST_LABELS][N_MARKER_BINS + 2]) {
	int i, j;
	for (i = 0; i < N_REAL_TEST_LABELS; i++) {
		statistics->count[i] = 0;
		statistics->ave[i] = 0.0;
		statistics->sd[i] = 0;
		for (j = 0; j < N_MARKER_BINS + 2; j++) {
			statistics->ave[i] += bins[i][j] * j;
			statistics->sd[i] += bins[i][j] * POW2(j);
			statistics->count[i] += bins[i][j];
		}
		if (statistics->count[i] > 1) {
			statistics->ave[i] /= statistics->count[i];
			statistics->sd[i] = sqrt(
					statistics->sd[i]
							/ statistics->count[i]-POW2(statistics->ave[i]));
		} else {
			statistics->sd[i] = NAN;
			if (statistics->count[i] == 0) {
				statistics->ave[i] = NAN;
			}
		}
	}
	return (0);
}

static int tests_define_marker_accuracy(struct tests_marker_accuracy *accuracy,
		struct tests_marker_statistics *statistics,
		size_t bins[N_REAL_TEST_LABELS][N_MARKER_BINS + 2]) {
	double limit_value;
	int i;
	int limit;
	accuracy->n_correct1 = accuracy->n_total1 = accuracy->n_correct2 =
			accuracy->n_total2 = 0;
	limit_value = statistics->ave[accuracy->type1]
			+ (statistics->ave[accuracy->type2]
					- statistics->ave[accuracy->type1])
					* statistics->sd[accuracy->type2]
					/ (statistics->sd[accuracy->type1]
							+ statistics->sd[accuracy->type2]);
	limit = (int) (limit_value + 0.5);
	if (statistics->ave[accuracy->type1] > statistics->ave[accuracy->type2]) {
		if (limit == limit_value + 0.5
				&& statistics->sd[accuracy->type1]
						< statistics->sd[accuracy->type2]) { // is the limit exactly on the border, and the sd of the bigger valued group smaller? -> give the border to the smaller sd group
			limit--;
		}
		for (i = 0; i <= limit; i++) {
			accuracy->n_correct2 += bins[accuracy->type2][i];
			accuracy->n_total1 += bins[accuracy->type1][i];
			accuracy->n_total2 += bins[accuracy->type2][i];
		}
		for (; i < N_MARKER_BINS + 2; i++) {
			accuracy->n_correct1 += bins[accuracy->type1][i];
			accuracy->n_total1 += bins[accuracy->type1][i];
			accuracy->n_total2 += bins[accuracy->type2][i];
		}
	} else {
		if (limit == limit_value + 0.5
				&& statistics->sd[accuracy->type1]
						> statistics->sd[accuracy->type2]) { // is the limit exactly on the border, and the sd of the bigger valued group smaller? -> give the border to the smaller sd group
			limit--;
		}
		for (i = 0; i <= limit; i++) {
			accuracy->n_correct1 += bins[accuracy->type1][i];
			accuracy->n_total1 += bins[accuracy->type1][i];
			accuracy->n_total2 += bins[accuracy->type2][i];
		}
		for (; i < N_MARKER_BINS + 2; i++) {
			accuracy->n_correct2 += bins[accuracy->type2][i];
			accuracy->n_total1 += bins[accuracy->type1][i];
			accuracy->n_total2 += bins[accuracy->type2][i];
		}
	}
	printf("Limit histogram index: %d\n", limit);
	return (0);
}

int tests_print_event_markers(char *base_filename, char *casename,
		struct test_events *events, struct test_event_markers *markers,
		int n_markers) {
	int n;
	char filename[FILENAME_MAX];
	size_t i, j;
	double step;
	double ave[N_REAL_TEST_LABELS], sd[N_REAL_TEST_LABELS];
	size_t count[N_REAL_TEST_LABELS];
	size_t bins[N_REAL_TEST_LABELS][N_MARKER_BINS + 2];
	FILE *fp;
	struct tests_marker_accuracy accuracy;
	struct tests_marker_statistics statistics;
	for (n = 0; n < n_markers; n++) {
		if (events->n_events != markers[n].n_values) {
			printf(
					"Error: there is different number of events (%lu) and event parameter values (%d)\n",
					events->n_events, markers[n].n_values);
			continue;
		}

		step = (markers[n].range_max_value - markers[n].range_min_value)
				/ (double) (N_MARKER_BINS);
		sprintf(filename, "%s.%s.txt", base_filename, markers[n].name);
//		printf("%d %s\n",n,filename);
		fp = fopen(filename, "r");
		if (fp) {  // does file exist already? -> open the file for append
			fclose(fp);
			fp = fopen(filename, "a");
		} else { // write the header line
			fp = fopen(filename, "w");
			fprintf(fp, "version\tfilename\tparameter_name");
			fprintf(fp,
					"\tN accuracy\tV accuracy\tcorrect N\ttotal N\tcorrect V\ttotal V");
			for (i = 0; i < N_REAL_TEST_LABELS; i++) {
				fprintf(fp, "\t%c_count\t%c_ave\t%c_sd",
						tests_trigger_type_to_c(i), tests_trigger_type_to_c(i),
						tests_trigger_type_to_c(i));
				fprintf(fp, "\t%c_->%lg", tests_trigger_type_to_c(i),
						markers[n].range_min_value);
				for (j = 1; j < N_MARKER_BINS + 1; j++) {
					fprintf(fp, "\t%c_%lg-%lg", tests_trigger_type_to_c(i),
							markers[n].range_min_value + (j - 1) * step,
							markers[n].range_min_value + j * step);
				}
				fprintf(fp, "\t%c_%lg->", tests_trigger_type_to_c(i),
						markers[n].range_max_value);
			}
			fprintf(fp, "\n");
		}
		fprintf(fp, "%d\t%s\t%s", TEST_FILE_VERSION_NO, casename,
				markers[n].name);

		// define histograms
		for (i = 0; i < N_REAL_TEST_LABELS; i++) {
			count[i] = 0;
			ave[i] = sd[i] = 0.0;
			for (j = 0; j < N_MARKER_BINS + 2; j++) {
				bins[i][j] = 0;
			}
			for (j = 0; j < markers[n].n_values; j++) {
				if (events->events[j].type
						== i&& markers[n].values[j]<0.5*FLT_MAX) {
					count[i]++;
					ave[i] += markers[n].values[j];
					sd[i] += POW2(markers[n].values[j]);
					if (markers[n].values[j] < markers[n].range_min_value) {
						bins[i][0]++;
					} else if (markers[n].values[j]
							>= markers[n].range_max_value) {
						bins[i][N_MARKER_BINS + 1]++;
					} else {
						bins[i][(int) ((markers[n].values[j]
								- markers[n].range_min_value) / step) + 1]++;
					}
				}
			}
			ave[i] /= (double) (count[i]);
			sd[i] = sqrt(sd[i] / (double) (count[i]) - (ave[i] * ave[i]));
		}

		// define and print basic statistics
		accuracy.type1 = TEST_NORMAL_BEAT;
		accuracy.type2 = TEST_VENTRICULAR;
		tests_define_marker_histogram_statistics(&statistics, bins);
		tests_define_marker_accuracy(&accuracy, &statistics, bins);
		fprintf(fp, "\t%lg\t%lg\t%lu\t%lu\t%lu\t%lu",
				(double) (accuracy.n_correct1) / accuracy.n_total1,
				(double) (accuracy.n_correct2) / accuracy.n_total2,
				accuracy.n_correct1, accuracy.n_total1, accuracy.n_correct2,
				accuracy.n_total2);

		// print the statistics and histograms for all the labels
		for (i = 0; i < N_REAL_TEST_LABELS; i++) {
			if (count[i] > 0) {
				fprintf(fp, "\t%lu\t%lg\t%lg", count[i], ave[i], sd[i]);
			} else {
				fprintf(fp, "\t0\tnan\tnan");
			}
			for (j = 0; j < N_MARKER_BINS + 2; j++) {
				fprintf(fp, "\t%lu", bins[i][j]);
			}
		}
		fprintf(fp, "\n");
		fclose(fp);
	}
	return (0);
}

int tests_analyze_event_markers(char *run_name, char *base_filename,
		struct test_event_markers *markers, int n_markers) {
	int n;
	char filename[FILENAME_MAX];
	size_t i, j;
	char c;
	int version_no;
	double step;
	size_t bins[N_REAL_TEST_LABELS][N_MARKER_BINS + 2];
	size_t one_bin[N_REAL_TEST_LABELS][N_MARKER_BINS + 2];
	FILE *fp;
	FILE *out;
	struct tests_marker_accuracy one_accuracy, accuracy, total_accuracy;
	struct tests_marker_statistics one_statistics, statistics, total_statistics;

	sprintf(filename, "%s.parameter_results.txt", base_filename);
	out = fopen(filename, "r");
	if (out) {  // does the file exist already?
		fclose(out);
		out = fopen(filename, "a");
	} else {
		out = fopen(filename, "w");

		fprintf(out, "version\trun_name\tparameter_name");
		fprintf(out,
				"\tN_case_accuracy\tV_case_accuracy\tcase_accuracy\tcase_correct_N\tcase_total_N\tcase_correct_V\tcase total_V");
		fprintf(out,
				"\tN_total_accuracy\tV_total_accuracy\ttotal_accuracy\ttotal_correct_N\ttotal_total_N\ttotal_correct_V\ttotal_total_V");

		for (i = 0; i < N_REAL_TEST_LABELS; i++) {
			fprintf(out, "\t%c_case_count\t%c_case_ave\t%c_case_sd",
					tests_trigger_type_to_c(i), tests_trigger_type_to_c(i),
					tests_trigger_type_to_c(i));
			fprintf(out, "\t%c_total_count\t%c_total_ave\t%c_total_sd",
					tests_trigger_type_to_c(i), tests_trigger_type_to_c(i),
					tests_trigger_type_to_c(i));
			fprintf(out, "\t%c_range_min\t%c_range_max",
					tests_trigger_type_to_c(i), tests_trigger_type_to_c(i));
			for (j = 0; j < N_MARKER_BINS + 2; j++) {
				fprintf(out, "\t%c_%lu", tests_trigger_type_to_c(i), j);
			}
		}
		fprintf(out, "\n");
	}

	for (n = 0; n < n_markers; n++) {
		// initialize values
		accuracy.n_correct1 = accuracy.n_total1 = accuracy.n_correct2 =
				accuracy.n_total2 = 0;
		for (i = 0; i < N_REAL_TEST_LABELS; i++) {
			statistics.count[i] = 0;
			statistics.ave[i] = 0.0;
			statistics.sd[i] = 0.0;
			for (j = 0; j < N_MARKER_BINS + 2; j++) {
				bins[i][j] = 0;
			}
		}

		sprintf(filename, "%s.%s.txt", base_filename, markers[n].name);
		fp = fopen(filename, "r");
		if (fp == NULL) {
			printf("Can't open %s for reading\n", filename);
			continue;
		}

		// skip the header
		do {
			c = fgetc(fp);
		} while (c != '\n' && c != EOF);

		// read and define the case by case accuracy values
		while (fscanf(fp, "%d\t%*s\t%*s\t%*g\t%*g\t%lu\t%lu\t%lu\t%lu",
				&version_no, &one_accuracy.n_correct1, &one_accuracy.n_total1,
				&one_accuracy.n_correct2, &one_accuracy.n_total2) == 5) {
			if (version_no != TEST_FILE_VERSION_NO) {
				printf("Mismatch in version numbers in %s\n", filename);
				continue;
			}
			accuracy.n_correct1 += one_accuracy.n_correct1;
			accuracy.n_total1 += one_accuracy.n_total1;
			accuracy.n_correct2 += one_accuracy.n_correct2;
			accuracy.n_total2 += one_accuracy.n_total2;

			// read and define the total histograms, and sums (for average) of ave and sd values
			for (i = 0; i < N_REAL_TEST_LABELS; i++) {
				fscanf(fp, "\t%lu\t%lg\t%lg", &one_statistics.count[i],
						&one_statistics.ave[i], &one_statistics.sd[i]);
				statistics.count[i] += one_statistics.count[i];
				if (one_statistics.count[i] > 0) {
					statistics.ave[i] += one_statistics.ave[i]
							* one_statistics.count[i];
					statistics.sd[i] += one_statistics.sd[i]
							* one_statistics.count[i];
				}
				for (j = 0; j < N_MARKER_BINS + 2; j++) {
					fscanf(fp, "\t%lu", &one_bin[i][j]);
					bins[i][j] += one_bin[i][j];
				}
			}
			fscanf(fp, "\n");
		}
		fclose(fp);

		for (i = 0; i < N_REAL_TEST_LABELS; i++) {
			if (statistics.count[i] > 1) {
				statistics.ave[i] /= statistics.count[i];
				statistics.sd[i] /= statistics.count[i];
			} else {
				statistics.ave[i] = NAN;
				statistics.sd[i] = NAN;
			}
		}

		// define total ave, sd and accuracy values
		total_accuracy.type1 = TEST_NORMAL_BEAT;
		total_accuracy.type2 = TEST_VENTRICULAR;
		tests_define_marker_histogram_statistics(&total_statistics, bins);
		tests_define_marker_accuracy(&total_accuracy, &total_statistics, bins);

		fprintf(out, "%d\t%s\t%s", TEST_FILE_VERSION_NO, run_name,
				markers[n].name);
		fprintf(out, "\t%lg\t%lg\t%lg\t%lu\t%lu\t%lu\t%lu",
				(double) (accuracy.n_correct1) / accuracy.n_total1,
				(double) (accuracy.n_correct2) / accuracy.n_total2,
				(double) (accuracy.n_correct1 + accuracy.n_correct2)
						/ (accuracy.n_total1 + accuracy.n_total2),
				accuracy.n_correct1, accuracy.n_total1, accuracy.n_correct2,
				accuracy.n_total2);
		fprintf(out, "\t%lg\t%lg\t%lg\t%lu\t%lu\t%lu\t%lu",
				(double) (total_accuracy.n_correct1) / total_accuracy.n_total1,
				(double) (total_accuracy.n_correct2) / total_accuracy.n_total2,
				(double) (total_accuracy.n_correct1 + total_accuracy.n_correct2)
						/ (total_accuracy.n_total1 + total_accuracy.n_total2),
				total_accuracy.n_correct1, total_accuracy.n_total1,
				total_accuracy.n_correct2, total_accuracy.n_total2);

		step = (markers[n].range_max_value - markers[n].range_min_value)
				/ (double) (N_MARKER_BINS);
		for (i = 0; i < N_REAL_TEST_LABELS; i++) {
			fprintf(out, "\t%lu\t%lg\t%lg", statistics.count[i],
					statistics.ave[i], statistics.sd[i]);
			fprintf(out, "\t%lu\t%lg\t%lg", total_statistics.count[i],
					(total_statistics.ave[i] - 1.0) * step
							+ markers[n].range_min_value,
					total_statistics.sd[i] * step);
			fprintf(out, "\t%lg\t%lg", markers[n].range_min_value,
					markers[n].range_max_value);
			for (j = 0; j < N_MARKER_BINS + 2; j++) {
				fprintf(out, "\t%lu", bins[i][j]);
			}
		}
		fprintf(out, "\n");
	}
	fclose(out);
	return (0);
}

 enum test_trigger_type tests_get_event_type(size_t sample, struct test_events *events, size_t pos_limit, size_t neg_limit){
	 size_t i;
	 if(events==NULL || events->n_events==0){
		 return(TEST_BEAT_UNDEFINED);
	 }
	 i=events->n_events/2;
	 if(sample<events->events[i].sample){
		 i=events->n_events/4;
		 if(sample<events->events[i].sample){
			 i=0;
		 }
	 }
	 else if(sample>events->events[events->n_events*3/4].sample){
		 i=events->n_events*3/4;
	 }
	 for(;events->events[i].sample<sample && i<events->n_events;i++);
	 if(i==0 || sample-events->events[i-1].sample>events->events[i].sample-sample){
		 if(events->events[i].sample-sample<pos_limit){
			 return(events->events[i].type);
		 }
		 else{
			 return(TEST_OTHER_PSEUDOBEAT);
		 }
	 }
	 else{
		 if(sample-events->events[i-1].sample<neg_limit){
			 return(events->events[i-1].type);
		 }
		 else{
			 return(TEST_OTHER_PSEUDOBEAT);
		 }
	 }
 }


double tests_relative_t_wave_correlation(int reference_id, int this_id, struct test_events *events, struct test_clusters *clusters){
	double relative_corr=0.0;

	return(relative_corr);
}

extern double array_kth_biggest(double a[], size_t n, size_t k);
int tests_define_cluster_event_timings(struct test_events *events,struct test_clusters *cluster_set,size_t start, size_t end, double *ratio_to_nn_, double *median_nn){
	int n;
	size_t i,n_samples,sample;
	double *ratio_to_nn=NULL;
	double *selected_ratio_to_nn=my_malloc(events->n_events*sizeof(double));
	if(ratio_to_nn_){ // Do we want to keep the values?
		ratio_to_nn=ratio_to_nn_;
	}
	else{
		ratio_to_nn=my_malloc(events->n_events*sizeof(double));
	}
	rhythm_normality_for_events(events,ratio_to_nn, median_nn,0.850*cluster_set->original_freq,1);

	for(n=0;n<cluster_set->n_clusters;n++){
		for(n_samples=0,i=0;i<cluster_set->clusters[n].n_beats;i++){
			sample=events->events[cluster_set->clusters[n].inds[i]].sample;
			if(start==end || (start<sample && end>sample)){
				selected_ratio_to_nn[n_samples]=ratio_to_nn[cluster_set->clusters[n].inds[i]];
				if(selected_ratio_to_nn[n_samples]>0){
					n_samples++;
				}
			}
		}
		if(n_samples==0){
			cluster_set->clusters[n].annotation_markers.event_timing=-1;
		}
		else{
			cluster_set->clusters[n].annotation_markers.event_timing=array_kth_biggest(selected_ratio_to_nn,n_samples,(n_samples)/2);
		}
	}
	if(!ratio_to_nn_){
		my_free(ratio_to_nn);
	}
	my_free(selected_ratio_to_nn);

	return(0);
}

#define TEST_MAX_EXTERNAL_FILTER_LEN 10000
int tests_filter_with_external_FIR(char *filename, struct data *from, struct data *to){
	FILE *fp=fopen(filename,"r");
	int ch;
	size_t i,j;
	double filter[TEST_MAX_EXTERNAL_FILTER_LEN];
	int filter_len=0,filter_offset;
	int normalize_filter=1;
	if(fp==NULL){
		printf("Couldn't open external FIR filter %s\n",filename);
		return(-1);
	}
	while(fscanf(fp,"%lf;\n",&(filter[filter_len++]))==1);

	fclose(fp);
	if(filter_len<3){
		printf("Something wrong with %s filter. Managed to read only %d samples.\n",filename,filter_len);
		return(-2);
	}

	if(normalize_filter){
		double total_power=0.0;
		for(i=0;i<filter_len/2;i++){
			total_power+=POW2(filter[i]);
		}
		total_power=4.0*sqrt(total_power);
		for(i=0;i<filter_len;i++){
			filter[i]/=total_power;
		}

	}


	filter_offset=filter_len/2;

	data_initiate_and_alloc_based_on_template(to,from);

	for(ch=0;ch<from->number_of_channels;ch++){
		for(i=0;i<from->samples_per_channel;i++){
			to->ch[ch].raw[i]=0.0;
				for(j=0;j<filter_len;j++){
					if(i+j<filter_offset || i+j>from->samples_per_channel+filter_offset){
						continue;
					}
					to->ch[ch].raw[i]+=from->ch[ch].raw[i+j-filter_offset]*filter[j];
//					to->ch[ch].raw[i]+=from->ch[ch].raw[i+j-filter_offset]*filter[j]/(filter[filter_offset]*20.0);
				}
			}
		}

	return(0);
}

int tests_create_external_filtered(struct data *original, struct data *filtered1, struct data *filtered2){
	int ret_val=0;
	ret_val+=tests_filter_with_external_FIR("/home/hvaanane/tests/time-frequency/s1/all.convolution.kernel.csv", original, filtered1);
	ret_val+=tests_filter_with_external_FIR("/home/hvaanane/tests/time-frequency/s2/all.convolution.kernel.csv", original, filtered2);
	return(ret_val);
}

int tests_get_cluster_tachyratio(struct test_events *events, struct test_cluster *this_cluster, double sfreq, double *tachy_ratio, double *extreme_tachy_ratio){
	ssize_t i,j;
	struct conf_rhythm *rhythm_conf=conf_rhythm_get();
	size_t total=0,tachy=0,extreme_tachy=0;
	double tachy_rr_limit=60.0/rhythm_conf->tachy_limit*sfreq;
	double extreme_tachy_rr_limit=60.0/rhythm_conf->extreme_tachy_limit*sfreq;
	for(i=0;i<this_cluster->n_beats;i++){
		for(j=this_cluster->inds[i]-1;j>=0 && events->events[j].type==TEST_ARTIFACT;j--);
		if(j<0){
			continue;
		}
		if(events->events[this_cluster->inds[i]].sample-events->events[j].sample<tachy_rr_limit){
			tachy++;
			if(events->events[this_cluster->inds[i]].sample-events->events[j].sample<extreme_tachy_rr_limit){
				extreme_tachy++;
			}
		}
		total++;
	}
	if(total){ // did we find any beats?
		*tachy_ratio=(double)(tachy)/(double)(total);
		*extreme_tachy_ratio=(double)(extreme_tachy)/(double)(total);
	}
	else{		// if not, then we'll just pick something from half way..
		*tachy_ratio=0.5;
		*extreme_tachy_ratio=0.5;
	}
	return(0.0);
}

int tests_define_cluster_annotation_markers(struct test_events *events,struct test_clusters *cluster_set,struct test_ch_info *ch_info,ssize_t *event_cluster_ids,double **correlations_mat){
	int n;
	int alloced_corr_mat=0,alloced_ec_ids=0;
//	tests_define_cluster_event_timings(events,cluster_set,0,0);
	if(!event_cluster_ids){
		event_cluster_ids=my_malloc(events->n_events*sizeof(ssize_t));
		tests_define_event_cluster_ids(events, event_cluster_ids, cluster_set);
		alloced_ec_ids=1;
	}

	if(!correlations_mat){
		size_t i,j;
		correlations_mat=matrix_malloc_d(cluster_set->n_clusters,cluster_set->n_clusters);
		for(i=0;i<cluster_set->n_clusters;i++){
			for(j=0;j<i;j++){
				correlations_mat[j][i]=-2.0;
			}
			correlations_mat[i][i]=1.0;
		}
		alloced_corr_mat=1;
	}

	for(n=0;n<cluster_set->n_clusters;n++){
		tests_get_cluster_tachyratio(events,cluster_set->clusters+n,cluster_set->original_freq,&(cluster_set->clusters[n].annotation_markers.tachy_ratio),&(cluster_set->clusters[n].annotation_markers.extreme_tachy_ratio));
		tests_get_n_corr(n, 5, 10, ch_info->ch_goodness, cluster_set, &(cluster_set->clusters[n].annotation_markers.similarity_to_normals), &(cluster_set->clusters[n].annotation_markers.diff_from_normals));
//		tests_get_one_ch_n_corr(n, 5, 10, cluster_set,&(cluster_set->clusters[n].annotation_markers.one_ch_similarity_to_normals));
//		cluster_set->clusters[n].annotation_markers.close_by_similarity=tests_define_close_by_similarity(n, cluster_set, events, event_cluster_ids, 15, ch_info->ch_goodness, correlations_mat);
	}
	if(alloced_ec_ids){
		my_free(event_cluster_ids);
	}
	if(alloced_corr_mat){
		matrix_free_d(correlations_mat);
	}
	return(0);
}

#define TESTS_MINIMUM_EVENT_PROBABILITY 0.1
double test_define_cluster_event_probability(struct test_events *events, struct test_cluster *this_cluster){
	size_t i;
	double cluster_probability=0.0;
	for(i=0;i<this_cluster->n_beats;i++){
		cluster_probability+=events->events[this_cluster->inds[i]].probability;
	}
	if(cluster_probability>TESTS_MINIMUM_EVENT_PROBABILITY){
		return(cluster_probability/(double)(this_cluster->n_beats));
	}
	return(TESTS_MINIMUM_EVENT_PROBABILITY);
}






int tests_free_ch_info(struct test_ch_info *ch_info){
	int ch;
	if(ch_info->ch_goodness){
		my_free(ch_info->ch_goodness);
		ch_info->ch_goodness=NULL;
	}
	if(ch_info->ch_ranges){
		for(ch=0;ch<ch_info->nch;ch++){
			if(ch_info->ch_ranges[ch].n_ranges>0){
				my_free(ch_info->ch_ranges[ch].ranges);
			}
		}
		my_free(ch_info->ch_ranges);
		ch_info->ch_ranges=NULL;
	}
	ch_info->nch=0;
	return(0);
}




int free_data(struct data *dat) {
	int ch;
	if (dat->ch == NULL) {
		return (0);
	}
	for (ch = 0; ch < dat->number_of_channels; ch++) {
		if (dat->ch[ch].raw)
			my_free(dat->ch[ch].raw);
	}
	dat->number_of_channels = 0;
	my_free(dat->ch);
	dat->ch=NULL;
	return (0);
}



int tests_read_mit_data(char *basename, struct data *data){
	FILE *fp;
	int ch;
	char filename[FILENAME_MAX];
	char file_id[100];
	char tmpstring[500];
	int narg,format,prev_format;
	unsigned char *data_buffer;
	size_t nread,i,help;
	int nch;
	int *dc_lvl;
	double *scale;
	int i_data_val;
//	r eve_mit_files.txt AINEISTO.conf skip_comparisons just_marker_prints print_to_json /var/www/RAecg_annotation/cache/
//	r all_but_vt_files.txt AINEISTO.conf print_cluster_markers print_cluster_probabilities print_energy print_filtered print_cluster_pics single_combined_annotation print_to_json /var/www/RAecg_annotation/cache/

	// 1. read the header

	strcpy(filename,basename);
	strcat(filename,".hea");
	fp=fopen(filename,"r");
	if(fp==NULL){
		printf("Couldn't open MIT header file: %s\n",filename);
		return(-1);
	}
	if(fgets (tmpstring,500,fp)==NULL){
		printf("Something wrong in MIT header file: %s\n",filename);
		return(-1);
	}
	while(tmpstring[0]=='#' || strlen(tmpstring)<2){ // skip the comment lines
		if(fgets (tmpstring,500,fp)==NULL){
			printf("Something wrong in MIT header file: %s\n",filename);
			return(-1);
		}
	}
	narg=sscanf(tmpstring,"%s %lu %lf %lu\n",file_id,&(data->number_of_channels),&(data->sample_freq),&(data->samples_per_channel));

	if(narg<4){
		printf("Only MIT files with nch, freq and nsamp info in header-file are supported atm.\n");
		return(-1);
	}

	dc_lvl=my_malloc(data->number_of_channels*sizeof(int));
	scale=my_malloc(data->number_of_channels*sizeof(double));
	for(ch=0;ch<data->number_of_channels;ch++){
		if(fgets (tmpstring,500,fp)==NULL){
			printf("Something wrong in MIT header file: %s\n",filename);
			return(-1);
		}
		while(tmpstring[0]=='#' || strlen(tmpstring)<2){ // skip the comment lines
			if(fgets (tmpstring,500,fp)==NULL){
				printf("Something wrong in MIT header file: %s\n",filename);
				return(-1);
			}
		}
		sscanf(tmpstring,"%*s %d %lf %*d %d %*d %*d %*d",&format,scale+ch,dc_lvl+ch);
		if(scale[ch]==0){
			scale[ch]=5.0; // Default gain is 200 -> default scale is 1000.0/200.0
		}
		else{
			scale[ch]=1000.0/scale[ch];
		}
		if(ch>0 && prev_format!=format){
			printf("Only one data format / case is allowed: %s\n",filename);
			return(-1);
		}
		prev_format=format;
	}
	fclose(fp);

	// 2. Read the data

	if(format!=212){
		printf("Only 212 MIT format supported atm\n");
		return(-1);
	}

	// 2.1 Open the file and alloc

	strcpy(filename,basename);
	strcat(filename,".dat");
	fp=fopen(filename,"rb");
	if(fp==NULL){
		printf("Couldn't open MIT data file: %s\n",filename);
		return(-1);
	}

	data->ch = my_calloc(data->number_of_channels, sizeof(struct data_channel));

	// 2.2 read and convert
	data_buffer=my_malloc(data->samples_per_channel*data->number_of_channels*3/2);
	nread=fread(data_buffer,1,data->samples_per_channel*data->number_of_channels*3/2,fp);
	if(nread<data->samples_per_channel*data->number_of_channels*3/2){
		printf("Couldn't read all the data %s\n",filename);
		return(-1);
	}
	nch=data->number_of_channels;
	for(ch=0;ch<nch;ch++){
		data->ch[ch].raw=my_malloc(data->samples_per_channel*sizeof(double));
		for(i=0;i<data->samples_per_channel;i++){
			help=ch+nch*i; // TODO: simpler data conversion
			if(help%2)
				i_data_val=((data_buffer[(help/2)*3+1]<<4) & 0xf00) | (data_buffer[(help/2)*3+2] & 0xff);
			else
				i_data_val=(short)((data_buffer[help/2*3+1] << 8) | (data_buffer[help/2*3+0] & 0xff));
			// Add sign from the 12th bit.
			if (i_data_val & 0x800)
				i_data_val |= ~(0xfff);
			else
				i_data_val &= 0xfff;
			data->ch[ch].raw[i]=(i_data_val-dc_lvl[ch])*scale[ch];
		}
	}

	// 2.3. free
	my_free(scale);
	my_free(dc_lvl);
	my_free(data_buffer);

	return(0);
}



