/*
 * af_detection.c
 *
 *  Created on: Nov 3, 2015
 *      Author: hvaanane
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>
#include <unistd.h>
#include <errno.h>
#include <limits.h>
#include <math.h>
#include <float.h>

#include "util.h"
#include "types.h"
#include "test_types.h"
#include "trig.h"
#include "memory_manager.h"
#include "rhythm.h"
#include "conf.h"


// High pass filter with 0.15 - 0.25 1/beat transition band
#define RHYTHM_FILTER_LEN 11

// Original version based to: Zhou X, Ding H, Wu W, Zhang Y. A Real-Time Atrial Fibrillation Detection Algorithm Based on the Instantaneous State of Heart Rate. PLoS One. 2015 Sep 16;10(9):e0136544

double *rhythm_simplified_bpm_entropy(struct test_events *events, double freq, int win_len, double step_size_){
	size_t i,j,k;
	int step_size=step_size_>0 ? step_size_+0.5 : 5;
	int word_len=3;
	int n_symbols=64;
	long int cons=1000000;
	double mult=freq*60.0/step_size;
	int min_rr=mult/n_symbols;
	int rr;
	int symbol_offset;
	int win_anchor=win_len/2+word_len/2;
	long int sum;
	char *symbols = my_malloc(events->n_events*sizeof(char));
	unsigned int *words = my_calloc(events->n_events,sizeof(int));
	long int	*pi_map = my_malloc(win_len*sizeof(long int));
	unsigned char  *word_bins=my_calloc(pow(n_symbols,word_len),sizeof(char)); // notice that there is a reason why the n_symbols and word_len are predefined small constants..
	double *entropy = my_malloc(events->n_events*sizeof(double));

 // define symbols
	for(i=1;i<events->n_events;i++){
		rr=events->events[i].sample-events->events[i-1].sample;
		if(rr<=min_rr){
			symbols[i]=n_symbols-1;
		}
		else{
			symbols[i]=mult/(rr);
		}
	}
// define words
	for(j=0;j<word_len;j++){
		symbol_offset=pow(n_symbols,j);
		for(i=1;i<events->n_events-word_len;i++){
			words[i-1]+=symbols[i+j]*symbol_offset;
		}
	}

// define pi_map
	pi_map[0]=0.0;
	for(i=1;i<win_len;i++){
		pi_map[i]=cons*(long int)(i)/(double)(win_len)*log2((double)(i)/win_len)/(log2((double)(win_len)));
	}

// define entropy

	// the first sample
	sum=0;
	for(k=0,i=0;i<=win_len;i++){
		if(word_bins[words[i]]>0){ // did we already add this word?
			continue;
		}
		word_bins[words[i]]++;
		for(j=i+1;j<=win_len;j++){
			if(words[i]==words[j]){
				word_bins[words[i]]++;
			}
		}
		sum+=pi_map[word_bins[words[i]]]; // update sum
		k++; // update count
	}
	entropy[0]=-k*sum/((double)(win_len)*cons);
	for(i=1;i<=win_anchor;i++){
		entropy[i]=entropy[0];
	}

	// update sum and count (k) for the next samples
	for(i=1;i<events->n_events-word_len-win_len;i++){

		// remove one ..
		sum-=pi_map[word_bins[words[i-1]]];
		word_bins[words[i-1]]--;
		if(word_bins[words[i-1]]>0){
			sum+=pi_map[word_bins[words[i-1]]];
		}
		else{
			k--;
		}

		// ..and add one sample
		if(word_bins[words[i+win_len]]>0){
			sum-=pi_map[word_bins[words[i+win_len]]];
		}
		else{
			k++;
		}
		word_bins[words[i+win_len]]++;
		sum+=pi_map[word_bins[words[i+win_len]]];
		entropy[i+win_anchor]=-k*sum/((double)(win_len)*cons);
	}
	for(j=i+win_anchor,i+=win_anchor+1;i<events->n_events;i++){
		entropy[i]=entropy[j];
	}

	my_free(symbols);
	my_free(words);
	my_free(pi_map);
	my_free(word_bins);

	return(entropy);

}

unsigned int *rhythm_define_words(struct test_events *events, double freq, int step_size, int word_len,int n_symbols){
	size_t i,j;
	double mult=freq*60.0/step_size;
	int min_rr=mult/n_symbols;
	int rr;
	int symbol_offset;
	char *symbols = my_malloc(events->n_events*sizeof(char));
	unsigned int *words = my_calloc(events->n_events,sizeof(int));

 // define symbols
	for(i=1;i<events->n_events;i++){
		rr=events->events[i].sample-events->events[i-1].sample;
		if(rr<=min_rr){
			symbols[i]=n_symbols-1;
		}
		else{
			symbols[i]=mult/(rr);
		}
	}
// define words
	for(j=0;j<word_len;j++){
		symbol_offset=pow(n_symbols,j);
		for(i=1;i<events->n_events-word_len;i++){
			words[i-1]+=symbols[i+j]*symbol_offset;
		}
	}
	my_free(symbols);
	return(words);
}

unsigned int *rhythm_define_good_words(double *rr, struct test_events *events, double freq, int step_size, int word_len,int n_symbols){
	size_t i,j,k;
	double mult=60.0/step_size;
	double min_rr=mult/n_symbols;
	int symbol_offset;
	char *symbols = my_malloc(events->n_events*sizeof(char));
	unsigned int *words = my_calloc(events->n_events,sizeof(int));

 // define symbols
	for(i=1;i<events->n_events;i++){
		if(rr[i]<0){ // break?
			symbols[i]=n_symbols+1;
		}
		else if(rr[i]==0){ // ignore?
			symbols[i]=n_symbols;
		}
		else if(rr[i]<=min_rr){
			symbols[i]=n_symbols-1;
		}
		else{
			symbols[i]=mult/(rr[i]);
		}
	}
// define words
	for(i=1;i<events->n_events-word_len;i++){
		if(rr[i]<=0){ // ignore?
			continue;
		}
		for(j=0,k=i;k<events->n_events && j<word_len;k++){
			if(rr[k]==0){ // ignore?
				continue;
			}
			if(rr[k]<0){ // break;
				words[i-1]=0; // 0 word is bad word..
				break;
			}
			symbol_offset=pow(n_symbols,j);
			if(rr[k]<=min_rr){
				words[i-1]+=(n_symbols-1)*symbol_offset;
			}
			else{
				words[i-1]+=(char)(mult/(rr[k]))*symbol_offset;
			}
			j++;
		}
		if(k==events->n_events){
			words[i-1]=0;
			break;
		}
	}
	my_free(symbols);
	return(words);
}


double rhythm_entropy(unsigned int *words, int win_len, double normalization_len){
	int i,j;
	double k,n;
	static char *checked=NULL;
	static int n_alloced=0;
	double entropy=0.0;

	if(win_len<0){
		my_free(checked);
		checked=NULL;
		n_alloced=0;
		return(0.0);
	}
	if(win_len>n_alloced){
		n_alloced=2*win_len;
		checked=my_malloc(n_alloced*sizeof(char));
	}
	memset(checked,0,win_len);
	for(k=0.0,i=0;i<win_len;i++){
		if(checked[i]){
			continue;
		}
		for(n=1.0,j=i+1;j<win_len;j++){
			if(words[i]==words[j]){
				checked[j]=1;
				n++;
			}
		}
		entropy+=n*log2(n/normalization_len); // update sum
		k++; // update count
	}

	entropy*=-k/(log2(normalization_len)*normalization_len*normalization_len);
	return(entropy);

}

double rhythm_good_entropy(unsigned int *words, int win_len, double min_win_len){
	int i,j;
	double k,n;
	static char *checked=NULL;
	static int n_alloced=0;
	double entropy=0.0;
	int normalization_len=0;

	if(win_len<0){
		my_free(checked);
		checked=NULL;
		n_alloced=0;
		return(0.0);
	}
	if(win_len>n_alloced){
		n_alloced=2*win_len;
		checked=my_malloc(n_alloced*sizeof(char));
	}
	memset(checked,0,win_len);

	for(i=0;i<win_len;i++){
		if(words[i]>0){
			normalization_len++;
		}
	}
	if(normalization_len<min_win_len){
		normalization_len=min_win_len;
	}

	for(k=0.0,i=0;i<win_len;i++){
		if(words[i]==0){ // ignore
			continue;
		}
		if(checked[i]){
			continue;
		}
		for(n=1.0,j=i+1;j<win_len;j++){
			if(words[i]==words[j]){
				checked[j]=1;
				n++;
			}
		}
		entropy+=n*log2(n/normalization_len); // update sum
		k++; // update count
	}

	entropy*=-k/(log2(normalization_len)*normalization_len*normalization_len);
	return(entropy);

}

double *rhythm_short_bpm_entropy(struct test_events *events, double freq, double win_duration, double step_size_){
	size_t i;
//	const int step_size=5; // plot original & 2
//	const int step_size=6; // plot 3
//	const int step_size=8; // plot 4
	int step_size=step_size_>0 ? step_size_+0.5 : 10; // plot 5
	int word_len=3;
	int n_symbols=64;
	double min_bpm_for_scaling = 45.0;
	double min_win_len = (size_t)(win_duration * min_bpm_for_scaling / 60.0 +0.5);
	double half_win_dur=win_duration*freq/2.0;
	int half_word_len=word_len/2;
	long int win_len;
	size_t start_i,start_i2,end_i;
	unsigned int *words = rhythm_define_words(events, freq, step_size, word_len, n_symbols);
	double *entropy = my_malloc(events->n_events*sizeof(double));

	for(end_i=0,start_i=0,i=0;i<events->n_events;i++){
		for(;events->events[i].sample-events->events[start_i].sample>half_win_dur;start_i++);
		for(;end_i<events->n_events-1 && events->events[end_i+1].sample-events->events[i].sample<half_win_dur;end_i++);
		if(start_i<half_word_len){
			start_i2=0;
		}else{
			start_i2=start_i-half_word_len;
		}
		if(end_i>=events->n_events-word_len-1){
			win_len=events->n_events-word_len-1-start_i2;
		}else{
			win_len=end_i-start_i2;
		}
		if(win_len>1){
			entropy[i]=rhythm_entropy(words+start_i2, win_len, win_len<min_win_len ? min_win_len : win_len);
		}else{
			if(i>0){
				entropy[i]=entropy[i-1];
			}else{
				entropy[i]=0.0;
			}
		}
	}

	my_free(words);
	return(entropy);
}

int rhythm_range_comparison(const void *event1,const void *event2){
	if(((struct test_range *)(event1))->start < ((struct test_range *)(event2))->start){
		return(-1);
	}
	else if(((struct test_range *)(event1))->start > ((struct test_range *)(event2))->start){
		return(1);
	}
	return(0);
}




static double rhythm_event_diff(size_t event1, size_t event2,struct test_events *events){
	if(event1>event2){
		return((events->events[event1].sample-events->events[event2].sample)-(events->events[event1].sub_sample-events->events[event2].sub_sample));
	}
	else{
		return((events->events[event2].sample-events->events[event1].sample)-(events->events[event2].sub_sample-events->events[event1].sub_sample));
	}
}



double rhythm_one_bpm_entropy(struct test_events *events, double freq){
	const int step_size=5;
	const int word_len=3;
	const int n_symbols=64;
	unsigned int *words = rhythm_define_words(events, freq, step_size, word_len, n_symbols);
	double entropy;
	if(events->n_events>word_len+1){
		entropy=rhythm_entropy(words, events->n_events-word_len-1, events->n_events-word_len-1);
	}else{
		entropy=-1.0;
	}
	my_free(words);
	return(entropy);
}

int rhythm_double_comparison(const void *event1,const void *event2){
	if(*((double *)(event1)) < *((double *)(event2))){
		return(-1);
	}
	else if(*((double *)(event1)) > *((double *)(event2))){
		return(1);
	}
	return(0);
}

static inline void rhythm_update_sorted(double old_val,double new_val, double *sorted,int len){
	int i;
	if(new_val==old_val){
		return;
	}

	if(new_val>old_val){
		for(i=0;i<len-1 && old_val>sorted[i];i++);
		for(;i<len-1 && new_val>sorted[i+1];i++){
			sorted[i]=sorted[i+1];
		}
		sorted[i]=new_val;
	}
	else{
		for(i=len-1;i>0 && old_val<sorted[i];i--);
		for(;i>0 && new_val<sorted[i-1];i--){
			sorted[i]=sorted[i-1];
		}
		sorted[i]=new_val;
	}
}

static inline double rhythm_median_of_the_sorted_positive(double *data, int len){
	int i,median_ind;
	for(i=0;i<len && data[i]<0.0;i++); // skip the negative and zero (bad) values
	if(i==len){
		return(-1.0);
	}
	if(i==len-1){
		return(data[i]);
	}
	if(i==len-2){
		return(0.5*(data[i]+data[i+1])); // not neccessarely the median, but always the average of the two middle samples
	}
	median_ind=i+(len-i)/2;
//	if(median_ind>3){
//		return(0.5*(data[median_ind-2]+data[median_ind-1])); // NOTICE: not actually the median, but something a bit smaller..
//	}
//	else{
		return(0.5*(data[median_ind-1]+data[median_ind])); // not necessarily the median, but we take rather ave of the two middle values than return the value of just one
//	}
}

int rhythm_create_median_filtered(double *data, double *filtered, int len, int median_len, double default_val){
	ssize_t i;
	double *sorted;
	int median_ind=median_len/2;
	if(median_len<3 || len<median_len){
		printf("Unsupported parameter values for median_filtering\n");
		if(len>0){
			memcpy(filtered,data,len*sizeof(double));
		}
		return(-1);
	}
	sorted=my_malloc(median_len*sizeof(double));
	memcpy(sorted,data,median_len*sizeof(double));
	qsort(sorted,median_len,sizeof(double),rhythm_double_comparison);
	filtered[0]=rhythm_median_of_the_sorted_positive(sorted,median_len);
	if(filtered[0]<0.0){ // Were there just bad values?
		filtered[0]=default_val; // we have to use the default val..
	}
	for(i=1;i<median_ind;i++){
		filtered[i]=filtered[0];
	}
	for(;i<len-median_len+median_ind;i++){
		rhythm_update_sorted(data[i-median_ind],data[i+median_len-median_ind], sorted,median_len);
		filtered[i]=rhythm_median_of_the_sorted_positive(sorted,median_len);
		if(filtered[i]<0.0){ // Were there just bad values?
			filtered[i]=(filtered[i-1]*median_len+default_val)/(double)(median_len+1); // we have to use the weighted ave of the default val and the previous val
		}
	}
	for(;i<len;i++){
		filtered[i]=filtered[len-median_len+median_ind-1];
	}
	my_free(sorted);
	return(0);
}

#define IS_NORMAL_RHYTHM_EVENT(type) (type==TEST_NORMAL_BEAT || type==TEST_BEAT_UNDEFINED)
#define ARTIFACT_EVENTS_INCLUDED

int rhythm_normality_for_events(struct test_events *events,double *ratio_to_nn, double *median_nn_, double default_nn,int ignore_artifacts){
	// TODO define 'typical RR range' and check for 30 % deviations from that
	size_t last_event,i,k;
	double *nn=my_malloc(events->n_events*sizeof(double));
	double *median_nn;
	int median_alloced=0;
	double nn_estimate;
	double nn_step;
	double rate_adaptation=3.0;
	int median_len=11;
	if(median_nn_){
		median_nn=median_nn_;
	}
	else{
		median_nn=my_malloc(events->n_events*sizeof(double));
		median_alloced=1;
	}

	// create nn timeseries for all the events

	// look for the first normal
	for(last_event=0,k=0,i=0;i<events->n_events;i++){
		for(;k<events->n_ranges && (events->ranges[k].type!=TEST_UNREADABLE || events->ranges[k].end<events->events[i].sample);k++);
		if((k<events->n_ranges && events->ranges[k].start<events->events[i].sample) || !IS_NORMAL_RHYTHM_EVENT(events->events[i].type)){ // is there bad range after the last event, or is the current event TEST_ARTIFACT, which should be skipped?
			nn[i]=-1; // nn undefined
		}
		else{
			nn[i]=-1; // nn still undefined
			last_event=i;
			i++;
			break;
		}
	}
	nn_estimate = default_nn; // 850.0 ms ~ 70.0 bpm

	// start estimating nn values
	for(;i<events->n_events;i++){
		for(;k<events->n_ranges && (events->ranges[k].type!=TEST_UNREADABLE || events->ranges[k].end<events->events[i].sample);k++);
		if((k<events->n_ranges && events->ranges[k].start<events->events[i].sample)){ // is there bad range after the last event, or is the current event TEST_ARTIFACT, which should be skipped?
			nn[i]=-1; // nn undefined in bad range
		}
		else if(!IS_NORMAL_RHYTHM_EVENT(events->events[i].type)){ // we don't estimate nn if the current beat is known / guessed non-sínus beat
			if(events->events[i].type==TEST_SUPRAVENTRICULAR){
				last_event=i;
			}
			nn[i]=-1;
		}
		else{
			nn[i]=events->events[i].sample-events->events[last_event].sample;
			if(i-last_event>1){ // beats skipped? -> we have to estimate if some of those are related to sinus activity..
				nn_step=(int)(((double)(nn[i])/nn_estimate)+0.5);
				if(nn_step>i-last_event){ // we don't allow adding new events here.. Not in the current implementation at least (TODO check this!)
					nn_step=i-last_event;
				}
				if(nn_step>1){
					nn[i]/=nn_step; // adjust nn..
				}
			}
			nn_estimate=(rate_adaptation*nn_estimate+nn[i])/(rate_adaptation+1.0); // adjust the nn_estimate
			last_event=i;
		}
	}

	rhythm_create_median_filtered(nn, median_nn, events->n_events, median_len, default_nn);

	if(ignore_artifacts){
		for(i=0;i<events->n_events && events->events[i].type==TEST_ARTIFACT;i++){
			ratio_to_nn[i]=-1;
		}
		ratio_to_nn[i]=-1;
		last_event=i++;
		for(;i<events->n_events;i++){
			ratio_to_nn[i]=(double)(events->events[i].sample-events->events[last_event].sample)/median_nn[i];
			if(events->events[i].type!=TEST_ARTIFACT){
				last_event=i;
			}
		}
	}
	else{
		ratio_to_nn[0]=-1;
		for(i=1;i<events->n_events;i++){
			ratio_to_nn[i]=(double)(events->events[i].sample-events->events[i-1].sample)/median_nn[i];
		}
	}
	if(median_alloced){
		my_free(median_nn);
	}
	my_free(nn);
	return(0);
}

#define RHYTHM_SKIP_THESE_FROM_IDENTIFY_MISSING(type) (type!=REGION_OF_INTEREST && type<N_TEST_RANGES)  // we skip all but the set ROI, and the reference ranges (if any)
int rhythm_identify_missing(struct test_events *events,double *ratio_to_nn, double *median_nn, double freq, struct trig_missing_events *missing_events){
	size_t i;
	size_t last_event=0;
	size_t prev_rr,rr;
	int k;
	double max_abs_diff=0.75*freq; // TODO set as rhythm or trig parameters
	double max_rel_diff=1.6;
	double max=3.0*freq;
	prev_rr=median_nn[0];
//	int *n_missing=my_calloc(events->n_events+1,sizeof(int));
	for(k=0;k<events->n_ranges && !RHYTHM_SKIP_THESE_FROM_IDENTIFY_MISSING(events->ranges[k].type);k++);
	for(i=0;i<events->n_events;i++){
		while(k<events->n_ranges && events->ranges[k].start<events->events[i].sample){ // Are we in bad range? Should we skip some beats / rr intervals?
			last_event=events->ranges[k].end;
			for(;i<events->n_events && last_event>=events->events[i].sample;i++);
			prev_rr=median_nn[i];
			for(k++;k<events->n_ranges && !RHYTHM_SKIP_THESE_FROM_IDENTIFY_MISSING(events->ranges[k].type);k++);
		}
		if(i==events->n_events){
			break;
		}
		if(events->events[i].type==TEST_ARTIFACT){
			continue;
		}
		rr=events->events[i].sample-last_event;
		if(((rr>median_nn[i]*max_rel_diff || rr>prev_rr+max_abs_diff) && (rr+prev_rr>2*1.3*(median_nn[i]))) || rr>max){ // Did we find too long rr interval -- probable missing beat?
			if(missing_events){
				missing_events->missing_range=my_realloc(missing_events->missing_range,(missing_events->n_missing+1)*sizeof(struct trig_missing_event));
				missing_events->missing_range[missing_events->n_missing].start=last_event;
				missing_events->missing_range[missing_events->n_missing].end=events->events[i].sample;
				missing_events->missing_range[missing_events->n_missing].expected_rr=median_nn[i];
				missing_events->n_missing++;
			}
		}
		last_event=events->events[i].sample;
		prev_rr=rr;

	}
	return(0);
}




// 1. Detect extra

// 2. Filter extra

// 3. Filter normal, respiratory, range

// 4. qvantify variability

// q= 88 %



double *rhythm_variability(struct test_events *events, double freq, double win_duration, double leave_out){
	double *variability = my_malloc(events->n_events*sizeof(double));
	double *rr = my_malloc(events->n_events*sizeof(double));
	double *tmp_sorted = my_malloc(events->n_events*sizeof(double));
	double half_win=win_duration*freq/2.0;
	size_t start_i,i,end_i,n;
	double range_limits[2]={0.06,0.94}; // keep 88 % -> drop 6 % of the lowest and the highest values

	if(leave_out>=0.0){
		range_limits[0]=leave_out/100.0;
		range_limits[1]=1.0-leave_out/100.0;
	}

	for(i=1;i<events->n_events;i++){
		rr[i]=events->events[i].sample-events->events[i-1].sample;
	}

	for(start_i=0,i=1,end_i=0;i<events->n_events;i++){
		for(;start_i<events->n_events && events->events[start_i].sample<events->events[i].sample-half_win;start_i++);
		for(;end_i<events->n_events && events->events[end_i].sample<events->events[i].sample+half_win;end_i++);
		if(end_i>start_i){
			n=end_i-start_i;
			memcpy(tmp_sorted,rr+start_i,n*sizeof(double));
			qsort(tmp_sorted,n,sizeof(double),rhythm_double_comparison);
			variability[i]=(tmp_sorted[(int)(range_limits[1]*n+0.5)]-tmp_sorted[(int)(range_limits[0]*n+0.5)])/freq;

		}
		else{
			variability[i]=0.0;
		}
	}
	variability[0]=variability[1]; // just to set something to the first sample..

	my_free(rr);
	my_free(tmp_sorted);
	return(variability);
}



/* Gauge:
 * this.draw_q_range = function() {
		var q = this.propability_range;
		var e = this.events;

		var _e = [];
		var total_surface = 0.0;
		for ( var i in e) {
			_e.push(e[i]);
			if (e[i].rr) {
				total_surface += e[i].rr;
			}
		}
		_e.sort(function(a, b) {
			return a.rr - b.rr;
		});

		var i = Math.floor(_e.length / 2);

		var limit_surface = q * total_surface;
		var sum_surface = _e[i].rr;
		var limit1_at, limit2_at;
		for (var j = 1; j < i && sum_surface < limit_surface; ++j) {
			if (i + j < _e.length) {
				sum_surface += _e[i + j].rr;
				limit2_at = i + j;
			}
			if (i - j >= 0) {
				sum_surface += _e[i - j].rr;
				limit1_at = i - j;
			}
		}
		--j;

		var limit1 = _e[limit1_at].rr;
		var limit2 = _e[limit2_at].rr;

		// console.log(i, j, sum_surface, limit_surface, total_surface, limit1,
		// limit2);

		var H = this.H;
		var W = this.W;

		var n = document.createElement('div');
		n.className = 'Histogram region1 filled';
		n.style.left = (W * limit1) + 'px';
		n.style.width = (W * (limit2 - limit1)) + 'px';
		n.title = Math.round(q * 100) + '% propability range\n' + 'center: '
				+ Math.round(1000.0 * _e[i].rr) + ' ms\n' + 'width: '
				+ Math.round(1000.0 * (limit2 - limit1)) + ' ms';
		this.node.appendChild(n);
		n._height = H;

		var _this = this;
		setTimeout(function() {
			_this.update_meter(limit1, limit2, 0);
		}, 100);
	}
 */

int rhythm_get_info(enum test_range_type type,size_t start, size_t end, struct test_events *events,size_t *n,size_t *this_dur){
	size_t i;
	*this_dur=0;
	*n=0;
	if(type==TEST_RHYTHM_UNDEFINED || type<0){ // calculate the normal -- undefined rhythm?
		*this_dur=end-start;
		for(i=0;i<events->n_ranges;i++){
			if(events->ranges[i].type!=TEST_RHYTHM_UNDEFINED  && events->ranges[i].end>start && events->ranges[i].start<end){
				(*n)++;
				*this_dur-=MIN(events->ranges[i].end,end)-MAX(events->ranges[i].start,start);
				start=events->ranges[i].end; // we don't want to calculate the overlapping ranges over and over again..
			}
		}

	}
	else{  // calculate the total duration of the defined rhythm
		for(i=0;i<events->n_ranges;i++){
			if(events->ranges[i].type==type && events->ranges[i].end>start && events->ranges[i].start<end){
				(*n)++;
				*this_dur+=MIN(events->ranges[i].end,end)-MAX(events->ranges[i].start,start);
			}
		}
	}
	return(0);
}

int rhythm_hrv(struct test_events *events, double freq, struct rhythm_hrv_markers *markers){
	double *nn;
	size_t i;

	nn=my_malloc(events->n_events*sizeof(double));
	for(i=1;i<events->n_events;i++){
		if(events->events[i].type==TEST_NORMAL_BEAT && events->events[i-1].type==TEST_NORMAL_BEAT){
			nn[i]=rhythm_event_diff(i,i-1,events)/freq;
		}
		else{
			nn[i]=-1.0;
		}
	}


	// 1. min, max and ave bpm, and SDNN
	double total_dur=0.0;
	double minute_dur=0;
	double bpm;
	size_t minute_n=0,total_n=0;
	markers->min_bpm=FLT_MAX;
	markers->max_bpm=0.0;
	markers->ave_bpm=0.0;
	markers->sdnn=0.0;
	for(i=1;i<events->n_events;i++){
		if(nn[i]>0){
			total_dur+=nn[i];
			minute_dur+=nn[i];
			markers->sdnn+=POW2(nn[i]);
			total_n++;
			minute_n++;
		}
		if(i>60){ // we don't exactly calculate nn/min, but just 60.0/ave_nn value during the last 60 beats. Easier this way.. TODO: calculate one min bpm estimate exactly!
			if(nn[i-60]>0){
				minute_dur-=nn[i-60];
				minute_n--;
			}
		}
		if(minute_n>30){ // update min & max? ..update only if there is enough normal beats in interval
			bpm=60.0/minute_dur*(double)(minute_n);
			if(bpm<markers->min_bpm){
				markers->min_bpm=bpm;
			}
			if(bpm>markers->max_bpm){
				markers->max_bpm=bpm;
			}
		}
	}
	markers->ave_bpm=60.0*(double)(total_n)/total_dur;
	markers->sdnn=sqrt(markers->sdnn/(double)(total_n)-POW2(total_dur/(double)(total_n)));

	// 2. RMSSD, pNN20 and pNN50
	double diff;
	size_t n_diff=0;
	markers->rmssd=0.0;
	markers->pnn20=0.0;
	markers->pnn50=0.0;
	for(i=1;i<events->n_events-1;i++){
		if(nn[i]>0 && nn[i+1]>0){
			diff=nn[i]-nn[i+1];
			n_diff++;
			markers->rmssd+=POW2(diff);
			if(fabs(diff)>0.020){
				markers->pnn20++;
			}
			if(fabs(diff)>0.050){
				markers->pnn50++;
			}
		}
	}
	markers->rmssd=sqrt(markers->rmssd/(double)(n_diff));
	markers->pnn20/=(double)(n_diff);
	markers->pnn50/=(double)(n_diff);
	my_free(nn);
	return(0);
}

double *rhythm_define_rr(struct test_events *events, double freq){
	size_t i,j=0;
	size_t prev_i;
	double *rr=my_calloc(events->n_events,sizeof(double));
//	for(prev_i=0;prev_i<events->n_events && events->events[prev_i].type==TEST_ARTIFACT;prev_i++);
	for(prev_i=0;prev_i<events->n_events && !IS_NORMAL_RHYTHM_EVENT(events->events[prev_i].type);prev_i++);
	for(i=prev_i+1;i<events->n_events;i++){
		if(events->events[i].type!=TEST_ARTIFACT){ // ignore the artifacts -- notice that it is recommended to remove the artifacts from the events first
			for(;j<events->n_ranges && (events->ranges[j].type!=TEST_UNREADABLE || events->ranges[j].end<events->events[prev_i].sample);j++); // find the next UNREADABLE range that ends after the prev beat
			if(j>=events->n_ranges || events->ranges[j].start>events->events[i].sample){ // is it ok to define RR?
				rr[i]=rhythm_event_diff(prev_i,i,events)/freq;
			}
			else{
				rr[i]=-1.0; // break in succeeding RRs
			}
			prev_i=i;
		}
		else{
			rr[i]=0.0; // skip
		}
	}
	return(rr);
}

double *rhythm_define_nn(struct test_events *events, double freq){
	size_t i,j=0;
	ssize_t prev_i;
	double *rr=my_calloc(events->n_events,sizeof(double));
	for(prev_i=0;prev_i<events->n_events && !IS_NORMAL_RHYTHM_EVENT(events->events[prev_i].type);prev_i++);
	for(i=prev_i+1;i<events->n_events;i++){
		if(events->events[i].type==TEST_ARTIFACT){
			rr[i]=0;	// ignore / skip the artifacts
		}
		if(IS_NORMAL_RHYTHM_EVENT(events->events[i].type)){
			if(prev_i>=0){
				for(;j<events->n_ranges && (events->ranges[j].type!=TEST_UNREADABLE || events->ranges[j].end<events->events[prev_i].sample);j++); // find the next UNREADABLE range that ends after the prev beat
				if(j>=events->n_ranges || events->ranges[j].start>events->events[i].sample){ // is it ok to define RR?
					rr[i]=rhythm_event_diff(prev_i,i,events)/freq;
				}
				else{
					rr[i]=-1; // break in succeeding NNs
				}
			}
			prev_i=i;
			rr[i]=-1;
		}
		else{ // can't define nn from the next
			prev_i=-1;
			rr[i]=-1; // break in succeeding NNs
		}
	}
	return(rr);
}

double rhythm_sorted_rr_and_median(double *rr, size_t len){
	int i,median_ind;
	qsort(rr,len,sizeof(double),rhythm_double_comparison);
	for(i=0;i<len && rr[i]<=0.0;i++); // skip the negative and zero (bad) values
	if(i==len){
		return(-1.0);
	}
	if(i==len-1){
		return(rr[i]);
	}
	if(i==len-2){
		return(0.5*(rr[i]+rr[i+1])); // not neccessarely the median, but always the average of the two middle samples
	}
	median_ind=i+(len-i)/2;
	return(0.5*(rr[median_ind-1]+rr[median_ind])); // not necessarily the median, but we take rather ave of the two middle values than return the value of just one
}

/************************/
int rhythm_estimate_sinus_rr(double *rr, size_t n){
	/**conf**/
	double max_relative_drop = 0.30;
	double prev_rr=1.0;
	double rr_ave;
	double rr_change;
	size_t i;
	for(i=0;i<n && rr[i]<=0.0;i++);
	if(i==n){ // not a single ok rr in data..
		for(i=0;i<n && rr[i]<=0.0;i++){
			rr[i]=prev_rr;
		}
		return(-1);
	}
	prev_rr=rr[i];
	for(i=0;i<n && rr[i]<=0.0;i++){
		rr[i]=prev_rr;  // fill the missing values
	}
	for(;i<n-1;i++){
		if(rr[i]<=0.0){
			if(rr[i+1]<=0.0){
				rr[i]=prev_rr; // fill the missing values
			}
			else{
				rr[i]=(prev_rr+rr[i+1])/2.0; // fill the missing values with linear estimate
				prev_rr=rr[i];
			}
		}
		else if(rr[i]<(1.0-1.5*max_relative_drop)*rr[i+1] && (i==0 || rr[i-1]*(1.0-max_relative_drop)>rr[i]) && (i<n-2 || rr[i+1]>(1.0 - 0.8 * max_relative_drop)*rr[i+2])){ // potential early / ectopic beat to be filtered out..
			rr_ave = rr[i]+rr[i+1];
			if(i<n-2){
				rr_change = rr[i+2]-prev_rr;
			}
			else{
				rr_change = 0.0;
			}
			prev_rr = rr[i] = rr_ave - rr_change/4.0;
			rr[i+1] = rr_ave + rr_change/4.0;
			i++;
		}
		else{
			prev_rr = rr[i];
		}
	}
	return(0);
}

// simple IIR filter (3rd order) for lowpass_filtering with 0.2 cutoff frequency
static inline void simple_RR_lowpass(double *in, double *out, size_t n){
	double xv[3+1]={0.0}, yv[3+1]={0.0};
	size_t i;
	for (i=0;i<n;i++){
		xv[0] = xv[1]; xv[1] = xv[2]; xv[2] = xv[3];
        xv[3] = in[i] / 1.014907356e+01;
        yv[0] = yv[1]; yv[1] = yv[2]; yv[2] = yv[3];
        yv[3] =   (xv[0] + xv[3]) + 3 * (xv[1] + xv[2])
                     + (  0.0562972365 * yv[0]) + ( -0.4217870487 * yv[1])
                     + (  0.5772405248 * yv[2]);
        out[i] = yv[3];
      }
  }

double rhythm_frequency_ratio(double *rr, size_t n){
	double *sinus_rr;
	double *filtered_rr;
	size_t i;
	double rr_ave;
	double low_energy=0.0,high_energy=0.0;
	if(n>0){
		sinus_rr=my_malloc(sizeof(double)*n);
		filtered_rr=my_malloc(sizeof(double)*n);
		memcpy(sinus_rr,rr,sizeof(double)*n);
		rhythm_estimate_sinus_rr(sinus_rr,n);
		rr_ave=0.0;
		for(i=1;i<n;i++){
			rr_ave+=sinus_rr[i];
		}
		rr_ave/=(double)(n-1);
		for(i=0;i<n;i++){
			sinus_rr[i]-=rr_ave;
		}
		simple_RR_lowpass(sinus_rr, filtered_rr,n);
//		dsp_low_pass(&filtered_rr, sinus_rr,n,1.0/4.0,1.0);
		for(i=0;i<n;i++){
			low_energy+=filtered_rr[i]*filtered_rr[i];
			high_energy+=(sinus_rr[i]-filtered_rr[i])*(sinus_rr[i]-filtered_rr[i]);
		}
		low_energy/=(double)(n);
		high_energy/=(double)(n);
	}
	if(0){
		FILE *fp=fopen("rr_ratio.txt","a");
		FILE *fp_o=fopen("original_rr.txt","a");
		FILE *fp_s=fopen("sinus_rr.txt","a");
		FILE *fp_l=fopen("lowpass_rr.txt","a");
		FILE *fp_h=fopen("highpass_rr.txt","a");
		fprintf(fp,"%lf;%lf;%lf;%lu\n",rr_ave,low_energy,high_energy,n);
		for(i=0;i<n;i++){
			fprintf(fp_o,"%lf;",rr[i]);
			fprintf(fp_s,"%lf;",sinus_rr[i]);
			fprintf(fp_l,"%lf;",filtered_rr[i]);
			fprintf(fp_h,"%lf;",sinus_rr[i]-filtered_rr[i]);
		}
		fprintf(fp_o,"\n");
		fprintf(fp_s,"\n");
		fprintf(fp_l,"\n");
		fprintf(fp_h,"\n");
		fclose(fp);
		fclose(fp_o);
		fclose(fp_s);
		fclose(fp_l);
		fclose(fp_h);
	}
	if(n>0){
		my_free(sinus_rr);
		my_free(filtered_rr);
	}
	return(low_energy/high_energy);
}
/*********************/
