/*
 * mit.c
 *
 *  Created on: Mar 9, 2016
 *      Author: hvaanane
 */


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <stdint.h>

#include "types.h"
#include "memory_manager.h"
#include "tests.h"

#define MIT_DATANAME_MAX 1000
#define MIT_FORMAT_MAX 20

static int correct_average=0;
static int correct_amplitude_scale=0;
static int verbose=0;


int mit_read_212_data(char *pathname, char *data_filename,int header_len, struct data_channel *data_ch, double *scale, int *dc_lvl, int nch, size_t len){
	char filename[FILENAME_MAX];
	FILE *fp;
	int ch;
	unsigned char *data_buffer;
	size_t nread,i,help;
	int i_data_val;
	strcpy(filename,pathname);
	strcat(filename,data_filename);

	fp=fopen(filename,"rb");
	if(fp==NULL){
		if(verbose){
			printf("Couldn't open MIT data file: %s\n",filename);
		}
		return(-1);
	}

	// read and convert
	data_buffer=my_malloc(len*nch*3/2);
	nread=fread(data_buffer,1,len*nch*3/2,fp);
	if(nread<len*nch*3/2){
		if(verbose){
			printf("Couldn't read all the data %s\n",filename);
		}
		return(-1);
	}
	for(ch=0;ch<nch;ch++){
		data_ch[ch].raw=my_malloc(len*sizeof(double));
		data_ch[ch].samples_per_mv=1000;
		for(i=0;i<len;i++){
			help=ch+nch*i; // TODO: simpler data conversion
			if(help%2)
				i_data_val=((data_buffer[(help/2)*3+1]<<4) & 0xf00) | (data_buffer[(help/2)*3+2] & 0xff);
			else
				i_data_val=(short)((data_buffer[help/2*3+1] << 8) | (data_buffer[help/2*3+0] & 0xff));
			// Add sign from the 12th bit.
			if (i_data_val & 0x800)
				i_data_val |= ~(0xfff);
			else
				i_data_val &= 0xfff;
			data_ch[ch].raw[i]=(i_data_val-dc_lvl[ch])*scale[ch];
		}
	}
	my_free(data_buffer);
	return(0);
}

int mit_read_16_data(char *pathname, char *data_filename,int header_len, struct data_channel *data_ch, double *scale, int *dc_lvl, int nch, size_t len){
	char filename[FILENAME_MAX];
	FILE *fp;
	int ch;
	int16_t *data_buffer;
	size_t nread,i;
	strcpy(filename,pathname);
	strcat(filename,data_filename);

	fp=fopen(filename,"rb");
	if(fp==NULL){
		if(verbose){
			printf("Couldn't open MIT data file: %s\n",filename);
		}
		return(-1);
	}

	// read and convert
	data_buffer=my_malloc(len*nch*sizeof(int16_t));
	fseek(fp,header_len,SEEK_SET);
	nread=fread(data_buffer,sizeof(int16_t),len*nch,fp);
	fclose(fp);
	if(nread<len*nch){
		if(verbose){
			printf("Couldn't read all the data %s\n",filename);
		}
		return(-1);
	}
	for(ch=0;ch<nch;ch++){
		data_ch[ch].raw=my_malloc(len*sizeof(double));
		data_ch[ch].samples_per_mv=1000;
		for(i=0;i<len;i++){
			data_ch[ch].raw[i]=data_buffer[ch+i*nch]*scale[ch];
		}
	}
	my_free(data_buffer);
	return(0);
}

int mit_read_data(char *basename, struct data *data){
	FILE *fp;
	int ch;
	int format;
	int header_len;
	char *p,filename[FILENAME_MAX],pathname[FILENAME_MAX];
	char file_id[100];
	char tmpstring[500];
	int narg;
	int ch_set_start,nch;
	int *dc_lvl;
	int return_val=0;
	double *scale;
	char (*data_filename)[FILENAME_MAX], (*dataname)[MIT_DATANAME_MAX],(*format_string)[MIT_FORMAT_MAX];
	// 1. read the header

	strcpy(pathname,basename);
	p=strrchr(pathname,'/');
	if(p){
		*(p+1)='\0';
	}

	strcpy(filename,basename);
	if(strlen(filename)<4 || strcmp(filename+strlen(filename)-4,".hea")){ // Do we need to add ".hea"?
		strcat(filename,".hea");
	}

	fp=fopen(filename,"r");
	if(fp==NULL){
		printf("Couldn't open MIT header file: %s\n",filename);
		return(-1);
	}
	if(fgets (tmpstring,500,fp)==NULL){
		printf("Something wrong in MIT header file: %s\n",filename);
		return(-1);
	}
	while(tmpstring[0]=='#' || strlen(tmpstring)<2){ // skip the comment lines
		if(fgets (tmpstring,500,fp)==NULL){
			printf("Something wrong in MIT header file: %s\n",filename);
			return(-1);
		}
	}
	narg=sscanf(tmpstring,"%s %lu %lf %lu\n",file_id,&(data->number_of_channels),&(data->sample_freq),&(data->samples_per_channel));

	if(narg<4){
		printf("Only MIT files with nch, freq and nsamp info in header-file are supported atm.\n");
		return(-1);
	}

	// alloc
	format_string=my_malloc(data->number_of_channels*sizeof(*format_string));
	dc_lvl=my_malloc(data->number_of_channels*sizeof(int));
	scale=my_malloc(data->number_of_channels*sizeof(double));
	data_filename=my_malloc(data->number_of_channels*sizeof(*data_filename));
	dataname=my_malloc(data->number_of_channels*sizeof(*dataname));

	for(ch=0;ch<data->number_of_channels;ch++){
		if(fgets (tmpstring,500,fp)==NULL){
			printf("Something wrong in MIT header file: %s\n",filename);
			return(-1);
		}
		while(tmpstring[0]=='#' || strlen(tmpstring)<2){ // skip the comment lines
			if(fgets (tmpstring,500,fp)==NULL){
				printf("Something wrong in MIT header file: %s\n",filename);
				return(-1);
			}
		}
		sscanf(tmpstring,"%s %s %lf %*d %d %*d %*d %*d %s",data_filename[ch], format_string[ch],scale+ch,dc_lvl+ch,dataname[ch]);
		if(scale[ch]==0){
			scale[ch]=5.0; // Default gain is 200 -> default scale is 1000.0/200.0
		}
		else{
			if(!strcmp(dataname[ch],"PCG")){  // phonocardiogram
				scale[ch]=1.0/scale[ch];    // trying to scale to microvolts.. or something in similar range than ECG in microvolts...
//				scale[ch]=0.5/scale[ch];
			}
			else{ // default -- ECG
				scale[ch]=1000.0/scale[ch]; // trying to scale to microvolts
//				scale[ch]=5000.0/scale[ch];
			}
		}

	}
	fclose(fp);

	// 2. Read the data

	data->ch = my_calloc(data->number_of_channels, sizeof(struct data_channel));
	for(ch=0;ch<data->number_of_channels;){
		ch_set_start=ch;
		for(ch++;ch<data->number_of_channels && !strcmp(data_filename[ch_set_start],data_filename[ch]);ch++); // collect the channels from one file to one set -- notice that current implementation doesn't support multidatafile setups, where channels in one file aren't sequental
		nch=ch-ch_set_start;
		narg=sscanf(format_string[ch_set_start],"%d+%d",&format,&header_len); // we assume all the data in one file to have the same format..
		if(narg==1){
			header_len=0;
		}
		switch(format){
		case 212:
			mit_read_212_data(pathname, data_filename[ch_set_start],header_len, data->ch+ch_set_start,scale+ch_set_start,dc_lvl+ch_set_start, nch,data->samples_per_channel);
			break;
		case 16:
			mit_read_16_data(pathname, data_filename[ch_set_start],header_len, data->ch+ch_set_start,scale+ch_set_start,dc_lvl+ch_set_start, nch,data->samples_per_channel);
			break;

		default:
			printf("Oops, MIT format %d isn't currently supported.\n",format);
			return_val=-1;
		}
	}

	if(correct_average){
		int ch;
		size_t i;
		size_t start=0;
		double ave;
		for(ch=0;ch<data->number_of_channels;ch++){
			ave=0.0;
			if(data->samples_per_channel>4000){
				start=1000;
			}
			for(i=start;i<data->samples_per_channel;i++){
				ave+=data->ch[ch].raw[i];
			}
			ave/=(double)(data->samples_per_channel-start);
			for(i=0;i<data->samples_per_channel;i++){
				data->ch[ch].raw[i]-=ave;
			}
		}
	}

	if(correct_amplitude_scale){
		int ch;
		size_t i;
		size_t start,end;
		double min,max,scale;
		double max_range=2000.0;
		if(data->samples_per_channel<4000){
			start=0;
			end=data->samples_per_channel;
		}
		else if(data->samples_per_channel<10000){
			start=1000;
			end=data->samples_per_channel-1000;
		}
		else{
			start=2000;
			end=9000;
		}
		for(ch=0;ch<data->number_of_channels;ch++){
			min=max=data->ch[ch].raw[start];
			for(i=start;i<end;i++){
				if(min>data->ch[ch].raw[i]){
					min=data->ch[ch].raw[i];
				}
				if(max<data->ch[ch].raw[i]){
					max=data->ch[ch].raw[i];
				}
			}
			scale=max_range/(max-min);
			for(i=0;i<data->samples_per_channel;i++){
				data->ch[ch].raw[i]*=scale;
			}
		}

	}

	// 2.3. free
	my_free(scale);
	my_free(dc_lvl);
	my_free(format_string);
	my_free(data_filename);
	my_free(dataname);

	return(return_val);
}

