function classifyResult = challenge(recordName)
%
% Sample entry for the 2017 PhysioNet/CinC Challenge.
%
% INPUTS:
% recordName: string specifying the record name to process
%
% OUTPUTS:
% classifyResult: integer value where
%                     N = normal rhythm
%                     A = AF
%                     O = other rhythm
%                     ~ = noisy recording (poor signal quality)
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% The challenge function requires that you have downloaded the challenge
% data 'training_set' in a subdirectory of the current directory.
%    http://physionet.org/physiobank/database/challenge/2017/
%
% This dataset is used by the generateValidationSet.m script to create
% the annotations on your training set that will be used to verify that
% your entry works properly in the PhysioNet testing environment.
%
%
% Version 1.0
%
%
% Written by: Chengyu Liu and Qiao Li January 20 2017
%             chengyu.liu@emory.edu  qiao.li@emory.edu
%
% Last modified by:
%
%


    % Read the file
	[~, ecg, fs] = rdmat(recordName);

    % Get the features
	f = getChallengeFeatures4(ecg, fs);
	
	% Load the mu and sigma parameters to scale the input according to the training dataset
	mu_sigma=load('DatasetMuSigma.mat');
	
	muOld = mu_sigma.muOld;
	mu    = mu_sigma.mu;
	sgOld = mu_sigma.sgOld;
	sigma = mu_sigma.sigma;
	
	% Scale the Outliers to mu+-sigma
	f = ScaleOutliers(f, muOld, sgOld);
	
	% If exists some NaN ---> Noise
	classifyResult='~';
	if sum(isnan(f)) > 0
		return;
	end
	
	
	% Scale the input with the mu and sigma of the dataset with filtered outliers
	nn_input=bsxfun(@minus,          f, mu);
	nn_input=bsxfun(@rdivide, nn_input, sigma);
	
	 
	% Filter columns (FEATURE SELECTION EXPERIMENT)
    % Experimento final
    % [22/72] bestF1( 1 20 40 32 68 36 8 15 27 49 60 42 70 56 57 52 71 62 9 3 5 37 48 50 24 7 67 47 45 41 39 61 69 46 33 23 16 34 66 65 22 35 18 28 59 44 10 55 43 58): 0.733649 [0.745038, 0.843647, 0.612263, 0.486486] 
    % col2filter   = [ 1 20 40 32 68 36 8 15 27 49 60 42 70 56 57 52 71 62 9 3 5 37 48 50 24 7 67 47 45 41 39 61 69 46 33 23 16 34 66 65 22 35 18 28 59 44 10 55 43 58];
    % >> columnas   =   2     4     6    11    12    13    14    17    19    21    25    26    29    30    31    38    51    53    54    63    64    72 
    nn_input = nn_input([2, 4, 6, 11, 12, 13, 14, 17, 19, 21, 25, 26, 29, 30, 31, 38, 51, 53, 54, 63, 64, 72]);

	
		% Load the Neural Network
	NN=load('SJimenezNets_Septiembre_v128v2.mat');

	label = nnpredict2(NN.nn, nn_input);
	switch label
		case 1
			classifyResult='A';
		case 2
			classifyResult='N';
		case 3
			classifyResult='O';
		case 4
			classifyResult='~';
	end

end



function [input] = ScaleOutliers(input, mu, sg)

    nsg = 3;
    ic1 = mu - nsg .* sg;
    ic2 = mu + nsg .* sg;
    nf  = numel(input);

    for i = 1 : nf

        % Vector de datos
        v = input(i);
		
		% If out of range...
		if v <= ic1(i) || v >= ic2(i)
		
		    % Si es menor que mu, tiene que estar por debajo de la mediana
			signo = v>=mu(i);
			if signo==0
				signo = -1; 
			end
		
			% Los ponemos al máximo o mínimo correspondiente
			input(i) = mu(i) + (signo .* nsg *sg(i));
		end
        
    end    

end