function alarmResult=challenge(recordName,alarm_type)
%
%  alarmResult=challenge(recordName,alarm_type)
%
% Sample entry for the 2015 PhysioNet/CinC Challenge.
%
% Inputs:
%   recordName
%       String specifying the record name to process
%   alarmType
%       String specifying the alarm type. Alarm types are:
%             Asystole, Bradycardia, Tachycardia,
%             Ventricular_Tachycardia, Ventricular_Flutter_Fib
%
%
% Outputs:
%   alarmResult
%       Integer value where 0 = false alarm and 1 is a true
%       alarm. 
%
%
% To run your entry on the entire training set in a format that is
% compatible with PhysioNet's scoring enviroment, run the script
% generateValidationSet.m
%
% Dependencies:
%
%       1) This function requires that you have the WFDB App Toolbox installed. 
%          For information on how to install the toolbox see:
%           http://physionet.org/physiotools/matlab/wfdb-app-matlab/
%
%       2) The CHALLENGE function requires that you have downloaded the challenge
%       data in the current directory. The training dataset can
%       be downloaded from PhysioNet at:
%           http://physionet.org/physiobank/database/challenge/2015/
%          
%         This dataset is used by the generateValidationSet.m script to
%         create the annotations on your traing set that will be used to 
%         verify that your entry works properly in the PhysioNet testing 
%         environment. 
%
% Version 0.1
%
%
% Written by Ikaro Silva, March 12, 2015.
%
% %Example using training data- 
% alarmResult=challenge('100','Asystole')
%

% Name of file containing answers
answers = 'answers.txt';

%Default answer is that the alarm is false.
alarmResult=0;

%Get Fs
info=wfdbdesc(recordName);
Fs=double(info(1).SamplingFrequency);
Nend=5*60*Fs;
Nstart=[];

%Run QRS detectors on first and second signals
wqrs(recordName,Nend,Nstart);

sqrs(recordName,Nend,Nstart,2);
movefile([recordName '.qrs'],[recordName '.sqrs2']);
sqrs(recordName,Nend,Nstart,1);
movefile([recordName '.qrs'],[recordName '.sqrs1']);

gqrs(recordName,Nend,Nstart,2);
movefile([recordName '.qrs'],[recordName '.gqrs2']);
gqrs(recordName,Nend,Nstart,1);
movefile([recordName '.qrs'],[recordName '.gqrs1']);

%Get Tachograms from all QRS detectors
hrw=tach(recordName,'wqrs',Nend,Nstart);
hrs1=tach(recordName,'sqrs1',Nend,Nstart);
hrs2=tach(recordName,'sqrs2',Nend,Nstart);
hrg1=tach(recordName,'gqrs1',Nend,Nstart);
hrg2=tach(recordName,'gqrs2',Nend,Nstart);

%Calculate cross correlation between the Tachograms
FsTach=2;
cmpInd=(30*FsTach);
r=corrcoef([hrw(end-cmpInd:end) hrs1(end-cmpInd:end) hrs2(end-cmpInd:end) ...
    hrg1(end-cmpInd:end) hrg2(end-cmpInd:end)]);
alarmResult=(sum(r(:))-5)/20;

%Make a decision (threshold set based on training data)
if(alarmResult>0.55)
    alarmResult=1;
else
    alarmResult=0;
end

% Write result to answers.txt
fid = fopen(answers, 'a');
if (fid == -1)
    error('Could not open answer file');
end

% Get base name of record (without directories)
i = strfind(recordName, filesep);
if (~isempty(i))
    basename = recordName(i(end)+1 : end);
else
    basename = recordName;
end

fprintf(fid, '%s,%d\n', basename, alarmResult);
fclose(fid);

end