function challenge(recordName)
%
% challenge(recordName)
%
% Octave-compatible code for the 2014 PhysioNet/CinC Challenge
%
% This function takes only one input argument:
%
% recordName
%       String specifying the name of the record to process.  Do not include
%       the '.dat' or '.hea' suffix in recordName.
%
% This function has no output arguments, but it writes an annotation file
% named "recordName.qrs" in the current directory, containing a annotation
% for each detected beat in the record.
%
% Dependencies:
%
%       1) This function requires the WFDB Toolbox for MATLAB and Octave.
%          For information on how to install the toolbox, please see:
%              http://www.physionet.org/physiotools/matlab/wfdb-app-matlab/
%
%       2) The Toolbox is supported only on 64-bit MATLAB 2013a and 2013b
%          and on GNU Octave 3.6.4 and later, on Linux, Mac OS X, and Windows.

[~,config] = wfdbloadlib;                               %Initializing Toolbox config

annName = 'qrs';                                        %Annotator name required by the Challenge rules.

siginfo = wfdbdesc(recordName);                        %Retrieving record informations
description = squeeze(struct2cell(siginfo));
description = description(5,:);
state=1;
res=0;

ecg_ind = get_index(description,'ECG');               %Retrieving ecg position in the record
bp_ind = get_index(description,'BP') ;                  %Retrieving abp position in the record
pl_ind = get_index(description,'PLETH');              %Retrieving pleth position in the record
fs = str2num(siginfo(ecg_ind).SamplingFrequency(1:end-3));
sp=1/fs;
if(fs~=125), res=1;                               
end                                                      
Onset=[];
if(~isempty(ecg_ind))
    gqrs(recordName,[],[],ecg_ind(1),[],annName);       %QRS detection on ECG
    [QRS,~,~,~,~]=rdann(recordName,annName);            %Loading QRS annotations into an array
    [~,SIGN]=rdsamp(recordName,ecg_ind);
    window=fs*300;

    if (~isempty(bp_ind))
        wabp(recordName,[],[],res,bp_ind);                        %ABP onset detection
        [Onset,~,~,~,~]=rdann(recordName,'wabp');       %Loading onset annotations into an array
        [~,SIGN2]=rdsamp(recordName,ecg_ind);
        k=min(numel(QRS),numel(Onset));                 
        for i=1:1:k                                             
            while(Onset(i)-QRS(i) >0.35*fs)       %Realigning ECG and ABP detection arrays.    
                temp_on=Onset(i:end);                     
                Onset(i)=NaN;
                Onset(i+1:end+1)=temp_on;
            end
            while(Onset(i)-QRS(i)<0)
                temp_qr=QRS(i:end);
                QRS(i)=NaN;
                QRS(i+1:numel(QRS)+1)=temp_qr;
            end
        end

        if(numel(Onset)> numel(QRS))                                             
            QRS=padarray(QRS,numel(Onset)-numel(QRS),NaN,'post');                
        elseif (numel(Onset)< numel(QRS))
            Onset=padarray(Onset,numel(QRS)-numel(Onset),NaN,'post');
        end
        delay=round(nanmean(Onset-QRS));                                                
        offset   = 1600;
        scale    = 20;
        Araw     = SIGN2*scale-offset;
        A = filter([1 0 0 0 0 -2 0 0 0 0 1],[1 -2 1],Araw)/24+30;
        A = (A(round(fs*0.2):end)+offset)/scale; 
        dypos= diff(A);
        dypos(dypos<0) = 0;
        ssf_abp = [0; 0; conv(ones(round(0.128*fs),1),dypos)];
        ssf_abp=ssf_abp+1;
        
        scale = 100;
        Araw= SIGN*scale;
        A = filter([1 0 0 0 0 -2 0 0 0 0 1],[1 -2 1],Araw)/24+30;
        A = A(round(0.2*fs):end)/scale; 
        dypos= diff(A);
        dypos(dypos<0) = 0;
        ssf_ecg = [0; 0; conv(ones(round(0.128*fs),1),dypos)];
        ssf_ecg=(ssf_ecg*100)+1;    
        ssf=ssf_ecg(1:end-delay+1).*ssf_abp(delay:end);
        ssf=resample(ssf,100,fs);
        QRS = wabp2(ssf);
        QRS=round(QRS*fs/100);
        QRS=QRS+round(0.16*fs);
        
        
        
    elseif(~isempty(pl_ind))
        wabp(recordName,[],[],res,pl_ind);                     %PLETH onset detection
        [Onset,~,~,~,~]=rdann(recordName,'wabp');     %Loading onset annotations into an array
        [~,SIGN2]=rdsamp(recordName,ecg_ind);
        k=min(numel(QRS),numel(Onset));                 
        for i=1:1:k                                             
            while(Onset(i)-QRS(i) >0.35*fs)             %Realigning PLETH and ECG detection arrays.    
                temp_on=Onset(i:end);                     
                Onset(i)=NaN;
                Onset(i+1:end+1)=temp_on;
            end
            while(Onset(i)-QRS(i)<0)
                temp_qr=QRS(i:end);
                QRS(i)=NaN;
                QRS(i+1:numel(QRS)+1)=temp_qr;
            end
        end
        if(numel(Onset)> numel(QRS))                                             
            QRS=padarray(QRS,numel(Onset)-numel(QRS),NaN,'post');                
        elseif (numel(Onset)< numel(QRS))
            Onset=padarray(Onset,numel(QRS)-numel(Onset),NaN,'post');
        end
        delay=round(nanmean(Onset-QRS));                         
        offset   = mean(SIG2);
        scale    = 20;
        Araw     = SIGN2*scale-offset;
        A = filter([1 0 0 0 0 -2 0 0 0 0 1],[1 -2 1],Araw)/24+30;
        A = (A(round(0.2*fs):end)+offset)/scale; 
        dypos= diff(A);
        dypos(dypos<0) = 0;
        ssf_pleth= [0; 0; conv(ones(round(0.128*fs),1),dypos)];
        ssf_pleth=ssf_pleth+1;
        scale = 100;
        Araw= SIGN*scale;
        A = filter([1 0 0 0 0 -2 0 0 0 0 1],[1 -2 1],Araw)/24+30;
        A = A(round(0.2*fs):end)/scale; 
        dypos= diff(A);
        dypos(dypos<0) = 0;
        ssf_ecg = [0; 0; conv(ones(round(0.128*fs),1),dypos)];
        ssf_ecg=(ssf_ecg*100)+1;
        
        ssf=ssf_ecg(1:end-delay+1).*ssf_pleth(delay:end);
        ssf=resample(ssf,100,fs);
        QRS = wabp2(ssf);
        QRS=round(QRS*fs/100);
        QRS=QRS+round(0.16*fs);
    end
end    
Detection=QRS;                                           %QRS now contains all the beats detected on the available signals.
k=1;
while(state==1)                                            %Beat selection loop
RR=diff(Detection).*(sp);
n=floor((numel(SIGN))/window);                        
 if(n>0)
    clear a;
    Dn=nan(numel(RR),1);
    Dind=nan(numel(RR),1);
    stop=(n-1)*window;
    if (stop==0), stop=1;
   end
    med=nan(n,1);  
    for i=1:window:stop                                  
        a=find(QRS>=i,1,'first');                                             
        b=find(QRS<i+window,1,'last');
        tempRR=RR(a:b);
        m=median(tempRR);
        D=abs(tempRR - m)./(1.483*median(abs(tempRR - m)));
        Dn(a:b)=D;
        Dind(a:b)=i;
        med(i)=m;
        clear D;
        clear tempRR;
        clear a;
     end                                        
    tempRR=RR(b+1:end);
    m=median(tempRR);
    D=abs(tempRR - m)./(1.483*median(abs(tempRR - m)));
    Dn(b+1:end)=D;
    Dind(b+1:end)=stop;
    med(stop)=m;
    clear D;
    clear tempRR;
    clear a;
else
    m=median(RR);
    Dn=abs(RR - m)./(1.483*median(abs(RR - m)));
    Dind(1:numel(Dn))=1;
    med=m;
    clear D;
    clear tempRR;
    clear a;
end

    a=find(Dn>4);
    siz(k)=numel(a);
    if(k>1)
       if(siz(k)>=siz(k-1))
       break;
       end
    end
    if(~isempty(a))
        for i=1:numel(a)
            beat=find(Detection==QRS(a(i)));
            if(~isempty(beat))
                if(RR(a(i))<med(Dind(a(i))))         
                   if(i>1)
                     d=find(Detection==QRS(a(i-1)));
                     if(beat-d>1)                
                              Detection(beat+1)=[];         
                     end
                   else
                       Detection(beat+1)=[]; 
                   end
                else                                
                   if(i>1)
                   d=find(Detection==QRS(a(i-1)));
                     if(beat-d>1)
                           n=round((Detection(beat+1)-Detection(beat))/((med(Dind(a(i))))*fs))-1;
                           for j=1:n   
                                Detect_temp=Detection(beat+1:end);                       
                                Detection(beat+1)=Detection(beat)+med(Dind(a(i)));   
                                Detection(beat+2:end+1)=Detect_temp;
                           end
                     end
                   else
                      n=round((Detection(beat+1)-Detection(beat))/((med(Dind(a(i))))*fs))-1;
                      for j=1:n                      
                          Detect_temp=Detection(beat+1:end);                       
                          Detection(beat+1)=Detection(beat)+med(Dind(a(i)));   
                          Detection(beat+2:end+1)=Detect_temp;
                      end
                     end
                end
                clear beat;
            end
        end
        Detection=sort(Detection);
        QRS=Detection;
        clear Dn;
        clear Dind;
        clear med;
        clear RR;
        k=k+1;
    else
        state=0;
    end
end
if((numel(SIGN)-Detection(end))>mean(diff(Detection)))
   Detection(end+1)=Detection(end)+mean(diff(Detection));
end
wrann(recordName,annName,Detection);
clear all;

%%%%%%%%%%%% Helper Function get_index %%%%%%%%%%%%%%%%%%%%%
% Inputs:
%   description  strings containing signal descriptions from wfdbdesc
%   pattern      a string to be found in the descriptions
% Output:
%   ind          indices of the descriptions that contain the pattern

function ind = get_index(description,pattern)
M = length(description);
tmp_ind = strfind(description,pattern);
clear ind;
ind = [];
for m = 1:M
    if(~isempty(tmp_ind{m}))
        ind(end+1) = m;
    end
end

