#include "mex.h"
//#include "string.h"
//#include "matrix.h"
#include <math.h>

/*Compute correlation coefficients */
void scalePDF(double *x, size_t x_length, double factor)
{
    int i;
    double sum;
    double min;
	
    min = x[0];
    for(i = 0; i < x_length; i++)
    {
        if(x[i] < min)
        {
            min = x[i];
        }
    }
	
    sum = 0.0;
    for(i = 0; i < x_length; i++)
    {
        x[i] -= min;
        sum += x[i];
    }
    
    for(i = 0; i < x_length; i++)
    {
        x[i] = x[i]*factor/sum + (1.0-factor)/(double)x_length;
    }
}

/*Compute correlation coefficients */
void calcADMF(double *d, const double *signal, size_t signal_length, int min_lag, int win)
{
    int i, j;
    double a_sample;
    double b_sample;
    double diff;
    int win_len;
    int mid;
    int b_start;
    
    mid = signal_length/2;
    
    /*Iterate through all possible lags */
    for(i = min_lag; i <= signal_length/2; ++i)
    {
        win_len = i < win ? win : i;
        b_start = mid - i;
      
        /*Calc. cov between a and b as well as */
        /*standard deviation for them */
        diff = 0.0;
        for(j = 0; j < win_len; ++j)
        {
            a_sample = signal[mid+j];
            b_sample = signal[b_start+j];
            diff += fabs(a_sample-b_sample);
        }
        
        /*Calculate the correlation coefficient for this window position */
        d[i-min_lag] = 1.0/(diff / ((double)win_len) + 2e-16);
    }
}


/*Compute correlation coefficients */
void calcMaxSpectrum(double *spectrum, const double *signal, size_t signal_length, int min_lag)
{
    int i, j;
    double max;
    double value;
    int mid = signal_length/2;        
    
    /*Iterate through all possible lags */
    for(i = min_lag; i <= signal_length/2; ++i)
    {
        /*mean max row value */
        max = (signal[mid-i] + signal[mid])/2.0;
        for(j = 1; j < i; ++j)
        {
            value = (signal[mid-i+j] + signal[mid+j])/2.0;
            if( value > max )
            {
                max = value;
            }
        }

        /*Calculate the correlation coefficient for this window position */
        spectrum[i-min_lag] = max;
    }
}

/*Compute correlation coefficients */
void calcCorrcoeffs(double *corrcoeffs, const double *signal, size_t signal_length, int min_lag, int win)
{
    int i, j;
    double cov;
    double a_sample;
    double b_sample;
    int win_len;
    int mid;
    int b_start;
    
    mid = signal_length/2;
    
    /*Iterate through all possible lags */
    for(i = min_lag; i <= signal_length/2; ++i)
    {
        win_len = i < win ? win : i;
        b_start = mid - i;
     
        cov = 0.0;
        for(j = 0; j < win_len; ++j)
        {
            a_sample = signal[mid+j];
            b_sample = signal[b_start+j];
            cov += a_sample*b_sample;
        }
        
        /*Calculate the correlation coefficient for this window position */
        corrcoeffs[i-min_lag] = cov / (double)(win_len);
    }
}

int max_lag_finder(const double* s, size_t s_length, int min_lag)
{
	int i;
    int peak1;
    int peak2;
    double ampl1;
    double ampl2;

    peak1 = peak2 = -1;
    ampl1 = ampl2 = 1e-16;
    
    
	for(i = 1; i < (s_length-1); i++)
	{
        /* is peak? */
		if( (s[i-1] < s[i]) && (s[i+1] < s[i]) )
		{
            /* is new largest peak? */
            if( s[i] > ampl1 ) 
            {
                ampl2 = ampl1;
                peak2 = peak1;
                ampl1 = s[i];
                peak1 = i;
            }
            /* is new 2nd largest peak? */
            else if( s[i] > ampl2 )
            {
                ampl2 = s[i];
                peak2 = i;
            }
		}
	}
    
	if( peak1 == -1 )
	{
		return -1;
	} 
	else if( (ampl1-ampl2)/ampl1 < 0.99 && abs((peak1+min_lag)/2 - (peak2+min_lag)) < 5)
    {
        return peak2;
    }
    else
    {
        return peak1;
    }
}

int single_interval_estimator(double *scaledSignals, size_t signal_length, size_t num_channels, int min_lag, int win, double *quality) 
{
    //double *scaledSignals;
    double *combined;
    double *estimator;
    double mean;
	double sum;
    int i, k;
    int lag;
    size_t estimator_length;
    
    estimator_length = (signal_length/2-min_lag+1);
    
    estimator = (double *)mxMalloc(estimator_length*sizeof(double));
    combined = mxMalloc(estimator_length*sizeof(double));
    //scaledSignals = (double *)mxMalloc(signal_length*num_channels*sizeof(double));
    
    for(i = 0; i < estimator_length; i++)
    {
        combined[i] = 1.0;
    }
    
	//for(i=0;i<signal_length;i++){
	//	scaledSignals[i] = signals[i];
	//}
    //memcpy(scaledSignals, signals, signal_length*num_channels*sizeof(double));
    
    for(k = 0; k < num_channels; k++)
    {
        /* remove mean from channels */
        mean = 0.0;
        for(i = 0; i < signal_length; i++)
        {
            mean += scaledSignals[k*signal_length+i];
        }
        mean /= (double)signal_length;
        for(i = 0; i < signal_length; i++)
        {
            scaledSignals[k*signal_length+i] -= mean;
        }
        
        /* compute estimators */
        /* xcorr */
        calcCorrcoeffs(estimator, &scaledSignals[k*signal_length], signal_length, min_lag, win);
        scalePDF(estimator, estimator_length, 0.9);
        for(i = 0; i < estimator_length; i++)
        {
            combined[i] *= estimator[i];
        }
                
        /* MAP */
        calcMaxSpectrum(estimator, &scaledSignals[k*signal_length], signal_length, min_lag);
        scalePDF(estimator, estimator_length, 0.9);
        for(i = 0; i < estimator_length; i++)
        {
            combined[i] *= estimator[i];
        }
        
        /* AMDF */
        calcADMF(estimator, &scaledSignals[k*signal_length], signal_length, min_lag, win);
        scalePDF(estimator, estimator_length, 0.9);
        for(i = 0; i < estimator_length; i++)
        {
            combined[i] *= estimator[i];
        }
        
    }
       
    lag = max_lag_finder(combined, estimator_length, min_lag);
    
	if(lag != -1)
	{
		sum = 0.0;
		for(i = 0; i < estimator_length; i++)
		{
			sum += combined[i];
		}
    	*quality = combined[lag]/sum;
		
		lag = lag+min_lag;
	}
	else
	{
		*quality = 0.0;
	}

    mxFree(estimator);
    //mxFree(scaledSignals);
    mxFree(combined);
    
    return lag;
}


void mexFunction(int nlhs, mxArray *plhs[], int nrhs, const mxArray *prhs[])
{
    /*Signal variables */
    double *signals;
	double *signals_window;
    size_t signal_length;
    size_t num_channels;
    int min_lag;
	int max_lag;
	int num_iter;
	int hop_size;
    int win;
    int lag;
    int i,k;
	double* lag_out;
	double* quality_out;


    /*Check if we got the correct # of input arguments */
    if(nrhs < 4)
    {
        mexErrMsgTxt("4 input arguments required: single_interval_estimator_c(signals, min_lag, max_lag, hop_size)");
    }
    
    /*Signal dimensions */
    signal_length = mxGetM(prhs[0]);
    num_channels = mxGetN(prhs[0]);
    
    /*Minimal lag */
    min_lag = (int)mxGetScalar(prhs[1]);
	
	/*Minimal lag */
    max_lag = (int)mxGetScalar(prhs[2]);

	/*Hop size */
    hop_size = (int)mxGetScalar(prhs[3]);
    
    /*Make sure minlag is valid */
    if( min_lag > signal_length/2 )
    {
        mexErrMsgTxt("min_lag argument must be smaller than size(signals,1)/2");
    }
	
	/*Make sure minlag is valid */
    if( max_lag <= min_lag )
    {
        mexErrMsgTxt("max_lag argument must be greater than min_lag");
    }
    
    /*Adapt window size parameter */
    if( nrhs > 4 )
    {
        if( mxIsInf(mxGetScalar(prhs[4])) ) 
        {
            win = signal_length/2;
        }
        else
        {
            win = mxGetScalar(prhs[4]);
            win = win > signal_length/2 ? signal_length/2 : win;
        }
    }
    else
    {
        win = 0;
    }
    
    /*Get vectors */
    signals = mxGetPr(prhs[0]);
	
	/*Compute number of iterations */
	num_iter = (int)((double)(signal_length-2*max_lag)/(double)hop_size + 0.5);
		
	/*Setup output matrices */
    plhs[0] = mxCreateDoubleMatrix(num_iter, 1, mxREAL);
	lag_out = mxGetPr(plhs[0]);
	plhs[1] = mxCreateDoubleMatrix(num_iter, 1, mxREAL);
	quality_out = mxGetPr(plhs[1]);
	
    /*Perform the magic... */
	signals_window = (double *)mxMalloc(num_channels*2*max_lag*sizeof(double));
	for(i = 0; i < num_iter; i++)
	{
		for(k = 0; k < num_channels; k++)
		{
			memcpy(&signals_window[k*2*max_lag], &signals[k*signal_length+i*hop_size], 2*max_lag*sizeof(double));
		}
		
		*lag_out = single_interval_estimator(signals_window, 2*max_lag, num_channels, min_lag, win, quality_out);
		if(*lag_out == -1 && i > 0)
		{
			*lag_out = *(lag_out-1);
		}
		lag_out++;
		quality_out++;
	}
	mxFree(signals_window);
}

