function challenge(record_name)

%
% Challenge Entry for the CinC 2014 Challenge
%
% Author: Alexandros Pantelopoulos
%
% 6/4/2014
%
% West Health Institute
%
% Dependencies:
%  - "get_index.m"
%  - "my_filters.mat"
%  - "remove_ECG_Baseline.m"
%  - "BaseLine1.m"
%  - "acorr_norm_sq.m"
%

% Using the WFDB Toolbox's wfdbloadlib function, initialize the Toolbox
% configuration to the default values.  (This is required for Octave.)
[~,config] = wfdbloadlib;
ann_name   = 'qrs';
load('my_filters.mat')

%% Get the data

% Get signal information
siginfo        = wfdbdesc(record_name);
description    = squeeze(struct2cell(siginfo));
description    = description(5,:);
% Get the signal
[~,signal,Fs]  = rdsamp( record_name );
% Extract ECG, BP, PPG, PLETH
ecg_ind        = get_index(description,'ECG');
bp_ind         = get_index(description,'BP');
bp_ind2        = get_index(description,'PPH');
bp_ind3        = get_index(description,'PLETH');
bp_ind         = [bp_ind bp_ind2 bp_ind3];
my_ind         = [ecg_ind bp_ind];
x              = signal(:,my_ind);
num_var        = size(x,2);

clear siginfo description my_ind

%% Change the sampling rate if needed
if (Fs ~= 250)
    tm2 = tm(1):(1/250):(tm(end)+1/Fs)-(1/250);
    x   = interp1(tm,x,tm2);
    Fs  = 250;
end


%% Preprocessing

% Parameters
win_len   = 330;
min_idx   = 80;

% Baseline removal using median filter or HPF
x2            = zeros(size(x));
x2(:,ecg_ind) = remove_ECG_Baseline(x(:,ecg_ind), Fs);
x2(:,ecg_ind) = x2(:,ecg_ind) - mean( x2(:,ecg_ind) );
x2(:,bp_ind)  = filter(b_hpf,a_hpf,x(:,bp_ind));
x2            = filter(b_hpf,a_hpf,x2);
x3            = filter(b_lpf,a_lpf,x2);

% Feature signal for autocorrelation
xdec          = abs( x3 );
% Do a few autocorrelations in order to find a reliable first peak
win_num       = 5;
acr           = acorr_norm_sq(xdec(1:win_num*win_len,:), win_len, min_idx);

%% Get the first peak

R_TO_PULSE = 55;

% sqi constants
c1  = 114;
c2  = 16;
c3  = 0.1;
sqi = zeros(win_num,num_var);
rr  = zeros(win_num,num_var);
for i = 1:win_num
    tmp         = acr(1+(i-1)*win_len:i*win_len,:);
    [maxV,maxI] = max(tmp);
    for j = 1:num_var
        if ((maxI(j)-min_idx) >= min_idx) 
            [maxV2,maxI2] = max(acr(1:maxI(j)-min_idx,j));
            if ((maxV2 >= 0.8 * maxV(j)) && (maxV2>0.5))
                maxV(j) = maxV2;
                maxI(j) = maxI2;
            end
        end
    end
    minV        = min(tmp(min_idx:end,:));
    tmp         = (maxV-minV) .* maxV;
    sqi(i,:)    = floor( c1 - c2 ./ (c3 + tmp) );
    rr(i,:)     = maxI;
end

% Find the signal with the best SQI
[ecg_sqi,ecg_num] = max( sqi(1,ecg_ind) );
[bp_sqi,bp_num]   = max( sqi(1,bp_ind) );
bp_num = bp_num + length(ecg_ind);

if ((ecg_sqi - bp_sqi > -10) && (ecg_sqi > 50))
    
    % Find the first R peak
    % Prepend zeros and remove them afterwards
    xdec = [zeros(7,2) ; xdec];
    p2p_pattern = [peak_loc zeros(1,rr(ecg_num,1)-length(peak_loc)) peak_loc];
    p2p_len = length( p2p_pattern );
    cpatt = zeros(rr(ecg_num,1),1);
    for i = 1:rr(ecg_num,1)
        cpatt(i) = p2p_pattern * xdec(i:i+p2p_len-1,ecg_num);
    end
    xdec = xdec(8:end,:);
    [~,idx] = max(cpatt);
    first_peak = idx;
    % Now find the exact position of the R peak based on the original data
    win = 30;
    if (first_peak <= win)
        low_lim = 1;
    else
        low_lim = first_peak-win;
    end
    high_lim = first_peak+win;
    [~,idx] = max(xdec(low_lim:high_lim,ecg_num));
    first_peak = low_lim + idx - 1;
    % Look for the first peak in the pulse signal
    low_lim  = first_peak + 20;
    high_lim = first_peak + 100;
    [~,idx] = max(xdec(low_lim:high_lim,bp_num));
    first_pulse = low_lim + idx - 1;

else
    
    % Find the first pulse
    % Prepend zeros and remove them afterwards
    xdec = [zeros(7,2) ; xdec];
    p2p_pattern = [peak_loc zeros(1,rr(bp_num,1)-length(peak_loc)) peak_loc];
    p2p_len = length( p2p_pattern );
    cpatt = zeros(rr(bp_num,1),1);
    for i = 1:rr(bp_num,1)
        cpatt(i) = p2p_pattern * xdec(i:i+p2p_len-1,bp_num);
    end
    xdec = xdec(8:end,:);
    [~,idx] = max(cpatt);
    first_pulse = idx;
    % Now find the exact position of the pulse based on the original data
    win = 30;
    if (first_pulse <= win)
        low_lim = 1;
    else
        low_lim = first_pulse-win;
    end
    high_lim = first_pulse+win;
    [~,idx] = max(xdec(low_lim:high_lim,bp_num));
    first_pulse = low_lim + idx - 1;
    % Look for the first R peak in the ECG signal
    if (ecg_sqi > 50)
        if (first_pulse <= 20)
            first_peak = 1;
        else
            if (first_pulse <= 100)
                low_lim = 1;
            else
                low_lim = first_pulse - 100;
            end
            high_lim = first_pulse - 20;
            [~,idx] = max(xdec(low_lim:high_lim,ecg_num));
            first_peak = low_lim + idx - 1;
        end
    else
        if (first_pulse <= R_TO_PULSE)
            first_peak = 1;
        else
            first_peak = first_pulse - R_TO_PULSE;
        end
    end
end

%% Autocorrelation with adaptive window and template size

% Mark an offset in case the first peak is too close to the origin
peak_offset = 0;
use_offset  = 0;
OFFSET      = 50;
if (first_peak <= OFFSET)
    i           = 1;
    peak_offset = OFFSET - first_peak;
    use_offset  = 1;
else
    i           = first_peak - OFFSET;
end
% Flags
use_prev    = 0;
% Parameters
x_len       = size(xdec,1);
ac_len      = win_len;
idx_jump    = 0;
tmpl_len    = 100;
new_rr      = rr(1,1);
% Append zeros for the autocorrelation algorithm
xdec        = [xdec ; zeros(5*win_len,num_var)];
% To hold the peaks and RR intervals
peak_idx    = first_peak;
pulse_idx   = first_pulse;
r_to_pulse  = first_pulse - first_peak;
rr_final    = [];

while (i + min_idx + r_to_pulse(end) < x_len)
    
    % Calculate the new autocorrelation window
    autocorr = zeros(ac_len,num_var);
    
    % New signal window
    my_signal            = zeros(tmpl_len+ac_len,num_var);
    my_signal(:,ecg_ind) = xdec(i:min(i+tmpl_len+ac_len-1,size(xdec,1)),ecg_ind);
    my_signal(:,bp_ind)  = xdec(i+r_to_pulse(end):min(i+r_to_pulse(end)+tmpl_len+ac_len-1,size(xdec,1)),bp_ind);
    
    % Comparison template
    if (use_prev == 1)
        use_prev = 0;
    else 
        tmpl        = my_signal(1:tmpl_len,:);
        tmpl_energy = sum( tmpl.^2 );
    end
    
    % Iterate over all lags
    for j = 1:ac_len
        if (j >= min_idx)
            tmpl2         = my_signal(j:tmpl_len+j-1,:);
            tmpl2_energy  = sum( tmpl2.^2 );
            autocorr(j,:) =	sum( tmpl .* tmpl2 ).^2;
            norm_coeff    = tmpl_energy .* tmpl2_energy;
            autocorr(j,:) = autocorr(j,:) ./ norm_coeff;
        end
    end

    % Find the peaks
    [maxV,maxI] = max(autocorr);
    if (~isempty(rr_final))
        for j = 1:num_var
            low_lim  = rr_final(end) - 30;
            high_lim = rr_final(end) + 30;
            if (high_lim > ac_len)
                high_lim = ac_len;
            end
            [maxV2,maxI2] = max(autocorr(low_lim:high_lim,1));
            if ((maxV2 >= 0.8 * maxV(j)) && (maxV2>0.5))
                maxV(j) = maxV2;
                maxI(j) = low_lim + maxI2 - 1;
            end
        end
    else
        for j = 1:num_var
            if ((maxI(j)-min_idx) >= min_idx) 
                [maxV2,maxI2] = max(autocorr(1:maxI(j)-min_idx,j));
                if ((maxV2 >= 0.8 * maxV(j)) && (maxV2>0.5))
                    maxV(j) = maxV2;
                    maxI(j) = maxI2;
                end
            end
        end
    end
    if (size(autocorr,1) > min_idx)
        minV    = min(autocorr(min_idx:end,:));
    else
        minV    = [0 0];
    end
    tmp         = (maxV-minV) .* maxV;
    sqi_new     = floor( c1 - c2 ./ (c3 + tmp) );
    rr_new      = maxI;

    % Find the signal with the best SQI
    [ecg_sqi,ecg_num] = max( sqi_new(ecg_ind) );
    [bp_sqi,bp_num]   = max( sqi_new(bp_ind) );
    bp_num = bp_num + length(ecg_ind);

    % Find the "best" channel
    channel_sel = 0;
    if (~isempty(rr_final))
        csqi1 = 0;
        csqi2 = 0;
        sel1 = 0;
        sel2 = 0;
        for j = 1:num_var
            if ( (rr_new(j)>=0.7*rr_final(end)) && (rr_new(j)<=1.3*rr_final(end)) && (sqi_new(j)>csqi1) )
                csqi1 = sqi_new(j);
                sel1  = j;
            end
            if (sqi_new(j)>csqi2)
                csqi2 = sqi_new(j);
                sel2  = j;
            end
            if ((csqi1-csqi2>-10) && (csqi1>50))
                channel_sel = sel1;
            else
                channel_sel = sel2;
            end
            if (channel_sel <= length(ecg_ind))
                ecg_num = channel_sel;
                channel_sel = 1;
            else
                bp_num = channel_sel;
                channel_sel = 2;
            end
        end
    else
        if ((ecg_sqi - bp_sqi > -5) && (ecg_sqi > 50))
            channel_sel = 1;
        else
            channel_sel = 2;
        end
    end
    
    if ( channel_sel == 1 )
        
        if ( (i > x_len - 300) && ((isempty(ecg_num))||(ecg_num==0)) )
            break;
        elseif ( (i > x_len - 300) && (abs(rr_new(ecg_num)-rr_final(end))/rr_final(end) > 0.5) )
            break;
        end
        
        % Now find the actual R peak
        low_lim    = peak_idx(end) + rr_new(ecg_num) - 30;
        high_lim   = peak_idx(end) + rr_new(ecg_num) + 30;
        [~,idx]    = max(xdec(low_lim:high_lim,ecg_num));
        new_peak   = low_lim + idx - 1;
        new_rr     = new_peak - peak_idx(end);
        idx_jump   = new_rr;
        ac_len_new = 2*new_rr;% + ceil(new_rr/2);
        rr_final   = [rr_final; new_rr];
        peak_idx   = [peak_idx; new_peak];
        % Find the pulse peak as well
        if (bp_sqi > 50)
            low_lim    = new_peak + 20;
            high_lim   = new_peak + 100;
            [~,idx]    = max(xdec(low_lim:high_lim,bp_num));
            new_pulse  = low_lim + idx - 1;
            pulse_idx  = [pulse_idx ; new_pulse];
            r_to_pulse = [r_to_pulse ; new_pulse-new_peak];
        else
            new_pulse  = new_peak + r_to_pulse(end);
            pulse_idx  = [pulse_idx ; new_pulse];
            r_to_pulse = [r_to_pulse ; r_to_pulse(end)];
        end
        
    else
        
        if ( (i > x_len - 300) && ((isempty(bp_num))||(bp_num==0)) )
            break;
        elseif ( (i > x_len - 300) && (abs(rr_new(bp_num)-rr_final(end))/rr_final(end) > 0.5) )
            break;
        end
        
        if (bp_sqi > 50)
            
            % Now find the actual R peak
            low_lim    = pulse_idx(end) + rr_new(bp_num) - 30;
            high_lim   = pulse_idx(end) + rr_new(bp_num) + 30;
            [~,idx]    = max(xdec(low_lim:high_lim,bp_num));
            new_pulse  = low_lim + idx - 1;
            pulse_idx  = [pulse_idx; new_pulse];
            % Find the R peak as well
            if (ecg_sqi > 50)
                low_lim    = new_pulse - 100;
                high_lim   = new_pulse - 20;
                [~,idx]    = max(xdec(low_lim:high_lim,ecg_num));
                new_peak   = low_lim + idx - 1;
                peak_idx   = [peak_idx ; new_peak];
                r_to_pulse = [r_to_pulse ; new_pulse-new_peak];
            else
                new_peak   = new_pulse - r_to_pulse(end);
                peak_idx   = [peak_idx ; new_peak];
                r_to_pulse = [r_to_pulse ; r_to_pulse(end)];
            end
            new_rr     = peak_idx(end) - peak_idx(end-1);
            idx_jump   = new_rr;
            ac_len_new = 2*new_rr;% + ceil(new_rr/2);
            rr_final   = [rr_final; new_rr];
            
        else
            idx_jump = 80;
            use_prev = 1;
        end
    end
            
    % Update the index and the autocorrelation window size
    i = i + idx_jump;
    if (use_offset == 1)
        i = i - peak_offset;
        use_offset = 0;
    end
    ac_len = ac_len_new;
    tmpl_len = ceil(new_rr/2);
    if (tmpl_len < 70)
        tmpl_len = 70;
        ac_len   = 280;
    end
    
end


%% Write the output file

wrann(record_name,ann_name,peak_idx);


