function challenge(recordName)
%
% challenge(recordName)
%
% Sample entry for the 2014 PhysioNet/CinC Challenge.
% This function should takes three parameters:
%
% recordName
%       String specifying the record name to process
%
% This function has no output arguments, its writes an "recordName.qrs" annotation file
% at the current directory. Do not include file extensions on the recordName string.
%
% The annotations file should contain all the beats that you were able
% to detect in the record.
%
% Dependencies:
%
%       1) This function requires that you have the WFDB
%       App Toolbox installed. For information on how to install the toolbox
%       please see:
%           http://www.physionet.org/physiotools/matlab/wfdb-app-matlab/
%
%         Please note that the Toolbox is supported only on MATLAB 2013a
%         and 2013b. Supported operating systems are 64 bit versions of:
%         Linux, Mac OS X, and Windows.
%
%       2) The CHALLENGE function requires that you have downloaded the challenge
%       data 'set-p' in a subdirectory of the current directory. The subdirectory
%       should be called '/challenge/2014/set-p/' . The 'set-p' dataset can
%       be downloaded from PhysioNet at:
%           http://physionet.org/physiobank/database/challenge/2014/
%          
%         This dataset is used by the generateValidationSet.m script to
%         create the annotations on your traing set that will be used to 
%         verify that your entry works properly in the PhysioNet testing 
%         environment. 
%
% Version 0.9
%
% See also: RDSAMP, RDANN, WRANN, GQRS, ECGPUWAVE, SQRS, WQRS, WABP
%
% Written by Ikaro Silva, December 10, 2013.
% Last Modified: February 11, 2014
%
%
%
% %Example using training data- 
% challenge('100')
%

%This line is required in order to properly load the WFDB Toolbox
%in the Octave testing framework
[~,config]=wfdbloadlib;
if(config.inOctave)
crash_dumps_octave_core(0);
end

%Used for discarding annotations that dont have a minimum number of beats
annName='qrs'; %All competitors are expected to save their annotation as *.qrs


%Get all ECG and blood pressure signals
siginfo=wfdbdesc(recordName);
description=squeeze(struct2cell(siginfo));
description=description(5,:);
ecg_ind=get_index(description,'ECG'); %Call subfunction to get channel indices
Mecg=length(ecg_ind);

%%Users can access the raw samples of the record by running the following
%command:
%[tm,signal]=rdsamp(recordName);
%
%%For more information please see the help in RDSAMP

%Annotated  the first ECG channel with GQRS and default values to generare QRS file
N=[];
N0=[];
threshold=[];
gqrs(recordName,N,N0,ecg_ind(1),threshold,annName);



%Run WABP on the record, which by default will analyze the first ABP, ART, or BP signal
bp_ind=get_index(description,'BP');
if(~isempty(bp_ind))
   wabp(recordName);
   
   %Read annotations back to compare with ECG annotations to find the one
   %with smallest RR variance (our best estimate)
  [rr_abp,~]=ann2rr(recordName,'wabp');
  [rr_ecg,~]=ann2rr(recordName,'qrs');
  [~,best]=min([var(rr_abp) var(rr_ecg)]);
   
   %If ABP has smallest annotation, overwrite the qrs file from GQRS
   if(best==1)
     movefile([recordName '.wabp'],[recordName '.' annName]);
   end
end





%%%%%%%%%%%% Helper Function %%%%%%%%%%%%%%%%%%%%%
function ind=get_index(description,pattern)
M=length(description);
tmp_ind=strfind(description,pattern);
ind=[];
for m=1:M
    if(~isempty(tmp_ind{m}))
        ind(end+1)=m;
    end
end




